// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cHistoryNews;

interface

{$INCLUDE Compiler.inc}

uses SysUtils, Classes, Windows, uTools, cChunkRc;

const MD5HashLen = 16;
type  MD5Hash = array[1..MD5HashLen] of Char;

type
  TNewsHistory = class( TChunkedRecords )
    private
      procedure MID2Hash( const MID: String;
                          out   Hash: MD5Hash );
                          
    public
      function  ContainsMID( const MID: String ) : Boolean;
      function  AddEntryDupes( const MID: String;
                               const GroupHash, ArtNo: LongInt;
                               const PurgeBase: LongInt ): Boolean;
      function  AddEntryFirst( const MID: String;
                               const GroupHash, ArtNo: LongInt;
                               const PurgeBase: LongInt ): Boolean;
      procedure RemoveEntry( const MID: String;
                             const GroupHash, ArtNo: LongInt );
      function  LocateMID( const MID: String;
                           out   Groupname: String;
                           out   ArtNo: Integer ): Boolean;

      procedure Rebuild;
      procedure Purge;

      procedure LoadFromFile; override;

      function  DiskspaceUsed: Int64;

      constructor Create( AFilePath: String; AChunkBits: Integer );
      destructor Destroy; override;
  end;

implementation

uses uConst, uConstVar, cArticle, cArtFiles, uMD5, uCRC32,
     uDateTime, cLogFileHamster, cHamster, cFileStream64;


// ------------------------------------------------------ THistoryRecords -----

const
  HISTORY_FILEBASE = 'Hist';
  HISTORY_FILEEXT  = '.dat';

  HISTKEYLEN_MID       = MD5HashLen;                          // multiple for crossposts
  HISTKEYLEN_MIDGRP    = HISTKEYLEN_MID    + sizeof(LongInt); // multiple after 'import -ih'
  HISTKEYLEN_MIDGRPART = HISTKEYLEN_MIDGRP + sizeof(LongInt); // always unique

type
  PHistoryEntry = ^THistoryEntry;
  THistoryEntry = record
    HashMid  : MD5Hash; //  16
    HashGrp  : LongInt; //   4
    ArtNo    : LongInt; //   4
    PurgeBase: LongInt; //   4
  end;                  // =28 Bytes

procedure TNewsHistory.MID2Hash( const MID: String; out Hash: MD5Hash );
var  s: String;
begin
     s := MD5ofStr( MID );
     System.Move( s[1], Hash[1], MD5HashLen );
end;

function TNewsHistory.ContainsMID( const MID: String ) : Boolean;
var  HE: THistoryEntry;
begin
     FillChar( HE, sizeof(HE), 0 );
     MID2Hash( MID, HE.HashMid );
     Result := ContainsKey( HE, HISTKEYLEN_MID );
end;

function TNewsHistory.LocateMID( const MID: String;
                                 out Groupname: String;
                                 out ArtNo: Integer ): Boolean;
var  HE: THistoryEntry;
     Chunk: Byte;
     GroupHash: LongInt;
     Index, i: Integer;
begin
     Result := False;

     FillChar( HE, sizeof(HE), 0 );
     MID2Hash( MID, HE.HashMid );
     GroupHash := 0;

     Chunk := ChunkOf( HE );
     Enter( Chunk );
     try
        Index := RecKeyIndexOf( Chunk, HISTKEYLEN_MID, HE );
        if Index>=0 then begin
           RecGet( Chunk, Index, HE );
           GroupHash := HE.HashGrp;
           ArtNo     := HE.ArtNo;
        end;
     finally Leave(Chunk) end;

     if Index >= 0 then begin
        Groupname := '';
        for i:=0 to Hamster.Config.Newsgroups.Count-1 do begin
           if StrToCRC32(Hamster.Config.Newsgroups.Name[i])=GroupHash then begin
              Groupname := Hamster.Config.Newsgroups.Name[i];
              break;
           end;
        end;
        if Groupname<>'' then begin
           // calling program has to check further,
           // if Groupname:ArtNo is *still* valid
           Result := True;
        end;
     end;
end;

function TNewsHistory.AddEntryDupes( const MID: String;
                                     const GroupHash, ArtNo: LongInt;
                                     const PurgeBase: LongInt ): Boolean;
var  HE: THistoryEntry;
begin
     FillChar( HE, sizeof(HE), 0 );
     MID2Hash( MID, HE.HashMid );
     HE.HashGrp := GroupHash;
     HE.ArtNo   := ArtNo;

     if ContainsKey( HE, HISTKEYLEN_MIDGRPART ) then begin // unique?
        Result := False;
     end else begin
        Result := True;
        if PurgeBase = 0 then HE.PurgeBase := DateTimeToUnixTime(Now)
                         else HE.PurgeBase := PurgeBase;
        Add( HE );
     end;
end;

function TNewsHistory.AddEntryFirst( const MID: String;
                                     const GroupHash, ArtNo: LongInt;
                                     const PurgeBase: LongInt ): Boolean;
begin
     if ContainsMid( MID ) then begin
        Result := False;
     end else begin
        Result := AddEntryDupes( MID, GroupHash, ArtNo, PurgeBase );
     end;
end;

procedure TNewsHistory.RemoveEntry( const MID: String; const GroupHash, ArtNo: LongInt );
var  HE: THistoryEntry;
begin
     FillChar( HE, sizeof(HE), 0 );
     MID2Hash( MID, HE.HashMid );
     HE.HashGrp := GroupHash;
     HE.ArtNo   := ArtNo;
     RemoveKey( HE, HISTKEYLEN_MIDGRPART );
end;

function TNewsHistory.DiskspaceUsed: Int64;
var  Chunk: Byte;
begin
   Result := 0;
   for Chunk := 0 to CHUNK_MAX do begin
      Result := Result + TFileStream64.FileSize( ChunkFilename(Chunk) );
   end;
end;

procedure TNewsHistory.Purge;
var  Idx : Integer;
     Base: TDateTime;
     Days, KeepDays: LongInt;
     PurgeCount: Integer;
     HE: THistoryEntry;
     Chunk: Byte;
begin
     SaveToFile;
     PurgeCount := 0;

     KeepDays := Hamster.Config.Settings.GetInt( hsPurgeHistoryKeepDays );
     if KeepDays > 0 then begin

        for Chunk := 0 to CHUNK_MAX do begin
           Enter( Chunk );
           try
              Idx := ChunkCount(Chunk) - 1;
              while Idx>=0 do begin
                 RecGet( Chunk, Idx, HE );
                 Base := UnixTimeToDateTime( HE.PurgeBase );
                 Days := Trunc( Now - Base );

                 if Days > KeepDays then begin
                    RemoveKey( HE, HISTKEYLEN_MIDGRPART );
                    inc( PurgeCount );
                 end;

                 dec( Idx );
              end;
           finally Leave(Chunk) end;
        end;

        SaveToFile;

     end;

     Log( LOGID_INFO, Format(
        'Purge History (%sd): %s entries purged.',
        [inttostr(Hamster.Config.Settings.GetInt(hsPurgeHistoryKeepDays)), inttostr(PurgeCount)]));
end;

procedure TNewsHistory.Rebuild;
var  LfdGrp, LfdArt, ArtMin, ArtMax: Integer;
     GrpHdl : LongInt;
     Article: TMess;
     GrpHash: LongInt;
     MessageID: String;
begin
     EnterAllChunks;
     try
        try
           Log( LOGID_SYSTEM, 'Rebuild history ...' );
           Clear;

           Article := TMess.Create;

           for LfdGrp:=0 to Hamster.Config.Newsgroups.Count-1 do begin
              GrpHdl := Hamster.ArticleBase.Open( Hamster.Config.Newsgroups.Name[LfdGrp] );
              if GrpHdl>=0 then begin
                 GrpHash := StrToCRC32( Hamster.ArticleBase.Name[GrpHdl] );
                 ArtMin  := Hamster.ArticleBase.GetInt(GrpHdl,gsLocalMin);
                 ArtMax  := Hamster.ArticleBase.GetInt(GrpHdl,gsLocalMax);

                 if (ArtMin>0) and (ArtMax>0) then begin
                    Log( LOGID_INFO, Format(
                                'Rebuild history: %s, %s articles',
                                [Hamster.Config.Newsgroups.Name[LfdGrp], inttostr(Hamster.ArticleBase.Count[GrpHdl])]));
                    for LfdArt:=ArtMin to ArtMax do begin
                       try
                          Article.FullText := Hamster.ArticleBase.ReadArticle( GrpHdl, LfdArt );
                       except
                          on E:Exception do begin
                             Log( LOGID_ERROR, 'Error in History.R.RA: ' + E.Message );
                             Log( LOGID_ERROR, 'Error at: ' + Hamster.Config.Newsgroups.Name[LfdGrp] + ':' + inttostr(LfdArt) );
                          end;
                       end;
                       try
                          if length(Article.HeaderText) > 0 then begin
                             MessageID := Trim( Article.HeaderValueByNameSL( 'Message-ID:' ) );
                             if MessageID<>'' then begin
                                AddEntryDupes( MessageID, GrpHash, LfdArt,
                                               DateTimeToUnixTime(Article.GetReceivedDT) );
                             end;
                          end;
                       except
                          on E:Exception do Log( LOGID_ERROR, 'Error in History.R.AMOR: ' + E.Message );
                       end;
                    end;
                 end;

                 Hamster.ArticleBase.Close( GrpHdl );

              end else begin
                 Log( LOGID_ERROR, 'Cannot open ' + Hamster.Config.Newsgroups.Name[LfdGrp] );
              end;
           end;

           Article.Free;

           Log( LOGID_INFO, 'Sort history ... ' );
           Sort;

           Log( LOGID_INFO, 'Save rebuilt history ... ' );
           SaveToFile;

           Log( LOGID_SYSTEM, Format(
               'History rebuilt (%s articles).', [inttostr(Count)]) );

        except
           on E:Exception do Log( LOGID_ERROR, 'Error in History.Rebuild: ' + E.Message );
        end;
     finally
        LeaveAllChunks;
     end;
end;

procedure TNewsHistory.LoadFromFile;
var  Chunk      : Byte;
     AutoRebuild: Boolean;
begin
     AutoRebuild := False;
     for Chunk:=0 to CHUNK_MAX do begin
        if not FileExists( ChunkFilename(Chunk) ) then begin
           AutoRebuild := True;
           break;
        end;
     end;

     if AutoRebuild then begin
        Log( LOGID_WARN, 'Rebuilding history-file!' );
        Rebuild;
     end else begin
        inherited LoadFromFile;
     end;
end;

constructor TNewsHistory.Create( AFilePath: String; AChunkBits: Integer );
begin
     if AFilePath='' then AFilePath:='.\';
     AFilePath := ExpandFilename( AFilePath );
     if AFilePath[length(AFilePath)]<>'\' then AFilePath:=AFilePath+'\';

     inherited Create( AFilePath + HISTORY_FILEBASE, HISTORY_FILEEXT,
                       AChunkBits, sizeof(THistoryEntry), True );

     LoadFromFile;
end;

destructor TNewsHistory.Destroy;
begin
     SaveToFile;
     inherited Destroy;
end;

end.

