// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cHistoryMail;

interface

{$INCLUDE Compiler.inc}

uses SysUtils, Classes, Windows, uTools, cChunkRc;

const MD5HashLen = 16;
type  MD5Hash = array[1..MD5HashLen] of Char;

type
  TMailHistory = class( TChunkedRecords )
    private
      procedure UIDL2Hash( const UIDL: String; out Hash: MD5Hash );
    public
      function  ContainsUIDL( const UIDL: String ): Boolean; overload;
      function  ContainsUIDL( const UIDL: String;
                              out PurgeBase: LongInt ): Boolean; overload;
      function  AddUIDL( const UIDL: String; const PurgeBase: LongInt ): Boolean;

      procedure Purge;

      constructor Create( AFilePath: String );
      destructor Destroy; override;
  end;

implementation

uses uConst, uConstVar, uMD5, uDateTime, cLogFileHamster, cHamster;

const
  MAILHISTORY_FILEBASE = 'MHistory';
  MAILHISTORY_FILEEXT  = '.dat';
  MAILHISTKEYLEN_UIDL  = MD5HashLen; // always unique

type
  PMailHistoryEntry = ^TMailHistoryEntry;
  TMailHistoryEntry = record
    HashUIDL : MD5Hash; //  16
    PurgeBase: LongInt; //   4
  end;                  // =28 Bytes

procedure TMailHistory.UIDL2Hash( const UIDL: String; out Hash: MD5Hash );
var  s: String;
begin
     s := MD5ofStr( UIDL );
     System.Move( s[1], Hash[1], MD5HashLen );
end;

function TMailHistory.ContainsUIDL( const UIDL: String ) : Boolean;
var  HE: TMailHistoryEntry;
begin
   FillChar( HE, sizeof(HE), 0 );
   UIDL2Hash( UIDL, HE.HashUIDL );
   Result := ContainsKey( HE, MAILHISTKEYLEN_UIDL );
end;

function TMailHistory.ContainsUIDL( const UIDL: String;
                                    out PurgeBase: LongInt ): Boolean;
var  HE   : TMailHistoryEntry;
     Chunk: Byte;
     Index: Integer;
begin
   Result := False;
   
   FillChar( HE, sizeof(HE), 0 );
   UIDL2Hash( UIDL, HE.HashUIDL );

   Chunk := ChunkOf( HE );
   Enter( Chunk );
   try
      Index := RecKeyIndexOf( Chunk, MAILHISTKEYLEN_UIDL, HE );
      if Index >= 0 then begin
         Result := True;
         RecGet( Chunk, Index, HE );
         PurgeBase := HE.PurgeBase;
      end;
   finally Leave(Chunk) end;
end;

function TMailHistory.AddUIDL( const UIDL: String; const PurgeBase: LongInt ): Boolean;
var  HE: TMailHistoryEntry;
begin
     FillChar( HE, sizeof(HE), 0 );
     UIDL2Hash( UIDL, HE.HashUIDL );

     if ContainsKey( HE, MAILHISTKEYLEN_UIDL ) then begin
        Result := False;
     end else begin
        Result := True;
        if PurgeBase = 0 then HE.PurgeBase := DateTimeToUnixTime(Now)
                         else HE.PurgeBase := PurgeBase;
        Add( HE );
     end;
end;

procedure TMailHistory.Purge;
var  Idx : Integer;
     Base: TDateTime;
     Days: LongInt;
     PurgeCount: Integer;
     HE: TMailHistoryEntry;
     Chunk: Byte;
begin
     SaveToFile;
     PurgeCount := 0;

     for Chunk:=0 to CHUNK_MAX do begin
        Enter( Chunk );
        try

           Idx := ChunkCount(Chunk) - 1;
           while Idx>=0 do begin
              RecGet( Chunk, Idx, HE );
              Base := UnixTimeToDateTime( HE.PurgeBase );
              Days := Trunc( Now - Base );

              if ( Days > Hamster.Config.Settings.GetInt(hsPurgeMHistoryKeepDays) ) and
                 ( Hamster.Config.Settings.GetInt(hsPurgeMHistoryKeepDays) > 0 ) then begin
                 RemoveKey( HE, MAILHISTKEYLEN_UIDL );
                 inc( PurgeCount );
              end;

              dec( Idx );
           end;


        finally Leave(Chunk) end;
     end;

     SaveToFile;

     Log( LOGID_INFO, Format(
              'Purge Mail-History (%sd): %s entries purged.',
              [inttostr(Hamster.Config.Settings.GetInt(hsPurgeMHistoryKeepDays)), inttostr(PurgeCount)]))
end;

constructor TMailHistory.Create( AFilePath: String );
begin
     if AFilePath='' then AFilePath:='.\';
     AFilePath := ExpandFilename( AFilePath );
     if AFilePath[length(AFilePath)]<>'\' then AFilePath:=AFilePath+'\';

     inherited Create( AFilePath + MAILHISTORY_FILEBASE, MAILHISTORY_FILEEXT,
                       0, sizeof(TMailHistoryEntry), True );

     LoadFromFile;
end;

destructor TMailHistory.Destroy;
begin
     SaveToFile;
     inherited Destroy;
end;

end.
