// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cHamster;

interface

uses SysUtils, Classes, Windows, SyncObjs, uConst, uType, cServerBase,
     cLiveServer, cLiveQueue, cHscFiles, cRasDialer, cNewsJobs, cArtFiles,
     cPasswords, cAccounts, cHamsterCfg, cHistoryNews, cHistoryMail,
     cSettings, cLiveMsg, tBase, cHscActions, cIPAccess, cMailDispatcher,
     cSmtpRouter, cMailLists, cMailTrap, cScheduler, uTools;

type
   TLiveClientQueueViewerThread = class( TLiveClientQueueHandlerThread )
      protected
         FLiveServer: TLiveServer;
         FLogAll    : Boolean;
         procedure HandleMessage( var LiveMsg: TLiveMsg ); override;
      public
         property LogAll: Boolean read FLogAll write FLogAll;
         constructor Create( ACreateSuspended: Boolean;
                             ALiveClient     : TLiveClient;
                             ALiveServer     : TLiveServer );
   end;

   THamster = class
      private
         FTimer            : Cardinal;
         FServers          : array[ TServerTypes ] of TServerBase;
         FCritSectServers  : TCriticalSection;

         FLiveServer       : TLiveServer;
         FLiveClientLocal  : TLiveClient;
         FLiveViewerThread : TLiveClientQueueViewerThread;
         FNewCheckPointFunc: TNewCheckPointFunc;
         FActiveThreads    : TActiveThreads;

         FHscFiles      : THscFiles;
         FHscActions    : THscActions;
         FRasDialer     : TRasDialer;
         FNewsJobs      : TNewsJobs;
         FArticleBase   : TArticleBase;
         FIPAccess      : TIPAccess;
         FPasswords     : TPasswords;
         FAccounts      : TAccounts;
         FMailLists     : TMailLists;
         FConfig        : TConfigHamster;
         FSmtpRouter    : TSmtpRouterFactory;
         FNewsHistory   : TNewsHistory;
         FMailHistory   : TMailHistory;
         FMailDispatcher: TMailDispatcher;
         FMailTrap      : TMailTrap;
         FScheduler     : TGlobalScheduler;
         FScriptDnsCache: TKeyValueCache;

         function GetWantsToTerminate: Boolean;
         function GetSSLAvailable: Boolean;

         procedure DoStartup;
         procedure DoShutdown;

      public
         property WantsToTerminate: Boolean read GetWantsToTerminate;
         property SSLAvailable    : Boolean read GetSSLAvailable;

         property LiveServer    : TLiveServer        read FLiveServer;
         property HscFiles      : THscFiles          read FHscFiles;
         property HscActions    : THscActions        read FHscActions;
         property RasDialer     : TRasDialer         read FRasDialer;
         property NewsJobs      : TNewsJobs          read FNewsJobs;
         property ArticleBase   : TArticleBase       read FArticleBase;
         property IPAccess      : TIPAccess          read FIPAccess;
         property Passwords     : TPasswords         read FPasswords;
         property Accounts      : TAccounts          read FAccounts;
         property MailLists     : TMailLists         read FMailLists;
         property Config        : TConfigHamster     read FConfig;
         property NewsHistory   : TNewsHistory       read FNewsHistory;
         property MailHistory   : TMailHistory       read FMailHistory;
         property MailDispatcher: TMailDispatcher    read FMailDispatcher;
         property MailTrap      : TMailTrap          read FMailTrap;
         property ActiveThreads : TActiveThreads     read FActiveThreads;
         property SmtpRouter    : TSmtpRouterFactory read FSmtpRouter;
         property Scheduler     : TGlobalScheduler   read FScheduler;
         property ScriptDnsCache: TKeyValueCache     read FScriptDnsCache;

         function ServerControl( ServerType : TServerTypes;
                                 ControlCode: TServerControls ): Integer;
         function ServerClients: String;

         function CheckConfiguration: Boolean;

         constructor Create( ANewCheckPointFunc: TNewCheckPointFunc );
         destructor Destroy; override;
   end;

var
   Hamster: THamster = nil;

implementation

uses uConstVar, uVar, cSyncObjects, cLogFileHamster, tMaintenance,
     uHamTools, cServerNNTP, cServerPOP3, cServerSMTP, cServerRECO,
     IdSSLOpenSSLHeaders, tUserTasks, cHscAction, cPCRE;

// --------------------------------------------------------- HamsterTimer -----

const
   HamsterTimerInterval = 2000;

var
   LastDaily : TDateTime = 0;
   DailyDelay: Integer = ( 2 * 60 * 1000 ) div HamsterTimerInterval;
   DoTestOnce: Boolean = True;

procedure HamsterTimerProc( hwnd: HWND; uMsg: UINT;
                            idEvent: UINT; dwTime: DWORD ); stdcall;

   procedure CountOutboxes;
   var  SR : TSearchRec;
   begin
      CounterEnter;
      try
         CounterOutboxChk := 0;
         CounterOutboxM   := 0;
         CounterOutboxN   := 0;

         if FindFirst( AppSettings.GetStr( asPathMailOut ) + '*.msg',
                       faAnyFile, SR ) = 0 then begin
            repeat inc( CounterOutboxM ) until FindNext( SR ) <> 0;
            SysUtils.FindClose( SR );
         end;

         if FindFirst( AppSettings.GetStr( asPathNewsOut ) + '*.msg',
                       faAnyFile, SR ) = 0 then begin
            repeat inc( CounterOutboxN ) until FindNext( SR ) <> 0;
            SysUtils.FindClose( SR );
         end;

      finally CounterLeave end;
   end;

   procedure BroadcastUpdate( REQ, INF: Integer );
   var  Reply: TLiveMsg;
   begin
      Reply := Hamster.LiveServer.ReplyFor(
                  TLiveMsg.Create( REQ, '' )
               );
      if Assigned( Reply ) then try
         Hamster.LiveServer.BroadCast( INF, Reply.MsgData );
      finally
         Reply.Free;
      end;
   end;

begin
   try
      if Assigned( Hamster ) then begin

         // start daily maintenance once per day if nothing else is running
         if Hamster.ActiveThreads.CountActiveTasks = 0 then begin // idle

            if DailyDelay > 0 then begin
               dec( DailyDelay ); // delay execution at startup
            end else begin
               if trunc(LastDaily) <> trunc(Now) then begin // day changed
                  // daily maintenance
                  LastDaily := Now;
                  with TThreadDailyMaintenance.Create do Resume;
               end else begin
                  // rebuild newsgroup lists if new groups were loaded
                  if GlobalListMarker( glTEST ) then begin
                     with TThreadRebuildGlobalLists.Create( tftFreeOnTerminate ) do Resume;
                  end;
               end;

               if DoTestOnce then begin
                  DoTestOnce := False;
                  //  with TUserTaskThread.Create( 'Test' ) do Resume;
               end;
            end

         end;

         // trigger scheduler
         if Assigned( Hamster.Scheduler ) then Hamster.Scheduler.Trigger;

         // refresh outbox counters
         if CounterOutBoxChk > 0 then CountOutboxes;

         // send Live info
         if Assigned( Hamster.LiveServer ) then begin

            // broadcast new counter values
            if CounterChanged then BroadcastUpdate( LMREQ_COUNTERS_LIST,
                                                    LMINF_COUNTERS_LIST );

            // broadcast task list
            if TasksChanged   then BroadcastUpdate( LMREQ_TASKS_LIST,
                                                    LMINF_TASKS_LIST );

            // broadcast client list
            if ClientsChanged then BroadcastUpdate( LMREQ_LOCALSERVER_CLIENTS,
                                                    LMINF_LOCALSERVER_CLIENTS );
         end;

      end;

   except end;
end;


// ----------------------------------------- TLiveClientQueueViewerThread -----

procedure TLiveClientQueueViewerThread.HandleMessage( var LiveMsg: TLiveMsg );
var  VLine  : String;
     VType: Integer;
begin
   if LiveMsg.MsgType <> LMINF_LOGFILE_ADD then exit;

   VType := strtointdef( '$' + copy(LiveMsg.MsgData,1,4), LOGID_INFO );

   if FLogAll or (VType=LOGID_ERROR) or (VType=LOGID_WARN)
              or (VType=LOGID_SYSTEM) then begin
      VLine := copy( LiveMsg.MsgData, 6, MaxInt );
      writeln( VLine );
   end;
end;

constructor TLiveClientQueueViewerThread.Create( ACreateSuspended: Boolean;
                                                 ALiveClient     : TLiveClient;
                                                 ALiveServer     : TLiveServer );
begin
   inherited Create( True, ALiveClient );
   FLiveServer := ALiveServer;
   FLogAll     := True;
   Priority    := tpLower;
   if not ACreateSuspended then Resume;
end;


// ------------------------------------------------------------- THamster -----

function THamster.ServerControl( ServerType : TServerTypes;
                                 ControlCode: TServerControls ): Integer;
var   StateChanged: Boolean;

   function SrvIsActive( ServerType: TServerTypes ): Boolean;
   begin
      Result := Assigned( FServers[ServerType] )
                and FServers[ServerType].Active;
   end;

   function SrvStart( ServerType: TServerTypes ): Boolean;
   begin
      Result := True;
      if SrvIsActive(ServerType) then exit;

      Result := False;

      try
         Log( LOGID_INFO, 'Starting ' + ServerNames[ServerType] + '-server ...' );
         StateChanged := True;
         IPAccess.WantReload := True;
         case ServerType of
            stNNTP: FServers[ ServerType ] := TServerNNTP.Create(
                       stNNTP,
                       Config.Settings.GetStr(hsLocalNntpBind),
                       Config.Settings.GetInt(hsLocalNntpPort),
                       Config.Settings.GetInt(hsLocalNntpClients),
                       Config.Settings.GetInt(hsLocalNntpSameCli),
                       IPACC_SCOPE_NNTP, TServerClientNNTP );
            stPOP3: FServers[ ServerType ] := TServerPOP3.Create(
                       stPOP3,
                       Config.Settings.GetStr(hsLocalPop3Bind),
                       Config.Settings.GetInt(hsLocalPop3Port),
                       Config.Settings.GetInt(hsLocalPop3Clients),
                       Config.Settings.GetInt(hsLocalPop3SameCli),
                       IPACC_SCOPE_POP3, TServerClientPOP3 );
            stSMTP: FServers[ ServerType ] := TServerSMTP.Create(
                       stSMTP,
                       Config.Settings.GetStr(hsLocalSmtpBind),
                       Config.Settings.GetInt(hsLocalSmtpPort),
                       Config.Settings.GetInt(hsLocalSmtpClients),
                       Config.Settings.GetInt(hsLocalSmtpSameCli),
                       IPACC_SCOPE_SMTP, TServerClientSMTP );
            stRECO: FServers[ ServerType ] := TServerRECO.Create(
                       stRECO,
                       Config.Settings.GetStr(hsLocalRecoBind),
                       Config.Settings.GetInt(hsLocalRecoPort),
                       Config.Settings.GetInt(hsLocalRecoClients),
                       Config.Settings.GetInt(hsLocalRecoSameCli),
                       IPACC_SCOPE_RECO, TServerClientRECO );
         end;
         FServers[ ServerType ].Active := True;
         Result := True;
         Log( LOGID_SYSTEM, ServerNames[ServerType] + '-server started.' );

      except
         on E:Exception do
            Log( LOGID_ERROR, 'Starting ' + ServerNames[ServerType]
                            + '-server failed: ' + E.Message );
      end;
   end;

   function SrvStop( ServerType: TServerTypes ): Boolean;
   var  Srv: TServerBase;
   begin
      Result := True;
      if not SrvIsActive(ServerType) then exit;

      try
         Log( LOGID_INFO, 'Stopping ' + ServerNames[ServerType] + '-server ...' );
         StateChanged := True;
         Srv := FServers[ ServerType ];
         FServers[ ServerType ] := nil;
         Srv.Free;
         Log( LOGID_SYSTEM, ServerNames[ServerType] + '-server stopped.' );

      except
         on E:Exception do
            Log( LOGID_ERROR, 'Stopping ' + ServerNames[ServerType]
                            + '-server failed: ' + E.Message );
      end;
   end;

   function SrvRestart( ServerType: TServerTypes ): Boolean;
   begin
      Result := True;
      if not SrvIsActive(ServerType) then exit;
      Result := SrvStop(ServerType) and SrvStart(ServerType);
   end;

   function SrvClientCount( ServerType: TServerTypes ): Integer;
   begin
      Result := 0;
      if not SrvIsActive(ServerType) then exit;
      Result := FServers[ ServerType ].ActiveConnections;
   end;

var  Reply: TLiveMsg;
begin
   StateChanged := False;

   FCritSectServers.Enter;
   try
      case ControlCode of
         scSTART:       Result := iif( SrvStart   (ServerType), iTRUE,iFALSE );
         scSTOP:        Result := iif( SrvStop    (ServerType), iTRUE,iFALSE );
         scRESTART:     Result := iif( SrvRestart (ServerType), iTRUE,iFALSE );
         scIsActive:    Result := iif( SrvIsActive(ServerType), iTRUE,iFALSE );
         scClientCount: Result := SrvClientCount(ServerType);
         else           Result := iFALSE;
      end;
   finally
      FCritSectServers.Leave;
   end;

   if StateChanged then begin
      Reply := LiveServer.ReplyFor(
                  TLiveMsg.Create( LMREQ_LOCALSERVER_CONTROL,
                                   '*' + CRLF + inttostr(ord(scIsActive)) )
               );
      if Assigned( Reply ) then try
         LiveServer.BroadCast( LMINF_LOCALSERVER_STATES, Reply.MsgData );
      finally
         Reply.Free;
      end;
   end;
end;

function THamster.ServerClients: String;
var  SL: TStringList;
     st: TServerTypes;
     i : Integer;
begin
   Result := '';
   SL := TStringList.Create;
   try
      for st := stNNTP to stRECO do begin
         if Assigned( FServers[ st ] ) then begin
            FServers[ st ].ActiveConnectionList( SL );
            for i := 0 to SL.Count-1 do begin
               Result := Result + inttostr(ord(st)) + TAB + SL[i] + CRLF;
            end;
         end;
      end;
   finally SL.Free end;
end;

function THamster.GetWantsToTerminate: Boolean;
begin
   Result := ( AppTerminateEvent.WaitFor( 0 ) = wrSignaled );
end;

function THamster.GetSSLAvailable: Boolean;
begin
   try
      Result := IdSSLOpenSSLHeaders.Load;
   except
      Result := False;
   end;
end;

function THamster.CheckConfiguration: Boolean;
var AllSuccessful: Boolean;

   procedure Fail( const reason: String );
   begin
      Log( LOGID_WARN, 'Configuration error: ' + reason );
      AllSuccessful := False;
   end;

   procedure Requires( const opt1, opt2: String );
   begin
      Fail( '"' + opt1 + '" requires "' + opt2 + '"' );
   end;
   
var  SL: TStringList;
     i, ID, GrpHdl: Integer;
     FQDN: String;
     HavePeers, HaveFeedOnly, GenerateMid, LocalInject: Boolean;
begin
   Log( LOGID_INFO, 'Checking configuration ...' );
   AllSuccessful := True;

   SL := TStringList.Create;
   try

      Config.BeginRead;
      try
         FQDN        := Config.Settings.GetStr( hsFQDNforMID );
         GenerateMid := Config.Settings.GetBoo( hsGenerateNewsMID );
         LocalInject := Config.Settings.GetBoo( hsLocalNntpLocalInjection );
      finally Config.EndRead end;

      if GenerateMid and (FQDN = '') then
         Requires( 'Generate Message-ID', 'FQDN' );

      if LocalInject and not GenerateMid then
         Requires( 'Local injection', 'Generate Message-ID' );

      HavePeers := False;
      Accounts.EnumUsernames( SL );
      for i := 0 to SL.Count - 1 do begin
         ID := Integer( SL.Objects[i] );
         if Accounts.Value[ ID, apNewsPeer ] = '1' then begin
            HavePeers := True;
            break;
         end;
      end;
      if HavePeers then begin
         if FQDN = '' then
            Requires( 'User may use feed commands', 'FQDN' );
         if not GenerateMid then
            Requires( 'User may use feed commands', 'Generate Message-ID' );
         if not LocalInject then
            Requires( 'User may use feed commands', 'Local injection' );
      end;

      HaveFeedOnly := False;
      for i := 0 to Config.Newsgroups.Count - 1 do begin
         GrpHdl := ArticleBase.Open( Config.Newsgroups.Name[i] );
         if GrpHdl >= 0 then try
            if ArticleBase.GetBoo( GrpHdl, gsFeedOnly ) then begin
               HaveFeedOnly := True;
               break;
            end;
         finally ArticleBase.Close( GrpHdl ) end;
      end;

      if HaveFeedOnly then begin
         if FQDN = '' then
            Requires( 'Group is feeded', 'FQDN' );
         if not GenerateMid then
            Requires( 'Group is feeded', 'Generate Message-ID' );
         if not LocalInject then
            Requires( 'Group is feeded', 'Local injection' );
      end;


   finally SL.Free end;

   Result := AllSuccessful;   
end;

procedure THamster.DoStartup;
begin
   if Assigned( FNewCheckPointFunc ) then FNewCheckPointFunc( 5000 );
   Log( LOGID_SYSTEM, GetMyStringFileInfo( 'ProductName', 'Hamster' ) + ' '
                    + GetMyVersionInfo + ' started' );

   // load configuration
   if Assigned( FNewCheckPointFunc ) then FNewCheckPointFunc( 30000 );
   Log( LOGID_INFO, 'Loading configuration ...' );

   Log( LOGID_INFO, 'Init ... HscFiles' );
   FHscFiles    := THscFiles.Create( AppSettings.GetStr(asPathScripts),
                                     AppSettings.GetStr(asPathModules) );
   Log( LOGID_INFO, 'Init ... HscActions' );
   FHscActions  := THscActions.Create( AppSettings.GetStr(asPathBase) );
   Log( LOGID_INFO, 'Init ... IPAccess' );
   FIPAccess    := TIPAccess.Create;
   Log( LOGID_INFO, 'Init ... Passwords' );
   FPasswords   := TPasswords.Create;
   Log( LOGID_INFO, 'Init ... Accounts' );
   FAccounts    := TAccounts.Create;
   Log( LOGID_INFO, 'Init ... NewsJobs' );
   FNewsJobs    := TNewsJobs.Create;
   Log( LOGID_INFO, 'Init ... ArticleBase' );
   FArticleBase := TArticleBase.Create;
   Log( LOGID_INFO, 'Init ... RasDialer' );
   FRasDialer   := TRasDialer.Create( FPasswords );
   Log( LOGID_INFO, 'Init ... MailLists' );
   FMailLists   := TMailLists.Create( AppSettings.GetStr(asPathBase) );
   Log( LOGID_INFO, 'Init ... SmtpRouter' );
   FSmtpRouter  := TSmtpRouterFactory.Create( AppSettings.GetStr(asPathBase)
                                              + CFGFILE_SMTPROUTER );
   Log( LOGID_INFO, 'Init ... Config' );
   FConfig      := TConfigHamster.Create( FNewsJobs, FArticleBase, FPasswords );
   Log( LOGID_INFO, 'Init ... CheckSettings' );
   SEM_LIMITTASKS.Count := Config.Settings.GetInt(hsTasksLimit);
   Log( LOGID_INFO, 'Init ... Scheduler' );
   FScheduler   := TGlobalScheduler.Create( AppSettings.GetStr(asPathBase)
                                            + CFGFILE_SCHEDULER, False );
   Log( LOGID_INFO, 'Init ... NewsHistory' );
   FNewsHistory := TNewsHistory.Create( AppSettings.GetStr(asPathGroups),
                                        Config.Settings.GetInt(hsHistoryChunkBits) );
   Log( LOGID_INFO, 'Init ... MailHistory' );
   FMailHistory := TMailHistory.Create( AppSettings.GetStr(asPathMails) );
   Log( LOGID_INFO, 'Init ... MailDispatcher' );
   FMailDispatcher := TMailDispatcher.Create;
   Log( LOGID_INFO, 'Init ... MailTrap' );
   FMailTrap    := TMailTrap.Create;
   Log( LOGID_INFO, 'Init ... Caches' );
   FScriptDnsCache := TKeyValueCache.Create( AppSettings.GetStr(asPathLogs)
                                           + CACHEFILE_SCRIPTDNSCACHE, 10000 );

   Log( LOGID_INFO, 'Check SSL ... '
                  + iif( SSLAvailable, 'Installed', 'Not installed' ) );

   if not FileExists( AppSettings.GetStr(asPathServer) + SRVFILE_ALLDESCS ) then begin
      if Assigned( FNewCheckPointFunc ) then FNewCheckPointFunc( 30000 );
      GlobalListMarker( glOPEN );
      with TThreadRebuildGlobalLists.Create( tftFreeByCode ) do try
         Resume;
         WaitFor;
      finally Free end;
   end;

   Log( LOGID_INFO, 'NewsServer.Count='  + inttostr(Config.NntpServers.Count  ) );
   Log( LOGID_INFO, 'Pop3Server.Count='  + inttostr(Config.Pop3Servers.Count  ) );
   Log( LOGID_INFO, 'SmtpServer.Count='  + inttostr(Config.SmtpServers.Count  ) );
   Log( LOGID_INFO, 'Newsgroups.Count='  + inttostr(Config.Newsgroups.Count   ) );
   Log( LOGID_INFO, 'Newspulls.Count='   + inttostr(Config.NewsPulls.Count    ) );
   Log( LOGID_INFO, 'Articles.Count='    + inttostr(Config.InitialArticleCount) );
   Log( LOGID_INFO, 'History.Count='     + inttostr(NewsHistory.Count) );
   Log( LOGID_INFO, 'MailHistory.Count=' + inttostr(MailHistory.Count) );

   // auto-start servers
   with Config.Settings do begin
      if Assigned( FNewCheckPointFunc ) then FNewCheckPointFunc( 5000 );
      if GetBoo( hsLocalNntpAutostart ) then ServerControl( stNNTP, scSTART );

      if Assigned( FNewCheckPointFunc ) then FNewCheckPointFunc( 5000 );
      if GetBoo( hsLocalPop3Autostart ) then ServerControl( stPOP3, scSTART );

      if Assigned( FNewCheckPointFunc ) then FNewCheckPointFunc( 5000 );
      if GetBoo( hsLocalSmtpAutostart ) then ServerControl( stSMTP, scSTART );

      if Assigned( FNewCheckPointFunc ) then FNewCheckPointFunc( 5000 );
      if GetBoo( hsLocalRecoAutostart ) then ServerControl( stRECO, scSTART );
   end;

   // start startup script
   if Assigned( FNewCheckPointFunc ) then FNewCheckPointFunc( 5000 );
   FHscActions.Execute( actStartup, '' );
end;

procedure THamster.DoShutdown;
begin
   // request stop for all active tasks
   FScheduler.Active := False;
   FActiveThreads.StopAll;

   // start shutdown script
   if Assigned( FNewCheckPointFunc ) then FNewCheckPointFunc( 5000 );
   FHscActions.Execute( actShutdown, '' );

   // shut down servers - throwing out any clients left
   if Assigned( FNewCheckPointFunc ) then FNewCheckPointFunc( 5000 );
   ServerControl( stNNTP, scSTOP );

   if Assigned( FNewCheckPointFunc ) then FNewCheckPointFunc( 5000 );
   ServerControl( stPOP3, scSTOP );

   if Assigned( FNewCheckPointFunc ) then FNewCheckPointFunc( 5000 );
   ServerControl( stSMTP, scSTOP );

   if Assigned( FNewCheckPointFunc ) then FNewCheckPointFunc( 5000 );
   ServerControl( stRECO, scSTOP );

   // wait until all active tasks are terminated
   if Assigned( FNewCheckPointFunc ) then FNewCheckPointFunc( 10000 );
   while FActiveThreads.CountAll > 0 do begin
      FActiveThreads.StopAll;
      AppTerminateEvent.SetEvent;
      Sleep( 250 );
   end;

   // terminate helper/configuration objects
   if Assigned( FNewCheckPointFunc ) then FNewCheckPointFunc( 5000 );
   if Assigned( FScriptDnsCache ) then FScriptDnsCache.Free;
   if Assigned( FMailDispatcher ) then FMailDispatcher.Free;
   if Assigned( FNewsHistory ) then FNewsHistory.Free;
   if Assigned( FMailHistory ) then FMailHistory.Free;
   if Assigned( FPasswords   ) then FPasswords.Free;
   if Assigned( FArticleBase ) then FArticleBase.Free;
   if Assigned( FNewsJobs    ) then FNewsJobs.Free;
   if Assigned( FRasDialer   ) then FRasDialer.Free;
   if Assigned( FMailTrap    ) then FMailTrap.Free;
   if Assigned( FScheduler   ) then FScheduler.Free;
   if Assigned( FMailLists   ) then FMailLists.Free;
   if Assigned( FAccounts    ) then FAccounts.Free;
   if Assigned( FIPAccess    ) then FIPAccess.Free;
   if Assigned( FHscFiles    ) then FHscFiles.Free;
   if Assigned( FHscActions  ) then FHscActions.Free;
   if Assigned( FSmtpRouter  ) then FSmtpRouter.Free;

   if Assigned( FConfig      ) then FConfig.Free;
end;

constructor THamster.Create( ANewCheckPointFunc: TNewCheckPointFunc );
var  st: TServerTypes;
begin
   inherited Create;

   Hamster := Self;

   FNewCheckPointFunc := ANewCheckPointFunc;

   FActiveThreads := TActiveThreads.Create;

   FCritSectServers := TCriticalSection.Create;
   for st := stNNTP to stRECO do FServers[ st ] := nil;

   SEM_LIMITTASKS := TSemaphoreObj.Create( 10 );

   CS_MAINTENANCE      := TCriticalSection.Create;
   CS_LOCK_MAILBOX_ALL := TCriticalSection.Create;
   CS_LOCK_MAILOUT_USE := TCriticalSection.Create;
   CS_LOCK_MAILOUT_ADD := TCriticalSection.Create;
   CS_LOCK_NEWSOUT     := TCriticalSection.Create;
   CS_LOCK_NEWSFEED    := TCriticalSection.Create;

   FLiveServer := TLiveServer.Create;
   FLiveClientLocal  := TLiveClient.Create( FLiveServer, LogFile.ViewMax );
   FLiveViewerThread := TLiveClientQueueViewerThread.Create( False, FLiveClientLocal, FLiveServer );

   DoStartup;
   FLiveViewerThread.LogAll := False;
   FScheduler.Active := True;

   FTimer := SetTimer( 0, 0, HamsterTimerInterval, @HamsterTimerProc );
end;

destructor THamster.Destroy;
begin
   // Do NOT use FreeAndNil() here, as it sets variable to nil BEFORE its
   // object is freed! Some dependent objects will still use their parent
   // while freeing, e. g. TLiveClients call LiveServer.ClientDisconnect.

   FLiveViewerThread.LogAll := True;
   DoShutdown;

   KillTimer( 0, FTimer );

   FLiveViewerThread.Terminate;
   FLiveClientLocal.ReceiveMsg( nil ); // wake up
   FLiveViewerThread.WaitFor;
   FLiveViewerThread.Free;
   FLiveClientLocal.Free;

   if Assigned( FLiveServer ) then FLiveServer.Free;

   FCritSectServers.Free;

   FActiveThreads.Free;

   inherited Destroy;
end;

//-----------------------------------------------------------------------------

initialization

finalization
   if Assigned( Hamster ) then begin
      try Hamster.Free except end;
      Hamster := nil;
   end;

end.
