// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cFiltersNews;

interface

{$INCLUDE Compiler.inc}

uses Classes, cArticle, cPCRE, uTools, cFiltersBase;

const
  XOVER_Number     =  0;
  XOVER_Subject    =  1;
  XOVER_From       =  2;
  XOVER_Date       =  3;
  XOVER_MessageID  =  4;
  XOVER_References =  5;
  XOVER_Bytes      =  6;
  XOVER_Lines      =  7;
  XOVER_Xref       =  8;
  XOVER_Xpost      =  9;
  XOVER_Age        = 10;
  XOVER_MAX        = 10;
  XOVER_INVALID    = -1;

  XOVERLIST_MARKFULL    = $40;
  XOVERLIST_MARKUNKNOWN = $80;
  XOVERLIST_DEFAULT     : array[0..8] of Integer = (
                             // '08,01,02,03,04,05,06,07,48'
                             8, // count
                             XOVER_Subject, XOVER_From, XOVER_Date,
                             XOVER_MessageID, XOVER_References,
                             XOVER_Bytes, XOVER_Lines,
                             XOVER_Xref or XOVERLIST_MARKFULL
                          );

  SCOREGROUP_FEED = '$FEED$';
  SCOREGROUP_POST = '$POST$';

type
  PXOverRec = ^TXOverRec;
  TXOverRec = record
    XFields: array[XOVER_Number..XOVER_MAX] of String;
  end;

  TFilterPatternNews = class( TFilterPatternBase )
      IsSameField: Boolean;
      XOverField : Integer;
      XFiltField : String;
  end;

  TFilterGetValueFunc = function ( const BaseData    : Pointer;
                                   const XOverField  : Integer;
                                   const XFiltField  : String;
                                   const DoMimeDecode: Boolean ): String;

  TFilterLineNews = class( TFilterLineBase )
    public
      IsFinal     : Boolean; // '='?
      ScoreValue  : Integer;
      IsBeforeLoad: Boolean;
      XOverField  : Integer;
      XFiltField: String;
      function  MatchesData( const RE: TPCRE;
                             const Data: Pointer;
                             const DataGetValue: TFilterGetValueFunc ): Boolean;
      function  SetFilterLine( FilterLine: String ): Boolean; override;
      function  AsString: String; override;
  end;

  TFiltersNews = class( TFiltersBase )
    private
      FHasXOverLines, FHasXFiltLines: Boolean;
    public
      property  HasXOverLines: Boolean read FHasXOverLines;
      property  HasXFiltLines: Boolean read FHasXFiltLines;
      procedure SelectSections( const SectionIdentifier: String ); override;
      function  LinesAdd( const LineText: String ): Integer; override;
      function  IsFilterLine( const Line: String ): Boolean; override;
      function  ScoreBeforeLoad( const XOverRec: TXOverRec;
                                 const MatchLog: PString ): Integer;
      function  ScoreAfterLoad ( const Article : TMess;
                                 const MatchLog: PString ): Integer;
      function  ScoreArticle   ( const Article : TMess ): Integer;
  end;

procedure XOverToXOverRec( var   XOverRec   : TXOverRec;
                           const ParsedXOver: TParser;
                           const XOverList  : array of integer );

procedure ArticleToXOverRec( const Article : TMess;
                             var   XOverRec: TXOverRec );

function GetXFiltVal( const BaseData    : Pointer;
                      const NotUsedHere : Integer;
                      const XFiltField  : String;
                      const DoMimeDecode: Boolean ): String;

implementation

uses SysUtils, uConst, uEncoding, uDateTime, cLogFileHamster;

procedure XOverToXOverRec( var   XOverRec: TXOverRec;
                           const ParsedXOver : TParser;
                           const XOverList   : array of integer );
var  PartNo, PartID: Integer;
     PartS         : String;
     IsFull        : Boolean;
     s             : String;
     i, k          : Integer;
begin
     // ParsedXOver (default):
     //    0:No. 1:Subject 2:From 3:Date 4:Message-ID 5:References 6:Bytes 7:Lines [8:Xref]
     // XOverList:
     //    [0]=Anzahl, [1..Anzahl]=Feldnummer [or XOVERLIST_MARKFULL] [or XOVERLIST_MARKUNKNOWN]

     for i:=XOVER_Number to XOVER_MAX do XOverRec.XFields[i]:='';
     XOverRec.XFields[XOVER_Number] := ParsedXOver.sPart( 0, '' );

     for PartNo:=1 to XOverList[0] do begin

        PartID := XOverList[PartNo];
        if (PartID and XOVERLIST_MARKFULL)<>0 then begin
           IsFull := True;
           PartID := PartID and not(XOVERLIST_MARKFULL);
        end else begin
           IsFull := False;
        end;

        if (PartID<0) or (PartID>XOVER_MAX) then PartID:=XOVERLIST_MARKUNKNOWN;

        if (PartID and XOVERLIST_MARKUNKNOWN)=0 then begin

           PartS := ParsedXOver.sPart( PartID, '' );

           if PartS<>'' then begin
              XOverRec.XFields[PartID] := PartS;

              if PartID=XOVER_Xref then begin // Xref->Xpost
                 s := LowerCase( PartS );
                 if IsFull and (copy(s,1,6)='xref: ') then System.Delete(s,1,6);
                 k := 0;
                 for i:=1 to length(s) do if s[i]=':' then inc(k);
                 if k=0 then k:=1;
                 XOverRec.XFields[XOVER_Xpost] := inttostr(k);
              end;

              if PartID=XOVER_Date then begin // Date->Age
                 try
                    k := Trunc( NowGMT - RfcDateTimeToDateTimeGMT( PartS, NowGMT ) );
                    XOverRec.XFields[XOVER_Age] := inttostr(k);
                 except
                    XOverRec.XFields[XOVER_Age] := '0';
                 end;
              end;
           end;

        end;
     end;
end;

function XFieldNameToNumber( Fieldname: String ): Integer;
begin
     if Fieldname[length(Fieldname)]=':' then System.Delete(Fieldname,length(Fieldname),1);
     Fieldname := UpperCase( Fieldname );

     if      Fieldname='NUMBER'     then Result := XOVER_Number
     else if Fieldname='SUBJECT'    then Result := XOVER_Subject
     else if Fieldname='FROM'       then Result := XOVER_From
     else if Fieldname='DATE'       then Result := XOVER_Date
     else if Fieldname='MESSAGE-ID' then Result := XOVER_MessageID
     else if Fieldname='REFERENCES' then Result := XOVER_References
     else if Fieldname='BYTES'      then Result := XOVER_Bytes
     else if Fieldname='LINES'      then Result := XOVER_Lines
     else if Fieldname='XREF'       then Result := XOVER_Xref
     else if Fieldname='XPOST'      then Result := XOVER_Xpost
     else if Fieldname='AGE'        then Result := XOVER_Age
     else                                Result := XOVER_INVALID;
end;

function XFieldNumberToName( Fieldnumber: Integer ): String;
begin
     case Fieldnumber of
        XOVER_Number     : Result:='Number';
        XOVER_Subject    : Result:='Subject';
        XOVER_From       : Result:='From';
        XOVER_Date       : Result:='Date';
        XOVER_MessageID  : Result:='Message-ID';
        XOVER_References : Result:='References';
        XOVER_Bytes      : Result:='Bytes';
        XOVER_Lines      : Result:='Lines';
        XOVER_Xref       : Result:='Xref';
        XOVER_Xpost      : Result:='Xpost';
        XOVER_Age        : Result:='Age';
        else               Result:='Invalid!';
     end;
end;

procedure ArticleToXOverRec( const Article: TMess;
                             var   XOverRec: TXOverRec );
var  i, k: Integer;
     s   : String;
begin
     for i:=XOVER_Number to XOVER_MAX do XOverRec.XFields[i]:=''; 

     with XOverRec do begin
        XFields[XOVER_Number]     := '42';
        XFields[XOVER_Subject]    := Article.HeaderValueByNameSL( 'Subject:' );
        XFields[XOVER_From]       := Article.HeaderValueByNameSL( 'From:' );
        XFields[XOVER_Date]       := Article.HeaderValueByNameSL( 'Date:' );
        XFields[XOVER_MessageID]  := Article.HeaderValueByNameSL( 'Message-ID:' );
        XFields[XOVER_References] := Article.HeaderValueByNameSL( 'References:' );

        XFields[XOVER_Bytes]      := Article.HeaderValueByNameSL( 'Bytes:' ); // simulation-only
        if XFields[XOVER_Bytes]='' then begin
           XFields[XOVER_Bytes]   := inttostr( Length(Article.FullText) );
        end;

        XFields[XOVER_Lines]      := Article.HeaderValueByNameSL( 'Lines:' );
        XFields[XOVER_Xref]       := Article.HeaderValueByNameSL( 'Xref:' );

        s := LowerCase( XFields[XOVER_Xref] ); // Xref->Xpost
        while copy(s,1,6)='xref: ' do System.Delete(s,1,6);
        k := 0;
        for i:=1 to length(s) do begin
           if s[i]=':' then inc(k);
        end;
        if k=0 then k:=1;
        XFields[XOVER_Xpost] := inttostr(k);

        try // Date->Age
           k := Trunc( NowGMT - RfcDateTimeToDateTimeGMT( XFields[XOVER_Date], NowGMT ) );
           XFields[XOVER_Age] := inttostr(k);
        except
           XFields[XOVER_Age] := '0';
        end;

     end;
end;

// ---------------------------------------------------------- TScoreLine ------

function GetXOverVal( const BaseData    : Pointer;
                      const XOverField  : Integer;
                      const NotUsedHere : String;
                      const DoMimeDecode: Boolean ): String;
// TFilterGetValueFunc for XOver-data in XOverRec
var  XOverRecPtr: PXOverRec absolute BaseData;
begin
   Result := XOverRecPtr^.XFields[ XOverField ];
   if (Result>'') and DoMimeDecode then Result := DecodeHeadervalue( Result );
end;

function GetXFiltVal( const BaseData    : Pointer;
                      const NotUsedHere : Integer;
                      const XFiltField  : String;
                      const DoMimeDecode: Boolean ): String;
// TFilterGetValueFunc for full article in TMess
var  Article: TMess absolute BaseData;
     s: String;
     i, k: Integer;
begin
   if          XFiltField = 'bytes' then begin
      Result := inttostr( length( Article.FullText ) );

   end else if XFiltField = 'age' then begin
      s := Article.HeaderValueByNameSL( 'date' );
      i := Trunc( NowGMT - RfcDateTimeToDateTimeGMT( s, NowGMT ) );
      Result := inttostr( i );

   end else if XFiltField = 'xpost' then begin
      s := LowerCase( Article.HeaderValueByNameSL( 'xref' ) );
      while copy(s,1,6)='xref: ' do System.Delete(s,1,6);
      k := 0;
      for i:=1 to length(s) do begin
         if s[i]=':' then inc(k);
      end;
      if k=0 then k:=1;
      Result := inttostr(k);

   end else if XFiltField = 'number' then begin
      s := LowerCase( Article.HeaderValueByNameSL( 'xref' ) );
      while copy(s,1,6)='xref: ' do System.Delete(s,1,6);
      i := pos( ':', s );
      if i > 0 then System.Delete( s, 1, i );
      i := PosWhSpace( s );
      if i > 0 then SetLength( s, i-1 );
      Result := inttostr( strtointdef( s, 0 ) );

   end else if XFiltField = 'header' then begin
      Result := Article.HeaderText;

   end else if XFiltField = 'body' then begin
      Result := Article.BodyText;

   end else if XFiltField = 'article' then begin
      Result := Article.FullText;

   end else begin
      Result := Article.HeaderValueByNameSL( XFiltField );
      if (Result>'') and DoMimeDecode then Result := DecodeHeadervalue(Result);
   end;
end;

function TFilterLineNews.MatchesData( const RE: TPCRE;
                                      const Data: Pointer;
                                      const DataGetValue: TFilterGetValueFunc ): Boolean;
const ONE=0; YES=1; NO=2;
var  Pat          : TFilterPatternNews;
     TestStr, DefStr : String;
     NeedOneOf, HaveOneOf: Boolean;
     PatNo: Integer;
     Matches : Boolean;
begin
     Result := True;

     DefStr := DataGetValue( Data, XOverField, XFiltField, DoMimeDecode );

     NeedOneOf := False;
     HaveOneOf := False;
     Matches   := False;

     for PatNo:=0 to PatternCount-1 do begin
        Pat := TFilterPatternNews( PatternItem[ PatNo ] );

        if Pat.IsSameField then begin
           TestStr := DefStr;
        end else begin
           TestStr := DataGetValue( Data, Pat.XOverField, Pat.XFiltField, DoMimeDecode );
        end;

        if (Pat.SelectType<>' ') or not(HaveOneOf) then begin
           if Pat.IsRegex then begin
              try
                 RE.OptCompile := PCRE_CASELESS;
                 Matches := RE.Match( PChar(Pat.Pattern), PChar(TestStr) );
              except
                 on E: Exception do begin
                    Log( LOGID_ERROR, 'Regex-error in {' + Pat.Pattern + '}:' + E.Message );
                    Matches := False;
                 end;
              end;
           end else begin
              Matches := MatchSimple( TestStr, Pat.Pattern );
           end;

           case Pat.SelectType of
              '+': if not Matches then begin Result:=False; break; end;
              '-': if Matches     then begin Result:=False; break; end;
              ' ': begin
                      NeedOneOf := True;
                      if Matches then HaveOneOf:=True;
                   end;
           end;
        end;
     end;

     if NeedOneOf and not HaveOneOf then Result:=False;
end;

function TFilterLineNews.SetFilterLine( FilterLine: String ): Boolean;
var  i, k: Integer;
     s: String;
     SelectType: Char;
     IsRegex: Boolean;
     TempIsSameField: Boolean;
     TempXOverField: Integer;
     TempXFiltField: String;
     Pattern: String;
     cEnd: Char;
     pat: TFilterPatternNews;
begin
     Result := False;
     LastSetError := 'invalid line';

     Clear;
     IsFinal      := False;
     DoUnless     := False;
     ScoreValue   := 0;
     DoMimeDecode := False;
     IsBeforeLoad := True;
     XOverField   := XOVER_INVALID;
     XFiltField   := '';

     FilterLine := TrimWhSpace( FilterLine );
     if FilterLine='' then exit;

     // ['?'] ['='] ('+'|'-') scorevalue [ WSP 'unless' ] WHSP ['~'] defaultfield WHSP pattern [WHSP pattern ...]

     // Score-After-Load marker
     if FilterLine[1] = '?' then begin
        IsBeforeLoad := False;
        System.Delete( FilterLine, 1, 1 );
     end;

     // Final-marker
     if FilterLine[1]='=' then begin
        IsFinal:=True;
        System.Delete( FilterLine, 1, 1 );
     end;

     // +/- Scorevalue
     i := PosWhSpace( FilterLine );
     if i<3 then begin LastSetError:='missing (+|-)scorevalue'; exit; end;
     s := copy( FilterLine, 1, i-1 );
     System.Delete( FilterLine, 1, i );
     FilterLine := TrimWhSpace( FilterLine );
     ScoreValue := strtoint( s );

     // unless?
     i := PosWhSpace( FilterLine );
     if i=0 then i:=length(FilterLine)+1;
     s := copy( FilterLine, 1, i-1 );
     if LowerCase(s) = FILTER_KEYWORD_UNLESS then begin
        DoUnless := true;
        System.Delete( FilterLine, 1, i );
        FilterLine := TrimWhSpace( FilterLine );
     end;

     // Default-Field with optional MIME-Decode
     i := PosWhSpace( FilterLine );
     if i<2 then begin LastSetError:='missing: [~]default-field'; exit; end;
     s := copy( FilterLine, 1, i-1 );
     if (i=2) and (s[1]='~') then begin
        LastSetError:='missing default-field or invalid space between ~ and field';
        exit
     end;
     System.Delete( FilterLine, 1, i );
     FilterLine := TrimWhSpace( FilterLine );
     if Copy(s,1,1)='~' then begin
        DoMimeDecode := True;
        System.Delete( s, 1, 1 );
     end;
     if IsBeforeLoad then begin
        XOverField := XFieldNameToNumber( s );
        if XOverField=XOVER_INVALID then begin LastSetError:='invalid default-field "'+s+'"'; exit; end;
     end else begin
        s := LowerCase( s );
        if (s>'') and (s[length(s)]=':') then System.Delete( s, length(s), 1 );
        XFiltField := s;
     end;

     // One or more patterns of the following forms:
     //    ['+'|'-'] ['@' fieldname ':'] '{' regex-pattern '}'
     //    ['+'|'-'] ['@' fieldname ':'] '"' simple-pattern '"'
     //    ['+'|'-'] ['@' fieldname ':'] simple-pattern without WHSP

     while FilterLine<>'' do begin
        SelectType := ' ';
        if FilterLine[1] in ['+','-'] then begin
           SelectType := FilterLine[1];
           System.Delete( FilterLine, 1, 1 );
        end;

        TempIsSameField := True;
        TempXOverField  := XOverField;
        TempXFiltField  := XFiltField;
        
        if FilterLine[1]='@' then begin
           TempIsSameField := False;
           i := Pos( ':', FilterLine );
           if i<3 then begin LastSetError:='missing: "fieldname:" on "@"'; exit; end;
           s := copy( FilterLine, 2, i-2 );
           System.Delete( FilterLine, 1, i );
           if IsBeforeLoad then begin
              TempXOverField := XFieldNameToNumber( s );
              if TempXOverField=XOVER_INVALID then begin LastSetError:='invalid field "'+s+'"'; exit; end;
           end else begin
              s := LowerCase( s );
              if (s>'') and (s[length(s)]=':') then System.Delete( s, length(s), 1 );
              TempXFiltField := s;
           end;
        end;

        if FilterLine='' then begin LastSetError:='missing: pattern'; exit; end;
        Pattern := '';

        if FilterLine[1]='{' then begin
           IsRegex := True;
           System.Delete( FilterLine, 1, 1 );
           k := 1;
           while FilterLine<>'' do begin
              if FilterLine[1]='{' then inc(k);
              if FilterLine[1]='}' then begin
                 dec(k);
                 if k=0 then break;
              end;
              Pattern := Pattern + FilterLine[1];
              System.Delete( FilterLine, 1, 1 );
           end;
           if copy(FilterLine,1,1)='}' then System.Delete( FilterLine, 1, 1 );
        end else begin
           IsRegex := False;
           if FilterLine[1]='"' then begin
              cEnd := '"';
              System.Delete( FilterLine, 1, 1 );
              i := Pos( cEnd, FilterLine );
           end else begin
              cEnd := #32;
              if (FilterLine[1] = '%') then
                 while ((FilterLine[3] = #32) or (FilterLine[3] = #9)) do
                    System.Delete( FilterLine, 3, 1 );
              i := PosWhSpace( FilterLine ); {WJ}
           end;
           // falls EOL:
           if (i<=0) then i := length(FilterLine) + 1;
           Pattern := copy( FilterLine, 1, i-1 );
           System.Delete( FilterLine, 1, i-1 );
           if copy(FilterLine,1,1)=cEnd then System.Delete( FilterLine, 1, 1 );
        end;

        if Pattern='' then begin LastSetError:='missing: pattern/-delimiter'; exit; end;

        pat := TFilterPatternNews.Create;
        pat.SelectType   := SelectType;
        pat.IsRegex      := IsRegex;
        pat.Pattern      := Pattern;
        pat.IsSameField  := TempIsSameField;
        pat.XOverField   := TempXOverField;
        pat.XFiltField   := TempXFiltField;
        PatternAdd( pat );

        FilterLine := TrimWhSpace( FilterLine );
        if FilterLine<>'' then begin
           if FilterLine[1] in ['#',';'] then FilterLine:=''; // rest of line is [valid] comment
        end;
     end;

     LastSetError := '';
     Result := True;
end;

function TFilterLineNews.AsString: String;
var  Pat  : TFilterPatternNews;
     PatNo: Integer;
begin
     if PatternCount=0 then begin
        Result := '(line not set)';
        exit;
     end;

     Result := '';

     if not IsBeforeLoad then Result := Result + '?';
     if IsFinal          then Result := Result + '=';
     if ScoreValue>=0    then Result := Result + '+'
                         else Result := Result + '-';
     Result := Result + inttostr( abs(ScoreValue) );

     if DoUnless then Result := Result + ' ' + FILTER_KEYWORD_UNLESS;

     Result := Result + ' ';
     
     if DoMimeDecode then Result:=Result+'~';
     if IsBeforeLoad then Result := Result + XFieldNumberToName( XOverField )
                     else Result := Result + XFiltField;

     for PatNo:=0 to PatternCount-1 do begin
        Pat := TFilterPatternNews( PatternItem[ PatNo ] );

        Result := Result + ' ';
        if Pat.SelectType in ['+','-'] then Result := Result + Pat.SelectType;

        if not Pat.IsSameField then begin
           if IsBeforeLoad then begin
              Result := Result + '@' + XFieldNumberToName( Pat.XOverField ) + ':';
           end else begin
              Result := Result + '@' + Pat.XFiltField + ':';
           end;
        end;

        if Pat.IsRegEx then begin
           Result := Result + '{' + Pat.Pattern + '}';
        end else begin
           if Pos(' ',Pat.Pattern)>0 then begin
              Result := Result + '"' + Pat.Pattern + '"';
           end else begin
              Result := Result + Pat.Pattern;
           end;
        end;
     end;
end;

// ---------------------------------------------------------- TScoreFile ------

procedure TFiltersNews.SelectSections( const SectionIdentifier: String );
begin
   FHasXOverLines := False;
   FHasXFiltLines := False;
   inherited;
end;

function TFiltersNews.LinesAdd( const LineText: String ): Integer;
var  lin: TFilterLineNews;
begin
     lin := TFilterLineNews.Create;
     if lin.SetFilterLine( LineText ) then begin
        Result := fFilterLines.Add( lin );
        if lin.IsBeforeLoad then FHasXOverLines := True
                            else FHasXFiltLines := True;
     end else begin
        Log( LOGID_WARN, 'Scorefile-line ignored: ' + LineText );
        Log( LOGID_WARN, 'Reason: ' + lin.LastSetError );
        lin.Free;
        Result := -1;
     end;
end;

function TFiltersNews.IsFilterLine( const Line: String ): Boolean;
begin
   Result := ( Line > '' ) and ( Line[1] in [ '?', '=', '+', '-' ] )
end;

function TFiltersNews.ScoreBeforeLoad( const XOverRec: TXOverRec;
                                       const MatchLog: PString ): Integer;
var  LineNo: Integer;
     Line  : TFilterLineNews;
     MatchResult: Boolean;
begin
     Result := 0;
     if Assigned(MatchLog) then MatchLog^ := '';
     if not FHasXOverLines then exit;

     for LineNo:=0 to LinesCount-1 do begin
        Line := TFilterLineNews( LinesItem[ LineNo ] );
        if Line.IsBeforeLoad then begin
           MatchResult := Line.MatchesData( RegexFilter, @XOverRec, GetXOverVal );
           if Line.DoUnless then MatchResult := not( MatchResult );
           if MatchResult then begin
              if Assigned(MatchLog) then MatchLog^ := MatchLog^ + Line.AsString + #13#10;
              if Line.IsFinal then begin
                 Result := Line.ScoreValue;
                 break;
              end;
              Result := Result + Line.ScoreValue;
           end;
        end;
     end;

     if Result<-9999 then Result:=-9999;
     if Result>+9999 then Result:=+9999;
end;

function TFiltersNews.ScoreAfterLoad( const Article: TMess;
                                      const MatchLog: PString ): Integer;
var  LineNo: Integer;
     Line  : TFilterLineNews;
     MatchResult: Boolean;
begin
     Result := 0;
     if Assigned(MatchLog) then MatchLog^ := '';
     if not FHasXFiltLines then exit;

     for LineNo:=0 to LinesCount-1 do begin
        Line := TFilterLineNews( LinesItem[ LineNo ] );
        if not Line.IsBeforeLoad then begin
           MatchResult := Line.MatchesData( RegexFilter, Article, GetXFiltVal );
           if Line.DoUnless then MatchResult := not( MatchResult );
           if MatchResult then begin
              if Assigned(MatchLog) then MatchLog^ := MatchLog^ + Line.AsString + #13#10;
              if Line.IsFinal then begin
                 Result := Line.ScoreValue;
                 break;
              end;
              Result := Result + Line.ScoreValue;
           end;
        end;
     end;

     if Result<-9999 then Result:=-9999;
     if Result>+9999 then Result:=+9999;
end;

function TFiltersNews.ScoreArticle( const Article: TMess ): Integer;
var  XOverRec: TXOverRec;
begin
   Result := 0;

   if FHasXOverLines then begin
      ArticleToXOverRec( Article, {var} XOverRec );
      Result := Result + ScoreBeforeLoad( XOverRec, nil );
   end;

   if FHasXFiltLines then begin
      Result := Result + ScoreAfterLoad( Article, nil );
   end;

   if Result<-9999 then Result:=-9999;
   if Result>+9999 then Result:=+9999;
end;

end.

