// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cFiltersMail;

interface

{$INCLUDE Compiler.inc}

uses SysUtils, Classes, cArticle, cFiltersBase, cPCRE;

type
  TFilterPatternMail = class( TFilterPatternBase )
    FilterHeader: String;
    ExtractField: boolean;
  end;

  TFilterLineMail = class( TFilterLineBase )
    protected
      function MatchesAccounts( const MailHeaders: String;
                                var   ResultAccounts: String ): Boolean;
      function TestHeaderValue( const HdrVal: String;
                                const Hdr: TMess;
                                const RE: TPCRE ): Boolean;
      function MatchesHeaders(  const RE: TPCRE;
                                const MailHeaders: String ): Boolean;
    public
      IsFinal      : Boolean; // '='?
      DoAllHeaders : Boolean; // '*'?
      ActionID     : Integer;
      ActionPars   : String;
      DefaultField : String;
      function  MatchesMailHeaders( const RE: TPCRE;
                                    const MailHeaders: String;
                                    var   ResultAccounts: String ): Boolean;
      function  SetFilterLine( FilterLine: String ): Boolean; override;
      function  AsString: String; override;
  end;

  TFiltersMail = class( TFiltersBase )
    public
      TOP_makes_sense  : Boolean;
      Filter_bytes_only: Boolean;
      function  LinesAdd( const LineText: String ): Integer; override;
      function  IsFilterLine( const Line: String ): Boolean; override;
      procedure SelectSections( const SectionIdentifier: String ); override;
      procedure FilterMail(       MailHeaders  : String;
                                  DefaultUser  : String;
                            var   ResultIgnore : Boolean;
                            var   ResultKillIt : Boolean;
                            const ResultNotifys: TStrings;
                            const ResultUsers  : TStrings;
                            const ResultGroups : TStrings;
                            var   NotifyReason : String;
                            out   Reason       : String );
  end;

procedure ExtractMailAddresses( const MailAddrs: TStringList; HdrNam, HdrVal: String );

  
implementation

uses uConst, uTools, cAccounts, uEncoding, cLogFileHamster, cHamster;

const
  MLFLT_ACTION_INVALID     =  0;
  MLFLT_ACTION_LOAD        =  1;
  MLFLT_ACTION_KILL        =  2;
  MLFLT_ACTION_IGNORE      =  3;
  MLFLT_ACTION_NOTIFY      =  4;
  MLFLT_ACTION_DEFAULT     = 10;
  MLFLT_ACTION_SET         = 11;
  MLFLT_ACTION_ADD         = 12;
  MLFLT_ACTION_DEL         = 13;
  MLFLT_ACTION_ADDACCOUNTS = 14;
  MLFLT_ACTION_POSTTO      = 15;

const
  ANY_SENDER        = 'Any-Sender:';
  ANY_SENDER_MAX    = 6;
  ANY_SENDER_LIST   : array[0..ANY_SENDER_MAX] of String =
                      ( 'From:', 'Apparently-From:', 'Sender:', 'Reply-To:',
                        'X-Sender:', 'Envelope-From:', 'X-Envelope-From:' );

  ANY_RECIPIENT     = 'Any-Recipient:';
  ANY_RECIPIENT_MAX = 10;
  ANY_RECIPIENT_LIST: array[0..ANY_RECIPIENT_MAX] of String =
                      ( 'To:', 'Apparently-To:', 'CC:', 'BCC:',
                        'Envelope-To:', 'X-Envelope-To:',
                        'Original-Recipient:', 'X-Resent-For:',
                        'X-Resent-To:', 'Resent-To:', 'Delivered-To:' );

procedure ExpandFilterGroups( HdrName: String; TS: TStrings );
var  i: Integer;
begin
     if HdrName[length(HdrName)]<>':' then HdrName:=HdrName+':';
     if CompareText( ANY_SENDER, HdrName )=0 then begin
        for i:=0 to ANY_SENDER_MAX do TS.Add( ANY_SENDER_LIST[i] );
        exit;
     end;
     if CompareText( ANY_RECIPIENT, HdrName )=0 then begin
        for i:=0 to ANY_RECIPIENT_MAX do TS.Add( ANY_RECIPIENT_LIST[i] );
        exit;
     end;
     TS.Add( HdrName );
end;

// ------------------------------------------------------ TFilterLineMail -----

procedure AddMailAddress( const MailAddrs: TStringList; TestStr: String );
var  at: Integer;
begin
    TestStr := TrimWhSpace( TestStr );
    if length( TestStr ) < 3 then exit;
    if PosWhSpace( TestStr ) > 0 then exit;

    if TestStr[1] = '<' then System.Delete( TestStr, 1, 1 );
    if TestStr[length(TestStr)] = '>' then begin
       System.Delete( TestStr, length(TestStr), 1 );
    end;

    at := Pos( '@', TestStr );
    if (at < 2) or (at = length(TestStr)) then exit;

    MailAddrs.Add( TestStr );
end;

procedure ExtractMailAddresses( const MailAddrs: TStringList; HdrNam, HdrVal: String );
// add all strings which look like mail-addresses to accounts-testlist
var p: Integer;
    addr: String;
begin
    if Pos( 'received', LowerCase(HdrNam) ) = 1 then begin
       // in Received header it has to be preceded by word 'for'
       // if not RE_Match( HdrVal, '\bfor\b.+@', PCRE_CASELESS ) then exit;
       HdrVal := RE_Extract( HdrVal, '\bfor\b.+@.+', PCRE_CASELESS );
       if HdrVal = '' then exit;
    end;

    addr := '';
    p    := 1;
    while p <= length(HdrVal) do begin
       if HdrVal[p] in [ #9, ' ', ',', '<', '>', '"', '(', ')', '''', ';' ] then begin
          AddMailAddress( MailAddrs, addr );
          addr := '';
       end else begin
          addr := addr + HdrVal[p];
       end;

       inc( p );
    end;

    AddMailAddress( MailAddrs, addr );
end;

function TFilterLineMail.MatchesAccounts( const MailHeaders: String;
                                          var ResultAccounts: String ): Boolean;
var  TestHdrs, MailAddrs: TStringList;
     Hdr: TMess;
     TestHdr, DefStr, AccountName: String;
     CurrHdr, LastPos, UID, LTY, i: Integer;
begin
     Hdr := TMess.Create;
     Hdr.HeaderText := MailHeaders;

     TestHdrs := TStringList.Create;
     ExpandFilterGroups( DefaultField, TestHdrs );

     MailAddrs := TStringList.Create;
     MailAddrs.Sorted := True;
     MailAddrs.Duplicates := dupIgnore;

     for CurrHdr := 0 to TestHdrs.Count - 1 do begin

        TestHdr := TestHdrs[CurrHdr];
        LastPos := -1;

        repeat
           LastPos := Hdr.IndexOfHeader( TestHdr, LastPos );
           if LastPos < 0 then break;

           DefStr := Hdr.HeaderValueByIndexSL( LastPos );
           if (DefStr > '') and DoMimeDecode then begin
              DefStr := DecodeHeadervalue( DefStr );
           end;

           ExtractMailAddresses( MailAddrs, TestHdr, DefStr );

           if not DoAllHeaders then break;
        until False;

     end;

     for i := 0 to MailAddrs.Count - 1 do begin

        if Hamster.Accounts.IsLocalMailbox( MailAddrs[i], UID,LTY ) then begin

           AccountName := Hamster.Accounts.Value[ UID, apUsername ];
           if ResultAccounts <> '' then ResultAccounts := ResultAccounts + ',';
           ResultAccounts := ResultAccounts + AccountName;

        end else begin

           Log( LOGID_DEBUG,
                'MailFilter.addaccounts: "' + MailAddrs[i] + '" unknown.' );

        end;

     end;

     if ResultAccounts <> '' then begin
        Log( LOGID_DEBUG,
             'MailFilter.addaccounts.final: "' + ResultAccounts + '"' );
     end;

     Result := ( ResultAccounts <> '' );

     MailAddrs.Free;
     TestHdrs.Free;
     Hdr.Free;
end;

function TFilterLineMail.TestHeaderValue( const HdrVal: String;
                                          const Hdr: TMess;
                                          const RE: TPCRE ): Boolean;
var  Matches, NeedOneOf, HaveOneOf: Boolean;
     PatNo, HPos: Integer;
     TestStr, Pattern: String;
     Pat: TFilterPatternMail;
begin
     Result := True;

     NeedOneOf := False;
     HaveOneOf := False;
     Matches   := False;

     for PatNo := 0 to PatternCount - 1 do begin

        Pat := TFilterPatternMail( PatternItem[ PatNo ] );

        if Pat.ExtractField then begin
           Pattern := '';
           HPos := Hdr.IndexOfHeader( Pat.Pattern, -1 );
           if HPos >= 0 then Pattern := Hdr.HeaderValueByIndexSL( Hpos )
        end else begin
           Pattern := Pat.Pattern
        end;
        if Pattern = '' then continue;

        if CompareText( Pat.FilterHeader, DefaultField ) = 0 then begin
           TestStr := HdrVal;
        end else begin
           TestStr := Hdr.HeaderValueByNameSL( Pat.FilterHeader );
           if (TestStr > '') and DoMimeDecode then begin
              TestStr := DecodeHeadervalue(TestStr);
           end;
        end;

        if (Pat.SelectType <> ' ') or not( HaveOneOf ) then begin
           if Pat.IsRegex then begin
              try
                 RE.OptCompile := PCRE_CASELESS;
                 Matches := RE.Match( PChar(Pattern), PChar(TestStr) );
              except
                 on E: Exception do begin
                    Log( LOGID_ERROR, 'Regex-error in {' + Pattern + '}:' + E.Message );
                    Matches := False;
                 end;
              end;
           end else begin
              Matches := MatchSimple( TestStr, Pattern );
           end;

           case Pat.SelectType of
              '+': if not Matches then begin Result := False; break; end;
              '-': if Matches     then begin Result := False; break; end;
              ' ': begin
                      NeedOneOf := True;
                      if Matches then HaveOneOf := True;
                   end;
           end;
        end;
     end;

     if NeedOneOf and not HaveOneOf then Result := False;
end;

function TFilterLineMail.MatchesHeaders( const RE: TPCRE;
                                         const MailHeaders: String ): Boolean;
var  TestHdr, DefStr: String;
     CurrHdr, LastPos: Integer;
     Hdr: TMess;
     TestHdrs: TStringList;
begin
     Result := False;

     Hdr := TMess.Create;
     Hdr.HeaderText := MailHeaders;

     TestHdrs := TStringList.Create;
     ExpandFilterGroups( DefaultField, TestHdrs );

     for CurrHdr := 0 to TestHdrs.Count - 1 do begin

        TestHdr := TestHdrs[CurrHdr];
        LastPos := -1;

        repeat
           LastPos := Hdr.IndexOfHeader( TestHdr, LastPos );
           if LastPos < 0 then break;

           DefStr := Hdr.HeaderValueByIndexSL( LastPos );
           if (DefStr > '') and DoMimeDecode then begin
              DefStr := DecodeHeadervalue( DefStr );
           end;

           // test given patterns
           Result := TestHeaderValue( DefStr, Hdr, RE ); // test patterns
           if Result then break; // current header matched the pattern-line

           if not DoAllHeaders then break;

        until False;

        if Result then break;

     end;

     TestHdrs.Free;
     Hdr.Free;
     exit;
end;

function TFilterLineMail.MatchesMailHeaders( const RE: TPCRE;
                                             const MailHeaders: String;
                                             var ResultAccounts: String ): Boolean;
begin
     Result := True;
     ResultAccounts := '';

     // notification is handled outside
     if ActionID = MLFLT_ACTION_NOTIFY then begin
        Result := True;
        exit;
     end;

     // new default-user is handled outside
     if ActionID = MLFLT_ACTION_DEFAULT then begin
        Result := True;
        exit;
     end;

     // test given header-name[s] for known mail-accounts
     if ActionID = MLFLT_ACTION_ADDACCOUNTS then begin
        Result := MatchesAccounts( MailHeaders, ResultAccounts );
        exit;
     end;

     // test given patterns
     if ActionID in [ MLFLT_ACTION_SET, MLFLT_ACTION_ADD, MLFLT_ACTION_DEL,
                      MLFLT_ACTION_POSTTO, MLFLT_ACTION_LOAD, MLFLT_ACTION_KILL,
                      MLFLT_ACTION_IGNORE ] then begin
        Result := MatchesHeaders( RE, MailHeaders );
        exit;
     end;
end;

Type TVTyp = (vtText, vtField, vtRegex);

function TFilterLineMail.SetFilterLine( FilterLine: String ): Boolean;
var  i, k: Integer;
     s, h: String;
     SelectType: Char;
     VTyp: TVTyp;
     FilterHeader: String;
     Pattern: String;
     cEnd: Char;
     pat: TFilterPatternMail;
begin
     Result := False;
     LastSetError := 'invalid line';

     Clear;
     IsFinal      := False;
     ActionID     := MLFLT_ACTION_INVALID;
     ActionPars   := '';
     DoMimeDecode := False;
     DoAllHeaders := False;
     DoUnless     := False;
     DefaultField := '';

     FilterLine := TrimWhSpace( FilterLine );
     if FilterLine='' then exit;

     // Final-marker
     if FilterLine[1]='=' then begin
        IsFinal:=True;
        System.Delete( FilterLine, 1, 1 );
     end;

     // action
     i := PosWhSpace( FilterLine );
     if i=0 then i:=length(FilterLine)+1;
     if i<3 then begin LastSetError:='missing filter-action'; exit; end;
     s := copy( FilterLine, 1, i-1 );
     System.Delete( FilterLine, 1, i );
     FilterLine := TrimWhSpace( FilterLine );

     i := Pos( '(', s );
     if i=0 then begin
        h := s;
        s := '';
     end else begin
        h := copy( s, 1, i-1 );
        System.Delete( s, 1, i );
        if copy(s,length(s),1)<>')' then begin LastSetError:='missing ")" in "'+h+'('+s+'"'; exit; end;
        System.Delete( s, length(s), 1 );
     end;
     ActionPars := TrimWhSpace(s);

     h := LowerCase( TrimWhSpace( h ) );
     if      h='default'     then ActionID:=MLFLT_ACTION_DEFAULT
     else if h='set'         then ActionID:=MLFLT_ACTION_SET
     else if h='add'         then ActionID:=MLFLT_ACTION_ADD
     else if h='del'         then ActionID:=MLFLT_ACTION_DEL
     else if h='addaccounts' then ActionID:=MLFLT_ACTION_ADDACCOUNTS
     else if h='postto'      then ActionID:=MLFLT_ACTION_POSTTO
     else if h='load'        then ActionID:=MLFLT_ACTION_LOAD
     else if h='kill'        then ActionID:=MLFLT_ACTION_KILL
     else if h='ignore'      then ActionID:=MLFLT_ACTION_IGNORE
     else if h='notify'      then ActionID:=MLFLT_ACTION_NOTIFY
     else                         ActionID:=MLFLT_ACTION_INVALID;
     if ActionID=MLFLT_ACTION_INVALID then begin
        LastSetError := 'Invalid action "' + h + '"';
        exit;
     end;

     // check filter-params
     if not( ActionID in [MLFLT_ACTION_DEFAULT, MLFLT_ACTION_NOTIFY] ) then begin

        // unless?
        i := PosWhSpace( FilterLine );
        if i=0 then i:=length(FilterLine)+1;
        s := copy( FilterLine, 1, i-1 );
        if LowerCase(s) = FILTER_KEYWORD_UNLESS then begin
           DoUnless := true;
           System.Delete( FilterLine, 1, i );
           FilterLine := TrimWhSpace( FilterLine );
        end;

        // Default-Field with optional MIME-Decode
        i := PosWhSpace( FilterLine );
        if i=0 then i:=length(FilterLine)+1;
        if i<2 then begin LastSetError:='missing: [~][*]default-field'; exit; end;
        s := copy( FilterLine, 1, i-1 );
        System.Delete( FilterLine, 1, i );
        FilterLine := TrimWhSpace( FilterLine );
        while (s>'') and (s[1] in ['~','*']) do begin
           if s[1]='~' then DoMimeDecode := True;
           if s[1]='*' then DoAllHeaders := True;
           System.Delete( s, 1, 1 );
        end;
        if s = '' then begin LastSetError:='missing: [~][*]default-field'; exit; end;
        if s[length(s)]<>':' then s:=s+':';
        DefaultField := s;

        if ActionID<>MLFLT_ACTION_ADDACCOUNTS then begin

           // One or more patterns of the following forms:
           //    ['+'|'-'] ['@' fieldname ':'] '{' regex-pattern '}'
           //    ['+'|'-'] ['@' fieldname ':'] '"' simple-pattern '"'
           //    ['+'|'-'] ['@' fieldname ':'] simple-pattern without WHSP

           while FilterLine>'' do begin
              SelectType := ' ';
              if FilterLine[1] in ['+','-'] then begin
                 SelectType := FilterLine[1];
                 System.Delete( FilterLine, 1, 1 );
              end;

              FilterHeader := DefaultField;
              if FilterLine[1]='@' then begin
                 i := Pos( ':', FilterLine );
                 if i<3 then begin LastSetError:='missing: "fieldname:" on "@"'; exit; end;
                 s := copy( FilterLine, 2, i-2 );
                 System.Delete( FilterLine, 1, i );
                 if s[length(s)]<>':' then s:=s+':';
                 FilterHeader := s;
              end;

              if FilterLine='' then begin LastSetError:='missing: pattern'; exit; end;
              Pattern := '';

              if FilterLine[1]='{' then begin
                 VTyp := vtRegex;
                 System.Delete( FilterLine, 1, 1 );
                 k := 1;
                 while FilterLine<>'' do begin
                    if FilterLine[1]='{' then inc(k);
                    if FilterLine[1]='}' then begin
                       dec(k);
                       if k=0 then break;
                    end;
                    Pattern := Pattern + FilterLine[1];
                    System.Delete( FilterLine, 1, 1 );
                 end;
                 if copy(FilterLine,1,1)='}' then System.Delete( FilterLine, 1, 1 );
              end else begin
                 if FilterLine[1]='$' then begin
                    VTyp := vtField;
                    cEnd := '$';
                    System.Delete( FilterLine, 1, 1 );
                    i := Pos( cEnd, FilterLine )
                 end else begin
                    VTyp := vtText;
                    if FilterLine[1]='"' then begin
                       cEnd := '"';
                       System.Delete( FilterLine, 1, 1 );
                       i := Pos( cEnd, FilterLine )
                    end else begin
                       cEnd := #32;
                       if (FilterLine[1] = '%') then
                          while ((FilterLine[3] = #32) or (FilterLine[3] = #9)) do
                             System.Delete( FilterLine, 3, 1 );
                       i := PosWhSpace( FilterLine )
                    end
                 end;
                 if (i<=0) then i := length(FilterLine) + 1;
                 Pattern := copy( FilterLine, 1, i-1 );
                 System.Delete( FilterLine, 1, i-1 );
                 if copy(FilterLine,1,1)=cEnd then System.Delete( FilterLine, 1, 1 )
              end;

              if Pattern='' then begin LastSetError:='missing: pattern/-delimiter'; exit; end;

              pat := TFilterPatternMail.Create;
              pat.SelectType   := SelectType;
              pat.IsRegex      := ( VTyp = vtRegEx );
              pat.ExtractField := ( VTyp = vtField );
              pat.Pattern      := Pattern;
              pat.FilterHeader := FilterHeader;
              PatternAdd( pat );

              FilterLine := TrimWhSpace( FilterLine );
              if FilterLine<>'' then begin
                 if FilterLine[1] in ['#',';'] then FilterLine:=''; // rest of line is [valid] comment
              end;
           end;
        end;
     end;

     LastSetError := '';
     Result := True;
end;

function TFilterLineMail.AsString: String;
var  Pat  : TFilterPatternMail;
     PatNo: Integer;
begin
     Result := '';

     if IsFinal then Result:=Result+'=';

     case ActionID of
        MLFLT_ACTION_DEFAULT:     Result:=Result+'default';
        MLFLT_ACTION_SET:         Result:=Result+'set';
        MLFLT_ACTION_ADD:         Result:=Result+'add';
        MLFLT_ACTION_DEL:         Result:=Result+'del';
        MLFLT_ACTION_ADDACCOUNTS: Result:=Result+'addaccounts';
        MLFLT_ACTION_POSTTO:      Result:=Result+'postto';
        MLFLT_ACTION_LOAD:        Result:=Result+'load';
        MLFLT_ACTION_KILL:        Result:=Result+'kill';
        MLFLT_ACTION_IGNORE:      Result:=Result+'ignore';
        MLFLT_ACTION_NOTIFY:      Result:=Result+'notify';
        else                      Result:=Result+'???';
     end;
     Result := Result + '(' + ActionPars + ')';

     if DoUnless then Result := Result + ' ' + FILTER_KEYWORD_UNLESS;
     
     if DefaultField<>'' then begin
        Result := Result + ' ';
        if DoMimeDecode then Result:=Result+'~';
        if DoAllHeaders then Result:=Result+'*';
        Result := Result + DefaultField;

        for PatNo:=0 to PatternCount-1 do begin
           Pat := TFilterPatternMail( PatternItem[ PatNo ] );
           Result := Result + ' ';
           if Pat.SelectType in ['+','-'] then Result:=Result+Pat.SelectType;
           if CompareText( Pat.FilterHeader, DefaultField )<>0 then begin
              Result := Result + '@' + Pat.FilterHeader + ':';
           end;
           if Pat.IsRegEx then begin
              Result := Result + '{' + Pat.Pattern + '}';
           end else begin
              if Pos(' ',Pat.Pattern)>0 then begin
                 Result := Result + '"' + Pat.Pattern + '"';
              end else begin
                 Result := Result + Pat.Pattern;
              end;
           end;
        end;
     end;
end;

// --------------------------------------------------------- TFiltersMail -----

function TFiltersMail.LinesAdd( const LineText: String ): Integer;
var  lin: TFilterLineMail;
begin
     lin := TFilterLineMail.Create;
     if lin.SetFilterLine( LineText ) then begin
        Result := fFilterLines.Add( lin );
     end else begin
        Log( LOGID_WARN, 'Filter-line ignored: ' + LineText );
        Log( LOGID_WARN, 'Reason: ' + lin.LastSetError );
        lin.Free;
        Result := -1;
     end;
end;

function TFiltersMail.IsFilterLine( const Line: String ): Boolean;
begin
   Result := (Line>'') and (Line[1] in ['=','a'..'z','A'..'Z'])
end;

procedure TFiltersMail.SelectSections( const SectionIdentifier: String );
var  LineNo: Integer;
     Line  : TFilterLineMail;
begin
     inherited SelectSections( SectionIdentifier );

     TOP_makes_sense := False;
     for LineNo:=0 to LinesCount-1 do begin
        Line := TFilterLineMail( LinesItem[ LineNo ] );
        if Line.ActionID in [MLFLT_ACTION_KILL, MLFLT_ACTION_IGNORE] then begin
           TOP_makes_sense := True;
           break;
        end;
     end;
end;

procedure TFiltersMail.FilterMail(       MailHeaders  : String;
                                         DefaultUser  : String;
                                   var   ResultIgnore : Boolean;
                                   var   ResultKillIt : Boolean;
                                   const ResultNotifys: TStrings;
                                   const ResultUsers  : TStrings;
                                   const ResultGroups : TStrings;
                                   var   NotifyReason : String;
                                   out   Reason       : String );
var  LineNo, i, k  : Integer;
     Line          : TFilterLineMail;
     Parser        : TParser;
     ResAccounts, s: String;
     WantNotify, MatchResult: Boolean;
begin
     ResultIgnore := False;
     ResultKillIt := False;
     WantNotify   := False;
     ResultNotifys.Clear;
     ResultUsers.Clear;
     ResultGroups.Clear;
     NotifyReason := '';
     Reason       := '';

     Parser := TParser.Create;

     if DefaultUser = '' then DefaultUser := 'admin';
     i := Pos( #13#10#13#10, MailHeaders );
     if i > 0 then MailHeaders := copy( MailHeaders, 1, i+1 );

     for LineNo := 0 to LinesCount - 1 do begin

        Line := TFilterLineMail( LinesItem[ LineNo ] );

        MatchResult := Line.MatchesMailHeaders(
           RegexFilter, MailHeaders, ResAccounts
        );
        
        if Line.DoUnless then MatchResult := not( MatchResult );

        if MatchResult then begin

           Log( LOGID_DEBUG, 'MailFilter (+): ' + Line.AsString );
           case Line.ActionID of

              // undo a previous kill/ignore
              MLFLT_ACTION_LOAD: begin
                 ResultIgnore := False;
                 ResultKillIt := False;
                 NotifyReason := '';
                 Reason       := '';
              end;

              // don't load, but delete
              MLFLT_ACTION_KILL: begin
                 ResultKillIt := True;
                 NotifyReason := NotifyReason + '-> ' + Line.AsString + #13#10;
                 Reason := iif( Reason <> '', Reason + #9 ) + Line.AsString;
              end;

              // don't load, don't delete
              MLFLT_ACTION_IGNORE: begin
                 ResultKillIt := False;
                 ResultIgnore := True;
                 NotifyReason := NotifyReason + '-> ' + Line.AsString + #13#10;
                 Reason := iif( Reason <> '', Reason + #9 ) + Line.AsString;
              end;

              // notify if "killed" or "ignored"
              MLFLT_ACTION_NOTIFY: begin
                 WantNotify := True;
                 ResultNotifys.Clear;
                 Parser.Parse( Line.ActionPars, ',' );
                 for i:=0 to Parser.Count-1 do begin
                    s := TrimWhSpace( Parser.sPart( i, '' ) );
                    if s<>'' then ResultNotifys.Add( s );
                 end;
              end;

              // new default-user
              MLFLT_ACTION_DEFAULT: begin
                 DefaultUser := Line.ActionPars;
              end;

              // set/start new recipient-list
              MLFLT_ACTION_SET: begin
                 ResultUsers.Clear;
                 Parser.Parse( Line.ActionPars, ',' );
                 for i:=0 to Parser.Count-1 do begin
                    s := TrimWhSpace( Parser.sPart( i, '' ) );
                    if s<>'' then ResultUsers.Add( s );
                 end;
              end;

              // add to recipient-list
              MLFLT_ACTION_ADD: begin
                 Parser.Parse( Line.ActionPars, ',' );
                 for i:=0 to Parser.Count-1 do begin
                    s := TrimWhSpace( Parser.sPart( i, '' ) );
                    k := ResultUsers.IndexOf( s );
                    if (k<0) and (s<>'') then ResultUsers.Add( s );
                 end;
              end;

              // delete from recipient-list
              MLFLT_ACTION_DEL: begin
                 Parser.Parse( Line.ActionPars, ',' );
                 for i:=0 to Parser.Count-1 do begin
                    s := TrimWhSpace( Parser.sPart( i, '' ) );
                    k := ResultUsers.IndexOf( s );
                    if k>=0 then ResultUsers.Delete( k );
                 end;
              end;

              // add matching accounts to recipient-list
              MLFLT_ACTION_ADDACCOUNTS: begin
                 Parser.Parse( ResAccounts, ',' );
                 for i:=0 to Parser.Count-1 do begin
                    s := TrimWhSpace( Parser.sPart( i, '' ) );
                    if s<>'' then ResultUsers.Add( s );
                 end;
              end;

              // post to newsgroups
              MLFLT_ACTION_POSTTO: begin
                 Parser.Parse( Line.ActionPars, ',' );
                 for i:=0 to Parser.Count-1 do begin
                    s := TrimWhSpace( Parser.sPart( i, '' ) );
                    if Hamster.Config.Newsgroups.IndexOf(s) < 0 then
                       Log( LOGID_WARN, 'Unknown group ' + s + ' in postto-mailfilter!' );
                    if s<>'' then ResultGroups.Add( s );
                 end;
              end;

           end;

           if Line.IsFinal then break; // line preceded with "="

        end else begin
           Log( LOGID_FULL, 'MailFilter (-): ' + Line.AsString );
        end;
     end;

     // use default-user, if no recipients/newsgroups were selected
     if (ResultUsers.Count=0) and (ResultGroups.Count=0) then begin
        if DefaultUser <> '' then begin
           Parser.Parse( DefaultUser, ',' );
           for i := 0 to Parser.Count - 1 do begin
              s := TrimWhSpace( Parser.sPart( i, '' ) );
              if s <> '' then ResultUsers.Add( s );
           end;
        end;
     end;

     // fill list of notification-recipients if none were given with "notify()"
     if WantNotify and (ResultIgnore or ResultKillIt) then begin
        // send notifications to determined recipients ...
        if ResultNotifys.Count=0 then ResultNotifys.Text:=ResultUsers.Text;
        // .. or finally to "admin"
        if ResultNotifys.Count=0 then ResultNotifys.Add( 'admin' );
     end;

     Parser.Free;
end;

end.
