// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cFiltersBase;

interface

{$INCLUDE Compiler.inc}

uses SysUtils, Classes, cPCRE, uTools;

const
   FILTER_KEYWORD_UNLESS = 'unless';

type
  TFilterPatternBase = class
    SelectType : Char;    // '+' | '-' | ' '
    IsRegex    : Boolean;
    Pattern    : String;
  end;

  TFilterLineBase = class
    private
    protected
      fFilterPatterns: TList;
      DoMimeDecode : Boolean; // '~'?
      DoUnless     : Boolean; // 'unless'?
      LastSetError : String;
      function GetPatternCount: Integer;
      function GetPatternItem( Index: Integer ): TFilterPatternBase;
    public
      property  PatternCount: Integer read GetPatternCount;
      property  PatternItem[ Index: Integer ]: TFilterPatternBase read GetPatternItem;
      function  PatternAdd( Pat: TFilterPatternBase ): Integer;
      procedure PatternDelete( Index: Integer );
      function  SetFilterLine( FilterLine: String ): Boolean; virtual; abstract;
      function  AsString: String; virtual; abstract;
      procedure Clear;

      constructor Create;
      destructor Destroy; override;
  end;

  TFiltersBase = class
    protected
      fFilterFilename: String;
      fFilterFile : TStringList;
      fFilterLines: TList;
      RegExSection: TPCRE;
      RegExFilter : TPCRE;
      function GetLinesCount: Integer;
      function GetLinesItem( Index: Integer ): TFilterLineBase;
    public
      property  LinesCount: Integer read GetLinesCount;
      property  LinesItem[ Index: Integer ]: TFilterLineBase read GetLinesItem;

      function  LinesAdd( const LineText: String ): Integer; virtual; abstract;
      procedure LinesDelete( Index: Integer );
      procedure Clear;

      function  IsFilterLine( const Line: String ): Boolean; virtual; abstract;
      procedure SelectSections( const SectionIdentifier: String ); virtual;
      function  SelectedLines: String;

      constructor Create( const AFilterFilename: String );
      destructor Destroy; override;
  end;

function MatchSimpleString( const TestStr, Pattern : String ): Boolean;
function MatchSimpleNumber( const TestStr, Pattern : String ): Boolean;
function MatchSimple      ( const TestStr, Pattern : String ): Boolean;
function MatchPatterns    ( const TestStr, Patterns: String;
                            const RegEx: TPCRE ): Boolean;

implementation

uses uConst, cLogFileHamster;

// --------------------------------------------------------------- Tools ------

function MatchSimpleString( const TestStr, Pattern: String ): Boolean;
begin
     if Pattern='*' then begin // '*' matcht alles
        Result := True;
     end else begin
        Result := ( Pos( LowerCase(Pattern), LowerCase(TestStr) ) > 0 );
     end;
end;

function MatchSimpleNumber( const TestStr, Pattern: String ): Boolean;
var  iBase, iTest: Integer;
     vgl         : Char;
begin
     try
        if TestStr='' then iBase := 0
                      else iBase := strtoint( TestStr );
        If Length(Pattern)<2 then Abort;
        iTest := strtoint( copy(Pattern,2,9) );
        vgl := Pattern[1];

        Result := False;
        if (vgl='>') and (iBase>iTest) then Result:=True;
        if (vgl='<') and (iBase<iTest) then Result:=True;
        if (vgl='=') and (iBase=iTest) then Result:=True;
     except

        Result := False;
     end;
end;

function MatchSimple( const TestStr, Pattern: String ): Boolean;
begin
     if Pattern[1]='%' then begin
        Result := MatchSimpleNumber( TestStr, copy(Pattern,2,255) );
     end else begin
        if TestStr='' then begin
           if Pattern='*' then Result:=True else Result:=False;
        end else begin
           Result := MatchSimpleString( TestStr, Pattern );
        end;
     end;
end;

function MatchPatterns( const TestStr, Patterns: String; const RegEx: TPCRE ): Boolean;
const ONE=0; YES=1; NO=2;
var  NeedOneOf, HaveOneOf: Boolean;
     Matches : Boolean;
     MPatterns, Pattern: String;
     i: Integer;
     SelectType: Char;
begin
     Result := True;

     MPatterns := Patterns;
     NeedOneOf := False;
     HaveOneOf := False;
     Matches   := False;

     while MPatterns<>'' do begin
        MPatterns := TrimWhSpace( MPatterns );
        i := PosWhSpace( MPatterns );
        if i=0 then begin
           Pattern   := MPatterns;
           MPatterns := '';
        end else begin
           Pattern  := copy( MPatterns, 1, i-1 );
           System.Delete( MPatterns, 1, i );
        end;

        // Pattern:
        //    ['+'|'-']  '{'  regex-pattern   '}'
        //    ['+'|'-'] ['"'] simple-pattern ['"']

        SelectType := ' ';
        if (Pattern>'') and (Pattern[1] in ['+','-']) then begin
           SelectType := Pattern[1];
           System.Delete( Pattern, 1, 1 );
           if Pattern='' then Log( LOGID_ERROR,
                                   'Missing pattern after +/-:' + Patterns );
        end;

        if (SelectType<>' ') or not(HaveOneOf) then begin
           if copy(Pattern,1,1)='{' then begin
              if Pattern[length(Pattern)]='}' then System.Delete(Pattern,length(Pattern),1);
              System.Delete( Pattern, 1, 1 );
              try
                 Matches := RegEx.Match( PChar(Pattern), PChar(TestStr) );
              except
                 on E: Exception do begin
                    Log( LOGID_ERROR, 'Regex-error in {' + Pattern + '}:' + E.Message );
                    Matches := False;
                 end;
              end;
           end else begin
              if copy(Pattern,1,1)='"' then begin
                 System.Delete( Pattern, 1, 1 );
                 if Pattern[length(Pattern)]='"' then System.Delete(Pattern,length(Pattern),1);
              end;
              Matches := MatchSimpleString( TestStr, Pattern );
           end;

           case SelectType of
              '+': if not Matches then begin Result:=False; break; end;
              '-': if Matches     then begin Result:=False; break; end;
              ' ': begin
                      NeedOneOf := True;
                      if Matches then HaveOneOf:=True;
                   end;
           end;
        end;
     end;

     if NeedOneOf and not HaveOneOf then Result:=False;
end;

// -------------------------------------------------------- TFilterLine ------

function TFilterLineBase.GetPatternCount: Integer;
begin
     Result := fFilterPatterns.Count;
end;

function TFilterLineBase.GetPatternItem( Index: Integer ): TFilterPatternBase;
begin
     if (Index>=0) and (Index<PatternCount) then begin
        Result := fFilterPatterns[ Index ];
     end else begin
        Result := nil;
     end;
end;

function TFilterLineBase.PatternAdd( Pat: TFilterPatternBase ): Integer;
begin
     Result := fFilterPatterns.Add( Pat );
end;

procedure TFilterLineBase.PatternDelete( Index: Integer );
begin
     if (Index>=0) and (Index<PatternCount) then begin
        TFilterPatternBase( fFilterPatterns[ Index ] ).Free;
        fFilterPatterns.Delete( Index );
     end;
end;


procedure TFilterLineBase.Clear;
begin
     while PatternCount>0 do PatternDelete( PatternCount-1 );
end;

constructor TFilterLineBase.Create;
begin
     inherited Create;
     fFilterPatterns := TList.Create;
     Clear;
end;

destructor TFilterLineBase.Destroy;
begin
     Clear;
     fFilterPatterns.Free;
     inherited Destroy;
end;

// -------------------------------------------------------- TFiltersBase ------

function TFiltersBase.GetLinesCount: Integer;
begin
     Result := fFilterLines.Count;
end;

function TFiltersBase.GetLinesItem( Index: Integer ): TFilterLineBase;
begin
     if (Index>=0) and (Index<LinesCount) then begin
        Result := fFilterLines[ Index ];
     end else begin
        Result := nil;
     end;
end;

procedure TFiltersBase.LinesDelete( Index: Integer );
begin
     if (Index>=0) and (Index<LinesCount) then begin
        TFilterLineBase( fFilterLines[ Index ] ).Free;
        fFilterLines.Delete( Index );
     end;
end;

procedure TFiltersBase.Clear;
begin
     while LinesCount>0 do LinesDelete( LinesCount-1 );
end;

procedure TFiltersBase.SelectSections( const SectionIdentifier: String );
var  LineNo : Integer;
     LineTx : String;
     GroupOK: Boolean;

     procedure DoLine( LineText: String );
     var  j, k: Integer;
     begin
          if IsFilterLine( LineText ) then begin // filter-line
             if GroupOK then LinesAdd( LineText );
             exit;
          end;

          if (LineText>'') and (LineText[1]='[') then begin // group-selection
             System.Delete( LineText, 1, 1 );
             k := 0;
             for j:=1 to length(LineText) do begin
                case LineText[j] of
                   '{': inc(k);
                   '}': dec(k);
                   ']': if (k=0) then begin
                           LineText := copy( LineText, 1, j-1 );
                           break;
                        end;
                end;
             end;
             GroupOK := MatchPatterns( SectionIdentifier, LineText, RegExSection );
             exit;
          end;

          if (LineText > '') and (LineText[1] in ['#',';']) then exit; // comment

          Log( LOGID_WARN, 'Filterfile-line ignored: ' + LineText );
          Log( LOGID_WARN, 'Filterfile: ' + fFilterFilename );
          Log( LOGID_WARN, 'Reason: starts with invalid char' );
     end;

begin
     Clear;
     GroupOK := False;

     for LineNo:=0 to fFilterFile.Count-1 do begin
        LineTx := TrimWhSpace( fFilterFile[LineNo] );
        if LineTx <> '' then try
           DoLine( LineTx );
        except
           on E: Exception do begin
              Log( LOGID_WARN, 'Filterfile-line ignored: ' + LineTx );
              Log( LOGID_WARN, 'Filterfile: ' + fFilterFilename );
              Log( LOGID_WARN, 'Error: ' + E.Message );
           end;
        end;
     end;
end;

function TFiltersBase.SelectedLines: String;
var  LineNo: Integer;
begin
     Result := '';
     for LineNo:=0 to LinesCount-1 do begin
        Result := Result + LinesItem[LineNo].AsString + #13#10;
     end;
end;

constructor TFiltersBase.Create( const AFilterFilename: String );
begin
   inherited Create;

   fFilterFile  := TStringList.Create;
   fFilterLines := TList.Create;
   Clear;

   fFilterFilename := aFilterFilename;
   if fFilterFilename<>'' then begin
      try
         if FileExists( fFilterFilename ) then begin
            FFilterFile.LoadFromFile( fFilterFilename );
         end;
      except
         on E: Exception do begin
            Log( LOGID_WARN, 'Couldn''t load filter-file ' + fFilterFilename );
            Log( LOGID_WARN, 'Load-Error: ' + E.Message );
         end;
      end;
   end;

   RegExSection := TPCRE.Create( False, PCRE_CASELESS );
   RegExFilter  := TPCRE.Create( False, 0 );
end;

destructor TFiltersBase.Destroy;
begin
   Clear;
   if Assigned(fFilterFile ) then fFilterFile.Free;
   if Assigned(fFilterLines) then fFilterLines.Free;
   if Assigned(RegExFilter ) then RegExFilter.Free;
   if Assigned(RegExSection) then RegExSection.Free;
   inherited Destroy;
end;

end.
