// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cDnsResolver;

interface

{$INCLUDE Compiler.inc}
             
uses SysUtils, Classes, IdAssignedNumbers, IdDNSResolver;

type
   TDnsAutoDetect = ( dnsNoAutoDetect, dnsDetectPreferred, dnsDetectFallback );
   
   TDnsMxResolver = class 
      private
         FDnsList   : TStringList;
         FDnsAuto   : TDnsAutoDetect;
         FResolver  : TIdDNSResolver;
         FMXRecords : TStringList;
         FLimitDNS  : Integer;
         FLimitMX   : Integer;

         procedure DoPrepareDnsList( const ADnsServers: String );
         procedure DoResolve( const Domain       : String;
                              var   NameHost, AIP: String;
                              const QueryType    : TQueryType );
         function  DoQueryMXRecords( const DnsServer: String;
                                     const DnsPort  : Integer;
                                     const Domain: String ): Boolean;

      public
         property MXRecords: TStringList read FMXRecords;

         function QueryMXRecords( const Domain: String ): Boolean;

         constructor Create( const ADnsAuto   : TDnsAutoDetect = dnsDetectFallback;
                             const ADnsServers: String  = '';
                             const ADnsTimeout: Integer = 10000;
                             const ALimitDNS  : Integer = 3;
                             const ALimitMX   : Integer = 3 );
         destructor Destroy; override;
   end;


implementation

uses TypInfo, uConst, uTools, uIpHelper, cLogFileHamster;

// ------------------------------------------------------- TDnsMxResolver -----

constructor TDnsMxResolver.Create( const ADnsAuto   : TDnsAutoDetect = dnsDetectFallback;
                                   const ADnsServers: String  = '';
                                   const ADnsTimeout: Integer = 10000;
                                   const ALimitDNS  : Integer = 3;
                                   const ALimitMX   : Integer = 3 );
begin
   inherited Create;

   FResolver := TIdDNSResolver.Create( nil );
   // FResolver.Host := ADnsServer;
   // FResolver.Port := IdPORT_DOMAIN;
   FResolver.ReceiveTimeout := ADnsTimeout;

   FLimitDNS  := ALimitDNS;
   FLimitMX   := ALimitMX;
   FMXRecords := TStringList.Create;
   FDnsList   := TStringList.Create;
   FDnsAuto   := ADnsAuto;
   DoPrepareDnsList( ADnsServers );
end;

destructor TDnsMxResolver.Destroy;
begin
   if Assigned( FDnsList   ) then FDnsList.Free;
   if Assigned( FResolver  ) then FResolver.Free;
   if Assigned( FMXRecords ) then FMXRecords.Free;

   inherited Destroy;
end;

procedure TDnsMxResolver.DoPrepareDnsList( const ADnsServers: String );
// prepare list of all DNS servers (given and/or auto-detected ones)

   function DnsIndexOf( const Server: String; const Port: Integer ): Integer;
   var  i: Integer;
   begin
      Result := -1;
      for i := 0 to FDnsList.Count - 1 do begin
         if FDnsList[i] = Server then begin
            if Integer( FDnsList.Objects[i] ) = Port then begin
               Result := i;
               break;
            end;
         end;
      end;
   end;

var  Lst, Tmp, Srv, Prt: String;
     SL: TStringList;
     i, Port: Integer;
begin
   FDnsList.Clear;

   SL := TStringList.Create;
   try

      // add given DNS servers
      Lst := TrimWhSpace( ADnsServers );
      repeat
         Tmp := NextWhSpacePart( Lst );
         Srv := NextSepPart( Tmp, ':' );
         Prt := NextSepPart( Tmp, ':' );
         if Srv = '' then break;

         if Prt = '' then Port := IdPORT_DOMAIN
                     else Port := strtointdef( Prt, IdPORT_DOMAIN );

         if DnsIndexOf( Srv, Port ) < 0 then begin
            FDnsList.AddObject( Srv, Pointer(Port) );
         end;
      until False;

      // auto-detect DNS servers and add them
      if FDnsAuto <> dnsNoAutoDetect then GetKnownDnsServerList( SL );
      for i := 0 to SL.Count - 1 do begin
         Srv  := SL[i];
         Port := IdPORT_DOMAIN;

         if FDnsAuto = dnsDetectPreferred then begin

            // remove a given duplicate
            if DnsIndexOf( Srv, Port ) > i then begin
               FDnsList.Delete( DnsIndexOf( Srv, Port ) );
            end;
            
            // add auto-detected DNS server at top of list
            FDnsList.InsertObject( i, Srv, Pointer(Port) );

         end else if FDnsAuto = dnsDetectFallback then begin

            // add auto-detected DNS server at end of list
            if DnsIndexOf( Srv, Port ) < 0 then begin
               FDnsList.AddObject( Srv, Pointer(Port) );
            end;

         end;
      end;

   finally SL.Free end;

   // limit number of DNS servers to query
   if FLimitDNS > 0 then begin
      while FDnsList.Count > FLimitDNS do FDnsList.Delete( FDnsList.Count-1 );
   end;
end;

procedure TDnsMxResolver.DoResolve( const Domain       : String;
                                    var   NameHost, AIP: String;
                                    const QueryType    : TQueryType );
// get MX records for given Domain; also return first found CNAME and A entry
var  i: Integer;
     s: String;
     r: TResultRecord;
begin
   FMXRecords.Clear;

   try
      FResolver.QueryRecords := QueryType;
      FResolver.Resolve( Domain );

      for i := 0 to FResolver.QueryResult.Count-1 do begin
         r := FResolver.QueryResult.Items[i];

         try
            s := GetEnumName( TypeInfo(TQueryRecordTypes), ord(r.RecType) );
         except
            s := 'unknown_rectype';
         end;
         if copy( s, 1, 2 ) = 'qt' then System.Delete( s, 1, 2 );
         s := r.Name + ' ' + s;

         case r.RecType of
            qtMX:    with TMXRecord(r) do begin
                        FMXRecords.AddObject(
                           ExchangeServer,
                           Pointer(
                              // sort key: 1.) Preference 2.) Random number
                              ( (Preference and $7FFFFF) shl 8 ) or Random($FF)
                           )
                        );
                        s := s + ' '+ExchangeServer + ' '+inttostr(Preference);
                     end;
            qtNAME:  with TNAMERecord(r) do begin
                        if NameHost = '' then NameHost := HostName;
                        s := s + ' ' + HostName;
                     end;
            qtA:     with TARecord(r) do begin
                        if AIP = '' then AIP := IPAddress;
                        s := s + ' ' + IPAddress;
                     end;
            qtNS:    with TNSRecord(r) do begin
                        s := s + ' ' + HostName;
                     end;
            qtSOA:   with TSOARecord(r) do begin
                        s := s + ' ' + Primary + '/' + ResponsiblePerson;
                     end;
         end;

         Log( LOGID_DEBUG, Format( '[DNS MX] Reply: %s', [ s ] ) );

      end;

   except
      on E: Exception do begin
         if FMXRecords.Count > 0 then i := LOGID_DEBUG else i := LOGID_WARN;
         Log( i, Format( '[DNS MX] Error: %s (server %s, domain %s)',
                         [ E.Message, FResolver.Host, Domain ] ) );
      end;
   end;
end;

function CompareByObjectAsInt( List: TStringList;
                               Index1, Index2: Integer ): Integer;
begin
   Result := Integer( List.Objects[Index1] )
           - Integer( List.Objects[Index2] );
end;

function TDnsMxResolver.DoQueryMXRecords( const DnsServer: String;
                                          const DnsPort  : Integer;
                                          const Domain   : String ): Boolean;
// Fill MXRecords with MX servers for given domain.
// Returns True if at least one was found.
var  NameHost, AIP: String;
begin
   Result := False;
   if DnsServer = '' then exit;

   // prepare resolver
   with FResolver do begin
      Host := DnsServer;
      Port := DnsPort;
      Log( LOGID_DETAIL,
           Format( '[DNS MX] Query: %s (port %d, timeout %d) for MX of %s',
                   [ Host, Port, ReceiveTimeout, Domain ] ) );
   end;

   // resolve domain
   NameHost := '';
   AIP      := '';
   DoResolve( Domain, NameHost, AIP, [ qtMX ] );

   if (FMXRecords.Count = 0) and (NameHost <> '') then begin // try CNAME
      Log( LOGID_DEBUG, Format( '[DNS MX] Query: %s for MX of CNAME %s',
                                [ FResolver.Host, NameHost ] ) );
      DoResolve( NameHost, NameHost, AIP, [ qtMX ] );
   end;
   
   if (FMXRecords.Count = 0) and (AIP = '') then  begin // try A    
      Log( LOGID_DEBUG, Format( '[DNS MX] Query: %s for A of %s',
                                [ FResolver.Host, Domain ] ) );
      DoResolve( Domain, NameHost, AIP, [ qtA ] );
   end;

   if (FMXRecords.Count = 0) and (AIP <> '') then begin // use A
      FMXRecords.AddObject( AIP, Pointer(0) );
   end;

   // remove own FQDN from list
   (*
   if (FOwnFQDN <> '') and (FMXRecords.Count > 1) then begin
      i := FMXRecords.IndexOf( FOwnFQDN );
      if i > 0 then begin
         FMXRecords.Delete( i );
         Log( LOGID_DEBUG, Format( '[DNS MX] Removed own FQDN %s from results',
                                   [ FOwnFQDN ] ) );
      end;
   end;
   *)

   // sort list by "randomized preference"
   if FMXRecords.Count > 1 then FMXRecords.CustomSort( CompareByObjectAsInt );

   // limit no. of MX to return
   if FLimitMX > 0 then begin
      while FMXRecords.Count > FLimitMX do begin
         FMXRecords.Delete( FMXRecords.Count-1 );
      end;
   end;

   // final result
   Result := ( FMXRecords.Count > 0 );
   if Result then begin
      Log( LOGID_DETAIL,
           Format( '[DNS MX] Result: %s -> %d found, %s (preferred)',
                   [ Domain, FMXRecords.Count, FMXRecords[0] ] ) );
   end else begin
      Log( LOGID_DETAIL,
           Format( '[DNS MX] No MX server found for %s', [Domain] ) );
   end;
end;

function TDnsMxResolver.QueryMXRecords( const Domain: String ): Boolean;
// Fill MXRecords with MX servers for given domain.
// Returns True if at least one was found.
var  i: Integer;
begin
   Result := False;
   FMXRecords.Clear;

   if FDnsList.Count = 0 then begin

      // no DNS server
      if FDnsAuto = dnsNoAutoDetect then
         Log( LOGID_WARN, '[DNS MX] No DNS server given!' )
      else
         Log( LOGID_WARN, '[DNS MX] No DNS server found!' );

   end else begin

      // try all given/detected DNS servers until we have a MX server
      for i := 0 to FDnsList.Count - 1 do begin
         Result := DoQueryMXRecords(
                      FDnsList[i], Integer(FDnsList.Objects[i]), Domain
                   );
         if Result then break;
      end;

      if not Result then begin
         Log( LOGID_WARN,
              Format( '[DNS MX] No MX servers found for %s', [Domain] ) );
      end;

   end;
end;

initialization
   Randomize;
   
end.
