// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cClientSMTP;

interface

{$INCLUDE Compiler.inc}

uses Classes, uType, cClientBase;

type
   TSendMailResult = ( SMR_INPROGRESS, SMR_DELIVERED, SMR_TRYAGAINLATER,
                       SMR_ABORTED, SMR_DISCONNECT );

type
   TClientSocketSMTP = class( TClientBase )
      protected
         function  HResultListFollows: Boolean; override;
         procedure HResultLineToCode; override;
      public
         EHLOResult: String;
         function  HLogin( AServerAlias, AUser, APass: String ): Boolean; override;
   end;

   TClientSMTP = class( TClientProtocolBase )
      private
         FMXMode: Boolean;
         ServerAlias, Server, Port, User, Pass: String;
         ServerDir: String;
         SMTP: TClientSocketSMTP;
         LastSMResult: TSendMailResult;
         FailNotifyRcpt, FailNotifyText: String;

         function DoSendMailfile( const MailOutFile: String;
                                  out   MailText   : String ): TSendMailResult;
         function DoDeliveryFailed( const MailFile: String ): Boolean;

      public
         procedure Terminate;
         procedure Connect;
         procedure Disconnect;
         function  Connected: Boolean;

         function  SendMailfile( const MailFile: String;
                                 const RaiseErrorCounter: Boolean = True ): Boolean;
         function  SendMailfileMX( const MailFile: String ): Boolean;

         constructor Create( AReportSubState: TReportSubStateInfo;
                             const AMXMode: Boolean;
                             const AServerAlias, AServer, APort, AUser, APass: String );
         destructor Destroy; override;
   end;

implementation

uses uConst, uConstVar, uVar, SysUtils, uTools, uWinSock, cArticle, uDateTime,
     uEncoding, uSASL, cLogFileHamster, uHamTools, cHamster, cMailDispatcher,
     IdAssignedNumbers, cSmtpRouter, IdSSLOpenSSL;

procedure TClientSocketSMTP.HResultLineToCode;
begin
   if FResultLine='' then FResultLine := '999 (timeout or connection lost)';
   FResultCode := strtoint( copy( FResultLine, 1, 3 ) );
   if not HConnected then FResultCode := 999;
end;

function TClientSocketSMTP.HResultListFollows: Boolean;
begin
   Result := False;
end;

function TClientSocketSMTP.HLogin( AServerAlias, AUser, APass: String ): Boolean;
var  HeloName: String;
     UseAUTH, SslOk: Boolean;
     TS: TStringList;
     SASL_REMOTE, SASL_WANTED, s: String;
     LfdServer, i: Integer;
begin
   Result := False;
   if not HConnected then exit;

   try
      Log( LOGID_DEBUG, Self.Classname + '.Login' );
      EHLOResult := '';

      HeloName := Hamster.Config.Settings.GetStr(hsFQDNforMID);
      if HeloName='' then HeloName := LookupLocalHostName;
      if not IsDomain(HeloName) then HeloName := 'localhost';

      UseAUTH     := False;
      SASL_WANTED := '';
      if (AUser<>'') and (APass<>'') then begin
         Hamster.Config.BeginRead;
         try
            with Hamster.Config.SmtpServers do begin
               LfdServer := IndexOfAlias( AServerAlias, True );
               if LfdServer >= 0 then begin
                  UseAUTH     := Settings( LfdServer ).GetBoo( ssSmtpUseAuth );
                  SASL_WANTED := Settings( LfdServer ).GetStr( ssSmtpUseSASL );
               end;
            end;
         finally Hamster.Config.EndRead end;
      end;

      if UseAUTH then begin

         HRequest( 'EHLO ' + HeloName, 0, False );
         if not HConnected then exit;
         if ResultCode = 250 then EHLOResult := ResultLine // CRLF-separated
                             else UseAUTH := False;     

      end;

      // SSL-TLS handshake
      if FUseSslMode = sslTLS then begin

         // determine if server supports TLS handshake
         SslOk := False;

         if EHLOResult <> '' then begin
            SslOk := pos( 'starttls' + CRLF, LowerCase(EHLOResult) ) > 0;
         end;

         if SslOk then begin

            // Handle TLS handshake
            s := StartTlsHandshake( 'STARTTLS', 220, 'EHLO ' + HeloName, 250, False );
            if s = '' then begin
               Log( LOGID_ERROR,
                    'Required SSL with TLS handshake was not activated!' );
               exit;
            end;

            EHLOResult := s;

         end else begin

            Log( LOGID_ERROR,
                 FServer + ' does not support required TLS handshake!' );
            exit;

         end;

      end;

      if UseAUTH then begin

         // get SASL mechanisms supported by server
         SASL_REMOTE := '';
         TS := TStringList.Create;
         TS.Text := EHLOResult;
         for i:=0 to TS.Count-1 do begin
            s := UpperCase( copy( TS[i], 5, 999 ) ); // skip '250[- ]'
            if (copy(s,1,4)='AUTH') and (length(s)>5) and (s[5] in [' ','=']) then begin
               SASL_REMOTE := TrimWhSpace( copy( s, 6, 999 ) ); // skip 'AUTH[ =]'
               break;
            end;
         end;
         TS.Free;
         if SASL_REMOTE='' then begin
            Log( LOGID_ERROR, 'SMTP ' + FServer + ': Missing AUTH in EHLO-reply!' );
            Log( LOGID_ERROR, 'SMTP ' + FServer + ': ' + ResultLine );
            exit;
         end;

         if not Remote_SASL_Login(
                   Self, 'AUTH %s %s', 334, 235, 5,
                   SASL_WANTED, SASL_REMOTE, AUser, APass ) then exit;

      end else begin

         HRequest( 'HELO ' + HeloName, 0, False );
         if not HConnected then exit;
         if ResultCode <> 250 then exit;

      end;

      Result := HConnected;

   finally
      if (not Result) and HConnected then HDisconnect;
   end;
end;

procedure TClientSMTP.Terminate;
begin
   if Assigned( SMTP ) then SMTP.Terminate;
end;

function TClientSMTP.Connected: Boolean;
begin
   if Assigned( SMTP ) then Result := SMTP.HConnected else Result := False;
end;

procedure TClientSMTP.Connect;
var  s: String;
     UseSOCKS: Boolean;
     UseSslMode: TSslTlsMode;
     LfdServer, TimeoutConnect, TimeoutCommand, i: Integer;
     SSLVersion: TIdSSLVersion;
     SSLCipher : String;
begin
   if Assigned( SMTP ) then Disconnect;

   if (not FMXMode) and (ServerDir = '') then begin
      Log( LOGID_ERROR, 'SMTP server "' + ServerAlias + '" is not configured!' );
      exit;
   end;

   TimeoutConnect := Hamster.Config.Settings.GetInt(hsRemoteTimeoutConnect);
   TimeoutCommand := Hamster.Config.Settings.GetInt(hsRemoteTimeoutCommand);
   UseSOCKS := FMXMode;
   UseSslMode := sslNone;
   SSLVersion := sslvSSLv3;
   SSLCipher  := '';

   Hamster.Config.BeginRead;
   try
      with Hamster.Config.SmtpServers do begin
         LfdServer := IndexOfAlias( ServerAlias, FMXMode );
         if LfdServer >= 0 then begin
            i := Settings( LfdServer ).GetInt( ssTimeoutConnect );
            if i > 0 then TimeoutConnect := i;
            i := Settings( LfdServer ).GetInt( ssTimeoutCommand );
            if i > 0 then TimeoutCommand := i;
            UseSOCKS := Settings( LfdServer ).GetBoo( ssUseSocks );
            UseSslMode := TSslTlsMode( Settings(LfdServer).GetInt(ssSSLMode) );
            SSLVersion := TIdSSLVersion( Settings(LfdServer).GetInt(ssSSLVersion) );
            SSLCipher  := Settings( LfdServer ).GetStr( ssSSLCipher );
         end;
      end;
   finally Hamster.Config.EndRead end;

   SMTP := TClientSocketSMTP.Create( nil, 'smtp' );
   SMTP.AcceptSmtpMultiline := True;

   if SMTP.HConnect( Server, Port, UseSOCKS, UseSslMode, SSLVersion, SSLCipher,
                     0, TimeoutConnect*1000, TimeoutCommand*1000 ) then begin
      if not SMTP.HLogin( ServerAlias, User, Pass ) then begin
         s := SMTP.LastErrResult;
         if s = '' then s := SMTP.ResultLine;
         if s <> '' then s := ' ("' + s + '")';
         Log( LOGID_WARN, 'Login failed ' + s );
      end;
      if (not FMXMode) and (SMTP.Greeting <> '') then begin
         HamFileRewriteLine( ServerDir + SRVFILE_GREETING, SMTP.Greeting );
      end;
   end;
end;

procedure TClientSMTP.Disconnect;
begin
   if not Assigned( SMTP ) then exit;
   try SMTP.HDisconnect except end;
   FreeAndNil( SMTP );
end;

function TClientSMTP.DoSendMailfile( const MailOutFile: String;
                                     out   MailText: String ): TSendMailResult;
var  CmdLine, Cmd, CmdReplies, MailFrom, s : String;
     PreData, HadValidRcpt                 : Boolean;
     LineNo, i                             : Integer;
     TheMail                               : TStringList;
     LastProgress                          : Integer;

     function TryToReset: Boolean;
     begin
        Result := False;
        SMTP.HRequest( 'RSET', 0, False );
        if SMTP.ResultCode=250 then Result:=True;
     end;

begin
   Result := SMR_DISCONNECT;
   MailText := '';

   if not FileExists( MailOutFile ) then exit;
   if not Connected then exit;

   Log( LOGID_DETAIL, '[' + Server + ', ' + Port + '] SendMail '
                          + ExtractFilename(MailOutFile) );

   TheMail := TStringList.Create;
   try

      TheMail.LoadFromFile( MailOutFile );
      MailText := TheMail.Text;

      PreData      := True;
      LineNo       := 0;
      MailFrom     := '';
      HadValidRcpt := False;
      CmdReplies   := '';

      Result := SMR_INPROGRESS;

      LastProgress := -1; 

      while LineNo < TheMail.Count do begin

         if Result=SMR_ABORTED    then break;
         if Result=SMR_DISCONNECT then break;
         if not Connected then break;

         s := TheMail[ LineNo ];

         if PreData and (copy(s,1,1)='!') then begin

            CmdLine := s;
            System.Delete( CmdLine, 1, 1 );

            i := PosWhSpace( CmdLine );
            if i=0 then Cmd:=CmdLine else Cmd:=copy(CmdLine,1,i-1);
            Cmd := UpperCase( Cmd );

            if (Cmd='MAIL') or (Cmd='RCPT') then begin

               i := Pos( ':', CmdLine );
               if i > 0 then begin
                  CmdLine := copy( CmdLine, 1, i )
                           + Trim( copy( CmdLine, i+1, MaxInt ) );
               end;

               CmdReplies := CmdReplies + '| ' + CmdLine + #13#10;
               SMTP.HRequest( CmdLine, 0, False );

               if (SMTP.ResultLine='') or (SMTP.ResultCode=999) then begin // Timeout
                  CmdReplies := CmdReplies + '| -> (timeout)' + #13#10;
                  Log( LOGID_ERROR, 'SendMail aborted: '
                                    + CmdLine + ' -> ' + 'Timeout (' + Server + ')' );
                  Result := SMR_DISCONNECT;
                  break;
               end;

               CmdReplies := CmdReplies + '| -> ' + SMTP.ResultLine + #13#10;

               case SMTP.ResultCode of

                  250, 251: begin // ok
                     // 250 Requested mail action okay, completed
                     // 251 User not local; will forward to <forward-path>
                     if Cmd='MAIL' then begin
                        MailFrom := TrimWhSpace(copy(CmdLine,11,99));
                     end;
                     if Cmd='RCPT' then begin
                        // mark recipient as delivered
                        TheMail[ LineNo ] := '!X-OK: ' + CmdLine;
                        HadValidRcpt := True;
                     end;
                  end;

                  450, 451, 452, 500, 501, 550, 551, 552, 553: begin // delivery failed
                     // 450 Requested mail action not taken: mailbox unavailable
                     // 451 Requested action aborted: local error in processing
                     // 452 Requested action not taken: insufficient system storage
                     // 500 Syntax error, command unrecognized
                     // 501 Syntax error in parameters or arguments
                     // 550 Requested action not taken: mailbox unavailable
                     //   | Unknown local part
                     //   | relaying to <x@y> prohibited by administrator
                     // 551 User not local; please try <forward-path>
                     // 552 Requested mail action aborted: exceeded storage allocation
                     // 553 Requested action not taken: mailbox name not allowed
                     if Cmd='MAIL' then begin
                        Log( LOGID_ERROR, 'SendMail aborted: '
                                          + CmdLine + ' -> ' + SMTP.ResultLine );
                        if TryToReset then Result:=SMR_ABORTED else Result:=SMR_DISCONNECT;
                        break;
                     end;
                     if Cmd='RCPT' then begin
                        Log( LOGID_WARN, 'SendMail failed: '
                                         + CmdLine + ' -> ' + SMTP.ResultLine );
                        Result := SMR_TRYAGAINLATER;
                     end;
                  end;

                  else begin
                     Log( LOGID_ERROR, 'SendMail aborted: '
                                       + CmdLine + ' -> ' + SMTP.ResultLine );
                     if TryToReset then Result:=SMR_ABORTED else Result:=SMR_DISCONNECT;
                     break;
                  end;

               end; // case SMTP.ResultCode of

            end; // if (Cmd='MAIL') or (Cmd='RCPT')

         end else begin

            if PreData then begin

               // first line after envelope-headers

               if MailFrom='' then begin
                  Log( LOGID_ERROR, 'SendMail aborted: No valid envelope-from!' );
                  if TryToReset then Result:=SMR_ABORTED else Result:=SMR_DISCONNECT;
                  break;
               end;
               if not(HadValidRcpt) then begin
                  Log( LOGID_ERROR, 'SendMail aborted: No valid recipients!' );
                  if TryToReset then Result:=SMR_ABORTED else Result:=SMR_DISCONNECT;
                  break;
               end;

               CmdReplies := CmdReplies + '| DATA' + #13#10;
               SMTP.HRequest( 'DATA', 0, False );

               if (SMTP.ResultLine='') or (SMTP.ResultCode=999) then begin // Timeout
                  CmdReplies := CmdReplies + '| -> (timeout)' + #13#10;
                  Log( LOGID_ERROR, 'SendMail aborted: '
                                    + 'DATA' + ' -> ' + 'Timeout (' + Server + ')' );
                  Result := SMR_DISCONNECT;
                  break;
               end;

               CmdReplies := CmdReplies + '| -> ' + SMTP.ResultLine + #13#10;
               if SMTP.ResultCode<>354 then begin
                  Log( LOGID_WARN, 'SendMail failed: '
                                   + 'DATA' + ' -> ' + SMTP.ResultLine );
                  if TryToReset then Result:=SMR_ABORTED else Result:=SMR_DISCONNECT;
                  break;
               end;

               PreData := False;

            end;

            SMTP.HWriteLnQ( s );

            if TheMail.Count > 0  then begin
               if trunc( LineNo / TheMail.Count * 100 ) <> LastProgress then begin
                 LastProgress := trunc( LineNo / TheMail.Count * 100 );
                 s := 'to ' + Server + ' - ' + inttostr(LastProgress) + '%';
                 Log( LOGID_STATUS, s );
                 ReportSubState( s );
               end;
            end;

          end;

         inc( LineNo );
      end;

      if Connected then begin
         if (Result<>SMR_ABORTED) and (Result<>SMR_DISCONNECT) then begin
            if not PreData then begin
               CmdReplies := CmdReplies + '| "." (end of data)' + #13#10;
               SMTP.HRequest( '.', 0, False ); // send "end of data"

               if (SMTP.ResultLine='') or (SMTP.ResultCode=999) then begin // Timeout
                  CmdReplies := CmdReplies + '| -> (timeout)' + #13#10;
                  Log( LOGID_ERROR, 'SendMail aborted: '
                                    + '"." (end of data)' + ' -> ' + 'Timeout (' + Server + ')' );
                  Result := SMR_DISCONNECT;
               end else begin
                  CmdReplies := CmdReplies + '| -> ' + SMTP.ResultLine + #13#10;
                  if SMTP.ResultCode=250 then begin
                     if Result=SMR_INPROGRESS then Result:=SMR_DELIVERED;
                     if (Result=SMR_TRYAGAINLATER) and HadValidRcpt then begin
                        // mark successful recipients as done in envelope-headers
                        TheMail.SaveToFile( MailOutFile );
                     end;
                  end else begin
                     Log( LOGID_WARN, 'SendMail failed: '
                                      + '"." (end of data)' + ' -> ' + SMTP.ResultLine );
                  end;
               end;
            end;
         end;
      end;

      // notify sender of errors
      if Result <> SMR_DELIVERED then begin
         s := '[Hamster]' + #13#10#13#10
            + 'The following mail could not be delivered to any or all '
            + 'recipients:' + #13#10#13#10
            + '> ' + MailOutFile + #13#10#13#10
            + 'Delivery-results:' + #13#10#13#10
            + CmdReplies + #13#10
            + 'Header-lines of undelivered mail:' + #13#10#13#10;
         for i:=0 to TheMail.Count-1 do begin
            if TheMail[i] = '' then break;
            s := s + '| ' + TheMail[i] + #13#10;
         end;
         FailNotifyRcpt := MailFrom;
         FailNotifyText := s;
         // SendLocalInfoMail( MailFrom, '[Hamster] Mail-delivery failed!', s );
         // Note: Now delayed until DoDeliveryFailed() because of multiple MX
         //       attempts and fallback servers. So notification will only be
         //       sent if mail finally did not get out in current run.
      end;

   finally TheMail.Free end;
end;

function TClientSMTP.DoDeliveryFailed( const MailFile: String ): Boolean;
var  MailPath, MailName, s: String;
     Attempts, i: Integer;
begin
   // notify sender of errors
   if (FailNotifyRcpt<>'') and (FailNotifyText<>'') then begin
      SendLocalInfoMail( FailNotifyRcpt,
                         '[Hamster] Mail-delivery failed!', FailNotifyText );
   end;

   // Mail could not be delivered to any or all recipients, so keep the
   // mail in outbox to allow corrections in it and/or to try it again
   // later.
   // After a configurable number of (unsuccessful) attempts, the mailfile
   // is renamed to "*.err", so it's not sent again without prior user-
   // interaction.
   //
   // Note: Successful recipients were marked as such in the mailfile, so
   //       they won't get the mail again and again.
   // Note: Sender (or at least "admin") already got an error-notification.

   Result := False; // = try again next time

   MailPath := ExtractFilePath( MailFile );
   MailName := ExtractFilename( MailFile );
   i := Pos( '.msg', lowercase(MailName) );
   if i > 0 then MailName := copy( MailName, 1, i-1 );

   i := Pos( '-', MailName );
   if i = 0 then begin
      Attempts := 0;
   end else begin
      Attempts := strtoint( copy( MailName, i+1, 3 ) );
      MailName := copy( MailName, 1, i-1 );
   end;

   inc( Attempts );

   if Attempts >= Hamster.Config.Settings.GetInt(hsSendMailAttemptsMax) then begin

      // maximum number of attempts reached

      Result := True; // = do not try again next time

      if Hamster.Config.Settings.GetBoo(hsSendMailAttemptsDel) then begin

         // delete file to prevent from further attempts
         DeleteFile( MailFile );

      end else begin

         // rename file to prevent from further attempts
         MailName := MailName + '-' + inttostr(Attempts-1);
         RenameFile( MailFile, MailPath + MailName + '.err' );

         // notify admin about orphaned file
         s := '[Hamster]' + #13#10#13#10
            + 'After ' + inttostr(Attempts) + ' attempts, an undeliverable '
            + 'mail was renamed to:' + #13#10#13#10
            + '> ' + MailPath + MailName + '.err' + #13#10#13#10
            + 'This file will NOT be deleted or sent out again by Hamster.' + #13#10;
         SendLocalInfoMail( 'admin', '[Hamster] Mail-delivery aborted!', s );
         
      end;

   end else begin

      // mark file with new attempts-count
      MailName := MailName + '-' + inttostr(Attempts);
      RenameFile( MailFile, MailPath + MailName + '.msg' );

   end;
end;

function TClientSMTP.SendMailfile( const MailFile: String;
                                   const RaiseErrorCounter: Boolean = True ): Boolean;
var  ResLog, MailText, MsgName, MsgValue, Envelope, s: String;
     LogNow: TDateTime;
     Msg: TMess;
     Index: Integer;
begin
   Result := True; // =continue

   FailNotifyRcpt := '';
   FailNotifyText := '';
   LastSMResult := DoSendMailfile( MailFile, MailText );
   if LastSMResult = SMR_INPROGRESS then LastSMResult := SMR_DISCONNECT;

   case LastSMResult of
      SMR_DELIVERED    : ResLog:='250 OK.';
      SMR_TRYAGAINLATER: ResLog:='991 Invalid recipient.';
      SMR_ABORTED      : ResLog:='992 Aborted due to errors.';
      SMR_DISCONNECT   : ResLog:='993 Timeout or unrecoverable error.';
   end;

   if LastSMResult = SMR_DELIVERED then begin

      // Mail could be delivered to all recipients, so delete it:

      Log( LOGID_INFO, Format('Mail %s sent: %s',
                       [ ExtractFilename(MailFile), SMTP.ResultLine ] ) );
      DeleteFile( MailFile );

   end else begin

      // Mail could not be delivered to any or all recipients. Either keep
      // it for additional attempts or delete it after final attempt.

      if RaiseErrorCounter then begin
         if DoDeliveryFailed( MailFile ) then begin
            ResLog := '999 Delivery failed; no further attempts.';
         end;
      end;

      if LastSMResult = SMR_DISCONNECT then begin
         // Delivery failed with a fatal error (e.g. timeout) or with server
         // left in an unrecoverable, undefined state.
         Result := False;
         if Connected then SMTP.HDisconnect;
         exit;
      end;

   end;

   // Note result in mailout.log:
   LogNow := Now;
   Msg := TMess.Create;
   try
      Msg.FullText := MailText;

      Index := 0;
      Envelope := '';
      while Msg.NextHeader( Index, MsgName, MsgValue ) do begin
         if copy( MsgName, 1, 1 ) <> '!' then break;
         if length(Envelope) > 0 then Envelope := Envelope + ', ';
         if CompareText( MsgName, '!mail from' ) = 0 then MsgName := 'From';
         if CompareText( MsgName, '!rcpt to'   ) = 0 then MsgName := 'To';
         Envelope := Envelope + MsgName + ': ' + MsgValue;
      end;

      s := DateTimeToLogTime( LogNow )
         + #9 + 'File='       + ExtractFilename( MailFile )
         + #9 + 'Server='     + ServerAlias
         + #9 + 'Result='     + Logify( ResLog )
         + #9 + 'Envelope='   + Logify( Envelope )
         + #9 + 'From='       + Logify( Msg.HeaderValueByNameSL(HDR_NAME_FROM) )
         + #9 + 'To='         + Logify( Msg.HeaderValueByNameSL(HDR_NAME_TO) )
         + #9 + 'Subject='    + Logify( Msg.HeaderValueByNameSL(HDR_NAME_SUBJECT) )
         ;
   finally Msg.Free end;
   HamFileAppendLine( AppSettings.GetStr(asPathLogs)
                      + LOGFILE_MAILOUT
                      + FormatDateTime( '"-"yyyy"-"mm', LogNow )
                      + LOGFILE_EXTENSION,
                      s );
end;

function TClientSMTP.SendMailfileMX( const MailFile: String ): Boolean;
var  TheMail: TStringList;
     DestRcptTo, DestDomain: String;
     i, k: Integer;
     s: String;
     SmtpRouter: TSmtpRouter;
begin
   Result       := False;
   LastSMResult := SMR_ABORTED;
   ServerAlias  := '';
   Server       := '';
   Port         := '';
   User         := '';
   Pass         := '';

   // load mail and get its (first and only) recipient
   TheMail := TStringList.Create;
   DestRcptTo := '';
   DestDomain := '';
   try
      TheMail.LoadFromFile( MailFile );
      for i := 0 to TheMail.Count - 1 do begin
         s := TheMail[i];
         if copy( s, 1, 1 ) <> '!' then break;

         if UpperCase( copy(s,1,9) ) = '!RCPT TO:' then begin
            DestRcptTo := TrimWhSpace( copy( s, 10, MaxInt ) );
            k := Pos( '@', DestRcptTo );
            if k > 0 then DestDomain := copy( DestRcptTo, k+1, MaxInt );
            k := Pos( '>', DestDomain );
            if k > 0 then SetLength( DestDomain, k-1 );
            break;
         end;
      end;
   finally TheMail.Free end;

   if DestDomain = '' then begin
      Log( LOGID_WARN, 'No valid recipient found in mail: ' + MailFile );
      exit;
   end;

   // get delivery information for recipient's domain
   SmtpRouter := Hamster.SmtpRouter.CreateSmtpRouter( DestDomain );
   try

      if SmtpRouter.OverrideServer <> '' then begin

         // send to specific server that was chosen in config file
         ServerAlias := SmtpRouter.OverrideServer;
         i := Hamster.Config.SmtpServers.IndexOfAlias( ServerAlias );
         if i >= 0 then begin

            Server      := Hamster.Config.SmtpServers.SrvName[ i ];
            Port        := Hamster.Config.SmtpServers.SrvPort[ i ];
            User        := Hamster.Config.SmtpServers.SrvUser[ i ];
            Pass        := Hamster.Config.SmtpServers.SrvPass[ i ];

            // send mail
            Log( LOGID_INFO,
                 Format( '[MX-override %s, %s] domain %s, mail %s',
                 [ Server, Port, DestDomain, ExtractFilename(MailFile) ] ));
            Connect;
            try
               Result := SendMailfile( MailFile, False );
            finally
               try Disconnect except end;
            end;
            
         end else begin
            Log( LOGID_WARN, Format( 'Unknown override server for mail "%s"!',
                                     [ MailFile ] ) );
         end;

      end else if SmtpRouter.MXServers.Count = 0 then begin

         // we should send to MX servers, but we have not found any
         Log( LOGID_WARN,
              Format( 'Could not deliver mail "%s": No mail exchange server '
                    + 'found for domain "%s"!',
                    [ ExtractFilename(MailFile), SmtpRouter.DestDomain ] ) );

      end else begin

         // loop through all MX servers until successfully delivered
         for i := 0 to SmtpRouter.MXServers.Count - 1 do begin

            // send to MX server
            ServerAlias := SmtpRouter.MXServers[ i ];
            Server      := SmtpRouter.MXServers[ i ];
            Port        := inttostr( IdPORT_SMTP );
            User        := '';
            Pass        := '';

            // send mail
            Log( LOGID_INFO,
                 Format( '[MX %s, %s] domain %s, mail %s, MX %d of %d',
                 [ Server, Port, DestDomain, ExtractFilename(MailFile),
                   i+1, SmtpRouter.MXServers.Count ] ) );
            Connect;
            try
               Result := SendMailfile( MailFile, False );
            finally
               try Disconnect except end;
            end;

            if Result and (LastSMResult = SMR_DELIVERED) then break; // success    
         end;
         
      end;

      // try fallback server, if mail could not be delivered for any reason
      if (not Result) or (LastSMResult <> SMR_DELIVERED) then begin
         if SmtpRouter.FallbackServer <> '' then begin
            if (SmtpRouter.OverrideServer='') or
               (SmtpRouter.OverrideServer<>SmtpRouter.FallbackServer) then begin

               ServerAlias := SmtpRouter.FallbackServer;
               i := Hamster.Config.SmtpServers.IndexOfAlias( ServerAlias );

               if i >= 0 then begin
                  Server      := Hamster.Config.SmtpServers.SrvName[ i ];
                  Port        := Hamster.Config.SmtpServers.SrvPort[ i ];
                  User        := Hamster.Config.SmtpServers.SrvUser[ i ];
                  Pass        := Hamster.Config.SmtpServers.SrvPass[ i ];

                  // send mail
                  Log( LOGID_INFO,
                       Format( '[MX-fallback %s, %s] domain %s, mail %s',
                       [ Server, Port, DestDomain, ExtractFilename(MailFile) ] ) );
                  Connect;
                  try
                     Result := SendMailfile( MailFile, False );
                  finally
                     try Disconnect except end;
                  end;

               end else begin
                  Log( LOGID_WARN, Format( 'Unknown fallback server for mail "%s"!',
                                   [ MailFile ] ) );
               end;

            end;
         end;
      end;

      // if all failed, increment attempts counter of mail
      if (not Result) or (LastSMResult <> SMR_DELIVERED) then begin
         DoDeliveryFailed( MailFile );
      end;

   finally SmtpRouter.Free end;
end;

constructor TClientSMTP.Create( AReportSubState: TReportSubStateInfo;
                                const AMXMode: Boolean;
                                const AServerAlias, AServer, APort, AUser, APass: String );
var  LfdServer: Integer;
begin
   inherited Create( AReportSubState );

   FMXMode := AMXMode;

   ServerAlias := AServerAlias;
   Server      := AServer;
   Port        := APort;
   User        := AUser;
   Pass        := APass;
   SMTP        := nil;
   ServerDir   := '';

   if not FMXMode then with Hamster.Config do begin
      BeginRead;
      try
         LfdServer := SmtpServers.IndexOfAlias( ServerAlias );
         if LfdServer >= 0 then ServerDir := SmtpServers.Path[ LfdServer ];
      finally EndRead end;
   end;
end;

destructor TClientSMTP.Destroy;
begin
   if Assigned(SMTP) then Disconnect;
   inherited Destroy;
end;

end.

