// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cClientPOP3;

interface

{$INCLUDE Compiler.inc}

uses Classes, uType, cClientBase, cFiltersMail, cArticle;

type
   TMailLoadState = (
      mlsContinue,            // continue processing
      mlsAlreadyKnown,        // mail was already loaded/filtered some time ago
      mlsAlreadyKnownExpired, // ... but has expired now and can be deleted
      mlsIgnore,              // ignore mail
      mlsDelete,              // delete mail
      mlsLoaded,              // mail loaded and processed
      mlsError                // severe error, stop processing
   );

   TClientSocketPOP3 = class( TClientBase )
      protected
         function  HResultListFollows: Boolean; override;
         procedure HResultLineToCode; override;
      public
         function  HLogin( AServerAlias, AUser, APass: String ): Boolean; override;
   end;

   TClientPOP3 = class( TClientProtocolBase )
      private
         ServerAlias, Server, Port, User, Pass, LocalUser, FilterSection: String;
         ServerDir, TaskInfo: String;
         POP3       : TClientSocketPOP3;
         FilterFile : TFiltersMail;
         AutoDistribute: Boolean;

         ReportSize    : Integer;
         ReportLogline : String;
         ReportProgress: Integer;

         Mail         : TMess;
         MailNo       : Integer;
         MailUidl     : String;
         MailSize     : Integer;
         MailCount    : Integer;
         MailLoadState: TMailLoadState;
         MailRetrieved: Boolean;

         FiltOrigin     : String;
         FiltReason     : String;
         FiltTopLines   : String;
         FiltTrapDele   : Boolean;
         FiltTrapScore  : Integer;
         FiltDestUsers  : TStringList;
         FiltDestGroups : TStringList;
         FiltDestNotifys: TStringList;
         FiltNewHeaders : String;

         procedure ReportSizeProc( DataSize:integer );
         procedure SendNotification( const NotifyType: Integer;
                                     const NotifyReason: String );
         procedure CheckIfAlreadyKnown( const ListUIDL: TStringList );
         procedure ApplySizeFilters();
         function  RetrieveMailHeaders(): Boolean;
         procedure FindAutoDistributeAccounts( const MailHeaders: TMess;
                                               const AccountList: TStringList );
         procedure AutoDistributeMail( const DefaultDestUser: String );
         function  ApplyFilters( const DefaultDestUser: String ): Boolean;
         function  RetrieveMail: Boolean;
         procedure ProcessHeaders;
         function  PostToNewsgroups: Boolean;
         function  StoreInMailboxes: Boolean;
         procedure BackupFilteredMail;
         function  DeleteMail: Boolean;

      public
         procedure Terminate;
         procedure Connect;
         procedure Disconnect;
         function  Connected: Boolean;

         procedure GetNewMails( LeaveOnServer: Boolean;
                                DefaultDestUser: String;
                                DefFilterSection: String );

         constructor Create( AReportSubState: TReportSubStateInfo;
                             const AServerAlias, AServer, APort, AUser, APass: String;
                             const ALocalUser, AFilterSection: String);
         destructor Destroy; override;
   end;

implementation

uses uConst, uConstVar, uVar, SysUtils, cAccounts, cHamster, uTools, cPCRE,
     uMD5, uDateTime, cLogFileHamster, uHamTools, cArtFiles, cMailUsers,
     IdIOHandlerSocket, IdStack, cMailItem, cMailDispatcher, IdSSLOpenSSL;

const
   NOTIFYTYPE_TEST   = 0;
   NOTIFYTYPE_KILL   = 1;
   NOTIFYTYPE_IGNORE = 2;
   NOTIFYTYPE_GLOBALSIZE_IGNORE = 10;
   NOTIFYTYPE_GLOBALSIZE_KILL   = 11;
   NOTIFYTYPE_GLOBALSIZE_NOTIFY = 12;

   UIDLMARKER_NOTIFY = 'notified-';

   RESULT_OK      =   0;
   RESULT_ERR     = 500;
   RESULT_TIMEOUT = 999;

   OriginGlobalSize = 'GlobalSize';
   OriginAction     = 'Action';
   OriginMailTrap   = 'MailTrap';
   OriginMailFilt   = 'MailFilt';

procedure TClientSocketPOP3.HResultLineToCode;
begin
   if FResultLine = '' then begin
      FResultCode := RESULT_TIMEOUT;
      FResultLine := '-ERR (timeout or connection lost)';

   end else if copy( ResultLine, 1, 1 ) = '+' then begin
      FResultCode := RESULT_OK; // +OK

   end else begin
      FResultCode := RESULT_ERR; // -ERR
   end;

   if not HConnected then FResultCode := RESULT_TIMEOUT;
end;

function TClientSocketPOP3.HResultListFollows: Boolean;
begin
   Result := ( ResultCode = RESULT_OK );
end;

function TClientSocketPOP3.HLogin( AServerAlias, AUser, APass: String ): Boolean;
var  SASL_WANTED, SASL_REMOTE, APOP_Stamp, s: String;
     UsePASS, UseAPOP, UseAUTH, SslOk: Boolean;
     LfdServer, i: Integer;
     SL: TStringList;
begin
   Result := False;
   if not HConnected then exit;

   try
      Log( LOGID_DEBUG, Self.ClassName + '.Login' );

      // SSL-TLS handshake
      if FUseSslMode = sslTLS then begin

         // determine if server supports TLS handshake
         SslOk := False;

         HRequestText( 'CAPA', 0 , nil );
         if not HConnected then exit;
         if ResultCode = RESULT_OK then begin
            SslOk := pos( 'stls' + CRLF, LowerCase(ResultText) ) > 0;
         end;

         if SslOk then begin

            // Handle TLS handshake
            s := StartTlsHandshake( 'STLS', RESULT_OK, 'CAPA', RESULT_OK, True );
            if s = '' then begin
               Log( LOGID_ERROR,
                    'Required SSL with TLS handshake was not activated!' );
               exit;
            end;

         end else begin

            Log( LOGID_ERROR,
                 FServer + ' does not support required TLS handshake!' );
            exit;

         end;

      end;

      if (AUser = '') or (APass = '') then begin
         Result := HConnected;
         exit;
      end;

      // determine enabled authentication mechanisms
      UsePASS := False;
      UseAPOP := False;
      UseAUTH := False;
      SASL_WANTED := '';

      if copy( APass, 1, 5 ) = 'PASS:' then begin
         Log( LOGID_WARN, 'Note: Use of "PASS:" passwords is deprecated!' );
         System.Delete( APass, 1, 5 );
         UsePASS := True;

      end else if copy( APass, 1, 5 ) = 'APOP:' then begin
         Log( LOGID_WARN, 'Note: Use of "APOP:" passwords is deprecated!' );
         System.Delete( APass, 1, 5 );
         UseAPOP := True;

      end else begin
         Hamster.Config.BeginRead;
         try
            with Hamster.Config.Pop3Servers do begin
               LfdServer := IndexOfAlias( AServerAlias, True );
               if LfdServer >= 0 then begin
                  UsePASS     := Settings(LfdServer).GetBoo(ssPop3UseAuthPass);
                  UseAPOP     := Settings(LfdServer).GetBoo(ssPop3UseAuthApop);
                  UseAUTH     := Settings(LfdServer).GetBoo(ssPop3UseAuthAuth);
                  SASL_WANTED := Settings(LfdServer).GetStr(ssPop3UseSasl);
               end;
            end;
         finally Hamster.Config.EndRead end;
      end;

      if (not Result) and UseAUTH then begin // try with AUTH/SASL

         // determine supported SASL mechanisms
         SASL_REMOTE := '';

         HRequestText( 'CAPA', 0 , nil ); // try CAPA -> SASL (list)
         if not HConnected then exit;
         if ResultCode = RESULT_OK then begin
            SL := TStringList.Create;
            try
               SL.Text := ResultText;
               for i := 0 to SL.Count - 1 do begin
                  if UpperCase( copy(SL[i],1,5) ) = 'SASL ' then begin
                     SASL_REMOTE := copy( SL[i], 6, MaxInt );
                     break;
                  end;
               end;
            finally SL.Free end;
         end;

         if SASL_REMOTE = '' then begin
            HRequestText( 'AUTH', 0 , nil ); // try AUTH -> (list)
            if not HConnected then exit;
            if ResultCode = RESULT_OK then begin
               SL := TStringList.Create;
               try
                  SL.Text := ResultText;
                  for i := 0 to SL.Count - 1 do begin
                     if SASL_REMOTE<>'' then SASL_REMOTE := SASL_REMOTE + ' ';
                     SASL_REMOTE := SASL_REMOTE + SL[i];
                  end;
               finally SL.Free end;
            end;
         end;

         if SASL_REMOTE = '' then begin

            Log( LOGID_WARN, 'Requested AUTH authentication not supported!' );

         end else begin

            Result := Remote_SASL_Login(
                         Self, 'AUTH %s %s', RESULT_OK, RESULT_OK, 3,
                         SASL_WANTED, SASL_REMOTE, AUser, APass );   
            if not HConnected then exit;

         end;

      end;

      if (not Result) and UseAPOP then begin // try with APOP

         APOP_Stamp := RE_Extract( Greeting, '(<.*?\@.*?>)', 0 );

         if APOP_Stamp = '' then begin

            Log( LOGID_WARN, 'Requested APOP authentication not supported!' );

         end else begin

            Log( LOGID_DEBUG, 'APOP timestamp="' + APOP_Stamp + '"' );
            s := MD5toHex( MD5ofStr( APOP_Stamp + APass ) );

            HRequest( 'APOP ' + AUser + ' ' + s, 0, True );
            if not HConnected then exit;
            if ResultCode = RESULT_OK then Result := True;

         end;

      end;

      if (not Result) and UsePASS then begin // try with USER/PASS

         HRequest( 'USER ' + AUser, 0, True );
         if not HConnected then exit;

         if ResultCode = RESULT_OK then begin
            HRequest( 'PASS ' + APass, 0, True );
            if not HConnected then exit;
            if ResultCode = RESULT_OK then Result := True;
         end;

      end;

   finally
      if (not Result) and HConnected then HDisconnect;
   end;
end;

procedure TClientPOP3.Terminate;
begin
   if Assigned( POP3 ) then POP3.Terminate;
end;

function TClientPOP3.Connected: Boolean;
begin
   if Assigned( POP3 ) then Result := POP3.HConnected else Result := False;
end;

procedure TClientPOP3.Connect;
var  s: String;
     UseSOCKS: Boolean;
     UseSslMode: TSslTlsMode;
     LfdServer, TimeoutConnect, TimeoutCommand, i: Integer;
     SSLVersion: TIdSSLVersion;
     SSLCipher : String;
begin
   if Assigned( POP3 ) then Disconnect;

   if ServerDir = '' then begin
      Log( LOGID_ERROR, 'POP3 server "' + ServerAlias + '" is not configured!' );
      exit;
   end;

   TimeoutConnect := Hamster.Config.Settings.GetInt(hsRemoteTimeoutConnect);
   TimeoutCommand := Hamster.Config.Settings.GetInt(hsRemoteTimeoutCommand);
   UseSOCKS := False;
   UseSslMode := sslNone;
   SSLVersion := sslvSSLv3;
   SSLCipher  := '';

   Hamster.Config.BeginRead;
   try
      with Hamster.Config.Pop3Servers do begin
         LfdServer := IndexOfAlias( ServerAlias );
         if LfdServer >= 0 then begin
            i := Settings( LfdServer ).GetInt( ssTimeoutConnect );
            if i > 0 then TimeoutConnect := i;
            i := Settings( LfdServer ).GetInt( ssTimeoutCommand );
            if i > 0 then TimeoutCommand := i;
            UseSOCKS := Settings( LfdServer ).GetBoo( ssUseSocks );
            UseSslMode := TSslTlsMode( Settings(LfdServer).GetInt(ssSSLMode) );
            SSLVersion := TIdSSLVersion( Settings(LfdServer).GetInt(ssSSLVersion) );
            SSLCipher  := Settings( LfdServer ).GetStr( ssSSLCipher );
         end;
      end;
   finally Hamster.Config.EndRead end;

   POP3 := TClientSocketPOP3.Create( nil, 'pop' );

   if POP3.HConnect( Server, Port, UseSOCKS, UseSslMode, SSLVersion, SSLCipher,
                     0, TimeoutConnect*1000, TimeoutCommand*1000 ) then begin
      if not POP3.HLogin( ServerAlias, User, Pass ) then begin
         s := POP3.LastErrResult;
         if s = '' then s := POP3.ResultLine;
         if s <> '' then s := ' ("' + s + '")';
         Log( LOGID_WARN, 'Login failed ' + s );
      end;
      if POP3.Greeting<>'' then begin
         HamFileRewriteLine( ServerDir + SRVFILE_GREETING, POP3.Greeting );
      end;
   end;
end;

procedure TClientPOP3.Disconnect;
begin
   if not Assigned( POP3 ) then exit;
   try POP3.HDisconnect except end;
   FreeAndNil( POP3 );
end;

procedure TClientPop3.ReportSizeProc( DataSize: Integer );
var  s: String;
begin
   if ReportSize = 0 then exit;
   if ReportProgress <> trunc(DataSize/ReportSize*100) then begin
      ReportProgress := trunc(DataSize/ReportSize*100);
      s := ReportLogline + ' - ' + inttostr(ReportProgress) + '%';
      Log( LOGID_STATUS, s );
      ReportSubState( s );
   end;
end;

procedure TClientPOP3.SendNotification( const NotifyType: Integer;
                                        const NotifyReason: String );

   function buildNotifySubject( const marker: String ): String;
   var  hs, hf: String;
   begin
      hs := Mail.HeaderValueByNameSL( HDR_NAME_SUBJECT );
      hs := StringReplace( hs, #9, ' ', [rfReplaceAll] );
      if hs = '' then hs := '(none)';

      hf := Mail.HeaderValueByNameSL( HDR_NAME_FROM );
      hf := StringReplace( hf, #9, ' ', [rfReplaceAll] );
      if hf = '' then hf := '(unknown)';

      Result := '[' + marker + ']' + ' '
              + '"' + hs + '"'
              + ' from ' + '"' + hf + '"';
      if length(Result) > 500 then Result := copy( Result, 1, 500 ) + ' [...]';
   end;


var  i: Integer;
     Subject, Body: String;
     MsgIsMail: Boolean;
begin
   MsgIsMail := True;

   case NotifyType of

      NOTIFYTYPE_KILL: begin
         Subject := buildNotifySubject( OriginMailFilt + ' KILL' );
         Body := '[Hamster]' + #13#10#13#10
               + Format(
                    'A mail-message on %s was deleted due to mail-filters.',
                    [Server+' (' + User + ')'])
               + #13#10 + NotifyReason
               + #13#10 + Format(
                    'Size of mail: %s bytes', [inttostr( MailSize )])
               + #13#10
      end;

      NOTIFYTYPE_IGNORE: begin
         Subject := buildNotifySubject( OriginMailFilt + ' IGNORE' );
         Body := '[Hamster]' + #13#10#13#10
               + Format(
                    'A mail-message on %s was ignored due to mail-filters.',
                    [Server+' (' + User + ')'])
               + #13#10 + NotifyReason
               + #13#10 + Format(
                    'Size of mail: %s bytes', [inttostr( MailSize )])
               + #13#10
      end;

      NOTIFYTYPE_GLOBALSIZE_KILL: begin
         Subject := '[Hamster] '+ 'Mail killed due to its size!';
         Body := '[Hamster]' + #13#10#13#10
               + Format(
                    'A mail-message on %s was deleted due to global size-limit.',
                    [Server+' (' + User + ')'])
               + #13#10 + Mail.FullText;
         MsgIsMail := false
      end;

      NOTIFYTYPE_GLOBALSIZE_IGNORE: begin
         Subject := '[Hamster] '+ 'Mail ignored due to its size!';
         Body := '[Hamster]' + #13#10#13#10
               + Format(
                    'A mail-message on %s was ignored due to global size-limit.',
                    [Server+' (' + User + ')'])
               + #13#10 + Mail.FullText;
         MsgIsMail := false
      end;

      NOTIFYTYPE_GLOBALSIZE_NOTIFY: begin
         Subject := '[Hamster] '+'Mail size notification!';
         Body := '[Hamster]' + #13#10#13#10
               + Format('A mail-message from %s was loaded.',
                    [Server+' (' + User + ')'])
               + #13#10 + Mail.FullText;
         MsgIsMail := false
      end;

      else begin
         Subject := buildNotifySubject( OriginMailFilt + ' NOTIFY' );
         Body := '[Hamster]' + #13#10#13#10
               + Format(
                    'Please ignore, a mail-message on %s was ... just tested.',
                    [Server+' (' + User + ')'])
               + #13#10
      end;
   end;

   if MsgIsMail then begin
      Body := Body + #13#10
                   + '---------- ' + Format(
                   'Headers and first %s lines of message follow',
                   [inttostr(Hamster.Config.Settings.GetInt(hsNoOfTopLinesToLoad))])
                   + ': ----------'
                   + #13#10#13#10 + Mail.FullText;
   end;

   for i := 0 to FiltDestUsers.Count - 1 do begin
      SendLocalInfoMail( FiltDestUsers[i], Subject, Body );
   end;
end;

procedure TClientPOP3.CheckIfAlreadyKnown( const ListUIDL: TStringList );
// check if current mail was already loaded
var  i, delDays, ageDays, PurgeBase: Integer;
begin
   if ListUIDL.Count = 0 then exit; // no UIDL list available

   Log( LOGID_DEBUG, 'Check if already loaded' );

   for i := 0 to ListUIDL.Count - 1 do begin
      if LongInt( ListUIDL.Objects[i] ) = MailNo then begin

         MailUidl := ListUIDL.Strings[i];
         if Hamster.MailHistory.ContainsUIDL( MailUidl, {out} PurgeBase ) then begin

            // already loaded mail (its UIDL is recorded in mail history)
            MailLoadState := mlsAlreadyKnown;
            Log( LOGID_DETAIL, TaskInfo + 'Already loaded/filtered:'
                 + ' #=' + inttostr(MailNo)
                 + ' Time=' + DateTimeToTimeStamp( UnixTimeToDateTime(PurgeBase) )
                 + ' Size=' + inttostr( MailSize )
                 + ' UIDL=' + MailUidl );

            // check if already loaded mail should be deleted after a
            // given number of days although "Leave on server" is enabled.
            delDays := Hamster.Config.Settings.GetInt( hsMailDeleteLoadedAfter );
            if delDays > 0 then begin // activated

               ageDays := (DateTimeToUnixTime(Now) - PurgeBase) div SECONDS_PER_DAY;
               if ageDays >= delDays then begin // loaded before given number of days

                  MailLoadState := mlsAlreadyKnownExpired;
                  Log( LOGID_INFO, TaskInfo + Format(
                     'Delete mail %d of %d (already loaded/filtered %s ago)',
                     [ MailNo, MailCount,
                       SecondsToDHMS(DateTimeToUnixTime(Now)-PurgeBase) ] ));

               end;
               
            end;

         end;

         break;

      end;
   end;
end;

procedure TClientPOP3.ApplySizeFilters();
// check global size limits for notify, ignore and kill
var  NotifyType, size: Integer;
begin
   Log( LOGID_DEBUG, 'Check global size limits' );

   NotifyType := NOTIFYTYPE_GLOBALSIZE_NOTIFY;

   // check ignore size limit
   size := Hamster.Config.Settings.GetInt( hsMailSizeIgnore );
   if (size > 0) and (MailSize > size) then begin

      Log( LOGID_INFO, TaskInfo + Format( 'Ignore mail %d of %d (%s)',
                       [ MailNo, MailCount, 'global size ignore' ] ) );

      MailLoadState := mlsIgnore;
      NotifyType    := NOTIFYTYPE_GLOBALSIZE_IGNORE;

   end;

   // check delete size limit
   size := Hamster.Config.Settings.GetInt( hsMailSizeKill );
   if (size > 0) and (MailSize > size) then begin

      Log( LOGID_INFO, TaskInfo + Format( 'Delete mail %d of %d (%s)',
                       [ MailNo, MailCount, 'global size delete' ] ) );

      MailLoadState := mlsDelete;
      NotifyType    := NOTIFYTYPE_GLOBALSIZE_KILL;

      FiltOrigin   := OriginGlobalSize;
      FiltReason   := Format( 'Mail size larger than %d byte', [ size ] );
      FiltTopLines := Format( '(no text available; reported size was %d byte)',
                             [ MailSize ] );

   end;

   // check notify size limit
   size := Hamster.Config.Settings.GetInt( hsMailSizeNotify );
   if (size > 0) and (MailSize > size) then begin

      if (MailUidl = '') or
         (not Hamster.MailHistory.ContainsUIDL( UIDLMARKER_NOTIFY + MailUidl )) then begin

         // send notify mail
         FiltDestNotifys.Text := 'admin';
         Mail.FullText := Format( 'Size of mail: %d byte', [ MailSize ]);
         SendNotification( NotifyType, '' );
         
         if MailUidl <> '' then begin
            // mark UIDL as notified
            Hamster.MailHistory.AddUIDL( UIDLMARKER_NOTIFY + MailUidl, 0 );
         end;

      end;

   end;
end;

function TClientPOP3.RetrieveMailHeaders(): Boolean;
// pre-load header lines with TOP if they are needed for filters
var  loadTop: Boolean;
     skipSize, topLines: Integer;
begin
   Result := False;

   // check if TOP lines are needed
   loadTop := Hamster.MailTrap.IsActive  // required for mail traps
           or FilterFile.TOP_makes_sense // required for mail filters
           or AutoDistribute;            // required for auto-distrib.

   // skip TOP if reported size <= limit
   skipSize := Hamster.Config.Settings.GetInt( hsMailTopSkipSize );
   if (MailSize > 1) and (MailSize <= skipSize) then loadTop := False;

   // retrieve TOP lines
   if loadTop then begin

      Log( LOGID_INFO, TaskInfo + Format( 'Retrieve mail %d of %d (headers)',
                                          [ MailNo, MailCount ] ) );

      topLines := Hamster.Config.Settings.GetInt( hsNoOfTopLinesToLoad );
      POP3.HRequestText( 'TOP ' + inttostr( MailNo )
                       + ' '    + inttostr( topLines ), 0, nil );

      if not Connected then exit;
      if POP3.FTerminated then exit;
      if (POP3.ResultLine='') or (POP3.ResultCode=999) then begin // Timeout
         Log( LOGID_WARN, 'GetNewMails canceled, no reply for TOP (' + Server + ')' );
         try Disconnect except end;
         exit;
      end;

      if POP3.ResultCode <> RESULT_OK then loadTop := False; // not supported

   end;

   // retrieve full mail if TOP is not needed/wanted/supported
   if not loadTop then begin

      ReportSize     := MailSize;
      ReportProgress := -1;
      ReportLogline  := TaskInfo + Format( 'Retrieve mail %d of %d (%d byte)',
                                           [ MailNo, MailCount, MailSize ] );
      Log( LOGID_INFO, ReportLogline);

      POP3.HRequestText( 'RETR ' + inttostr( MailNo ), 0, ReportSizeProc );

      if not Connected then exit;
      if POP3.FTerminated then exit;
      if (POP3.ResultLine='') or (POP3.ResultCode=999) then begin // Timeout
         Log( LOGID_WARN, 'GetNewMails canceled, no reply for RETR.1 (' + Server + ')' );
         try Disconnect except end;
         exit;
      end;

      if POP3.ResultCode <> RESULT_OK then exit; // failed

      MailRetrieved := True;
      MailSize      := POP3.ResultStrm.Size;

   end;

   // set preliminary mail text for filtering, either headers only
   // (MailRetrieved=False) or full message (MailRetrieved=True).
   Mail.FullText := POP3.ResultText;

   Result := True;
end;

procedure TClientPOP3.FindAutoDistributeAccounts( const MailHeaders: TMess;
                                                  const AccountList: TStringList );
// fill "AccountList" with all account names whose mail address is found in
// any of the provided "MailHeaders" that may contain recipient information.
const ADHeadersMax = 11;
      ADHeadersList: array[ 0 .. ADHeadersMax ] of String =
                     ( 'to', 'apparently-to', 'cc', 'bcc', 'envelope-to',
                       'x-envelope-to', 'original-recipient', 'x-resent-for',
                       'x-resent-to', 'resent-to', 'delivered-to', 'received' );
var  MailAddrs: TStringList;
     HdrName, HdrValue, AccountName: String;
     Index, UID, LTY, i: Integer;
begin
   MailAddrs := TStringList.Create;
   MailAddrs.Sorted := True;
   MailAddrs.Duplicates := dupIgnore;

   try
      // loop through all headers of message to find recipient related ones
      Index := 0;
      while MailHeaders.NextHeader( Index, HdrName, HdrValue ) do begin

         HdrName := LowerCase( HdrName );
         for i := 0 to ADHeadersMax do begin
            if HdrName = ADHeadersList[i] then begin
               ExtractMailAddresses( MailAddrs, HdrName, HdrValue );
               break;
            end;
         end;

      end;

      // look for known local accounts in list of found mail addresses
      for i := 0 to MailAddrs.Count - 1 do begin
         if length( MailAddrs[i] ) = 0 then continue;
         if Hamster.Accounts.IsLocalMailbox( MailAddrs[i], UID, LTY ) then begin
            AccountName := Hamster.Accounts.Value[ UID, apUsername ];
            AccountList.Add( AccountName );
            Log( LOGID_INFO, 'Mail recipient "' + MailAddrs[i]
               + '" was auto-distributed to account "' + AccountName + '"' );
         end else begin
            Log( LOGID_DEBUG, 'Mail recipient "' + MailAddrs[i]
               + '" was unknown -> no auto-distribution.' );
         end;
      end;

   finally MailAddrs.Free end;
end;

procedure TClientPOP3.AutoDistributeMail( const DefaultDestUser: String );
// automatically distribute mail to recipients found in header lines
begin
   // skip if mail filters selected someone/something different
   if FiltDestGroups.Count > 0 then exit; // mail-to-news
   if FiltDestUsers.Count <> 1 then exit; // other than 1 default user
   if FiltDestUsers[0] <> DefaultDestUser then exit; // not default user

   Log( LOGID_DEBUG, 'Check Auto Distribution' );

   // clear default recipient
   FiltDestUsers.Clear;

   // find recipient accounts in headers of mail
   FindAutoDistributeAccounts( Mail, FiltDestUsers );

   // if no known recipients were found, reset to default recipient
   if FiltDestUsers.Count = 0 then FiltDestUsers.Add( DefaultDestUser );
end;

function TClientPOP3.ApplyFilters( const DefaultDestUser: String ): Boolean;
// apply mail filters (action, mailtrap, mailfilt, auto-distribution)
var  NotifyReason, reason: String;
     DeleteIt, IgnoreIt: Boolean;
begin
   DeleteIt := False;
   IgnoreIt := False;

   // process "MailHeaderIn" actions to check headers of mail
   if ProcessAction_MailInHeader( moPOP3, Mail, FiltNewHeaders ) then begin

      // ok, just activate any modified header lines
      if FiltNewHeaders <> '' then Mail.HeaderText := FiltNewHeaders;

   end else begin

      // action script has invalidated mail
      MailLoadState := mlsDelete;
      DeleteIt      := True;
      FiltOrigin    := OriginAction;
      FiltReason    := 'Action "MailInHeader[File]" has invalidated mail.';
      FiltTopLines  := Mail.FullText;
      Log( LOGID_DETAIL, 'Action MailInHeader/File:  result=kill' + ' ' + Mail.MailSummary );

   end;

   // apply mail traps (MailTrap.hst)
   if not DeleteIt then begin

      Log( LOGID_DEBUG, 'Check Mail Traps' );

      if Hamster.MailTrap.Reject( Mail, {out} reason, {out} FiltTrapDele,
                                        {out} FiltTrapScore ) then begin
         MailLoadState := mlsDelete;
         DeleteIt      := True;
         FiltOrigin    := OriginMailTrap;
         FiltReason    := reason;
         FiltTopLines  := Mail.FullText;
      end;

      if DeleteIt or FiltTrapDele or (FiltTrapScore <> 0) then begin
         Log( LOGID_DETAIL, 'MailTrap: '
            + ' result=' + iif( DeleteIt, 'kill', 'load' )
            + iif( FiltTrapDele, ' delete=yes', '' )
            + ' score=' + inttostr( FiltTrapScore )
            + ' ' + Mail.MailSummary );
      end;

   end;

   // apply mail filters (MailFilt.hst)
   if not DeleteIt then begin

      Log( LOGID_DEBUG, 'Check mail filters' );

      Mail.SetHeaderSL( 'Bytes:', inttostr(MailSize), '' );

      FilterFile.FilterMail( Mail.HeaderText,
                             DefaultDestUser,
                             {var} IgnoreIt, {var} DeleteIt,
                             FiltDestNotifys, FiltDestUsers, FiltDestGroups,
                             {var} NotifyReason, {out} reason );

      if IgnoreIt and (MailLoadState = mlsContinue) then begin
         MailLoadState := mlsIgnore;
      end;

      if DeleteIt then begin
         MailLoadState := mlsDelete;
         FiltOrigin    := OriginMailFilt;
         FiltReason    := reason;
         FiltTopLines  := Mail.FullText;
      end;

      Mail.DelHeaderML( 'Bytes:' ); // just added above for any "Bytes"-filter

      if IgnoreIt or DeleteIt then begin
         Log( LOGID_DETAIL, 'MailFilt: '
            + ' result=' + iif( DeleteIt, 'kill', 'ignore' )
            + ' ' + Mail.MailSummary );
      end;
      
   end;

   // auto-distribute to recipients found in header lines
   if (not DeleteIt) and AutoDistribute then begin
      AutoDistributeMail( DefaultDestUser );
   end;

   // notify if result is deleted or ignored
   if (DeleteIt or IgnoreIt) and (FiltDestNotifys.Count > 0) then begin

      Log( LOGID_DEBUG, 'Handle filter results' );

      if (MailUidl = '') or
         (not Hamster.MailHistory.ContainsUIDL( UIDLMARKER_NOTIFY + MailUidl )) then begin

         if FiltDestNotifys.Count = 0 then FiltDestNotifys.Add( 'admin' );
         if DeleteIt then SendNotification( NOTIFYTYPE_KILL,   NotifyReason )
                     else SendNotification( NOTIFYTYPE_IGNORE, NotifyReason );

         if MailUidl <> '' then begin // mark as notified
            Hamster.MailHistory.AddUIDL( UIDLMARKER_NOTIFY + MailUidl, 0 );
         end;

      end;

   end;

   Result := True;
end;

function TClientPOP3.RetrieveMail(): Boolean;
// retrieve full mail text (if not done already)
var  s: String;
     LogNow: TDateTime;
begin
   Result := False;

   if not MailRetrieved then begin

      ReportSize     := MailSize;
      ReportProgress := -1;
      ReportLogline  := TaskInfo + Format( 'Retrieve mail %d of %d (%d bytes)',
                                           [ MailNo, MailCount, MailSize ] );
      Log( LOGID_INFO, ReportLogline);

      POP3.HRequestText( 'RETR ' + inttostr(MailNo), 0, ReportSizeProc );

      if not Connected then exit;
      if POP3.FTerminated then exit;
      if (POP3.ResultLine='') or (POP3.ResultCode=999) then begin // Timeout
         Log( LOGID_WARN, 'GetNewMails canceled, no reply for RETR (' + Server + ')' );
         try Disconnect except end;
         exit;
      end;

      // note: POP3.ResultCode is checked below after logging

      MailRetrieved := True;
      Mail.FullText := POP3.ResultText;

      // re-apply any modified headers of action script
      if FiltNewHeaders <> '' then Mail.HeaderText := FiltNewHeaders;

   end;

   // save result in MailIn.log
   Log( LOGID_DEBUG, 'Update log files' );
   LogNow := Now;
   s := DateTimeToLogTime( LogNow )
      + #9 + 'Server='     + ServerAlias
      + #9 + 'Result='     + Logify( POP3.ResultLine )
      + #9 + 'From='       + Logify( Mail.HeaderValueByNameSL(HDR_NAME_FROM) )
      + #9 + 'To='         + Logify( Mail.HeaderValueByNameSL(HDR_NAME_TO) )
      + #9 + 'Subject='    + Logify( Mail.HeaderValueByNameSL(HDR_NAME_SUBJECT) )
      + #9 + 'Date='       + Logify( Mail.HeaderValueByNameSL(HDR_NAME_DATE) )
      ;
   HamFileAppendLine( AppSettings.GetStr(asPathLogs)
                      + LOGFILE_MAILIN
                      + FormatDateTime( '"-"yyyy"-"mm', LogNow )
                      + LOGFILE_EXTENSION,
                      s );

   if POP3.ResultCode = RESULT_OK then Result := True;
end;

procedure TClientPOP3.ProcessHeaders();
var  MessageID, s: String;
     i: Integer;
begin
   Log( LOGID_DEBUG, 'Process message headers' );

   // add missing Message-ID
   MessageID := Mail.HeaderValueByNameSL( HDR_NAME_MESSAGE_ID );
   if MessageID = '' then begin // add MID with .nomid marker
      MessageID := MidGenerator( Hamster.Config.Settings.GetStr(hsFQDNforMID), '.nomid' );
      Mail.AddHeaderSL( HDR_NAME_MESSAGE_ID, MessageID );
   end;

   // add missing Date
   s := Mail.HeaderValueByNameSL( HDR_NAME_DATE );
   if s = '' then begin // add Date marked with -0000 timezone
      s := DateTimeGMTToRfcDateTime( NowGMT, '-0000' );
      Mail.AddHeaderSL( HDR_NAME_DATE, s );
   end;

   // add filter results
   s := '';
   for i := 0 to FiltDestUsers.Count - 1 do begin
      if s <> '' then s := s + ',';
      s := s + 'account:' + FiltDestUsers[i];
   end;
   for i := 0 to FiltDestGroups.Count - 1 do begin
      if s <> '' then s := s + ',';
      s := s + 'news:' + FiltDestGroups[i];
   end;
   if s<>'' then Mail.AddHeaderSL( HDR_NAME_X_HAMSTER_TO, s );

   // add info line
   s := 'Score='    + inttostr( FiltTrapScore ) + ' '
      + 'UIDL='     + UIDLGenerator + ' '
      + 'Received=' + DateTimeToTimeStamp(Now) + ' '
      + 'UID=' + IntToStr( GetUID(0) );
   Mail.SetHeaderSL( HDR_NAME_X_HAMSTER_INFO, s, '' );
end;

function TClientPOP3.PostToNewsgroups(): Boolean;
// post loaded mail to (local) newsgroups
var  PostText: String;
     i: Integer;
begin
   Result := True;

   if FiltDestGroups.Count = 0 then exit; // not redirected to any group

   PostText := Mail.FullText;

   for i := 0 to FiltDestGroups.Count - 1 do begin

      Log( LOGID_DEBUG, 'Post to ' + FiltDestGroups[i] + ' ...' );

      if not SaveMailToNews( FiltDestGroups[i], PostText ) then begin

         Log( LOGID_ERROR, 'Could not post mail to ' + FiltDestGroups[i] );

         // don't stop but make sure that someone receives this mail
         if FiltDestUsers.Count = 0 then FiltDestUsers.Add( 'admin' );

      end;

   end;
end;

function TClientPOP3.StoreInMailboxes(): Boolean;
// store loaded mail in local mailboxes
var  MailItem: TMailItem;
     i, uid: Integer;
begin
   Result := True;

   if FiltDestUsers.Count = 0 then exit; // no recipients (mail-to-news)

   MailItem := TMailItem.Create( moPOP3 );
   with MailItem do try

      try
         // set sender identification (just server IP for POP3)
         Sender.UserID := ACTID_ADMIN;
         MailItem.Sender.IPAddr := GStack.StringToTInAddr(
            TIdIOHandlerSocket( POP3.IOHandler ).Binding.PeerIP
         ).S_addr;

         // check recipients
         for i := 0 to FiltDestUsers.Count - 1 do begin
            Log( LOGID_DEBUG, 'Save for ' + FiltDestUsers[i] + ' ...' );
            uid := Hamster.Accounts.UserIDOf( FiltDestUsers[i] );
            if uid = ACTID_INVALID then uid := ACTID_ADMIN;
            Recipients.Add( uid );
         end;

         // set mail text
         MailText.FullText := Mail.FullText;

         // dispatch mail
         Hamster.MailDispatcher.Process( MailItem );

      except
         on E: Exception do begin
            Log( LOGID_ERROR, 'Error processing mail: ' + E.Message );
            MailLoadState := mlsError;
            Result := False;
         end;
      end;

      // check delivery success for all recipients
      for i := 0 to Recipients.Count - 1 do begin
         if Recipients[i].DeliveryResult <> mdrSuccess then begin
            MailLoadState := mlsError;
            Result := False;
            break;
         end;
      end;

   finally MailItem.Free end;
end;

procedure TClientPop3.BackupFilteredMail();
// backup mail deleted by any of the filters
var  i: Integer;
     Account, Subject, Body, LogFile, LogHeader, LogMarker, s, hs, hf, ho: String;
     LogNow: TDateTime;
begin
   // got something to back up?
   if (length(FiltOrigin) = 0) or (length(FiltReason) = 0) then exit;

   LogNow := Now;

   // prepare info for logfile-entries and backup-mail
   hs := TMess.HeaderValueFromText( FiltTopLines, HDR_NAME_SUBJECT );
   hs := StringReplace( hs, #9, ' ', [rfReplaceAll] );
   if hs = '' then hs := '(none)';

   hf := TMess.HeaderValueFromText( FiltTopLines, HDR_NAME_FROM );
   hf := StringReplace( hf, #9, ' ', [rfReplaceAll] );
   if hf = '' then hf := '(unknown)';

   ho := FiltOrigin;
   if FiltOrigin = OriginMailTrap then begin
      if FiltTrapDele then ho := ho + ' DELETE'
                      else ho := ho + ' SCORE ' + inttostr(FiltTrapScore);
   end;

   Subject := '[' + ho + ']';
   Subject := Subject + ' ' + '"' + hs + '"';
   Subject := Subject + ' from ' + '"' + hf + '"';
   if length(Subject) > 500 then Subject := copy( Subject, 1, 500 ) + ' [...]';

   // info in standard logfile
   Log( LOGID_INFO, 'Mail deleted: ' + Subject );

   // marker for daily spam report
   s := DateTimeToLogTime( LogNow ) + #9 + ho + #9 + hs + #9 + hf;
   HamFileAppendLine( AppSettings.GetStr(asPathLogs) + LOGFILE_SPAMREPORT, s );

   // add matches to MailFilters.log
   LogFile := AppSettings.GetStr( asPathLogs )
            + LOGFILE_MAILFILTERS
            + FormatDateTime( '"-"yyyy"-"mm', LogNow )
            + LOGFILE_EXTENSION;
   LogHeader := DateTimeToLogTime( LogNow ) + #9 + ServerAlias + #9 + FiltOrigin;
   LogMarker := '1'; // = first line
   s := FiltReason;
   HamFileEnter; // keep all entries together for easier evaluation
   try
      repeat
         i := Pos( #9, s );
         if i = 0 then begin
            HamFileAppendLine( LogFile, LogHeader + #9 + LogMarker
                                                  + #9 + s );
            SetLength( s, 0 );
         end else begin
            HamFileAppendLine( LogFile, LogHeader + #9 + LogMarker
                                                  + #9 + copy( s, 1, i-1 ) );
            System.Delete( s, 1, i );
         end;
         LogMarker := '0'; // = continuation lines
      until length( s ) <= 1;
   finally HamFileLeave end;
                      
   // backup account configured?
   Account := Hamster.Config.Settings.GetStr( hsMailJunkAccount );
   if Account = '' then exit;

   // backup disabled for specific deletion type?
   if Hamster.Config.Settings.GetBoo( hsMailJunkSkipFiltKill ) then begin
      if FiltOrigin = OriginMailFilt then exit; // skip MailFilt-KILL
   end;
   if Hamster.Config.Settings.GetBoo( hsMailJunkSkipTrapDelete ) then begin
      if (FiltOrigin = OriginMailTrap) and FiltTrapDele then exit; // skip MailTrap-DELETE
   end;
   if Hamster.Config.Settings.GetBoo( hsMailJunkSkipTrapScore ) then begin
      if (FiltOrigin = OriginMailTrap) and // skip MailTrap-SCORE
         (FiltTrapScore < Hamster.Config.Settings.GetInt(hsMailJunkSkipScoreLimit)) then exit;
   end;

   // build backup mail and send it to configured mailbox
   Body := ''
         + '| Server: ' + ServerAlias + ' (' + Server + ':' + Port
                        + iif( User<>'', ', ' + User ) + ')' + #13#10
         + '| ' + #13#10
         + '| Filter: ' + FiltOrigin + #13#10
         + '|    ' + StringReplace( FiltReason, #9, #13#10'|    ',
                                    [rfReplaceAll] ) + #13#10
         + iif( FiltOrigin = OriginMailTrap,
                '| Final score: ' + inttostr(FiltTrapScore) + #13#10 )
         + '| ' + #13#10
         + '| Headers and top lines of deleted mail follow:' + #13#10
         + #13#10
         + FiltTopLines;
   SendLocalInfoMail( Account, Subject, Body );
end;

function TClientPOP3.DeleteMail(): Boolean;
// delete mail on server
begin
   Result := False;

   // log/backup filtered mail that is about to be deleted now
   if length( FiltOrigin ) > 0 then begin
      Log( LOGID_DEBUG, 'Backup mail info' );
      BackupFilteredMail();
   end;

   // delete mail on server
   Log( LOGID_DEBUG, 'Mark #' + inttostr(MailNo) + ' for delete' );

   POP3.HRequest( 'DELE ' + inttostr(MailNo), 0, False );

   if (POP3.ResultLine='') or (POP3.ResultCode=999) then begin // Timeout
      Log( LOGID_WARN, 'GetNewMails canceled, no reply for DELE (' + Server + ')' );
      try Disconnect except end;
      exit;
   end;

   if POP3.ResultCode <> RESULT_OK then begin
      Log( LOGID_ERROR, TaskInfo + 'Mail delete failed (' + inttostr(MailNo) + ')' );
      exit;
   end;

   Result := True;
end;


procedure TClientPOP3.GetNewMails( LeaveOnServer: Boolean;
                                   DefaultDestUser: String;
                                   DefFilterSection: String );
var  OvrLfd, SumSize, DefaultDestUserID, i: Integer;
     s: String;
     Overview, ListUIDL: TStringList;
     Parser: TParser;
begin
   if not Connected then exit;

   Log( LOGID_INFO, TaskInfo + 'GetNewMails' );

   // check given default-recipient
   if DefaultDestUser  = '' then DefaultDestUser := LocalUser;
   if DefaultDestUser  = '' then DefaultDestUser := 'admin';
   if DefFilterSection = '' then DefFilterSection := FilterSection;
   if DefFilterSection = '' then DefFilterSection := '*';
   DefaultDestUserID := Hamster.Accounts.UserIDOf( DefaultDestUser );
   if DefaultDestUserID = ACTID_INVALID then begin
      Log( LOGID_ERROR, TaskInfo + 'Unknown user "' + DefaultDestUser + '"!' );
      exit;
   end;
   if not Hamster.Accounts.HasMailbox( DefaultDestUserID ) then begin
      Log( LOGID_ERROR, TaskInfo + 'User "' + DefaultDestUser + '" has no mailbox!' );
      exit;
   end;

   // get specific filter-list for current selection
   FilterFile.SelectSections( DefFilterSection );

   // get list of available mails
   Log( LOGID_DEBUG, 'Load mail LIST' );

   POP3.HRequestText( 'LIST', SPECIAL_ACCEPTWEAKENDOFLIST
                           or SPECIAL_ACCEPTNODOTENDOFLIST, nil );

   if not Connected then exit;
   if (POP3.ResultLine = '') or (POP3.ResultCode = 999) then begin // Timeout
      Log( LOGID_WARN, 'GetNewMails canceled, no reply for LIST (' + Server + ')' );
      try Disconnect except end;
      exit;
   end;
   if POP3.ResultCode <> RESULT_OK then begin
      Log( LOGID_WARN, 'GetNewMails failed: LIST' );
      Log( LOGID_WARN, '"' + POP3.ResultCmnd + '" -> "' + POP3.ResultLine + '"' );
      exit;
   end;
   if POP3.FTerminated then exit;

   // are any mails available on server?
   if length( POP3.ResultText ) = 0 then begin
      Log( LOGID_INFO, TaskInfo + 'No mails to load.' );
      exit;
   end;

   // load mails
   Overview := TStringList.Create;
   Parser   := TParser.Create;
   ListUIDL := TStringList.Create;

   try
      Overview.Text := POP3.ResultText;
      MailCount := Overview.Count;

      // report total number and total size of available mails
      SumSize := 0;
      for OvrLfd := 0 to Overview.Count - 1 do begin
         Log( LOGID_DEBUG, 'Mail list entry: ' + Overview[OvrLfd] );
         Parser.Parse( Overview[OvrLfd], ' ' ); // -> (0,)=No. (1,)=Bytes
         inc( SumSize, Parser.iPart( 1, 1 ) );
      end;
      Log( LOGID_INFO, TaskInfo + Format( 'Checking %d mail(s) (%d byte)',
                                          [ MailCount, SumSize ] ));

      // prepare list of unique mail identifiers (UIDL) if already loaded
      // or filtered mails are to be ignored.
      if Hamster.Config.Settings.GetBoo(hsIgnoreAlreadyLoaded) or
         Hamster.Config.Settings.GetBoo(hsIgnoreAlreadyFiltered) then begin

         Log( LOGID_DEBUG, 'Load UIDL list' );

         POP3.HRequestText( 'UIDL', SPECIAL_ACCEPTWEAKENDOFLIST
                                 or SPECIAL_ACCEPTNODOTENDOFLIST, nil );

         if (POP3.ResultLine='') or (POP3.ResultCode=999) then begin // Timeout
            Log( LOGID_WARN, Format(
               'GetNewMails canceled, no reply for UIDL (%s)', [Server] ) );
            try Disconnect except end;
            exit;
         end;

         if POP3.ResultCode = RESULT_OK then begin
            ListUIDL.Text := POP3.ResultText;
            for i := 0 to ListUIDL.Count - 1 do begin
               Log( LOGID_DEBUG, 'UIDL list entry: ' + ListUIDL[i] );
               Parser.Parse( ListUIDL[i], ' ' );
               s := TrimWhSpace( Parser.SPart(1,'') ); // UIDL
               if s = '' then begin
                  ListUIDL.Objects[i] := Pointer( -1 );
                  ListUIDL.Strings[i] := ''
               end else begin
                  ListUIDL.Objects[i] := Pointer( Parser.iPart(0,-1) ); // No.
                  ListUIDL.Strings[i] := Server + '::' + s;
               end;
            end;
         end;

      end;

      // loop through all available mails
      for OvrLfd := 0 to Overview.Count - 1 do begin

         Log( LOGID_DEBUG, 'Process mail #' + inttostr(MailNo) );

         // still connected?
         if not Connected then break;
         if POP3.FTerminated then break;

         // init mail info
         Parser.Parse( Overview[OvrLfd], ' ' );
         MailNo        := Parser.iPart( 0, -1 ); // mail number
         MailSize      := Parser.iPart( 1,  1 ); // mail size
         Mail.Clear;                             // mail message
         MailUidl      := '';                    // UIDL of mail (if requested)
         MailRetrieved := False;                 // true = fill mail retrieved
         MailLoadState := mlsContinue;           // mail processing status
         if MailNo <= 0 then continue;

         // init filter info
         FiltOrigin     := '';    // type (GlobalSize,Action,MailTrap,MailFilt)
         FiltReason     := '';    // specific matching filter
         FiltTopLines   := '';    // TOP lines of filtered mail
         FiltTrapDele   := False; // true if 'delete' result of a mail trap
         FiltTrapScore  := 0;     // score of mail trap
         FiltNewHeaders := '';    // modified headers of action
         FiltDestUsers.Clear;     // local mail recipients
         FiltDestGroups.Clear;    // local mail-to-news newsgroups
         FiltDestNotifys.Clear;   // local notification recipients

         // check if mail was already loaded/filtered
         if MailLoadState = mlsContinue then begin
            CheckIfAlreadyKnown( ListUIDL );
         end;

         // check global mail size limits
         if MailLoadState = mlsContinue then begin
            ApplySizeFilters();
         end;

         // pre-load header lines with TOP
         if MailLoadState = mlsContinue then begin
            if not RetrieveMailHeaders() then break;
         end;

         // apply mail filters (actions, traps, mailfilt, auto-distribution)
         if MailLoadState = mlsContinue then begin
            ApplyFilters( DefaultDestUser );
         end;

         // retrieve full mail and process it
         if MailLoadState = mlsContinue then begin

            if not RetrieveMail() then break;         // retrieve full mail

            ProcessHeaders();                         // add/modify headers
            if not PostToNewsgroups() then break;     // post to newsgroups
            if not StoreInMailboxes() then break;     // store in mailboxes

            if MailLoadState = mlsContinue then begin // success

               MailLoadState := mlsLoaded;
               CounterInc( CounterMailNew );

               // mark as loaded in mail history
               if MailUidl <> '' then Hamster.MailHistory.AddUIDL( MailUidl, 0 );

            end;

         end;

         // handle "leave mais on server, i. e. don't delete"
         if LeaveOnServer then begin

            // if "ignore already filtered mails" is enabled, add the
            // filtered mail to history as well so it will be ignored
            if MailLoadState in [ mlsIgnore, mlsDelete ] then begin
               if Hamster.Config.Settings.GetBoo(hsIgnoreAlreadyFiltered) then begin
                  if MailUidl <> '' then Hamster.MailHistory.AddUIDL( MailUidl, 0 );
               end;
            end;

            // don't execute a 'delete' filter result unless the
            // "leave ... but delete filtered mails" option is enabled
            if MailLoadState = mlsDelete then begin
               if not Hamster.Config.Settings.GetBoo(hsLeaveButDeleteFiltered) then begin
                  MailLoadState := mlsIgnore;
               end;
            end;

            // don't delete loaded mail
            if MailLoadState = mlsLoaded then MailLoadState := mlsIgnore;

         end;

         // delete mail
         case MailLoadState of
            mlsAlreadyKnownExpired: if not DeleteMail() then break;
            mlsDelete:              if not DeleteMail() then break;
            mlsLoaded:              if not DeleteMail() then break;
         end;

         // don't continue after severe errors
         if MailLoadState = mlsError then break;

      end; // for OvrLfd := ...

      // flush mail history
      Hamster.MailHistory.SaveToFile;

   finally
      Parser.Free;
      ListUIDL.Free;
      Overview.Free;
   end
end;

constructor TClientPOP3.Create( AReportSubState: TReportSubStateInfo;
                                const AServerAlias, AServer, APort, AUser, APass,
                                ALocalUser, AFilterSection: String );
var  LfdServer: Integer;
begin
   inherited Create( AReportSubState );

   ServerAlias    := AServerAlias;
   Server         := AServer;
   Port           := APort;
   User           := AUser;
   Pass           := APass;
   LocalUser      := ALocalUser;
   FilterSection  := AFilterSection;

   POP3           := nil;
   ServerDir      := '';
   AutoDistribute := False;
   TaskInfo       := '[' + Server + iif( length(User)>0,' ('+User+')','' ) + ', POP3] ';

   // init server settings
   with Hamster.Config do begin
      BeginRead;
      try
         LfdServer := Pop3Servers.IndexOfAlias( ServerAlias, True );
         if LfdServer >= 0 then begin
            ServerDir      := Pop3Servers.Path[ LfdServer ];
            AutoDistribute := Pop3Servers.Settings(LfdServer).GetBoo(ssPop3AutoDistribute);
         end;
      finally EndRead end;
   end;

   FilterFile := TFiltersMail.Create( AppSettings.GetStr(asPathBase) + CFGFILE_MAILFILT );

   Mail := TMess.Create;

   FiltDestUsers := TStringList.Create;
   FiltDestUsers.Sorted     := True;
   FiltDestUsers.Duplicates := dupIgnore;

   FiltDestGroups := TStringList.Create;
   FiltDestGroups.Sorted     := True;
   FiltDestGroups.Duplicates := dupIgnore;

   FiltDestNotifys := TStringList.Create;
   FiltDestNotifys.Sorted     := True;
   FiltDestNotifys.Duplicates := dupIgnore;
end;

destructor TClientPOP3.Destroy;
begin
   if Assigned( FiltDestNotifys ) then FreeAndNil( FiltDestNotifys );
   if Assigned( FiltDestGroups  ) then FreeAndNil( FiltDestGroups );
   if Assigned( FiltDestUsers   ) then FreeAndNil( FiltDestUsers );
   if Assigned( Mail            ) then FreeAndNil( Mail );

   if Assigned(POP3) then Disconnect;
   if Assigned(FilterFile) then FilterFile.Free;

   inherited Destroy;
end;

end.
