// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cClientNNTP;

interface

{$INCLUDE Compiler.inc}

uses Classes, uType, cClientBase, cFiltersNews;

type
   TClientSocketNNTP = class( TClientBase )
      protected
         function  HResultListFollows: Boolean; override;
         procedure HResultLineToCode; override;
      public
         function  HLogin( AServerAlias, AUser, APass: String; FeedMode: Boolean ): Boolean; reintroduce;
   end;

   TClientNNTP = class( TClientProtocolBase )
      private
         ServerAlias, Server, Port, User, Pass: String;
         ServerDir  : String;
         NNTP: TClientSocketNNTP;
         ScoreFile: TFiltersNews;
         ServerXOverList: array[0..10] of Integer;
         PullInfo: String;
         FFeedMode: Boolean;

         function DoPullSelectGroup( const GroupName: String;
                                     const GroupHandle: Integer;
                                     out   SrvHasMin, SrvHasMax: Integer ): Boolean;
         function DoPullSelectRange( const GroupName: String;
                                     const GroupHandle: Integer;
                                     const SrvHasMin, SrvHasMax: Integer;
                                     out   WeWantMin, WeWantMax: Integer ): Boolean;
         function DoPullGetOverview( const WeWantMin, WeWantMax: Integer;
                                     var   Overview: TStringList;
                                     var   ByNumOnly: Boolean ): Boolean;
         function DoPullGetArticle ( const GroupName: String;
                                     const GroupHandle: Integer;
                                     const WeWantCur: Integer;
                                     var   Score, ScoreAL: Integer ): Boolean;
         function DoPullGetArticles( const GroupName: String;
                                     const GroupHandle: Integer;
                                     const Overview: TStringList;
                                     const ByNumOnly: Boolean;
                                     var   FirstPullDone: Boolean;
                                     var   RaiseLowMark: Boolean;
                                     var   NewLowMark: Integer ): Boolean;

      public
         procedure Terminate;
         function  Terminated: Boolean;

         procedure Connect;
         procedure Disconnect;
         function  Connected: Boolean;

         procedure GetServerInfos;
         procedure GetNewGroups;
         procedure GetListOfMIDs;
         procedure GetNewArticlesForGroup( const GroupName: String;
                                           const GroupHandle: Integer );
         procedure FeedNewArticlesOfGroup( const GroupName: String;
                                           const GroupHandle: Integer );
         function  PostArticle( const PostFile, ArtText: String;
                                var   SrvResult: String ): Boolean;

         constructor Create( AReportSubState: TReportSubStateInfo;
                             const AServerAlias, AServer, APort, AUser, APass: String;
                             AFeedMode: Boolean );
         destructor Destroy; override;
   end;

implementation

uses uConst, uConstVar, uVar, Windows, SysUtils, uTools,
     uCRC32, cArtFiles, cArticle, uDateTime, cSyncObjects,
     cLogFileHamster, cHamster, uHamTools, cHscAction, IdSSLOpenSSL;

var
   ArtCurLoading: TThreadStringList;

{ TClientSocketNNTP }

procedure TClientSocketNNTP.HResultLineToCode;
begin
   if FResultLine='' then FResultLine := '999 (timeout or connection lost)';
   FResultCode := strtoint( copy( FResultLine, 1, 3 ) );
   if not HConnected then FResultCode := 999;
end;

function TClientSocketNNTP.HResultListFollows: Boolean;
begin
   Result := ( ResultCode < 400 );
end;

function TClientSocketNNTP.HLogin( AServerAlias, AUser, APass : String; FeedMode: Boolean ): Boolean;
var  SslOk: Boolean;
     s: string;
begin
   Result := False;
   if not HConnected then exit;

   try
      Log( LOGID_DEBUG, 'NntpClient.Login' );

      // SSL-TLS handshake
      if FUseSslMode = sslTLS then begin

         // determine if server supports TLS handshake
         SslOk := False;

         HRequestText( 'LIST EXTENSIONS', 0 , nil );
         if not HConnected then exit;
         if ResultCode = 202 then begin
            SslOk := pos( 'starttls', LowerCase(ResultText) ) > 0;
         end;

         if not SslOk then begin
            HRequestText( 'HELP', 0 , nil );
            if not HConnected then exit;
            if ResultCode = 100 then begin
               SslOk := pos( 'starttls', LowerCase(ResultText) ) > 0;
            end;
         end;

         if SslOk then begin

            // Handle TLS handshake
            s := StartTlsHandshake( 'STARTTLS', 399, 'HELP', 100, True );
            if s = '' then begin
               Log( LOGID_ERROR,
                    'Required SSL with TLS handshake was not activated!' );
               exit;
            end;

         end else begin

            Log( LOGID_ERROR,
                 FServer + ' does not support required TLS handshake!' );
            exit;

         end;
      end;

      if not FeedMode then begin
         HRequest( 'MODE READER', 0, False );
         if not HConnected then exit;
         if ResultCode >= 400 then {ignore};
      end;

      if (AUser<>'') and (APass<>'') then begin
         HRequest( 'AUTHINFO USER ' + AUser, 0, True );
         if not HConnected then exit;
         if ResultCode >= 400 then exit;

         HRequest( 'AUTHINFO PASS ' + APass, 0, True );
         if not HConnected then exit;
         if ResultCode >= 400 then exit;
      end;

      Result := HConnected;

   finally
      if (not Result) and HConnected then HDisconnect;
   end;
end;


{ TClientNNTP }

procedure TClientNNTP.Terminate;
begin
   if Assigned( NNTP ) then NNTP.Terminate;
end;

function TClientNNTP.Terminated: Boolean;
begin
   Result := Assigned( NNTP ) and NNTP.FTerminated;
end;

function TClientNNTP.Connected: Boolean;
begin
   if Assigned( NNTP ) then Result := NNTP.HConnected else Result := False;
end;

procedure TClientNNTP.Connect;
var  s: String;
     UseSOCKS: Boolean;
     UseSslMode: TSslTlsMode;
     LfdServer, TimeoutConnect, TimeoutCommand, i: Integer;
     SSLVersion: TIdSSLVersion;
     SSLCipher : String;
begin
   if Assigned( NNTP ) then Disconnect;

   if ServerDir = '' then begin
      Log( LOGID_ERROR, 'NNTP server "' + ServerAlias + '" is not configured!' );
      exit;
   end;

   TimeoutConnect := Hamster.Config.Settings.GetInt(hsRemoteTimeoutConnect);
   TimeoutCommand := Hamster.Config.Settings.GetInt(hsRemoteTimeoutCommand);
   UseSOCKS   := False;
   UseSslMode := sslNone;
   SSLVersion := sslvSSLv3;
   SSLCipher  := '';

   Hamster.Config.BeginRead;
   try
      with Hamster.Config.NntpServers do begin
         LfdServer := IndexOfAlias( ServerAlias );
         if LfdServer >= 0 then begin
            i := Settings( LfdServer ).GetInt( ssTimeoutConnect );
            if i > 0 then TimeoutConnect := i;
            i := Settings( LfdServer ).GetInt( ssTimeoutCommand );
            if i > 0 then TimeoutCommand := i;
            UseSOCKS   := Settings( LfdServer ).GetBoo( ssUseSocks );
            UseSslMode := TSslTlsMode( Settings(LfdServer).GetInt(ssSSLMode) );
            SSLVersion := TIdSSLVersion( Settings(LfdServer).GetInt(ssSSLVersion) );
            SSLCipher  := Settings( LfdServer ).GetStr( ssSSLCipher );
         end;
      end;
   finally Hamster.Config.EndRead end;

   NNTP := TClientSocketNNTP.Create( nil, 'nntp' );

   if NNTP.HConnect( Server, Port, UseSOCKS, UseSslMode, SSLVersion, SSLCipher,
                     0, TimeoutConnect*1000, TimeoutCommand*1000 ) then begin
      if not NNTP.HLogin( ServerAlias, User, Pass, FFeedMode ) then begin
         s := NNTP.LastErrResult;
         if s = '' then s := NNTP.ResultLine;
         if s <> '' then s := ' ("' + s + '")';
         Log( LOGID_WARN, 'Login failed ' + s );
      end;
      if NNTP.Greeting<>'' then begin
         HamFileRewriteLine( ServerDir + SRVFILE_GREETING, NNTP.Greeting );
      end;
   end;
end;

procedure TClientNNTP.Disconnect;
begin
   if not Assigned( NNTP ) then exit;
   try NNTP.HDisconnect except end;
   FreeAndNil( NNTP );
end;

procedure TClientNNTP.GetServerInfos;

   procedure GetList( Cmnd, Filename: String );
   begin
      if not Connected then exit;

      Log( LOGID_DETAIL, '[' + Server + '] Loading <' + Cmnd + '> ...' );

      NNTP.HRequestText( Cmnd, 0, nil );
      if not Connected then exit;

      if NNTP.ResultStrm.Size = 0 then begin
         if Cmnd='LIST' then begin
            if NNTP.ResultCode<400 then
               Log( LOGID_WARN, 'No newsgroups available on "LIST"' )
            else
               Log( LOGID_WARN, 'Couldn''t load "' + Cmnd + '" (' + NNTP.ResultLine + ')' );
            exit;
         end else begin
            StreamWriteLn( NNTP.ResultStrm, '# Command: ' + NNTP.ResultCmnd );
            StreamWriteLn( NNTP.ResultStrm, '# Result : ' + NNTP.ResultLine );
            Log( LOGID_DEBUG, 'Couldn''t load "' + Cmnd + '" (' + NNTP.ResultLine + ')' );
         end;
      end;

      try
         StreamToFile( NNTP.ResultStrm, Filename );
      except
         on E: Exception do begin
            Log( LOGID_ERROR, 'Couldn''t save "' + Cmnd + '" to "' + Filename + '"' );
            Log( LOGID_ERROR, 'ERROR: ' + E.Message );
         end;
      end;
   end;

var  SrvFile, LastNewGroups: String;
     LfdServer: Integer;
     ReloadGroups, ReloadDescs: Boolean;
begin
   SrvFile := ServerDir + SRVFILE_HELPTEXT;
   if not FileExists( SrvFile ) then GetList( 'HELP', SrvFile );

   if not FFeedMode then begin
      SrvFile := ServerDir + SRVFILE_OVERVIEWFMT;
      if not FileExists( SrvFile ) then GetList( 'LIST OVERVIEW.FMT', SrvFile );
   end;

   ReloadGroups := False;
   ReloadDescs  := False;
   LastNewGroups := '';

   Hamster.Config.BeginRead;
   try
      with Hamster.Config.NntpServers do begin
         LfdServer := IndexOfAlias( ServerAlias );
         if LfdServer >= 0 then begin
            ReloadGroups  := Settings(LfdServer).GetBoo( ssNntpReloadGroups  );
            ReloadDescs   := Settings(LfdServer).GetBoo( ssNntpReloadDescs   );
            LastNewGroups := Settings(LfdServer).GetStr( ssNntpLastNewGroups );
         end;
      end;
   finally Hamster.Config.EndRead end;

   if not FileExists( ServerDir + SRVFILE_GROUPS ) then ReloadGroups := True;

   if ReloadGroups then begin
      // load complete list of available groups
      SrvFile := ServerDir + SRVFILE_GROUPS;
      GetList( 'LIST', SrvFile );

      Hamster.Config.BeginWrite;
      try
         with Hamster.Config.NntpServers do begin
            LfdServer := IndexOfAlias( ServerAlias );
            if LfdServer >= 0 then begin
               if Uppercase( LastNewGroups ) <> 'NEVER' then begin
                  Settings(LfdServer).SetDT( ssNntpLastNewGroups, NowGMT );
               end;
               Settings(LfdServer).SetBoo( ssNntpReloadGroups, False );
               Settings(LfdServer).SetDT ( ssNntpReloadGroupsDT, Now );
               Settings(LfdServer).Flush;
            end;
         end;
      finally Hamster.Config.EndWrite end;

      GlobalListMarker( glOPEN );
   end else begin
      // load new groups only
      GetNewGroups;
   end;

   if ReloadDescs then begin
      SrvFile := ServerDir + SRVFILE_GRPDESCS;
      GetList( 'LIST NEWSGROUPS', SrvFile );

      Hamster.Config.BeginWrite;
      try
         with Hamster.Config.NntpServers do begin
            LfdServer := IndexOfAlias( ServerAlias );
            if LfdServer >= 0 then begin
               Settings(LfdServer).SetBoo( ssNntpReloadDescs, False );
               Settings(LfdServer).SetDT ( ssNntpReloadDescsDT, Now );
               Settings(LfdServer).Flush;
            end;
         end;
      finally Hamster.Config.EndWrite end;

      GlobalListMarker( glOPEN );
   end;
end;

procedure TClientNNTP.GetNewGroups;
var  LastGMT : TDateTime;
     s, cmd, NewGroups, GrpName, GrpDesc, LastNewGroups: String;
     i, GrpCurr, DscCurr, LoadDescs, LfdServer, FixNewGrpY2K: Integer;
     TS, TSD, SL: TStringList;
     Changed : Boolean;
begin
   if not Connected then exit;

   if DateTimeToTimeStamp(NowGMT)<'19990906' then begin
      Log( LOGID_WARN, 'GetNewGroups ' + Server + ', invalid pc-date: '
                     + DateTimeToTimeStamp(NowGMT) );
      exit;
   end;

   LoadDescs     := 0;  // Load descs? 0=auto-detect, 1=always, 2=never
   LastNewGroups := '';
   FixNewGrpY2K  := 0;
   Changed       := False;

   Hamster.Config.BeginRead;
   try
      with Hamster.Config.NntpServers do begin
         LfdServer := IndexOfAlias( ServerAlias );
         if LfdServer >= 0 then begin
            LoadDescs     := Settings(LfdServer).GetInt( ssNntpLoadDescsMode );
            LastNewGroups := Settings(LfdServer).GetStr( ssNntpLastNewGroups );
            FixNewGrpY2K  := Settings(LfdServer).GetInt( ssNntpNewGroupsY2K  );
         end;
      end;
   finally Hamster.Config.EndRead end;

   s := LastNewGroups;
   if Uppercase(s) = 'NEVER' then exit;
   if s<>'' then LastGMT := TimeStampToDateTime( s )
            else LastGMT := NowGMT;

   if ( LastGMT > NowGMT ) or ( NowGMT-LastGMT > 365 {days} ) then begin
      Log( LOGID_WARN, 'GetNewGroups ' + Server + ', invalid date assumed: '
                     + ' Last=' + DateTimeToTimeStamp(LastGMT)
                     + ' Now='  + DateTimeToTimeStamp(NowGMT ) );
      exit;
   end;

   try
      if FixNewGrpY2K = 1 then begin
         s := FormatDateTime( 'yyyymmdd hhnnss', LastGMT ) + ' GMT';
      end else begin
         s := FormatDateTime( 'yymmdd hhnnss',   LastGMT ) + ' GMT';
      end;
      LastGMT := NowGMT;
      cmd := 'NEWGROUPS ' + s;
      NNTP.HRequestText( cmd, 0, nil );

      if NNTP.ResultCode < 400 then begin

         Changed := True;

         if NNTP.ResultStrm.Size > 0 then begin
            NewGroups := NNTP.ResultText;
            TS  := TStringList.Create;
            TSD := TStringList.Create;

            // add new groups to group-list (no "dupe"-check!)
            TS.Text := NewGroups;
            HamFileAppendList( ServerDir + SRVFILE_GROUPS, TS );
            GlobalListMarker( glOPEN );

            // load descriptions of new groups
            TS.Text := NewGroups;
            TSD.Clear;
            if LoadDescs <> 2 then begin
               for GrpCurr:=0 to TS.Count - 1 do begin
                  GrpName := TS[GrpCurr];
                  i := PosWhSpace( GrpName );
                  if i>0 then GrpName:=copy(GrpName,1,i-1);

                  NNTP.HRequestText( 'LIST NEWSGROUPS ' + GrpName, 0, nil );
                  if not Connected then break;
                  if NNTP.FTerminated then break;
                  if (NNTP.ResultLine='') or (NNTP.ResultCode=999) then begin // Timeout
                     Log( LOGID_WARN, Format(
                        'Pull canceled, no reply for LIST NEWSGROUPS (%s)', [Server] ));
                     try Disconnect except end;
                     break;
                  end;

                  GrpDesc := '';
                  if (NNTP.ResultCode<400) and (NNTP.ResultStrm.Size>0) then begin
                     if (LoadDescs=0) and (NNTP.ResultStrm.Size>10*60) then begin
                        // Server returned a large list, so it looks like the server
                        // does not support LIST NEWSGROUPS with a groupname pattern.
                        // Disable immediate loading of descriptions to prevent from
                        // further trials:
                        LoadDescs := 2; // never
                        Changed := True;
                        break;
                     end;

                     SL := TStringList.Create;
                     try
                        SL.Text := NNTP.ResultText;
                        for DscCurr := 0 to SL.Count-1 do begin
                           s := SL[DscCurr];
                           i := PosWhSpace( s );
                           if i>0 then begin
                              if AnsiCompareText( GrpName, copy(s,1,i-1) )=0 then begin
                                 GrpDesc := TrimWhSpace( copy( s, i+1, 255 ) );
                                 break;
                              end;
                           end;
                        end;
                     finally SL.Free end;
                  end;
                  TSD.Add( GrpDesc );
               end;
            end;

            // post note to hamster.misc
            TS.Text := NewGroups;
            s := '';
            for i:=0 to TS.Count-1 do begin
               s := s + TS[i] + #13#10;
               GrpDesc := '';
               if i<TSD.Count then begin
                  if TSD[i]<>'' then GrpDesc:=TSD[i];
               end;
               if GrpDesc='' then GrpDesc:='?';
               s := s + ' ' + #9 + GrpDesc + #13#10;
            end;
            s := s + #13#10
                   + '-- ' + #13#10
                   + 'Result of "' + cmd + '"' + #13#10
                   + 'Sent at ' + FormatDateTime( 'yymmdd hhnnss', LastGMT ) + ' GMT';
            SaveInInternalGroup( hsHamGroupNewGroups,
                                 '[' + Server + '] New groups', s );

            // save new groups in global list of groups (may cause duplicates,
            // which will be fixed when rebuilding global lists)
            TS.Text := NewGroups;
            s := '';
            for GrpCurr:=0 to TS.Count-1 do begin
               GrpName := TS[GrpCurr];
               i := PosWhSpace( GrpName );
               if i>0 then GrpName:=copy(GrpName,1,i-1);
               if GrpCurr<TSD.Count then GrpDesc:=TSD[GrpCurr]
                                    else GrpDesc:='?';
               if GrpDesc='' then GrpDesc:='?';
               s := s + GrpName + #9 + GrpDesc + #13#10;
            end;
            TS.Text := s;
            HamFileAppendList( ServerDir + SRVFILE_GRPDESCS, TS );
            CS_MAINTENANCE.Enter;
            try
               HamFileAppendList( AppSettings.GetStr(asPathServer) + SRVFILE_ALLDESCS, TS );
               GlobalListMarker( glOPEN );
            finally CS_MAINTENANCE.Leave end;

            TSD.Free;
            TS.Free;
         end;
      end;

   finally
      if Changed then begin
         Hamster.Config.BeginWrite;
         try
            with Hamster.Config.NntpServers do begin
               LfdServer := IndexOfAlias( ServerAlias );
               if LfdServer >= 0 then begin
                  Settings(LfdServer).SetDT ( ssNntpLastNewGroups, LastGMT   );
                  Settings(LfdServer).SetInt( ssNntpLoadDescsMode, LoadDescs );
                  Settings(LfdServer).Flush;
               end;
            end;
         finally Hamster.Config.EndWrite end;
      end;
   end;
end;

procedure TClientNNTP.GetListOfMIDs;
var  MessageID    : String;
     MidList      : TStringList;
     LfdArticle   : TMess;
     MidListLfd   : Integer;
     IgnoreHistory: Boolean;
begin
   if not FileExists( ServerDir + SRVFILE_GETMIDLIST ) then exit;

   MidList := TStringList.Create;
   MidList.LoadFromFile( ServerDir + SRVFILE_GETMIDLIST );
   DeleteFile( ServerDir + SRVFILE_GETMIDLIST );

   LfdArticle := TMess.Create;

   try
      for MidListLfd := MidList.Count - 1 downto 0 do begin
         if NNTP.FTerminated then break;
         if not Connected then break;

         MessageID     := MidList[MidListLfd];
         IgnoreHistory := False;
         if copy( MessageID, 1, 1 ) = '!' then begin
            IgnoreHistory := True;
            System.Delete( MessageID, 1, 1 );
         end;

         if (MessageID <> '') and not IgnoreHistory then begin
            if Hamster.NewsHistory.ContainsMID( MessageID ) then begin
               Log( LOGID_Debug, '[' + Server + '] GetByMID '
                                     + MessageID + ': already in history' );
               MidList.Delete( MidListLfd );
               MessageID := '';
            end;
         end;

         if MessageID <> '' then try
            NNTP.HRequestText( 'ARTICLE ' + MessageID, 0, nil );
            if not Connected then break;
            Log( LOGID_Debug, '[' + Server + '] GetByMID '
                                  + MessageID + ': ' + NNTP.ResultLine );

            if NNTP.ResultCode < 400 then begin
               CounterInc( CounterArtNew );
               LfdArticle.FullText := NNTP.ResultText;
               if SaveArticle( LfdArticle, '', -1, '(none-getbymid)',
                               IgnoreHistory, '', actIgnore ) then begin
                  CounterInc( CounterArtLoad );
               end else begin
                  CounterInc( CounterArtHist );
               end;
            end else begin
               if NNTP.ResultCode >= 500 then break;
            end;
         finally
            MidList.Delete( MidListLfd );
         end;
      end;

   finally
      if MidList.Count > 0 then begin
         HamFileAppendList( ServerDir + SRVFILE_GETMIDLIST, MidList );
      end;
      MidList.Free;
      LfdArticle.Free;
   end;
end;

function TClientNNTP.DoPullSelectGroup( const GroupName: String;
                                        const GroupHandle: Integer;
                                        out   SrvHasMin, SrvHasMax: Integer ): Boolean;
// send GROUP and return server's article range
var  SeemsOK: Boolean;
     Parser : TParser;
     i: Integer;
begin
   Result := False;

   NNTP.HRequest( 'GROUP ' + GroupName, 0, False );
   if NNTP.FTerminated then exit;
   if not Connected then exit;
   if (NNTP.ResultLine='') or (NNTP.ResultCode=999) then begin // Timeout
      Log( LOGID_WARN, Format(
         'Pull canceled, no reply for GROUP (%s)', [Server] ));
      try Disconnect except end;
      exit;
   end;
   if NNTP.ResultCode >= 400 then begin
      Log( LOGID_WARN, Format(
         'Pull failed: %s', [Groupname] ));
      Log( LOGID_WARN, '[' + Server + '] "'   + NNTP.ResultCmnd + '"'
                                    + ' -> "' + NNTP.ResultLine + '"' );
      SaveInInternalGroup( hsHamGroupPullErrors, PullInfo + 'Pull error!',
                           'GROUP-reply was: ' + #13#10 + NNTP.ResultLine );
      exit;
   end;

   // read group-infos
   SeemsOK := True;
   Parser := TParser.Create;
   try
      Parser.Parse( NNTP.ResultLine, ' ' ); // 0:err 1:count 2:first 3:last 4:name
      i         := Parser.iPart( 0, -1 );  if i        <0 then SeemsOK:=False;
      i         := Parser.iPart( 1, -1 );  if i        <0 then SeemsOK:=False;
      SrvHasMin := Parser.iPart( 2, -1 );  if SrvHasMin<0 then SeemsOK:=False;
      SrvHasMax := Parser.iPart( 3, -1 );  if SrvHasMax<0 then SeemsOK:=False;
   finally
      Parser.Free;
   end;
   if not SeemsOK then begin
      Log( LOGID_WARN, 'Pull canceled: ' + GroupName );
      Log( LOGID_WARN, 'Invalid GROUP-reply assumed: "'+NNTP.ResultLine+'"' );
      SaveInInternalGroup( hsHamGroupPullErrors, PullInfo + 'Pull error!',
                  'Invalid GROUP-reply assumed: ' + #13#10 + NNTP.ResultLine );
      try Disconnect except end;
      exit;
   end;

   Hamster.ArticleBase.SetDT( GroupHandle, gsLastServerPull, Now );
   Result := True;
end;

function TClientNNTP.DoPullSelectRange( const GroupName: String;
                                        const GroupHandle: Integer;
                                        const SrvHasMin, SrvHasMax: Integer;
                                        out   WeWantMin, WeWantMax: Integer ): Boolean;
// return the article range to load (returns False if none or on errors)
var  s: String;
     i: Integer;
begin
   Result := False;

   s := 'Old .Min/.Max: ' + inttostr(Hamster.ArticleBase.ServerMin[GroupHandle,Server])
                    + '-' + inttostr(Hamster.ArticleBase.ServerMax[GroupHandle,Server])
            + '  Limit: ' + inttostr(Hamster.ArticleBase.PullLimit[GroupHandle,Server]);
   Log( LOGID_DEBUG, PullInfo + s );
   s := 'Srv .Min/.Max: ' + inttostr(SrvHasMin)
                    + '-' + inttostr(SrvHasMax);
   Log( LOGID_DEBUG, PullInfo + s );

   // find out artno.-range to load
   WeWantMin := Hamster.ArticleBase.ServerMin[GroupHandle,Server] + 1; // default: from (last loaded + 1)
   WeWantMax := SrvHasMax;                                     // default: to (current max.)
   if WeWantMin<SrvHasMin then WeWantMin:=SrvHasMin; // adjust for expire-gap or new group
   if WeWantMin>SrvHasMax+100 then begin // server restarted with new numbers?
      s := 'Old('     + inttostr(Hamster.ArticleBase.ServerMin[GroupHandle,Server])
            + '-'     + inttostr(Hamster.ArticleBase.ServerMax[GroupHandle,Server])
         + ') Limit(' + inttostr(Hamster.ArticleBase.PullLimit[GroupHandle,Server])
         + ') New('   + inttostr(SrvHasMin) + '-' + inttostr(SrvHasMax) + ')';
      Log( LOGID_DEBUG, PullInfo + 'Renumber triggered: ' + s );
      WeWantMin := SrvHasMin;
      WeWantMax := SrvHasMax;
   end;
   i := Hamster.ArticleBase.PullLimit[GroupHandle,Server]; // adjust for pull-limit per session
   if (i<>0) and ( (WeWantMax-WeWantMin)>abs(i) ) then begin
      if i>0 then WeWantMin := WeWantMax - i + 1  // only (limit) newest
             else WeWantMax := WeWantMin - i - 1; // only (limit) oldest
   end;
   if WeWantMin<SrvHasMin then WeWantMin:=SrvHasMin;
   if WeWantMax>SrvHasMax then WeWantMax:=SrvHasMax;
   Hamster.ArticleBase.ServerMin[GroupHandle,Server] := WeWantMin - 1; // save first art-no we have (maybe ficticious)
   Hamster.ArticleBase.ServerMax[GroupHandle,Server] := SrvHasMax;     // save last available art-no. (for info only)
   if WeWantMin>WeWantMax then exit; // no new articles
   if WeWantMax=0 then exit; // no articles

   Log( LOGID_INFO, PullInfo + Format( 'Server has %s-%s. We want %s-%s.',
      [inttostr(SrvHasMin), inttostr(SrvHasMax), inttostr(WeWantMin), inttostr(WeWantMax) ]));

   Result := True;
end;

function TClientNNTP.DoPullGetOverview( const WeWantMin, WeWantMax: Integer;
                                        var   Overview: TStringList;
                                        var   ByNumOnly: Boolean ): Boolean;
// return overview for the given range; if server doesn't support XOVER,
// ByNumOnly is set to False and Overview is just filled with wanted numbers
var  OvrLfd: Integer;
begin
   Result := False;
   Overview.Clear;

   if not ByNumOnly then begin

      // get overview of articles
      Log( LOGID_STATUS, PullInfo + Format(
           'Loading overview %d-%d ...', [ WeWantMin, WeWantMax] ) );
      NNTP.HRequestText( 'XOVER ' + inttostr(WeWantMin)
                            + '-' + inttostr(WeWantMax), 0, nil );
      if not Connected then exit;
      if NNTP.FTerminated then exit;
      if (NNTP.ResultLine='') or (NNTP.ResultCode=999) then begin // Timeout
         Log( LOGID_WARN, 'Pull canceled, no reply for XOVER (' + Server + ')' );
         try Disconnect except end;
         exit;
      end;

      // prepare overview-list for scoring/download
      if NNTP.ResultCode >= 400 then begin
         Log( LOGID_INFO, 'XOVER-result'+': ' + NNTP.ResultLine );
         Log( LOGID_INFO, 'XOVER-failed -> ignore scorefile, load by number' );
         ByNumOnly := True;
      end else begin
         ByNumOnly := False;
         Overview.Text := NNTP.ResultText;
      end;

   end;

   if ByNumOnly then begin
      for OvrLfd:=WeWantMin to WeWantMax do OverView.Add( inttostr(OvrLfd) );
   end;

   Result := True;
end;

function TClientNNTP.DoPullGetArticle( const GroupName: String;
                                       const GroupHandle: Integer;
                                       const WeWantCur: Integer;
                                       var   Score, ScoreAL: Integer ): Boolean;
// load article from server and save it in local group
var  Art: TMess;
     ResultCode, ScoreBL: Integer;
     ScoreStr: String;
begin
   Result  := False;
   ScoreAL := 0;

   // load article
   NNTP.HRequestText( 'ARTICLE ' + inttostr(WeWantCur), 0, nil );
   if not Connected then exit;
   if NNTP.FTerminated then exit;
   if (NNTP.ResultLine='') or (NNTP.ResultCode=999) then begin
      Log(LOGID_WARN, PullInfo+'Pull cancelled, no reply for ARTICLE');
      try Disconnect except end;
      exit;
   end;

   ResultCode := NNTP.ResultCode;
   if ResultCode < 400 then begin
      if NNTP.ResultStrm.Size < 32 then ResultCode := 423;
   end;

   if ResultCode < 400 then begin
      Art := TMess.Create;
      try
         // get loaded article
         Art.FullText := NNTP.ResultText;

         // get "Score-After-Load" value for article
         ScoreAL := ScoreFile.ScoreAfterLoad( Art, nil );

         // calculate final score value (= before + after load)
         ScoreBL := Score;
         inc( Score, ScoreAL );
         if Score < -9999 then Score := -9999;
         if Score > +9999 then Score := +9999;

         // save article if "Score-After-Load" is also >= 0
         if ScoreAL >= 0 then try
            if ScoreAL = 0 then begin
               ScoreStr := inttostr(Score);
            end else begin
               ScoreStr := Format( '%d ScoreLoad=%d ScoreSave=%d',
                                   [ Score, ScoreBL, ScoreAL ] );
            end;

            if SaveArticle( Art, GroupName, GroupHandle, ScoreStr,
                            False, '', actNewsIn ) then begin
               CounterInc( CounterArtLoad );
            end else begin
               CounterInc( CounterArtHist );
            end;
         except
            NNTP.Terminate;
            raise;
         end;

      finally
         Art.Free;
      end;
   end else begin
      // 423 no such article number in this group
      // 430 no such article found
      CounterInc( CounterArtHist );
      if (ResultCode <> 423) and (ResultCode <> 430) then exit;
   end;

   Result := True;
end;

function TClientNNTP.DoPullGetArticles( const GroupName: String;
                                        const GroupHandle: Integer;
                                        const Overview: TStringList;
                                        const ByNumOnly: Boolean;
                                        var   FirstPullDone: Boolean;
                                        var   RaiseLowMark: Boolean;
                                        var   NewLowMark: Integer ): Boolean;
// score overview and load wanted articles from server
var  OvrLfd, WeWantCur, Score, ScoreAL, Counter: Integer;
     ActMsgID, s: String;
     Parser: TParser;
     XOverRec: TXOverRec;
     CanLoad: Boolean;
begin
   Result := False;

   Parser := TParser.Create;

   try
      // load articles
      if OverView.Count=0 then begin
         Log( LOGID_INFO, PullInfo + 'No articles to load.' );
      end else begin
         Log( LOGID_INFO, PullInfo + Format(
                 'Loading up to %s articles ...', [inttostr(Overview.Count)]));
      end;

      Counter := 0;
      
      for OvrLfd:=0 to Overview.Count-1 do begin
         Log( LOGID_STATUS, PullInfo + Format(
              'Loading article %s of max. %s ...', [inttostr(OvrLfd+1),
              inttostr(Overview.Count)]));

         if not Connected then exit;
         if NNTP.FTerminated then exit;

         CounterInc( CounterArtNew );

         // XOVER: 0:No. 1:Subject 2:From 3:Date 4:Message-ID 5:References
         //              6:Bytes 7:Lines [8:Xref full]
         Parser.Parse( Overview[OvrLfd], #9 );
         WeWantCur := Parser.iPart( 0, -1 );
         if WeWantCur>=0 then begin

            Score    := 0;
            ScoreAL  := 0;
            ActMsgID := '';

            if not ByNumOnly then begin

               if Score>=0 then begin // check if article is already in history
                  ActMsgID := Parser.sPart( 4, '' ); // Message-ID
                  if (ActMsgID='') or Hamster.NewsHistory.ContainsMID(ActMsgID) then begin
                     Score := -10000; // already in history
                     CounterInc( CounterArtHist );
                  end;
               end;

               if Score>=0 then begin // check scorefile
                  XOverToXOverRec( XOverRec, Parser, ServerXOverList );
                  Score := ScoreFile.ScoreBeforeLoad( XOverRec, nil );
                  if Score<0 then CounterInc( CounterArtKill );
               end;

            end;

            CanLoad := True;
            if (Score>=0) and (ActMsgID<>'') then begin
               // check, if current MID is currently loaded by another thread
               CanLoad := ArtCurLoading.Add( ActMsgID );
            end;

            if (Score>=0) and CanLoad then begin

               try // load article
                  inc( Counter );
                  if (Counter mod 10) = 0 then begin
                     ReportSubState(
                        inttostr( trunc( (OvrLfd+1) / Overview.Count * 100 ) ) + '%'
                     );
                  end;

                  if not DoPullGetArticle( GroupName, GroupHandle,
                                           WeWantCur,
                                           Score, ScoreAL ) then exit;
               finally
                  if ActMsgID<>'' then ArtCurLoading.Remove( ActMsgID );
               end;

               if ScoreAL < 0 then begin
                  // article loaded but not saved due to "Score-After-Load"
                  CounterInc( CounterArtKill );
                  if ScoreAL >= Hamster.Config.Settings.GetInt(hsScoreLogLimit) then begin
                     s := Server + #9 + GroupName
                                 + #9 + inttostr(ScoreAL)
                                 + #9 + Overview[OvrLfd];
                     HamFileAppendLine( AppSettings.GetStr(asPathGroups) + GRPFILE_KILLSLOG, s );
                  end else begin
                     Log( LOGID_DEBUG, 'Skipped (ScoreAL=' + inttostr(ScoreAL)
                                                 + '): ' + Overview[OvrLfd] );
                  end;
               end;

            end else begin

               if (Score>=0) and not CanLoad then begin
                  RaiseLowMark := False;
                  Log( LOGID_DEBUG, 'Skipped (Concurrent): ' + Overview[OvrLfd] );
               end else if Score=-10000 then begin
                  Log( LOGID_DEBUG, 'Skipped (History): ' + Overview[OvrLfd] );
               end else begin
                  if Score >= Hamster.Config.Settings.GetInt(hsScoreLogLimit) then begin
                     s := Server + #9 + GroupName
                                 + #9 + inttostr(Score)
                                 + #9 + Overview[OvrLfd];
                     HamFileAppendLine( AppSettings.GetStr(asPathGroups) + GRPFILE_KILLSLOG, s );
                  end else begin
                     Log( LOGID_DEBUG, 'Skipped (score=' + inttostr(Score)
                                                 + '): ' + Overview[OvrLfd] );
                  end;
               end;
            end;

            if RaiseLowMark then NewLowMark := WeWantCur;
            FirstPullDone := True;
         end;
      end;

      Result := True;

   finally
      Parser.Free;
   end;
end;

procedure TClientNNTP.GetNewArticlesForGroup( const GroupName: String;
                                              const GroupHandle: Integer );
const SPLIT_SIZE  = 200;
      SPLIT_LIMIT = SPLIT_SIZE + SPLIT_SIZE div 2; 
var  SrvHasMin, SrvHasMax, NewLowMark: Integer;
     WeWantMin, WeWantMax, WeWantTempMax: Integer;
     Overview : TStringList;
     ByNumOnly, FirstPullDone, RaiseLowMark: Boolean;
begin
   if not Connected then exit;

   PullInfo := '[' + Server + ', ' + GroupName + '] ';
   Log( LOGID_DETAIL, PullInfo + 'GetNewArticles' );

   // select group, get available article range
   if not DoPullSelectGroup( GroupName, GroupHandle,
                             SrvHasMin, SrvHasMax ) then exit;

   // get wanted article range
   if not DoPullSelectRange( GroupName, GroupHandle,
                             SrvHasMin, SrvHasMax,
                             WeWantMin, WeWantMax ) then exit;

   // pull articles
   FirstPullDone := False;
   RaiseLowMark  := True;
   NewLowMark    := -1;
   ByNumOnly     := False;
   Overview      := TStringList.Create;

   try
      // prepare scorelist for current group
      ScoreFile.SelectSections( GroupName );

      // pull
      repeat
         // split large pulls
         if (WeWantMax-WeWantMin) > SPLIT_LIMIT then begin
            WeWantTempMax := WeWantMin + SPLIT_SIZE - 1;
            if WeWantTempMax > WeWantMax then WeWantTempMax := WeWantMax;
         end else begin
            WeWantTempMax := WeWantMax;
         end;
         if WeWantMin > WeWantTempMax then break;

         // load overview for wanted range
         if not DoPullGetOverview( WeWantMin, WeWantTempMax,
                                   Overview, ByNumOnly ) then exit;
         if (NewLowMark<0) and (Overview.Count=0) then NewLowMark := WeWantMax;

         // score and load articles
         if not DoPullGetArticles( GroupName, GroupHandle,
                                   Overview, ByNumOnly,
                                   FirstPullDone,
                                   RaiseLowMark, NewLowMark ) then break;

         WeWantMin := WeWantTempMax + 1;
      until False;

   finally
      Overview.Free;
   end;

   if FirstPullDone then begin
      // first pull for Group/Server is from now on assumed as "done"
      Hamster.ArticleBase.SetFirstPullDone( GroupHandle, Server );
   end;

   if NewLowMark >= 1 then begin
      // save last art-no processed
      Hamster.ArticleBase.ServerMin[ GroupHandle, Server ] := NewLowMark;
   end;
end;

function TClientNNTP.PostArticle( const PostFile, ArtText: String;
                                  var   SrvResult: String ): Boolean;
var  Info, SendBuf, s: String;
     LogNow: TDateTime;
     Msg: TMess;
begin
   Result    := False;
   SrvResult := '';

   if not Connected then exit;
   if ArtText='' then exit;

   Info := '[' + Server + '] ';
   Log( LOGID_DETAIL, Info + 'PostArticle' );

   NNTP.HRequest( 'POST', 0, False );

   if NNTP.ResultCode = 340 then begin

      // send text and end-of-text dot
      SendBuf := TextToRfcWireFormat( ArtText ) + '.' + #13#10;
      NNTP.HWrite( SendBuf );
      if not Connected then exit;
      Log( LOGID_DETAIL, Info + 'Message sent, waiting for confirmation ...' );

      // read confirmation line from server
      NNTP.HReadLnResult;
      if not Connected then exit;
      if NNTP.ResultCode = 240 then Result:=True;

   end;

   SrvResult := NNTP.ResultLine;

   // save result in NewsIn.log
   LogNow := Now;
   Msg := TMess.Create;
   try
      Msg.FullText := ArtText;
      s := DateTimeToLogTime( LogNow )
         + #9 + 'File='       + ExtractFilename( PostFile )
         + #9 + 'Server='     + ServerAlias
         + #9 + 'Result='     + Logify( NNTP.ResultLine )
         + #9 + 'MID='        + Logify( Msg.HeaderValueByNameSL('Message-ID') )
         + #9 + 'From='       + Logify( Msg.HeaderValueByNameSL('From') )
         + #9 + 'Newsgroups=' + Logify( Msg.HeaderValueByNameSL('Newsgroups') )
         + #9 + 'Subject='    + Logify( Msg.HeaderValueByNameSL('Subject') )
         ;
   finally Msg.Free end;
   HamFileAppendLine( AppSettings.GetStr(asPathLogs)
                      + LOGFILE_NEWSOUT
                      + FormatDateTime( '"-"yyyy"-"mm', LogNow )
                      + LOGFILE_EXTENSION,
                      s );
end;

procedure TClientNNTP.FeedNewArticlesOfGroup( const GroupName: String;
                                              const GroupHandle: Integer );
var  Msg: TMess;
     ArtNoMin, ArtNoMax, FeedNo: Integer;
     CountSkip, CountAccept, CountReject: Integer;
     MessageId, SendBuf, Reason, s: String;
     OK: Boolean;

     procedure skip( const r: String );
     begin
        OK := False;
        Reason := r;
     end;

begin
   if not Connected then exit;

   PullInfo := '[' + Server + ', ' + GroupName + '] ';
   Log( LOGID_DETAIL, PullInfo + 'FeedNewArticles' );

   // get available article numbers
   ArtNoMin := Hamster.ArticleBase.GetInt( GroupHandle, gsLocalMin );
   ArtNoMax := Hamster.ArticleBase.GetInt( GroupHandle, gsLocalMax );

   // get last article number sent
   FeedNo := Hamster.ArticleBase.FeederLast[ GroupHandle, Server ];

   // are new articles available?
   if FeedNo >= ArtNoMax then begin
      Log( LOGID_INFO, PullInfo + 'No new articles to feed.' );
      exit;
   end else begin
      Log( LOGID_INFO, PullInfo + Format(
           'Feeding up to %s articles ...', [inttostr(ArtNoMax-FeedNo)]));
   end;

   CountSkip   := 0;
   CountAccept := 0;
   CountReject := 0;

   Msg := TMess.Create;
   try

      while Connected do begin

         CS_LOCK_NEWSFEED.Enter;
         try

            // get next article number to feed (=last+1)
            FeedNo := Hamster.ArticleBase.FeederLast[ GroupHandle, Server ] + 1;
            if FeedNo < ArtNoMin then FeedNo := ArtNoMin;
            if FeedNo > ArtNoMax then exit; // done, no new articles left

            Log( LOGID_STATUS, PullInfo + Format(
                 'Feeding article number %s of max. %s ...',
                 [ inttostr(FeedNo), inttostr(ArtNoMax) ] ) );

            Msg.FullText := Hamster.ArticleBase.ReadArticle( GroupHandle, FeedNo );
            MessageId := Msg.HeaderValueByNameSL( HDR_NAME_MESSAGE_ID );

            // check article
            OK := True;

            if Msg.HeaderLineCountSL = 0 then Skip( 'expired or cancelled' );

            s := MessageId;
            if s = '' then Skip( 'missing Message-ID:' );

            s := Msg.HeaderValueByNameSL( HDR_NAME_PATH );
            if s = '' then Skip( 'missing Path:' );

            s := Msg.HeaderValueByNameSL( HDR_NAME_DISTRIBUTION );
            if s = 'local' then Skip( 'Distribution: is local' );

            s := Msg.HeaderValueByNameSL( HDR_NAME_PATH );
            if Pos( Server + '!', s ) = 1 then Skip( 'server in Path:' );
            if Pos( '!' + Server + '!', s ) > 0 then Skip( 'server in Path:' );

            if OK then begin

               // remove Hamster specific X-headers
               Msg.DelHeaderML( HDR_NAME_X_HAMSTER_TO );
               Msg.DelHeaderML( HDR_NAME_X_HAMSTER_INFO );
               Msg.DelHeaderML( HDR_NAME_X_HAMSTER_FEED );
               Msg.DelHeaderML( HDR_NAME_X_NEWSGROUPS );
               Msg.DelHeaderML( HDR_NAME_X_OLD_XREF );

               // feed article
               Log( LOGID_DETAIL, 'Feeding article ' + inttostr(FeedNo)
                                + ': ' + MessageId );
               inc( CountReject );
               
               NNTP.HRequest( 'IHAVE ' + MessageId, 0, False );
               if not Connected then exit;

               // Initial responses:
               //    335 Send article to be transferred
               //    435 Article not wanted
               //    436 Transfer not possible; try again later

               if (NNTP.ResultCode <> 335) and (NNTP.ResultCode <> 435) then begin
                  Log( LOGID_WARN, PullInfo + 'IHAVE failed: ' + NNTP.ResultLine );
                  exit;
               end;

               if NNTP.ResultCode = 335 then begin

                  // send text and end-of-text dot
                  SendBuf := TextToRfcWireFormat( Msg.FullText ) + '.' + #13#10;
                  NNTP.HWrite( SendBuf );
                  if not Connected then exit;

                  // read confirmation line from server
                  NNTP.HReadLnResult;
                  if not Connected then exit;

                  // Subsequent responses:
                  //    235 Article transferred OK
                  //    436 Transfer failed; try again later
                  //    437 Transfer rejected; do not retry

                  if (NNTP.ResultCode <> 235) and (NNTP.ResultCode <> 437) then begin
                     Log( LOGID_WARN, PullInfo + 'IHAVE/2 failed: ' + NNTP.ResultLine );
                     exit;
                  end;

                  if NNTP.ResultCode = 235 then begin
                     dec( CountReject );
                     inc( CountAccept );
                  end;

               end;


            end else begin

               Log( LOGID_DETAIL, 'Skipped article ' + inttostr(FeedNo)
                                + ': ' + MessageId + ' ' + Reason );
               inc( CountSkip );
                                
            end;

            // raise last fed number
            Hamster.ArticleBase.FeederLast[ GroupHandle, Server ] := FeedNo;

         finally CS_LOCK_NEWSFEED.Leave end;

      end;

   finally
      Msg.Free;
      Log( LOGID_INFO, 'Feed result: '
                     + inttostr(CountAccept+CountReject+CountSkip) + ' processed, '
                     + inttostr(CountAccept) + ' accepted, '
                     + inttostr(CountReject) + ' rejected, '
                     + inttostr(CountSkip  ) + ' skipped' );
   end;
end;

constructor TClientNNTP.Create( AReportSubState: TReportSubStateInfo;
                                const AServerAlias, AServer, APort, AUser, APass: String;
                                AFeedMode: Boolean );
var  LfdServer: Integer;
begin
   inherited Create( AReportSubState );

   ServerAlias := AServerAlias;
   Server      := AServer;
   Port        := APort;
   User        := AUser;
   Pass        := APass;
   NNTP        := nil;
   ServerDir   := '';
   FFeedMode   := AFeedMode;

   with Hamster.Config do begin
      BeginRead;
      try
         LfdServer := NntpServers.IndexOfAlias( ServerAlias );
         if LfdServer >= 0 then ServerDir := NntpServers.Path[ LfdServer ];
      finally EndRead end;
   end;

   ScoreFile := TFiltersNews.Create( AppSettings.GetStr(asPathBase) + CFGFILE_SCORES );

   Move( XOVERLIST_DEFAULT[0], ServerXOverList[0], sizeof(XOVERLIST_DEFAULT) );
end;

destructor TClientNNTP.Destroy;
begin
   if Assigned(NNTP) then Disconnect;
   if Assigned(ScoreFile) then ScoreFile.Free;

   inherited Destroy;
end;


initialization
   ArtCurLoading := TThreadStringList.Create( True, dupIgnore );

finalization
   ArtCurLoading.Free;

end.