// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cClientBase;

interface

{$INCLUDE Compiler.inc}

uses Classes, IdTCPClient, IdSocks, uType, IdSSLOpenSSL;

const
   SPECIAL_ACCEPTWEAKENDOFLIST  = $00000001;
   SPECIAL_ACCEPTNODOTENDOFLIST = $00000002;
   // SPECIAL_ACCEPTSMTPMULTILINE  = $00000004; -> .AcceptSmtpMultiline

type
   TReportSizeProc = procedure( DataSize: Integer ) of object;

   TClientProtocolBase = class
      protected
         FReportSubState: TReportSubStateInfo;
         procedure ReportSubState( newSubState: String );
         
      public
         constructor Create( AReportSubState: TReportSubStateInfo );
   end;

   TClientBase = class( TIdTCPClient )
      protected
         FTerminated : Boolean;
         FServiceName: String;
         FServer     : String;
         FPort       : Integer;
         FPortStr    : String;
         FSocksInfo  : TIdSocksInfo;
         FUseSslMode : TSslTlsMode;
         FGreeting   : String;
         FResultCmnd : String;
         FResultLine : String;
         FResultCode : Integer;
         FResultStrm : TMemoryStream;
         FTimeoutConnectMS: Integer;
         FTimeoutCommandMS: Integer;
         FAcceptSmtpMultiline: Boolean;
         FLastErrResult: String;

         function  HResultListFollows: Boolean; virtual;
         procedure HResultLineToCode; virtual;

         function  GetResultStrmText: String;

         function StartTlsHandshake( const StartCmd : String;
                                     const StartCode: Integer;
                                     const TestCmd  : String;
                                     const TestCode : Integer;
                                     const TestMulti: Boolean ): String; virtual;

      public
         property  ResultCmnd: String        read FResultCmnd;
         property  ResultLine: String        read FResultLine;
         property  ResultCode: Integer       read FResultCode;
         property  ResultStrm: TMemoryStream read FResultStrm;
         property  ResultText: String        read GetResultStrmText;
         property  LastErrResult: String     read FLastErrResult;
         property  Greeting  : String        read FGreeting;
         property  AcceptSmtpMultiline: Boolean read  FAcceptSmtpMultiline
                                                write FAcceptSmtpMultiline;

         function  HReadLn( CloseOnTimeout: Boolean = True ): String;
         procedure HReadLnResult;

         procedure HWrite   ( const Txt: String );
         procedure HWriteLn ( const Txt: String );
         procedure HWriteLnQ( const Txt: String );

         procedure HRequest( const Cmnd     : String;
                             const Special  : Integer;
                             const FilterLog: Boolean ); virtual;
         procedure HRequestText( const Cmnd   : String;
                                 const Special: Integer;
                                 const ReportSizeProc: TReportSizeProc ); virtual;

         function  HConnect( const AServer, APort  : String;
                             const UseSOCKS        : Boolean;
                             const UseSslMode      : TSslTlsMode;
                             const SSLVersion      : TIdSSLVersion;
                             const SSLCipher       : String;
                             const Special         : Integer;
                             const TimeoutConnectMS: Integer;
                             const TimeoutCommandMS: Integer ): Boolean;
         function  HLogin( AServerAlias, AUser, APass: String ): Boolean; virtual;

         procedure HDisconnect( ASendQuit: Boolean = True );
         function  HConnected: Boolean;

         procedure Terminate;

         constructor Create( const AOwner: TComponent;
                             const AServiceName: String ); reintroduce;
         destructor  Destroy; override;
   end;

function Remote_SASL_Login( const Client: TClientBase;
                            const RequestFormat: String;
                            const CodeChallenge, CodeSuccess, ResultPos: Integer;
                            const SASL_WANTED, SASL_REMOTE: String;
                            const AUser, APass: String ): Boolean;


implementation

uses uConst, uConstVar, uVar, IdGlobal, IdIOHandlerSocket,
     cHamster, uHamTools, Windows, SysUtils, uTools, uWinSock, cLogFileHamster,
     cPasswords, uEncoding, uSASL, uSHA1, uMD5;

// ----------------------------------------------------------------------------

function Remote_SASL_Login( const Client: TClientBase;
                            const RequestFormat: String;
                            const CodeChallenge, CodeSuccess, ResultPos: Integer;
                            const SASL_WANTED, SASL_REMOTE: String;
                            const AUser, APass: String ): Boolean; 
var  Mechanism, sWanted, sLocal, sRemote, s, h: String;
     i: Integer;
     SL: TStringList;
     realm, nonce, cnonce, ncvalue, qop, digesturi, a1, a2: String;
begin
   Result := False;

   sRemote := UpperCase( SASL_REMOTE    ); // supported by server
   sLocal  := UpperCase( SASL_SUPPORTED ); // supported by Hamster
   sWanted := UpperCase( SASL_WANTED    ); // preference of user
   if sWanted='' then sWanted := sLocal; // no preference, use Hamster-default

   Mechanism := '';

   while sWanted<>'' do begin
      i := PosWhSpace( sWanted );
      if i=0 then begin
         s := sWanted;
         sWanted := '';
      end else begin
         s := copy( sWanted, 1, i-1 );
         System.Delete( sWanted, 1, i );
      end;

      if length(s) > 0 then begin
         if Pos( ' ' + s + ' ', ' ' + sLocal + ' ' ) = 0 then begin
            Log( LOGID_WARN, Client.FServer + ': Invalid SASL-setting "'
                                            + s + '"' );
         end else begin
            if Pos( ' ' + s + ' ', ' ' + sRemote + ' ' ) > 0 then begin
               Mechanism := s;
               break;
            end;
         end;
      end;
   end;

   if Mechanism='LOGIN' then begin

      Client.HRequest( Trim( Format( RequestFormat, [ Mechanism, '' ] ) ), 0, False );
      if not Client.HConnected then exit;
      if Client.ResultCode <> CodeChallenge then begin
         Log( LOGID_ERROR, Client.FServer + ': AUTH LOGIN/1 failed with "'
                                          + Client.ResultLine + '"' );
         exit;
      end;

      s := EncodeB64( AUser[1], length(AUser) ); // Username
      Client.HRequest( s, 0, True );
      if not Client.HConnected then exit;
      if Client.ResultCode <> CodeChallenge then begin
         Log( LOGID_ERROR, Client.FServer + ': AUTH LOGIN/2 failed with "'
                                          + Client.ResultLine + '"' );
         exit;
      end;

      s := EncodeB64( APass[1], length(APass) ); // Password
      Client.HRequest( s, 0, True );
      if not Client.HConnected then exit;
      if Client.ResultCode = CodeSuccess then begin
         Result := True;
      end else begin
         Log( LOGID_ERROR, Client.FServer + ': AUTH LOGIN/3 failed with "'
                                          + Client.ResultLine + '"' );
      end;

   end else if Mechanism='PLAIN' then begin

      s := AUTH_PLAIN_Encode( AUser, APass );
      Client.HRequest( Trim( Format( RequestFormat, [ Mechanism, s ] ) ), 0, True );
      if not Client.HConnected then exit;
      if Client.ResultCode = CodeSuccess then begin
         Result := True;
      end else begin
         Log( LOGID_ERROR, Client.FServer + ': AUTH PLAIN failed with "'
                                          + Client.ResultLine + '"' );
      end;

   end else if (Mechanism='CRAM-MD5' ) or
               (Mechanism='CRAM-SHA1') then begin

      Client.HRequest( Trim( Format( RequestFormat, [ Mechanism, '' ] ) ), 0, False ); 
      if not Client.HConnected then exit;
      if Client.ResultCode <> CodeChallenge then begin   
         Log( LOGID_ERROR, Client.FServer + ': AUTH CRAM/1 failed with "'
                                          + Client.ResultLine + '"' );
         exit;
      end;

      s := copy( Client.ResultLine, ResultPos, 999 ); // challenge
      s := DecodeB64( s[1], length(s) );
      if s='' then begin
         Log( LOGID_ERROR, Client.FServer + ': AUTH CRAM/2 failed with "'
                                          + Client.ResultLine + '"' );
         exit;
      end;

      if Mechanism='CRAM-MD5' then begin
         s := AUTH_CRAM_MD5_Encode( s, AUser, APass );
      end else if Mechanism='CRAM-SHA1' then begin
         s := AUTH_CRAM_SHA1_Encode( s, AUser, APass );
      end else begin
         exit;
      end;
      
      Client.HRequest( s, 0, True );
      if not Client.HConnected then exit;
      if Client.ResultCode = CodeSuccess then begin
         Result := True;
      end else begin
         Log( LOGID_ERROR, Client.FServer + ': AUTH CRAM/3 failed with "'
                                          + Client.ResultLine + '"' );
      end;

   end else if Mechanism='DIGEST-MD5' then begin

      Client.HRequest( Trim( Format( RequestFormat, [ Mechanism, '' ] ) ), 0, False ); 
      if not Client.HConnected then exit;
      if Client.ResultCode <> CodeChallenge then begin
         Log( LOGID_ERROR, Client.FServer + ': AUTH DIGEST-MD5/1 failed with "'
                                          + Client.ResultLine + '"' );
         exit;
      end;

      s := copy( Client.ResultLine, ResultPos, 999 ); // challenge
      s := DecodeB64( s[1], length(s) );
      if s='' then begin
         Log( LOGID_ERROR, Client.FServer + ': AUTH DIGEST-MD5/2 failed with "'
                                          + Client.ResultLine + '"' );
         exit;
      end;

      // check challenge, extract values
      SL := TStringList.Create;
      try
         log( LOGID_DEBUG, 'DIGEST-MD5 challenge: ' + s );
         ArgsSplitChar( s, SL, ',', True );

         realm := UnDQuoteStr( SL.Values[ 'realm' ] );
         nonce := UnDQuoteStr( SL.Values[ 'nonce' ] );
         
      finally SL.Free end;

      // build response

      s := 'username="' + AUser + '"';

      if realm <> '' then s := s + ',' + 'realm="' + realm + '"';

      h := Format( '%s%d%d', [ MidGenerator(''), GetTickCount, Random(999999) ] );
      h := MD5ofStr( h );
      cnonce := EncodeB64( h[1], length(h) );
      s := s + ',' + 'cnonce="' + cnonce + '"';

      ncvalue := '00000001';
      s := s + ',' + 'nc=' + ncvalue;

      qop := 'auth';
      s := s + ',' + 'qop=' + qop;

      digesturi := Client.FServiceName + '/' + Client.FServer;
      s := s + ',' + 'digest-uri="' + digesturi + '"';

      a1 := MD5OfStr( AUser + ':' + realm + ':' + APass )
          + ':' + nonce + ':' + cnonce;
      a2 := 'AUTHENTICATE:' + digesturi;
      h  := MD5toHex( MD5OfStr(
               MD5toHex( MD5OfStr( A1 ) )
               + ':' + nonce + ':' + ncvalue + ':' + cnonce + ':' + qop
               + ':' + MD5toHex( MD5OfStr( A2 ) )
            ) );
      s := s + ',' + 'response=' + h;

      // send response
      log( LOGID_DEBUG, 'DIGEST-MD5 response: ' + s );
      s := EncodeB64( s[1], length(s) );
      Client.HRequest( s, 0, True );
      if not Client.HConnected then exit;
      if Client.ResultCode <> CodeChallenge then begin
         Log( LOGID_ERROR, Client.FServer + ': AUTH DIGEST-MD5/3 failed with "'
                                          + Client.ResultLine + '"' );
         exit;
      end;

      // check for valid rspauth
      s := copy( Client.ResultLine, ResultPos, 999 ); // rspauth
      s := DecodeB64( s[1], length(s) );
      log( LOGID_DEBUG, 'DIGEST-MD5 rspauth: ' + s );

      a2 := ':' + digesturi;
      h := MD5toHex( MD5OfStr(
              MD5toHex( MD5OfStr( A1 ) )
              + ':' + nonce + ':' + ncvalue + ':' + cnonce + ':' + qop
              + ':' + MD5toHex( MD5OfStr( A2 ) )
           ) );
      if (Pos('rspauth=',s) > 0) and (Pos(h,s) > 0) then begin
         Client.HRequest( '', 0, False );
         if not Client.HConnected then exit;
         if Client.ResultCode = CodeSuccess then Result := True;
      end else begin
         Log( LOGID_ERROR, Client.FServer + ': AUTH DIGEST-MD5/4 failed with "'
                                          + Client.ResultLine + '"' );
         exit;
      end;

   end else begin

      Log( LOGID_ERROR, Client.FServer + ': No supported SASL mechanism for AUTH!' );
      Log( LOGID_ERROR, Client.FServer + ': "' + SASL_REMOTE    + '"'
                        + ' Hamster: "' + SASL_SUPPORTED + '"'
                        + ' User: "'    + SASL_WANTED    + '"' );

   end;
end;

// ----------------------------------------------------------------------------

function SOCKS_Enabled( const SocksProfile: TSocksProfile ): Boolean;
// Returns True, if given profile contains valid SOCKS settings.
begin
   with SocksProfile do begin
      Result :=  ( Version in [ svSocks4, svSocks4A, svSocks5 ] )
             and ( length( Server ) > 0 )
             and ( Port > 0 );
   end;
end;

// -------------------------------------------------- TClientProtocolBase -----

constructor TClientProtocolBase.Create( AReportSubState: TReportSubStateInfo);
begin
   inherited Create;
   FReportSubState := AReportSubState;
end;

procedure TClientProtocolBase.ReportSubState( newSubState: String );
begin
   if Assigned( FReportSubState ) then FReportSubState( newSubState );
end;

// ---------------------------------------------------------- TClientBase -----

function TClientBase.HReadLn( CloseOnTimeout: Boolean = True ): String;
begin
   SetLength( Result, 0 );
   try

      try
         ReadTimeout := FTimeoutCommandMS;
         Result := ReadLn( #10, IdTimeoutDefault );

         while ReadLnSplit do begin
            Result := Result + ReadLn( #10, IdTimeoutDefault );
            if length( Result ) > 50*1024*1024 then begin
               raise Exception.Create( 'Linelength-limit exceeds 50 MB!' );
            end;
         end;

      except
         on E:Exception do begin
            Log( LOGID_ERROR, Format(
                   'Exception receiving data from %s:', [FServer]) );
            Log( LOGID_ERROR, 'Exception[' + E.ClassName + '] ' + E.Message );
            try HDisconnect(False) except end;
            exit;
         end;
      end;

      CounterInc( CounterByteIn, length(Result) + 2 );

      if ReadLnTimedOut then begin
         Log( LOGID_WARN, Format(
              'Receiving data from %s failed (timeout)', [FServer]) );
         try if CloseOnTimeout then HDisconnect(False) except end;
         exit;
      end;

   finally
      if (LogFile.FileMask and LOGID_FULL) <> 0 then begin
         if length(Result) > 0 then Log( LOGID_DEBUG, '>*> ' + Result );
      end;
   end;
end;

procedure TClientBase.HWrite( const Txt: String );
begin
   if (LogFile.FileMask and LOGID_FULL) <> 0 then begin
      Log( LOGID_DEBUG, '<*< ' + Txt );
   end;

   CounterInc( CounterByteOut, length(Txt) );

   try
      Write( Txt );
   except
      on E:Exception do begin
         Log( LOGID_ERROR, Format(
              'Exception sending data to %s:', [FServer]) );
         Log( LOGID_ERROR, 'Exception[' + E.ClassName + '] ' + E.Message );
         try HDisconnect except end;
      end;
   end;
end;

procedure TClientBase.HWriteLn( const Txt: String );
begin
   HWrite( Txt + CRLF );
end;

procedure TClientBase.HWriteLnQ( const Txt: String );
var  Snd: String;
begin
   if copy( Txt, 1, 1 ) = '.' then Snd := '.' + Txt // quote leading dot
                              else Snd := Txt;
   HWrite( Snd + CRLF );
end;

procedure TClientBase.HReadLnResult;
var  ReplyLine : String;
     EndOfReply: Boolean;
begin
   FResultLine := '';
   FResultStrm.Clear;
   HResultLineToCode;
   FResultLine := '';
   if not HConnected then exit;

   // receive reply line
   EndOfReply := False;
   try
      while HConnected and not( FTerminated ) do begin

         ReplyLine := HReadLn;

         if FAcceptSmtpMultiline then begin
            if FResultLine <> '' then FResultLine := CRLF + FResultLine;
            FResultLine := ReplyLine + FResultLine; // note: reverse order
            if copy( ReplyLine, 4, 1 ) <> '-' then EndOfReply := True;
         end else begin
            FResultLine := ReplyLine;
            EndOfReply  := True;
         end;

         if EndOfReply then break;
      end;

   except
      on E:Exception do begin
         Log( LOGID_ERROR, Format(
              'Exception sending command to %s:', [FServer] ) );
         Log( LOGID_ERROR, 'Exception[' + E.ClassName + '] ' + E.Message );
      end;
   end;

   if not EndOfReply then begin
      try if HConnected then HDisconnect except end;
   end else begin
      Log( LOGID_DETAIL, Self.ClassName + ' < ' + FResultLine );
   end;
   
   HResultLineToCode;
   if FResultCode >= 400 then FLastErrResult := FResultLine;
end;

procedure TClientBase.HRequest( const Cmnd     : String;
                                const Special  : Integer;
                                const FilterLog: Boolean );
var  CmndLog   : String;
begin
   FResultCmnd := Cmnd;
   FResultLine := '';
   FResultStrm.Clear;
   HResultLineToCode;
   if not HConnected then exit;

   // log command
   CmndLog := Cmnd;
   if FilterLog then begin
      CmndLog := '[...authorization...]';
      if copy(Cmnd,1, 5)='USER '          then CmndLog:=copy(Cmnd,1, 5)+'[...]';
      if copy(Cmnd,1, 5)='PASS '          then CmndLog:=copy(Cmnd,1, 5)+'[...]';
      if copy(Cmnd,1,14)='AUTHINFO USER ' then CmndLog:=copy(Cmnd,1,14)+'[...]';
      if copy(Cmnd,1,14)='AUTHINFO PASS ' then CmndLog:=copy(Cmnd,1,14)+'[...]';
   end else begin
      CmndLog := Cmnd;
   end;
   Log( LOGID_DETAIL, Self.Classname + ' > ' + CmndLog );

   // send command
   HWriteLn( Cmnd );

   // receive result
   HReadLnResult;
end;

procedure TClientBase.HRequestText( const Cmnd: String;        
                                    const Special: Integer;
                                    const ReportSizeProc: TReportSizeProc );
var  ListLine, s: String;
     EndOfReply, CloseOnTimeout: Boolean;
     i: Integer;
     LoadedDataSize:Integer;
begin
   LoadedDataSize:=0;

   // send command
   HRequest( Cmnd, Special, False );
   if not HConnected then exit;
   if not HResultListFollows then exit;

   // receive list til end-of-list (CRLF "." CRLF) or timeout
   EndOfReply := False;
   CloseOnTimeout := ( (Special and SPECIAL_ACCEPTNODOTENDOFLIST) = 0 );

   try
      while HConnected and not( FTerminated ) do begin

         ListLine := HReadLn( CloseOnTimeout );
         if not HConnected then break;

         // end-of-list?
         if ListLine = '.' then begin

            // default end-of-list
            EndOfReply := True;

         end else begin

            if (Special and SPECIAL_ACCEPTNODOTENDOFLIST) <> 0 then begin
               // handle special case:
               //    (former?^H) POP3-LIST at Telda.net (no final '.' on 0 msg)
               if UpperCase( copy(FResultLine,1,6) ) = '+OK 0 ' then begin
                  if (FResultStrm.Size=0) and (length(ListLine)=0) then begin
                     EndOfReply := True; // no further data
                  end;
               end;
            end;

            if (Special and SPECIAL_ACCEPTWEAKENDOFLIST) <> 0 then begin
               // handle special case:
               //    (former?) POP3-LIST at Germanynet: end-of-list='  .CRLF  '
               //    (former?) POP3-LIST at uunet: #0...#0'.'
               s := '';
               for i:=1 to length(ListLine) do begin
                  if ListLine[i] in [#33..#126] then s := s + ListLine[i];
               end;
               if s = '.' then EndOfReply := True;
            end;

         end;

         if not EndOfReply then begin
            if copy(ListLine,1,2) = '..' then System.Delete( ListLine, 1, 1 );
            StreamWriteLn( FResultStrm, ListLine );
            inc( LoadedDataSize, length(ListLine) + 2 );
            if Assigned(ReportSizeProc) then ReportSizeProc( LoadedDataSize );
         end;

         if EndOfReply then break;
      end;

   except
      on E:Exception do begin
         Log( LOGID_ERROR, Format(
               'Exception receiving list from %s:', [FServer] ) );
         Log( LOGID_ERROR, 'Exception[' + E.ClassName + '] ' + E.Message );
      end;
   end;

   // no (valid) list -> give up
   if not EndOfReply then begin
      try if HConnected then HDisconnect except end;
   end;
end;

procedure TClientBase.HResultLineToCode;
begin
   if HConnected and ( FResultLine <> '' ) then FResultCode := 0
                                           else FResultCode := 999;
end;

function TClientBase.HResultListFollows: Boolean;
begin
   Result := False;
end;

function TClientBase.GetResultStrmText: String;
begin
   StreamToString( FResultStrm, Result );
end;

function TClientBase.StartTlsHandshake( const StartCmd : String;
                                        const StartCode: Integer;
                                        const TestCmd  : String;
                                        const TestCode : Integer;
                                        const TestMulti: Boolean ): String;
begin
   try
      // default = failed
      Result := '';

      // send start command, check reply
      HRequest( StartCmd, 0, False );
      if HConnected and (ResultCode <= StartCode) then begin
         Log( LOGID_INFO, 'SSL with TLS handshake started' );
      end else begin
         Log( LOGID_WARN, 'SSL negotation failed!');
         exit;
      end;

      // activate local SSL handler
      with IOHandler as TIdSSLIOHandlerSocket do begin
        PassThrough := False;
      end;
      if not HConnected then exit;

      // send test command, check reply
      if TestMulti then begin
         HRequestText( TestCmd, 0 , nil );
      end else begin
         HRequest( TestCmd, 0, False );
      end;
      if HConnected and (ResultCode <= TestCode) then begin
         Result := ResultLine;
         Log( LOGID_INFO, 'SSL with TLS handshake was activated' );
      end else begin
         Log( LOGID_WARN, 'SSL with TLS handshake was not activated!' );
      end;

   except
      Result := ''; // failed
      try HDisconnect( False ); except end;
   end;
end;

function TClientBase.HConnect( const AServer, APort  : String;
                               const UseSOCKS        : Boolean;
                               const UseSslMode      : TSslTlsMode;
                               const SSLVersion      : TIdSSLVersion;
                               const SSLCipher       : String;
                               const Special         : Integer;
                               const TimeoutConnectMS: Integer;
                               const TimeoutCommandMS: Integer ): Boolean;
var  ReplyLine, s, u, p: String;
     SocksProfile: TSocksProfile;
begin
   Result := False;

   FServer  := AServer;
   FPortStr := APort;
   FPort    := LookupServicePort( FPortStr );
   FUseSslMode := UseSslMode;
   FTimeoutConnectMS := TimeoutConnectMS;
   FTimeoutCommandMS := TimeoutCommandMS;

   case FUseSslMode of
      sslPort : s := ', SSL (secure port)';
      sslTLS  : s := ', SSL (TLS handshake)';
      else      s := '';
   end;
   Log( LOGID_DETAIL, Self.Classname
                    + '.Connect (' + FServer + ':' + FPortStr + s + ')' );

   // use SSL if configured
   if FUseSslMode <> sslNone then try
      if not Hamster.SSLAvailable then begin
         Log( LOGID_ERROR, 'Connection to ' + FServer + ' failed: '
                         + 'Required SSL libraries are not installed!' );
         exit;
      end;

      // create and assign SSL aware IOHandler
      IOHandler := TIdSSLIOHandlerSocket.Create( nil );
      IOHandler.OnStatus := OnStatus;
      FFreeIOHandlerOnDisconnect := True;
      with IOHandler as TIdSSLIOHandlerSocket do begin
         SSLOptions.Method     := SSLVersion;
         SSLOptions.Mode       := sslmClient;
         SSLOptions.VerifyMode := [];
         if SSLCipher <> '' then SSLOptions.CipherList := SSLCipher;
         if FUseSslMode = sslPort then begin
            PassThrough := False;
         end else begin
            PassThrough := True;
         end;
      end;

   except
      on E:Exception do begin
         Log( LOGID_ERROR, Format(
              'Exception enabling SSL support for %s:', [FServer] ) );
         Log( LOGID_ERROR, 'Exception[' + E.ClassName + '] ' + E.Message );
         exit;
      end;
   end;

   // use SOCKS if configured
   with Hamster.Config.Settings do begin
      SocksProfile.Version := TSocksVersion( GetInt( hsSocksVersion ) );
      SocksProfile.Server  := GetStr( hsSocksServer );
      SocksProfile.Port    := GetInt( hsSocksPort );
      SocksProfile.Auth    := TSocksAuthentication( GetInt( hsSocksAuth ) );
      SocksProfile.User    := GetStr( hsSocksUser );
      SocksProfile.Pass    := GetStr( hsSocksPass );
   end;

   if UseSOCKS and SOCKS_Enabled( SocksProfile ) then begin

      // init SocksInfo
      if not Assigned( FSocksInfo ) then begin
         FSocksInfo := TIdSocksInfo.Create( nil );
      end;

      FSocksInfo.Version        := IdSocks.TSocksVersion(SocksProfile.Version);
      FSocksInfo.Authentication := IdSocks.TSocksAuthentication(SocksProfile.Auth);
      FSocksInfo.Host           := SocksProfile.Server;
      FSocksInfo.Port           := SocksProfile.Port;

      u := SocksProfile.User;
      p := SocksProfile.Pass;
      if copy(u,1,1) = '$' then Hamster.Passwords.UsePassword( (u), u, p );
      FSocksInfo.Username := u;
      FSocksInfo.Password := p;

      // init IOHandler to use SocksInfo
      if not Assigned( IOHandler ) then begin
         IOHandler := TIdIOHandlerSocket.Create( Self );
         IOHandler.OnStatus := OnStatus;
         FFreeIOHandlerOnDisconnect := True;
      end;
      TIdIOHandlerSocket( IOHandler ).SocksInfo := FSocksInfo;

   end;

   try
      FResultLine := '';
      ReplyLine  := '';

      // connect
      Host := FServer;
      Port := FPort;
      Connect( FTimeoutConnectMS );

      // read initial greeting line(s)
      while HConnected and not( FTerminated ) do begin

         ReplyLine := HReadLn;
         if not HConnected then break;

         if FAcceptSmtpMultiline then begin
            if FResultLine <> '' then FResultLine := CRLF + FResultLine;
            FResultLine := ReplyLine + FResultLine; // note: reverse order
            if copy( ReplyLine, 4, 1 )<>'-' then Result := True;
         end else begin
            FResultLine := ReplyLine;
            Result := True;
         end;

         if Result then begin
            FGreeting := FResultLine;
            break;
         end;
         
      end;

   except
      on E:Exception do begin
         Log( LOGID_ERROR, Format(
              'Exception connecting to %s:', [FServer] ) );
         Log( LOGID_ERROR, 'Exception[' + E.ClassName + '] ' + E.Message );
      end;
   end;

   if not Result then begin
      Log( LOGID_WARN, Format( 'Connection to %s failed!', [FServer]) );
      try HDisconnect except end;
   end;
end;

function TClientBase.HLogin( AServerAlias, AUser, APass: String ): Boolean;
begin
   Result := False;
   if HConnected then HDisconnect;
end;

procedure TClientBase.HDisconnect( ASendQuit: Boolean = True );
begin
   if HConnected then begin
      Log( LOGID_DETAIL,
           Self.Classname + '.Disconnect (' + FServer + ':' + FPortStr + ')' );
      if ASendQuit then try HRequest( 'QUIT', 0, False ) except end;
      Disconnect;
   end;
end;

function TClientBase.HConnected: Boolean;
begin
   Result := Connected;
end;

procedure TClientBase.Terminate;
begin
   FTerminated := True;
end;

constructor TClientBase.Create( const AOwner: TComponent;
                                const AServiceName: String  );
begin
   inherited Create( AOwner );

   Log( LOGID_DEBUG, Self.Classname + '.Create' );

   FTerminated  := False;
   FServiceName := AServiceName;
   FServer      := '';
   FPort        := 0;
   FPortStr     := '';
   FUseSslMode  := sslNone;
   FSocksInfo   := nil;
   FGreeting    := '';
   FResultStrm  := TMemoryStream.Create;
   FAcceptSmtpMultiline := False;
   FLastErrResult := '';

   self.MaxLineLength := 32 * 1024;
   self.MaxLineAction := maSplit;
end;

destructor TClientBase.Destroy;
begin
   Log( LOGID_DEBUG, Self.Classname + '.Destroy' );

   try if HConnected then Disconnect except end;
   if Assigned( FResultStrm ) then FResultStrm.Free;
   if Assigned( FSocksInfo  ) then FSocksInfo.Free;

   inherited Destroy;
end;

// ----------------------------------------------------------------------------

initialization
   Randomize;

end.
