// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cArtFile; // Storage for a single newsgroup

// ----------------------------------------------------------------------------
// Contains a class to access and modify the files of a single newsgroup. This
// class is not used directly, it is covered by the class in "cArtFiles.pas".
// ----------------------------------------------------------------------------

interface

{$INCLUDE Compiler.inc}

uses Windows, SysUtils, Classes, IniFiles, cIndexRc, uDateTime, uConst,
     cSettings, cFileStream64;

const
   ARTFILE_DEFAULTMODE = fmReadWriteDenyWrite64;
   ARTFILE_EXCLUSIVE   = fmReadWriteExclusive64;

type
   ESizeLimitReached = class( Exception );

   TCompactCanPurgeFunc = function( Data: PChar;
                                    PurgePar: LongInt ) : Boolean of object;

   PIndexRec = ^TIndexRec;
   TIndexRec = record
      DatKey: LongWord;
      DatPos: LongWord;
   end;

   TArticleFile = class
      private
         CS_THISFILE: TRTLCriticalSection;
         IndexFile  : TIndexedRecords;
         FFilePath  : String;
         FGroupname : String;
         FSdat      : TFileStream64;
         FFileMode  : Word;
         PurgeCount : Integer;
         FSettings  : TSettingsPlain;
         FPullSettings: TSettingsQualified;

         function GetSettings: TSettingsPlain;
         function GetPullSettings: TSettingsQualified;

         function  GetDatKey( Index: Integer ): LongWord;
         function  GetDatPos( Index: Integer ): Int64;

         procedure WriteRec( Key: LongWord; const Data; DataSize: LongInt );
         function  ReadSize( Key: LongWord ): Integer;
         procedure ReadData( Key: LongWord; var Data; MaxSize: Integer );
         procedure Delete  ( Key: LongWord );

         function  Compact( CanPurgeFunc: TCompactCanPurgeFunc;
                            KeepDays: LongInt;
                            var NewKeyMin, NewKeyMax: LongInt): Boolean;

         function  GetServerMin( const Server: String ): Integer;
         procedure SetServerMin( const Server: String; NewArtMin: Integer );
         function  GetServerMax( const Server: String ): Integer;
         procedure SetServerMax( const Server: String; NewArtMax: Integer );
         function  GetPullLimit( const Server: String ): Integer;
         function  GetFeederLast( const Server: String ): Integer;
         procedure SetFeederLast( const Server: String; NewFeederLast: Integer );
         function  GetPurgeKeepDays: Integer;
         procedure SetPurgeKeepDays( NewKeepDays: Integer );

         function  IsOldEnoughToPurge( PArtText: PChar; KeepMaxDays: LongInt ) : Boolean;
         function GetCount: LongInt;

         procedure CheckSizeLimit( DataPos: Int64; DataSize: LongWord );

      protected
         property DatKey[ Index: Integer ]: LongWord read GetDatKey;
         property DatPos[ Index: Integer ]: Int64    read GetDatPos; 

      public
         property Count: LongInt read GetCount;
         property GroupName: String read FGroupname;

         property Settings: TSettingsPlain read GetSettings;
         function ServerSettingsAll( const Server: String ): String;
         procedure ServerSetStr( const Server: String;
                                 const ID: Integer;
                                 const NewValue: String );

         property  ServerMin[ const Server:String ]: Integer read GetServerMin write SetServerMin;
         property  ServerMax[ const Server:String ]: Integer read GetServerMax write SetServerMax;
         property  PullLimit[ const Server:String ]: Integer read GetPullLimit;
         property  FeederLast[ const Server:String ]: Integer read GetFeederLast write SetFeederLast;
         property  PurgeKeepDays: Integer read GetPurgeKeepDays write SetPurgeKeepDays;
         procedure SetFirstPullDone( const Server: String );
         function  DTCreated: TDateTime;

         function  IndexOfKey( Key: LongInt ): LongInt;
         function  Open( fmMode: Word ): Boolean; virtual;
         procedure Close; virtual;
         procedure Flush; virtual;

         function  ReserveArtNo: LongInt;
         function  ReadArticle( ArtNo: Integer; var MaxSize: Integer ): String;
         function  WriteArticle( ArtNo: Integer; const ArtText: String ): LongInt;
         procedure DeleteArticle( ArtNo: Integer );

         procedure Purge;
         procedure Reindex;

         procedure EnterThisFile;
         procedure LeaveThisFile;

         constructor Create( const AGroupName: String );
         destructor Destroy; override;
  end;

implementation

uses uConstVar, uVar, uTools, cArticle, cFiltersNews, cLogFileHamster, cHamster;

const
   DATMASK_SIZE     = $00FFFFFF;
   DATMASK_RESERVED = $7F000000;
   DATMASK_DELETED  = LongInt($80000000);

// --------------------------------------------------------- TArticleFile -----

function TArticleFile.GetCount: LongInt;
begin
   if Assigned(IndexFile) then Result := IndexFile.Count
                          else Result := 0;
end;

function TArticleFile.GetSettings: TSettingsPlain;
begin
   if not Assigned( FSettings ) then begin
      FSettings := TSettingsPlain.Create(
         SettingsDef_Groups,
         TSettingsHandler_IniFile.Create(
            AppSettings.GetStr(asPathGroups) + FGroupName + '\' + GRPFILE_INI
         ),
         True {AutoFlush}
      );
   end;

   Result := FSettings;
end;

function TArticleFile.GetPullSettings: TSettingsQualified;
begin
   if not Assigned( FPullSettings ) then begin
      FPullSettings := TSettingsQualified.Create(
         SettingsDef_Pulls,
         TSettingsHandler_IniFile.Create(
            AppSettings.GetStr(asPathGroups) + FGroupName + '\' + GRPFILE_INI
         ),
         True {AutoFlush}
      );
   end;

   Result := FPullSettings;
end;

procedure TArticleFile.CheckSizeLimit( DataPos: Int64; DataSize: LongWord );
// raise exception if DataSize would grow file above 32 Bit indexfile pointer.
var  NextPos: Int64;
begin
   NextPos := DataPos + sizeof(DataSize) + DataSize;
   if Int64Rec( NextPos ).Hi <> 0 then begin
      raise ESizeLimitReached.Create( 'Size limit reached (' + Groupname + ')!' );
   end;
end;

procedure TArticleFile.WriteRec( Key: LongWord; const Data; DataSize: LongInt );
var  idx: Integer;
     siz: LongInt;
     IRec : TIndexRec;
     DataPos: Int64;
begin
   try
      if DataSize > DATMASK_SIZE then DataSize := DATMASK_SIZE;

      idx := IndexFile.RecKeyIndexOf( sizeof(Key), Key );

      if idx < 0 then begin

         // add article
         DataPos := FSdat.Size;
         CheckSizeLimit( DataPos, DataSize );
         IRec.DatKey := Key;
         IRec.DatPos := DataPos;
         IndexFile.RecAdd( idx, IRec );

      end else begin

         // replace article
         DataPos := DatPos[idx];
         if (DataPos < 0) or (DataPos > FSdat.Size) then begin // damaged
            siz  := 0;
         end else begin
            FSdat.Seek( DataPos, soFromBeginning );
            FSdat.Read( siz, sizeof(siz) );
         end;
         if ( DataSize <> siz ) then begin
            // different size, so delete old one and write at new position
            if siz <> 0 then begin
               siz := siz or DATMASK_DELETED;
               FSdat.Seek( DataPos, soFromBeginning );
               FSdat.Write( siz, sizeof(siz) );
            end;
            DataPos := FSdat.Size;  
            CheckSizeLimit( DataPos, DataSize );
            PIndexRec( IndexFile.RecPtr(idx) )^.DatPos := DataPos;
            IndexFile.Changed := True;
         end;

      end;

      FSdat.Seek( DatPos[idx], soFromBeginning );
      FSdat.Write( DataSize, sizeof(DataSize) );
      FSdat.Write( Data, DataSize );

   except
      on E:Exception do begin
         Log( LOGID_ERROR, 'ArtFile.WriteRec-Error: ' + E.Message );
         raise;
      end;
   end;
end;

procedure TArticleFile.Delete( Key: LongWord );
var  idx: Integer;
     siz: LongInt;
begin
   idx := IndexFile.RecKeyIndexOf( sizeof(Key), Key );
   if idx<0 then exit;

   FSdat.Seek( DatPos[idx], soFromBeginning );
   FSdat.Read( siz, sizeof(siz) );
   siz := siz or DATMASK_DELETED;
   FSdat.Seek( DatPos[idx], soFromBeginning );
   FSdat.Write( siz, sizeof(siz) );

   IndexFile.RecDelete( idx );
end;

function TArticleFile.Compact( CanPurgeFunc: TCompactCanPurgeFunc;
                               KeepDays: LongInt;
                               var NewKeyMin, NewKeyMax: LongInt ): Boolean;
var  OldPos, NewPos   : Int64;
     RecKey, RecSize  : LongInt;
     RecData          : PChar;
     IsDeleted        : Boolean;
     DebugInfo        : String;
     TestIdx          : LongInt;
     MaxLimit         : Int64;
begin
   Log( LOGID_DEBUG, Format('Compact: %s', [FFilePath] ) );
   Result := False;

   // save changes; close files if currently open
   Close;

   // open exclusive
   if not Open( ARTFILE_EXCLUSIVE ) then begin
      Log( LOGID_WARN, Format(
           'Compact canceled; %s is in use.', [FFilePath] ));
      exit;
   end;

   MaxLimit  := FSdat.Size - 1;
   NewPos    := 0;
   NewKeyMin := NewKeyMax + 1;
   NewKeyMax := 0;

   // sort index entries by position in .dat file
   IndexFile.Sort( sizeof(LongWord){=Offset TIndexRec.DatPos} );

   TestIdx := 0;
   while TestIdx < IndexFile.Count do begin

      // get old values
      RecKey := DatKey[TestIdx];
      OldPos := DatPos[TestIdx];

      if ( OldPos >= 0 ) and ( OldPos <= MaxLimit-3 ) then begin
         FSdat.Seek( OldPos, soFromBeginning );
         FSdat.Read( RecSize, sizeof(RecSize) );
      end else begin
         // damaged pointer in index-file
         Log( LOGID_WARN, Format(
              'Damaged index-pointer deleted (ArtNo=%s)', [inttostr(RecKey)]) );
         RecSize := DATMASK_DELETED;
      end;

      IsDeleted := ( RecSize and DATMASK_DELETED ) <> 0;
      RecSize   := RecSize and DATMASK_SIZE;
      if ( OldPos + 3 + RecSize) > MaxLimit then begin
         // damaged pointer in data-file
         Log( LOGID_WARN, Format(
             'Damaged data-pointer deleted (ArtNo=%s)', [inttostr(RecKey)]) );
         IsDeleted := True;
      end;

      DebugInfo := inttostr(RecKey) + ': ('
                 + inttostr(OldPos) + '/' + inttostr(RecSize) + ') -> ';

      // old data is marked as deleted
      if IsDeleted then DebugInfo:=DebugInfo + 'DEL';

      RecData := nil;

      if not(IsDeleted) and (RecSize>0) then begin
         // read old data
         GetMem( RecData, RecSize + 1 );
         RecData^ := #0;
         FSdat.Read( RecData^, RecSize );

         if RecData^<#32 then begin
            DebugInfo := DebugInfo + 'INVALID';
            FreeMem( RecData, RecSize + 1 );
            RecData := nil;
         end else begin
            // check, if old data should be purged
            if Assigned(CanPurgeFunc) then begin
               RecData[RecSize] := #0; // ASCIIZ
               if CanPurgeFunc( RecData, KeepDays ) then begin
                  DebugInfo := DebugInfo + 'PURGE';
                  FreeMem( RecData, RecSize + 1 );
                  RecData := nil;
               end;
            end;
         end;
      end;

      if Assigned(RecData) then begin
         // article is valid and should be kept

         // adjust key-bounds
         if NewKeyMin=0 then NewKeyMin:=RecKey;
         if NewKeyMax=0 then NewKeyMax:=RecKey;
         if RecKey<NewKeyMin then NewKeyMin:=RecKey;
         if RecKey>NewKeyMax then NewKeyMax:=RecKey;

         if OldPos = NewPos then begin
            // remains at old position in .dat-file
            DebugInfo := DebugInfo + 'KEEP';
         end else begin
            // move to new position in .dat-file
            DebugInfo := DebugInfo + 'MOVE ' + inttostr(NewPos);
            PIndexRec( IndexFile.RecPtr(TestIdx) )^.DatPos := NewPos; // update index-entry
            IndexFile.Changed := True;
            try
               FSdat.Seek ( NewPos, soFromBeginning );
               FSdat.Write( RecSize, sizeof(RecSize) );
               FSdat.Write( RecData^, RecSize );
            except
               on E: Exception do begin
                  Log( LOGID_ERROR, 'ArtFile.Compact-Error: ' + E.Message );
                  break;
               end;
            end;
         end;

         // adjust position for next data-record
         NewPos := NewPos + sizeof(RecSize);
         NewPos := NewPos + RecSize;
         CheckSizeLimit( NewPos, 0 );

         FreeMem( RecData, RecSize + 1 );

         inc( TestIdx );

      end else begin

         // article is invalid or should be purged -> delete index-entry
         IndexFile.RecDelete( TestIdx );

      end;

      Log( LOGID_FULL, DebugInfo );

   end;

   if NewKeyMin=0 then NewKeyMin:=1;

   // cut unused space at end of .dat-file
   if NewPos <> FSdat.Size then begin
      FSdat.Seek( NewPos, soFromBeginning );
      SetEndOfFile( FSdat.Handle );
   end;

   // save new index
   IndexFile.Changed := True;
   IndexFile.Sort; // re-sort index by article numbers
   IndexFile.FlushToFile;

   // close files again
   Close;

   Result := True;
end;

function TArticleFile.Open( fmMode: Word ) : Boolean;
var  localMaxIdx, localMaxIni: LongWord;
begin
   Result := True;
   
   FFileMode := fmMode;
   Close;

   try
      FSdat := TFileStream64.Create( FFilePath + GRPFILE_DAT, FFileMode );
   except
      on E: Exception do begin
         Result := False;
         Log( LOGID_ERROR, 'ArtFile.Open-Error '
                         + FFilePath + GRPFILE_DAT + ': ' + E.Message );
         exit;
      end;
   end;

   IndexFile := TIndexedRecords.Create( FFilePath + GRPFILE_IDX, FFileMode,
                                        sizeof(TIndexRec), True );
   try
      IndexFile.LoadFromFile;

      if IndexFile.Count > 0 then begin
         localMaxIdx := DatKey[ IndexFile.Count - 1 ];
         localMaxIni := GetSettings.GetInt( gsLocalMax );
         if localMaxIdx > localMaxIni then begin
            GetSettings.SetInt( gsLocalMax, localMaxIdx );
            if Assigned(FSettings) then FSettings.Flush;
            Log( iif( localMaxIdx > localMaxIni + 10, LOGID_WARN, LOGID_INFO ),
                 Format( 'ArtFile.Open has fixed Local.Max'
                       + ' of group %s from %d to %d.',
                       [ FGroupname, localMaxIni, localMaxIdx ] ) );
         end;
      end;

   except
      on E: Exception do begin
         Result := False;
         Log( LOGID_ERROR, 'ArtFile.Open-Error '
                         + FFilePath + GRPFILE_IDX + ': ' + E.Message );
         Close;
         exit;
      end;
   end;

   if (IndexFile.Count=0) and (FSdat.Size>0) then Reindex;
end;

procedure TArticleFile.Flush;
begin
   if Assigned(IndexFile    ) then try IndexFile.FlushToFile except end;
   if Assigned(FSdat        ) then try FSdat.Flush           except end;
   if Assigned(FSettings    ) then try FSettings.Flush       except end;
   if Assigned(FPullSettings) then try FPullSettings.Flush   except end;
end;

procedure TArticleFile.Close;
begin
   try Flush; except end;
   if Assigned( IndexFile ) then try FreeAndNil( IndexFile ) except end;
   if Assigned( FSdat     ) then try FreeAndNil( FSDat     ) except end;
end;

function TArticleFile.IndexOfKey( Key: LongInt ): LongInt;
begin
   Result := IndexFile.RecKeyIndexOf( sizeof(Key), Key );
end;

function TArticleFile.ServerSettingsAll( const Server: String ): String;
begin
   Result := GetPullSettings.GetAll( Server );
end;

procedure TArticleFile.ServerSetStr( const Server: String;
                                     const ID: Integer;
                                     const NewValue: String );
begin
   GetPullSettings.SetStr( Server, ID, NewValue );
end;

function TArticleFile.GetServerMin( const Server: String ): Integer;
begin
   Result := GetPullSettings.GetInt( Server, psArtNoMin );
end;

procedure TArticleFile.SetServerMin( const Server: String; NewArtMin: Integer );
begin
   GetPullSettings.SetInt( Server, psArtNoMin, NewArtMin );
end;

function TArticleFile.GetServerMax( const Server: String ): Integer;
begin
   Result := GetPullSettings.GetInt( Server, psArtNoMax );
end;

procedure TArticleFile.SetServerMax( const Server: String; NewArtMax: Integer );
begin
   GetPullSettings.SetInt( Server, psArtNoMax, NewArtMax );
end;

function TArticleFile.GetFeederLast( const Server: String ): Integer;
begin
   Result := GetPullSettings.GetInt( Server, psFeederLast );
end;

procedure TArticleFile.SetFeederLast( const Server: String; NewFeederLast: Integer );
begin
   GetPullSettings.SetInt( Server, psFeederLast, NewFeederLast );
end;

procedure TArticleFile.SetFirstPullDone( const Server: String );
begin
   GetPullSettings.SetBoo( Server, psFirstPullDone, True );
end;

function TArticleFile.GetPullLimit( const Server: String ): Integer;
begin
   if not GetPullSettings.GetBoo( Server, psFirstPullDone ) then begin
      Result := Hamster.Config.Settings.GetInt(hsPullLimitFirst);
      exit;
   end;

   if GetSettings.GetStr( gsPullLimit ) = '' then begin
      Result := Hamster.Config.Settings.GetInt(hsPullLimit);
   end else begin
      Result := GetSettings.GetInt( gsPullLimit );
   end;
end;

function TArticleFile.GetPurgeKeepDays: Integer;
begin
   if GetSettings.GetStr( gsPurgeKeepDays ) = '' then begin
      Result := Hamster.Config.Settings.GetInt(hsPurgeArticlesKeepDays);
   end else begin
      Result := GetSettings.GetInt( gsPurgeKeepDays );
   end;
end;

procedure TArticleFile.SetPurgeKeepDays( NewKeepDays: Integer );
begin
   GetSettings.SetInt( gsPurgeKeepDays, NewKeepDays );
end;

function TArticleFile.DTCreated: TDateTime;
begin
   Result := GetSettings.GetDT( gsCreated );
   if Result = 0 then begin // missing or invalid
      Result := NowGMT;
      GetSettings.SetDT( gsCreated, Result );
   end;
end;

function TArticleFile.GetDatKey( Index: Integer ): LongWord;
begin
   if (Index>=0) and (Index<IndexFile.Count) then
      Result := PIndexRec( IndexFile.RecPtr(Index) )^.DatKey
   else
      Result := 0;
end;

function TArticleFile.GetDatPos( Index: Integer ): Int64;
begin
   if (Index>=0) and (Index<IndexFile.Count) then
      Result := PIndexRec( IndexFile.RecPtr(Index) )^.DatPos
   else
      Result := -1;
end;

function TArticleFile.ReadSize( Key: LongWord ): Integer;
var  idx: Integer;
     siz: LongInt;
begin
   try
      Result := 0;

      idx := IndexFile.RecKeyIndexOf( sizeof(Key), Key );
      if idx<0 then exit;

      FSdat.Seek( DatPos[idx], soFromBeginning );
      FSdat.Read( siz, sizeof(siz) );
      Result := siz and DATMASK_SIZE;
   except
      on E:Exception do begin
         Log( LOGID_ERROR, 'ArtFile.ReadSize-Error: ' + E.Message );
         raise;
      end;
   end;
end;

procedure TArticleFile.ReadData( Key: LongWord; var Data; MaxSize: Integer );
var  idx : Integer;
     dpos: Int64;
     siz : LongInt;
begin
   try
      idx := IndexFile.RecKeyIndexOf( sizeof(Key), Key );
      if idx<0 then exit;

      dpos := DatPos[idx];
      if dpos<0 then exit;

      FSdat.Seek( dpos, soFromBeginning );
      FSdat.Read( siz, sizeof(siz) );
      siz := siz and DATMASK_SIZE;
      if siz>MaxSize then siz:=MaxSize;
      if siz>0 then FSdat.Read( Data, siz );
   except
      on E:Exception do begin
         Log( LOGID_ERROR, 'ArtFile.ReadData-Error: ' + E.Message );
         raise;
      end;
   end;
end;

function TArticleFile.ReadArticle( ArtNo: Integer; var MaxSize: Integer ): String;
var  siz, rsiz: Integer;
     buf: PChar;
begin
   try
      siz := ReadSize( ArtNo );
      rsiz := siz;
      if (MaxSize>0) and (siz>MaxSize) then siz:=MaxSize;
      MaxSize := rsiz; // return real size

      if siz<=0 then begin
         Result := '';
      end else begin
         GetMem( buf, siz+1 );
         ReadData( ArtNo, buf^, siz );
         (buf+siz)^ := #0;
         Result := String( buf );
         FreeMem( buf, siz+1 );
      end;
   except
      on E:Exception do begin
         Log( LOGID_ERROR, 'ArtFile.ReadArticle-Error: ' + E.Message );
         raise;
      end;
   end;
end;

function TArticleFile.ReserveArtNo: LongInt;
begin
   Result := GetSettings.GetInt( gsLocalMax ) + 1;
   GetSettings.SetInt( gsLocalMax, Result );
end;

function TArticleFile.WriteArticle( ArtNo: Integer; const ArtText: String ): LongInt;
var  i: Integer;
begin
   try
      if ArtNo <= 0 then begin
         ArtNo := ReserveArtNo;
      end else begin
         i := GetSettings.GetInt( gsLocalMax );
         if (i=0) or (ArtNo>i) then GetSettings.SetInt( gsLocalMax, ArtNo )
      end;

      i := GetSettings.GetInt( gsLocalMin );
      if (i=0) or (ArtNo<i) then GetSettings.SetInt( gsLocalMin, ArtNo );

      WriteRec( ArtNo, ArtText[1], length(ArtText) );
      Result := ArtNo;
   except
      on E:Exception do begin
         Log( LOGID_ERROR, 'ArtFile.WriteArticle-Error: ' + E.Message );
         raise;
      end;
   end;
end;

procedure TArticleFile.DeleteArticle( ArtNo: Integer );
begin
   Delete( ArtNo );
end;

function TArticleFile.IsOldEnoughToPurge( PArtText: PChar; KeepMaxDays: LongInt ) : Boolean;
var  Art : TMess;
     Base: TDateTime;
     Days: LongInt;
     MID : String;
begin
   Result := False;
   if KeepMaxDays<=0 then exit;

   Art := TMess.Create;
   try
      Art.FullText := String( PArtText );
      Base := Art.GetReceivedDT;
      MID  := Art.HeaderValueByNameSL('Message-ID:');
   finally Art.Free end;

   if MID='' then begin // damaged?
      inc(PurgeCount);
      Result:=True;
      exit;
   end;

   Days := Trunc( Now - Base );
   if Days>KeepMaxDays then begin
      inc(PurgeCount);
      Result:=True;
      exit;
   end;
end;

procedure TArticleFile.Purge;
var  AMin, AMax: LongInt;
begin
   PurgeCount := 0;

   if PurgeKeepDays > 0 then begin
      AMin := GetSettings.GetInt( gsLocalMin );
      AMax := GetSettings.GetInt( gsLocalMax );
      if Compact( IsOldEnoughToPurge, PurgeKeepDays, AMin, AMax ) then begin
         if AMin<>GetSettings.GetInt( gsLocalMin ) then GetSettings.SetInt( gsLocalMin, AMin );
         if AMax> GetSettings.GetInt( gsLocalMax ) then GetSettings.SetInt( gsLocalMax, AMax );
      end;
   end;

   Log( LOGID_INFO, Format('Purge %s (%sd): %s articles purged.',
                    [GroupName, inttostr(PurgeKeepDays), inttostr(PurgeCount)]) );
end;

procedure TArticleFile.Reindex;
var  RecSize, RecKey: Integer;
     CurPos, FileLen: Int64;
     NewKeyMin, NewKeyMax, idx: Integer;
     IRec     : TIndexRec;
     RecData  : PChar;
     IsDeleted: Boolean;
     Art      : TMess;
     s: String;
     Cnt, i: Integer;
begin
   Log( LOGID_Debug, Format(
      'Auto-reindex of %s started due to missing index-file.', [Groupname]) );

   IndexFile.Clear;
   CurPos  := 0;
   Cnt     := 0;
   FileLen := FSdat.Size;

   NewKeyMin := GetSettings.GetInt( gsLocalMin );
   NewKeyMax := GetSettings.GetInt( gsLocalMax );
   if NewKeyMin<1 then NewKeyMin:=1;
   if NewKeyMax<NewKeyMin then NewKeyMax:=NewKeyMin-1;

   while CurPos < FileLen do begin
      FSdat.Seek( CurPos, soFromBeginning );
      FSdat.Read( RecSize, sizeof(RecSize) );

      IsDeleted := ( RecSize and DATMASK_DELETED ) <> 0;
      RecSize   := RecSize and DATMASK_SIZE;
      if (CurPos+RecSize)>=FileLen then break; // damaged pointer in data-file

      if not IsDeleted then begin
         RecKey := NewKeyMax + 1; // default: assign next article-number

         GetMem( RecData, RecSize + 1 );
         RecData^ := #0;
         FSdat.Read( RecData^, RecSize );
         RecData[RecSize] := #0; // ASCIIZ
         Art := TMess.Create;
         try
            Art.FullText := String( RecData );
            s  := LowerCase( Art.HeaderValueByNameSL('Xref:') );
         finally Art.Free end;
         FreeMem( RecData, RecSize + 1 );

         if s<>'' then begin
            i := Pos( LowerCase(Groupname)+':', s );
            if i>0 then begin
               System.Delete( s, 1, i+length(GroupName) );
               i := PosWhSpace( s );
               if i>0 then s := copy(s,1,i-1);
               if s<>'' then begin
                  i := strtointdef( s, -1 );
                  if (i>=NewKeyMin) and (i<=NewKeyMax) then RecKey:=i; // use Xref-number
               end;
            end;
         end;

         if NewKeyMin=0 then NewKeyMin:=RecKey;
         if NewKeyMax=0 then NewKeyMax:=RecKey;
         if RecKey<NewKeyMin then NewKeyMin:=RecKey;
         if RecKey>NewKeyMax then NewKeyMax:=RecKey;

         IRec.DatKey := RecKey;
         IRec.DatPos := CurPos;
         IndexFile.RecAdd( idx, IRec );
         inc( Cnt );
      end;

      inc( CurPos, sizeof(RecSize) );
      inc( CurPos, RecSize );
   end;

   // save new index
   IndexFile.Changed := True;
   IndexFile.FlushToFile;

   // save new bounds
   GetSettings.SetInt( gsLocalMin, NewKeyMin );
   GetSettings.SetInt( gsLocalMax, NewKeyMax );

   Log( LOGID_DEBUG, Format(
          'Auto-reindex of %s done; %s articles recovered.',
           [Groupname, inttostr(Cnt)]));
end;

procedure TArticleFile.EnterThisFile;
begin
   EnterCriticalSection( CS_THISFILE );
end;

procedure TArticleFile.LeaveThisFile;
begin
   LeaveCriticalSection( CS_THISFILE );
end;

constructor TArticleFile.Create( const AGroupName: String );
begin
   inherited Create;

   FGroupName := AGroupName;
   FFilePath  := AppSettings.GetStr(asPathGroups) + FGroupName + '\';

   // create path and files if they don't exist already
   ForceFileExists( FFilePath + GRPFILE_DAT );
   ForceFileExists( FFilePath + GRPFILE_IDX );

   InitializeCriticalSection( CS_THISFILE );

   FFileMode     := ARTFILE_DEFAULTMODE;
   FSdat         := nil;
   IndexFile     := nil;
   FSettings     := nil;
   FPullSettings := nil;
end;

destructor TArticleFile.Destroy;
begin
   try Close; except end;
   DeleteCriticalSection( CS_THISFILE );
   if Assigned( FPullSettings ) then try FPullSettings.Free; except end;
   if Assigned( FSettings     ) then try FSettings.Free;     except end;
   inherited Destroy;
end;

end.

