// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cAccounts;

interface

{$INCLUDE Compiler.inc}

uses Classes, SyncObjs, IniFiles;

type
   TAccounts = class
      private                                 
         FLock          : TCriticalSection;
         FUserList      : TStringList;
         FUserMailList  : TStringList;
         ActFile        : TIniFile;
         FUserIDMax     : Integer;
         LockedMailboxes: TList;
         AuthLogins     : TStringList;

         procedure SetValue( const UserID, PropertyID: Integer;
                             PropertyValue: String );
         function  GetValue( const UserID, PropertyID: Integer ): String;

         procedure Lock;
         procedure Unlock;
         procedure RefreshUserList; overload;
         procedure RefreshUserList( UID: Integer ); overload;
         procedure RefreshUserMailList( UID: Integer );
         function  IsUniqueUsername( const UserName: String ): Boolean;

      public
         property  Value[ const UserID, PropertyID: Integer ]: String read GetValue write SetValue;

         function  Add( NewName: String; UserID: Integer ): Integer;
         function  AddClone( NewName: String; TemplateID: Integer ): Integer;
         procedure Delete( const UserID: Integer );

         function  MailboxPath( const UserID: Integer ): String;
         function  HasMailbox( const UserID: Integer ): Boolean;
         function  MailboxLock( const UserID: Integer; const LockIt: Boolean ): Boolean;
         function  IsLocalMailbox( OrgMailAddr: String;
                                   out UserID, LocalType: Integer ): Boolean;
         function  HasMailForward( const UserID: Integer;
                                   out Account, Address, MailList: String;
                                   out KeepCopy: Boolean ): Boolean;

         procedure LogAuthenticatedLogin( const FromIP: LongInt );
         function  ChkAuthenticatedLogin( const FromIP: LongInt ): Boolean;

         function  UserIDOf( const UserName: String ): Integer;
         function  LoginID( const UserName, Password: String;
                            const ClientIPn: LongInt ): Integer;
         function  ListAll: String;
         procedure EnumUsernames( const SL: TStringList );

         constructor Create;
         destructor Destroy; override;
   end;

implementation

uses uConst, uConstVar, uVar, Windows, SysUtils, uTools, cPCRE, uBlowfish,
     uDateTime, cLogFileHamster, cIPAccess, uHamTools, cHamster, cSettingsDef,
     WinSock, uWinSock;

const
   ACT_FILENAME    = 'Accounts.!!!';
   ACT_CODEKEY     = 'Just to make it "unreadable", not "safe"!';
   ACTM_SECTION    = 'Common';
   ACTM_USERIDMAX  = 'UserIDMax';

function EncodeProperty( UserID: Integer; buf: String ): String;
var  i: Integer;
     s: String;
begin
   s := HamBlowfishEncipher( inttostr(UserID) + ACT_CODEKEY, buf );
   buf := '';
   for i := 1 to length(s) do begin
      buf := buf + inttohex( ord(s[i]), 2 );
   end;
   Result := buf;
end;

function DecodeProperty( UserID: Integer; buf: String ): String;
var  i: Integer;
     s: String;
begin
   SetLength( s, length(buf) div 2 );
   for i := 1 to length(buf) div 2 do begin
      s[i] := chr( strtointdef( '$' + copy(buf,i*2-1,2), 0 ) );
   end;
   if s='' then
      Result := #7#0#13#10
   else
      Result := HamBlowfishDecipher( inttostr(UserID) + ACT_CODEKEY, s );
end;

function TAccounts.Add( NewName: String; UserID: Integer ): Integer;
var  s: String;
     i: Integer;
begin
   Lock;
   try
      Result := ACTID_INVALID;
      if UserID = ACTID_NEW then UserID := FUserIDMax + 1;

      // delete all properties of old account
      if UserID <= FUserIDMax then Delete( UserID );

      // create account by setting some properties
      s := NewName;
      if (s = '') or not IsUniqueUsername( s ) then begin
         i := UserID;
         repeat
            s := 'newuser' + inttostr(i);
         until IsUniqueUsername( s );
      end;
      Value[ UserID, apUsername ] := s;
      Value[ UserID, apPassword ] := ACTPW_NOACCESS;
      Value[ UserID, apFullname ] := 'New User #' + inttostr(UserID);
      Value[ UserID, apMailbox  ] := '0';

      // adjust global values
      if UserID>FUserIDMax then begin
         FUserIDMax := UserID;
         ActFile.WriteInteger( ACTM_SECTION, ACTM_USERIDMAX, FUserIDMax );
      end;

      Result := UserID;

   finally Unlock; end;
end;

function TAccounts.AddClone( NewName: String; TemplateID: Integer ): Integer;
begin
   Lock;
   try
      Result := Add( NewName, ACTID_NEW );
      if Result = ACTID_INVALID then exit;

      if TemplateID <> ACTID_INVALID then begin
         if Value[TemplateID,apUsername] = '' then TemplateID := ACTID_INVALID;
      end;

      if TemplateID <> ACTID_INVALID then begin
         Value[Result,apIpRestriction  ] := Value[TemplateID,apIpRestriction  ];
         Value[Result,apNewsPost       ] := Value[TemplateID,apNewsPost       ];
         Value[Result,apNewsRead       ] := Value[TemplateID,apNewsRead       ];
         Value[Result,apNewsNewNews    ] := Value[TemplateID,apNewsNewNews    ];
         Value[Result,apNewsXHSearch   ] := Value[TemplateID,apNewsXHSearch   ];
         Value[Result,apNewsPeer       ] := Value[TemplateID,apNewsPeer       ];
         Value[Result,apNewsAutoSub    ] := Value[TemplateID,apNewsAutoSub    ];
         Value[Result,apMailbox        ] := Value[TemplateID,apMailbox        ];
         Value[Result,apMailSend       ] := Value[TemplateID,apMailSend       ];
         Value[Result,apRemoteControl  ] := Value[TemplateID,apRemoteControl  ];
         Value[Result,apMailFwdAccount ] := Value[TemplateID,apMailFwdAccount ];
         Value[Result,apMailFwdAddress ] := Value[TemplateID,apMailFwdAddress ];
         Value[Result,apMailFwdMailList] := Value[TemplateID,apMailFwdMailList];
         Value[Result,apMailFwdKeepCopy] := Value[TemplateID,apMailFwdKeepCopy];
         HasMailbox( Result ); // create/remove mailbox
      end;

   finally Unlock; end;
end;

procedure TAccounts.Delete( const UserID: Integer );
var  i, uid: Integer;
     UserName: String;
begin
   Lock;
   try
      // get account name of user to be deleted
      UserName := Value[ UserID, apUserName ];

      // remove account from mail lists
      Hamster.MailLists.RemoveAccountFromAllLists( UserName );

      // remove account from forward entries
      for i := 0 to FUserList.Count - 1 do begin
         uid := Integer( FUserList.Objects[i] );
         if Value[ uid, apMailFwdAccount ] = UserName then begin
            Value[ uid, apMailFwdAccount ] := '';
         end;
      end;

      // remove mailbox
      Value[ UserID, apMailbox ] := '0';

      // delete all properties of account by deleting whole ini-section
      if UserID <= FUserIDMax then begin
         ActFile.EraseSection( inttostr(UserID) );
         RefreshUserList( UserID );
         RefreshUserMailList( UserID );
      end;

   finally Unlock; end;
end;

function TAccounts.GetValue( const UserID, PropertyID: Integer ): String;
var  PropertyName: String;
begin
   if PropertyID = apUniqueID then begin

      Result := inttostr( UserID )

   end else begin

      Result := '';
      PropertyName := SettingsDef_Accounts.Def[ PropertyID ].Keyword;
      if length( PropertyName ) = 0 then exit;

      Lock;
      try
         try
            Result := ActFile.ReadString( inttostr(UserID), PropertyName, '' );
            if copy(PropertyName,1,1)='!' then Result:=DecodeProperty(UserID,Result);

            if (UserID=ACTID_ADMIN) and (PropertyID=apRemoteControl) then begin
               Result := inttostr( RC_PROFILE_FULLACCESS );
            end;
            
         except
            on E:Exception do
               Log( LOGID_ERROR, 'Account.GetValue.Exception ('
                  + 'u=' + inttostr(UserID)
                  + ' p=' + inttostr(PropertyID)
                  + ' pn="' + PropertyName + '"'
                  + ' r="' + Result + '"): ' + E.Message );
         end;
      finally Unlock; end;

   end;
end;

procedure TAccounts.SetValue( const UserID, PropertyID: Integer;
                              PropertyValue: String );
var  PropertyName: String;
begin
   if PropertyID = apUniqueID then exit;

   PropertyName := SettingsDef_Accounts.Def[ PropertyID ].Keyword;
   if copy(PropertyName,1,1)='!' then begin
      PropertyValue := EncodeProperty( UserID, PropertyValue );
   end;

   if (UserID=ACTID_ADMIN) and (PropertyID=apRemoteControl) then begin
      PropertyValue := inttostr( RC_PROFILE_FULLACCESS );
   end;

   Lock;
   try
      ActFile.WriteString( inttostr(UserID), PropertyName, PropertyValue );

      if PropertyID = apUsername then RefreshUserList( UserId );

      if (PropertyID = apMailbox) or (PropertyID = apUsername) then begin
         HasMailbox( UserID ); // create/remove mailbox
      end;

      if (PropertyID = apMailBox) or (PropertyID = apMailAddress) then begin
         RefreshUserMailList( UserId );
      end;

   finally Unlock; end;
end;

function TAccounts.MailboxPath( const UserID: Integer ): String;
var  s: String;
begin
   Lock;
   try
      s := Value[ UserID, apUsername ];
      if s='' then Result:=''
              else Result := AppSettings.GetStr(asPathMails) + s + '\';
   finally Unlock; end;
end;

function TAccounts.HasMailbox( const UserID: Integer ): Boolean;
begin
   Lock;
   try
      if UserID=ACTID_ADMIN then begin
         Result := True
      end else begin
         Result := ( Value[ UserID, apMailbox ] = '1' );
         if not Result then begin
            // forward functions require mailbox to be enabled
            Result := ( Value[ UserID, apMailFwdAccount  ] <> '' )
                   or ( Value[ UserID, apMailFwdAddress  ] <> '' )
                   or ( Value[ UserID, apMailFwdMailList ] <> '' );
            if Result then Value[ UserID, apMailbox ] := '1'; 
         end;
      end;

      if Result then begin
         if not DirectoryExists( MailboxPath(UserID) ) then begin
            ForceDirectories( MailboxPath(UserID) );
         end;
      end else begin
         if DirectoryExists( MailboxPath(UserID) ) then begin
            if not RemoveDir( MailboxPath(UserID) ) then begin
               Log( LOGID_WARN, Format(
                  'Mailbox "%s" could not be deleted!', [Value[ UserID, apUsername]]))
            end;
         end;
      end;
   finally Unlock; end;
end;

function TAccounts.HasMailForward( const UserID: Integer;
                                   out Account, Address, MailList: String;
                                   out KeepCopy: Boolean): Boolean;
begin
   Result := False;
   if UserID = ACTID_INVALID then exit;
   if not HasMailbox( UserID ) then exit;
   
   Account  := Value[ UserID, apMailFwdAccount  ];
   Address  := Value[ UserID, apMailFwdAddress  ];
   MailList := Value[ UserID, apMailFwdMailList ];
   KeepCopy := Value[ UserID, apMailFwdKeepCopy ] <> '0';

   Result := (Account<>'') or (Address<>'') or (MailList<>'');
end;

function TAccounts.MailboxLock( const UserID: Integer;
                                const LockIt: Boolean ): Boolean;
var  i: Integer;
begin
   Lock;
   try
      Result := False;
      i := LockedMailboxes.IndexOf( Pointer(UserID) );
      if LockIt then begin
         if i<0 then begin
            LockedMailboxes.Add( Pointer(UserID) );
            Result := True;
         end;
      end else begin
         if i>=0 then begin
            LockedMailboxes.Delete( i );
            Result := True;
         end;
      end;
   finally Unlock end;
end;

function TAccounts.IsLocalMailbox( OrgMailAddr: String;
                                   out UserID, LocalType: Integer ): Boolean;
var  i: Integer;
     MailAddr, MailNam, MailDom, Reason, s: String;
     IsLocal: Boolean;
begin
   Result     := False;
   Reason     := '';
   UserID     := ACTID_INVALID;
   MailAddr   := ExtractMailAddr( OrgMailAddr );
   LocalType  := LOCALMAILTYPE_NORMAL;

   try

      Lock;
      try
         // 1.) a) check list of assigned addresses if local mailboxes are enabled
         // 1.) b) if no specific one was found, use a given catch-all (*@domain)
         i := FUserMailList.IndexOf( MailAddr );
         if i >= 0 then begin
            Result := True;
            Reason := 'Is listed in account''s mail addresses';
            UserID := Integer( FUserMailList.Objects[i] );
         end else begin
            s := ExtractMailDomain( MailAddr );
            i := FUserMailList.IndexOf( '*@' + s );
            if i >= 0 then begin // catch-all '*@domain'
               Result := True;
               Reason := 'Is catch-all account of domain ' + s;
               UserID := Integer( FUserMailList.Objects[i] );
            end;
         end;

         // 2.) check for local domains
         if not Result then begin

            i := LastDelimiter( '@', MailAddr );
            if i = 0 then begin
               MailNam := MailAddr;
               MailDom := ''; // no domain -> always local
               IsLocal := True;
               Reason  := 'No domain in mail address';
            end else begin
               MailNam := copy( MailAddr, 1, i-1 );
               MailDom := copy( MailAddr, i+1, 255 );
               if (MailDom = 'test'     ) or EndsWith( MailDom, '.test'      )
               or (MailDom = 'example'  ) or EndsWith( MailDom, '.example'   )
               or (MailDom = 'invalid'  ) or EndsWith( MailDom, '.invalid'   )
               or (MailDom = 'localhost') or EndsWith( MailDom, '.localhost' ) then begin
                  IsLocal := True;
                  Reason  := MailDom + ' is a special, local-only domain';
                  LocalType := LOCALMAILTYPE_INVALID;
               end else begin
                  IsLocal := IsLocalDomain( MailDom );
                  if IsLocal then Reason := MailDom + ' is defined as local domain';
               end;
            end;

            if IsLocal then begin
               Result := True;
               UserID := UserIdOf( UnDQuoteStr( MailNam ) );
               if UserID=ACTID_INVALID then begin
                  if MailNam='postmaster' then UserID:=ACTID_ADMIN;
                  if MailNam='abuse'      then UserID:=ACTID_ADMIN;
                  if MailNam='usenet'     then UserID:=ACTID_ADMIN;
                  if MailNam='news'       then UserID:=ACTID_ADMIN;
               end;
               if UserID<>ACTID_ADMIN then begin
                  if not HasMailbox( UserID ) then UserID:=ACTID_INVALID;
               end;

               if UserID=ACTID_INVALID then begin
                  UserID := ACTID_ADMIN;
                  if LocalType=LOCALMAILTYPE_NORMAL then LocalType:=LOCALMAILTYPE_UNKNOWN;
               end;
            end;

         end;

      finally Unlock; end;

   finally
      if Result then Log( LOGID_DEBUG, '"' + OrgMailAddr + '" is local account "'
                        + Value[ UserID, apUsername ] + '": ' + Reason );
   end;
end;

procedure TAccounts.LogAuthenticatedLogin( const FromIP: LongInt );
var  ValidTil, i: LongInt;
begin
   Lock;
   try
      // add/extend authentication period for address
      ValidTil := DateTimeToUnixTime( NowGMT )
                + Hamster.Config.Settings.GetInt( hsLocalSmtpAfterPop3Time );
      i := AuthLogins.IndexOf( inttohex(FromIP,8) );
      if i<0 then begin
         AuthLogins.AddObject( inttohex(FromIP,8), Pointer(ValidTil) );
      end else begin
         AuthLogins.Objects[i] := Pointer( ValidTil );
      end;

      // remove expired periods
      ValidTil := DateTimeToUnixTime( NowGMT );
      for i:=AuthLogins.Count-1 downto 0 do begin
         if LongInt(AuthLogins.Objects[i])<ValidTil then AuthLogins.Delete(i);
      end;
   finally Unlock; end;
end;

function TAccounts.ChkAuthenticatedLogin( const FromIP: LongInt ): Boolean;
var  i: Integer;
begin
   Lock;
   try
      Result := False;
      i := AuthLogins.IndexOf( inttohex(FromIP,8) );
      if i>=0 then begin
         if LongInt(AuthLogins.Objects[i])>=DateTimeToUnixTime(NowGMT) then begin
            Result := True;
         end else begin
            AuthLogins.Delete( i ); // expired
         end;
      end;
   finally Unlock; end;
end;

function TAccounts.ListAll: String;
var  i, UID: Integer;
begin
   Lock;
   try
      Result := '';
      for i := 0 to FUserList.Count - 1 do begin
         UID := Integer( FUserList.Objects[i] );
         Result := Result + FUserList[i]
                 + TAB    + inttostr( UID )
                 + TAB    + DQuoteStr( Value[ UID, apFullname ] )
                 + CRLF;
      end;
   finally Unlock; end;
end;

procedure TAccounts.EnumUsernames( const SL: TStringList );
var  i: Integer;
begin
   Lock;
   try
      SL.Clear;
      for i := 0 to FUserList.Count - 1 do begin
         SL.AddObject( FUserList[i], FUserList.Objects[i] );
      end;
   finally Unlock; end;
end;

procedure TAccounts.RefreshUserList;
// refresh internal list for all users
var  i, k, UID: Integer;
     s: String;
     SL: TStringList;
     prs: TParser;
begin
   SL := TStringList.Create;
   prs := TParser.Create;

   Lock;
   try
      FUserList.Clear;

      try
         ActFile.ReadSections( SL );

         for i := 0 to SL.Count -1 do begin

            s := SL.Strings[i];
            if (length(s) > 0) and (s[1] in ['1'..'9']) then begin

               UID := strtointdef( s, ACTID_INVALID );
               if UID <> ACTID_INVALID then begin

                  FUserList.AddObject( Value[ UID, apUsername ], Pointer( UID ) );

                  if HasMailBox( UID ) then begin

                     prs.SplitWhSpace( Value[ UID, apMailAddress ] );
                     k := 0;
                     repeat
                        s := prs.sPart( k, '' );
                        if s = '' then break;
                        FUserMailList.AddObject( s, Pointer( UID ) );
                        inc( k );
                     until False;

                  end;

               end;

            end;

         end;

      except
         on E:Exception do
            Log( LOGID_ERROR, 'Account.RefreshUserList.Exception: ' + E.Message );
      end;

   finally
      Unlock;
      prs.Free;
      SL.Free;
   end;
end;

procedure TAccounts.RefreshUserList( UID: Integer );
// refresh internal username list for given ID
var  k: Integer;
begin
   if UID = ACTID_INVALID then exit;

   Lock;
   try

      // remove old name
      for k := FUserList.Count - 1 downto 0 do begin
         if Integer( FUserList.Objects[k] ) = UID then begin
            FUserList.Delete( k );
         end;
      end;

      // add new name
      FUserList.AddObject( Value[ UID, apUsername ], Pointer( UID ) );

   finally Unlock end;
end;

procedure TAccounts.RefreshUserMailList( UID: Integer );
// refresh internal list of known mail addresses for given ID
var  k: Integer;
     s: String;
     prs: TParser;

begin
   if UID = ACTID_INVALID then exit;

   Lock;
   try

      // remove all old addresses
      for k := FUserMailList.Count - 1 downto 0 do begin
         if Integer( FUserMailList.Objects[k] ) = UID then begin
            FUserMailList.Delete( k );
         end;
      end;

      // add new addresses
      if HasMailBox( UID ) then begin

         prs := TParser.Create;
         try

            prs.SplitWhSpace( Value[ UID, apMailAddress ] );
            k := 0;
            repeat
               s := prs.sPart( k, '' );
               if s = '' then break;
               FUserMailList.AddObject( s, Pointer( UID ) );
               inc( k );
            until False;

         finally prs.Free end;

      end;

   finally Unlock end;
end;

function TAccounts.UserIDOf( const UserName: String ): Integer;
var  i : Integer;
begin
   Lock;
   try
      Result := ACTID_INVALID;
      i := FUserList.IndexOf( UserName );
      if i >= 0 then Result := Integer( FUserList.Objects[i] );
   finally Unlock; end;
end;

function TAccounts.IsUniqueUsername( const UserName: String ): Boolean;
begin
   Lock;
   try
      Result := ( UserIDOf(UserName) = ACTID_INVALID );
   finally Unlock; end;
end;

function TAccounts.LoginID( const UserName, Password: String;
                            const ClientIPn: LongInt ): Integer;
var  iprList, ipr, pw, ipName: String;
     SL: TStringList;
     i, j, hChkIP, ipMin, ipMax: Integer;
     ok: Boolean;
begin
   Lock;
   try

      // identify account
      Result := UserIDOf( UserName );
      if Result <> ACTID_INVALID then begin

         // check IP restriction
         iprList := TrimWhSpace( Value[ Result, apIpRestriction ] );
         if iprList = '' then begin

            ok  := True; // ok, unrestricted

         end else begin

            ok := False;
            SL := TStringList.Create;
            try

               ArgsSplitChar( iprList, SL, ',' );
               for i := 0 to SL.Count - 1 do begin

                  ipr := TrimWhSpace( SL[i] );
                  if ipr = '' then continue;

                  if ipr[1] in ['0'..'9'] then begin
                     // compare with given IP [range]
                     j := Pos( '-', ipr );
                     if j = 0 then begin
                        ipMin := StrTohAddr( ipr );
                        ipMax := ipMin;
                     end else begin
                        ipMin := StrTohAddr( TrimWhSpace( copy( ipr, 1, j-1 ) ) );
                        ipMax := StrTohAddr( TrimWhSpace( copy( ipr, j+1, MaxInt ) ) );
                     end;
                     hChkIP := ntohl( ClientIPn );
                     if (inttohex(hChkIP,8) >= inttohex(ipMin,8)) and
                        (inttohex(hChkIP,8) <= inttohex(ipMax,8)) then begin
                        ok := True;
                        break;
                     end;

                  end else if ipr[1] = '?' then begin
                     // compare with current IP of given name
                     ipName := copy( ipr, 2, MaxInt );
                     ipMin  := LookupHostAddr( ipName );
                     if ClientIPn = ipMin then begin ok := True; break end;
                     
                  end else begin
                     // compare with given IP name
                     ipName := LookupHostName( ClientIPn );
                     if CompareText( ipName, ipr ) = 0 then begin
                        ok := True;
                        break;
                     end;
                     if length( ipName ) > length( ipr ) then begin
                        ipName := copy( ipName,
                                        length(ipName) - length(ipr) - 1,
                                        length(ipr) + 1 );
                        if CompareText( ipName, '.' + ipr ) = 0 then begin
                           ok := True;
                           break;
                        end;
                     end;

                  end;
                  
               end;

            finally SL.Free end;
            
         end;
         
         if not ok then begin
            Result := ACTID_INVALID; // reject
            Log( LOGID_WARN, 'Account "' + UserName + '" tried to login from '
                           + 'unwanted IP ' + nAddrToStr( ClientIPn ) + '! '
                           + 'Login rejected!' );
            exit;
         end;

         // check password
         pw := Value[ Result, apPassword ];
         if pw = ACTPW_NOACCESS then Result := ACTID_INVALID;
         if pw = ACTPW_NOTNEEDED then begin
            Log( LOGID_WARN, 'Account "' + UserName + '" needs a password!' );
         end else begin
            if pw <> Password then Result := ACTID_INVALID;
         end;

      end;

   finally Unlock; end;
end;

procedure TAccounts.Lock;
begin
   FLock.Enter;
end;

procedure TAccounts.Unlock;
begin
   FLock.Leave;
end;

constructor TAccounts.Create;
begin
   inherited Create;

   FLock := TCriticalSection.Create;

   FUserList := TStringList.Create;
   FUserList.Sorted := True;

   FUserMailList := TStringList.Create;
   FUserMailList.Sorted := True;
   FUserMailList.Duplicates := dupIgnore;

   ActFile := TIniFile.Create( AppSettings.GetStr(asPathBase) + ACT_FILENAME );

   LockedMailboxes := TList.Create;

   AuthLogins := TStringList.Create;
   AuthLogins.Sorted := True;

   RefreshUserList;

   FUserIDMax := ActFile.ReadInteger( ACTM_SECTION, ACTM_USERIDMAX, 0 );
   if FUserIDMax <= 0 then begin

      // create common section at top of file
      ActFile.WriteInteger( ACTM_SECTION, ACTM_USERIDMAX, FUserIDMax );

      // create default admin-account
      Add( 'admin', ACTID_ADMIN );
      Value[ ACTID_ADMIN, apUsername        ] := 'admin';
      Value[ ACTID_ADMIN, apPassword        ] := 'admin';
      Value[ ACTID_ADMIN, apFullname        ] := 'Hamster Administrator';
      Value[ ACTID_ADMIN, apIpRestriction   ] := '';
      Value[ ACTID_ADMIN, apNewsPost        ] := '.*';
      Value[ ACTID_ADMIN, apNewsRead        ] := '.*';
      Value[ ACTID_ADMIN, apNewsNewNews     ] := '1';
      Value[ ACTID_ADMIN, apNewsXHSearch    ] := '1';
      Value[ ACTID_ADMIN, apNewsPeer        ] := '0';
      Value[ ACTID_ADMIN, apNewsAutoSub     ] := inttostr(NEWSAUTOSUB_ALL);
      Value[ ACTID_ADMIN, apMailbox         ] := '1';
      Value[ ACTID_ADMIN, apMailSend        ] := '1';
      Value[ ACTID_ADMIN, apMailAddress     ] := '';
      Value[ ACTID_ADMIN, apRemoteControl   ] := inttostr(RC_PROFILE_FULLACCESS);
      Value[ ACTID_ADMIN, apMailFwdAccount  ] := '';
      Value[ ACTID_ADMIN, apMailFwdAddress  ] := '';
      Value[ ACTID_ADMIN, apMailFwdMailList ] := '';
      Value[ ACTID_ADMIN, apMailFwdKeepCopy ] := '1';

   end;
end;

destructor TAccounts.Destroy;
begin
   try
      ActFile.Free;
      LockedMailboxes.Free;
      AuthLogins.Free;
      FUserList.Free;
      FUserMailList.Free;
      FLock.Free;
   except end;

   inherited Destroy;
end;

end.
