unit fUserTasks;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ComCtrls;

type
  TfrmUserTasks = class(TForm)
    grpMaster: TGroupBox;
    grpDetail: TGroupBox;
    cbUserTasks: TComboBox;
    Label5: TLabel;
    btnEdit: TButton;
    btnAdd: TButton;
    btnDelete: TButton;
    Label1: TLabel;
    lbUserTasks: TListBox;
    btnSave: TButton;
    btnUndo: TButton;
    StatusBar1: TStatusBar;
    Label2: TLabel;
    lbTasks: TListBox;
    btnSelect: TButton;
    btnUnselect: TButton;
    btnUp: TButton;
    btnDown: TButton;
    procedure FormCreate(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure btnEditClick(Sender: TObject);
    procedure btnAddClick(Sender: TObject);
    procedure btnDeleteClick(Sender: TObject);
    procedure cbUserTasksClick(Sender: TObject);
    procedure btnSaveClick(Sender: TObject);
    procedure btnUndoClick(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure lbUserTasksClick(Sender: TObject);
    procedure lbTasksClick(Sender: TObject);
    procedure btnSelectClick(Sender: TObject);
    procedure btnUnselectClick(Sender: TObject);
    procedure btnUpClick(Sender: TObject);
    procedure btnDownClick(Sender: TObject);
    procedure lbUserTasksDblClick(Sender: TObject);
    procedure lbTasksDblClick(Sender: TObject);
    procedure FormResize(Sender: TObject);
  private
    { Private-Deklarationen }
    Editing: Boolean;
    function IndexOfUserTaskName( const Name: String ): Integer;
    function SelectedUserTaskName: String;
    function SelectName( SelName: String ): Boolean;
    function RefreshList( SelName: String ): Boolean;
    procedure RefreshAvailableTasks;
    function LoadAndShowUserTask( const Name: String ): Boolean;
    procedure ReflectEditmode;
    procedure ShowInfo( ctrl: TControl );
  public
    { Public-Deklarationen }
    class function ShowDialog( AOwner: TComponent; const SelName: String ): Integer;
  end;

var
  frmUserTasks: TfrmUserTasks;

implementation

{$R *.DFM}

uses uConst, uTools, uGlobal, cLiveMsg, dInput, dDialogs, cUserTasks;


class function TfrmUserTasks.ShowDialog( AOwner: TComponent; const SelName: String ): Integer;
begin
   with TfrmUserTasks.Create( AOwner ) do try
      if SelName <> '' then SelectName( SelName );
      Result := ShowModal;
   finally Free end;
end;

procedure TfrmUserTasks.FormCreate(Sender: TObject);
begin
   GrowDpiForm( self );

   Editing := False;
   ReflectEditmode;
   RefreshList( '' );
   RefreshAvailableTasks;
end;

procedure TfrmUserTasks.FormResize(Sender: TObject);
begin
   grpMaster.Width := ClientWidth - grpMaster.Left * 2;
   grpDetail.Width := ClientWidth - grpDetail.Left * 2;
   grpDetail.Height := ClientHeight - grpDetail.Top - grpMaster.Top - StatusBar1.Height;
end;

procedure TfrmUserTasks.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
var  i: Integer;
begin
   if Editing then begin
      i := HMessageDlg( 'Save changes?', mtConfirmation, [mbYes,mbNo,mbCancel] );
      if i = mrYes then btnSaveClick(nil);
      if i = mrNo  then btnUndoClick(nil);
      if (i = mrCancel) or Editing then CanClose := False;
   end;
end;

procedure TfrmUserTasks.FormClose(Sender: TObject; var Action: TCloseAction);
begin
   Action := caFree;
end;

procedure TfrmUserTasks.ReflectEditmode;
begin
   cbUserTasks.Enabled := not Editing;
   btnEdit.Enabled     := not Editing;
   btnAdd.Enabled      := not Editing;
   btnDelete.Enabled   := not Editing;

   btnSave.Enabled     := Editing;
   btnUndo.Enabled     := Editing;
   btnSelect.Enabled   := Editing;
   btnUnselect.Enabled := Editing;
   btnUp.Enabled       := Editing;
   btnDown.Enabled     := Editing;
end;

function TfrmUserTasks.IndexOfUserTaskName( const Name: String ): Integer;
var  i: Integer;
begin
   Result := -1;
   for i := 0 to cbUserTasks.Items.Count - 1 do begin
      if CompareText( cbUserTasks.Items[i], Name ) = 0 then begin
         Result := i;
         exit;
      end;
   end;
end;

function TfrmUserTasks.SelectedUserTaskName: String;
begin
   Result := '';
   if cbUserTasks.ItemIndex < 0 then exit;
   if cbUserTasks.ItemIndex >= cbUserTasks.Items.Count then exit;
   Result := cbUserTasks.Items[ cbUserTasks.ItemIndex ];
end;

function TfrmUserTasks.LoadAndShowUserTask( const Name: String ): Boolean;
var  Reply: TLiveMsg;
begin
   Result := False;
   lbUserTasks.Clear;
   if Name = '' then exit;

   Reply := LiveConnector.RCLiveRequest(
               TLiveMsg.Create( LMREQ_USERTASKS_GET, Name )
            );
   if Assigned( Reply ) then try
      if ( Reply.MsgType = LMREP_OK ) then begin
         lbUserTasks.Items.Text := Reply.MsgData;
         Result := True;
      end;
   finally Reply.Free end;
end;

function TfrmUserTasks.SelectName( SelName: String ): Boolean;
var  i: Integer;
begin
   Result := False;
   
   // select given name
   i := IndexOfUserTaskName( SelName );
   if i >= 0 then begin
      cbUserTasks.ItemIndex := i;
      Result := True;
   end;

   // show selected task
   Result := Result and LoadAndShowUserTask( SelectedUserTaskName );
end;

function TfrmUserTasks.RefreshList( SelName: String ): Boolean;
var  Reply: TLiveMsg;
begin
   if SelName = '' then SelName := SelectedUserTaskName;

   // refresh list
   cbUserTasks.Clear;

   // load list of User Tasks
   Reply := LiveConnector.RCLiveRequest(
      TLiveMsg.Create( LMREQ_USERTASKS_LIST, '' )
   );
   if Assigned( Reply ) then try
      if Reply.MsgType = LMREP_OK then cbUserTasks.Items.Text := Reply.MsgData;
   finally Reply.Free end;

   if (SelName = '') and (cbUserTasks.Items.Count > 0) then begin
      SelName := cbUserTasks.Items[0];
   end;

   // select given name
   Result := SelectName( SelName );
end;

procedure TfrmUserTasks.RefreshAvailableTasks;
var  SL: TStringList;
     Reply: TLiveMsg;
     i: Integer;
     s, s1, s2: String;
begin
   lbTasks.Clear;
   SL := TStringList.Create;
   try

      // RAS-DIAL
      Reply := LiveConnector.RCLiveRequest(
         TLiveMsg.Create( LMREQ_PASSWORD_LIST, '' )
      );
      if Assigned( Reply ) then try
         if Reply.MsgType = LMREP_OK then begin
            SL.Text := Reply.MsgData;
            for i := 0 to SL.Count - 1 do begin
               s  := SL[i]; // <identifier> TAB <description> TAB <pwset>
               s1 := UnDQuoteStr( NextSepPart( s, TAB ) );
               s2 := UnDQuoteStr( NextSepPart( s, TAB ) );
               if UpperCase(copy(s2,1,4))='RAS:' then begin
                  lbTasks.Items.Add( USERTASKCMD_RAS_DIAL + ' ' + '"' + s1 + '"' );
               end;
            end;
         end;
      finally FreeAndNil(Reply) end;

      // RAS-HANGUP, -REQUIRED
      lbTasks.Items.Add( USERTASKCMD_RAS_HANGUP );
      lbTasks.Items.Add( USERTASKCMD_RAS_REQUIRED );

      // MAILS-SEND
      Reply := LiveConnector.RCLiveRequest(
         TLiveMsg.Create( LMREQ_SRVSMTP_LIST, '' )
      );
      if Assigned( Reply ) then try
         if Reply.MsgType = LMREP_OK then begin
            SL.Text := Reply.MsgData;
            for i := 0 to SL.Count - 1 do begin
               lbTasks.Items.Add( USERTASKCMD_MAILS_SEND + ' ' + '"' + SL[i] + '"' );
            end;
         end;
      finally FreeAndNil(Reply) end;

      // MAILS-SEND-MX
      lbTasks.Items.Add( USERTASKCMD_MAILS_SEND_MX );

      // MAILS-FETCH
      Reply := LiveConnector.RCLiveRequest(
         TLiveMsg.Create( LMREQ_SRVPOP3_LIST, '' )
      );
      if Assigned( Reply ) then try
         if Reply.MsgType = LMREP_OK then begin
            SL.Text := Reply.MsgData;
            for i := 0 to SL.Count - 1 do begin
               lbTasks.Items.Add( USERTASKCMD_MAILS_FETCH + ' ' + '"' + SL[i] + '"' );
            end;
         end;
      finally FreeAndNil(Reply) end;

      // NEWS
      Reply := LiveConnector.RCLiveRequest(
         TLiveMsg.Create( LMREQ_SRVNNTP_LIST, '' )
      );
      if Assigned( Reply ) then try
         if Reply.MsgType = LMREP_OK then begin
            SL.Text := Reply.MsgData;
            for i := 0 to SL.Count - 1 do begin
               lbTasks.Items.Add( USERTASKCMD_NEWS + ' ' + '"' + SL[i] + '"' );
            end;
         end;
      finally FreeAndNil(Reply) end;

      // WAIT
      lbTasks.Items.Add( USERTASKCMD_WAIT );
      lbTasks.Items.Add( USERTASKCMD_WAIT_ALL );

      // SCRIPT, SCRIPT-WAIT
      Reply := LiveConnector.RCLiveRequest(
                  TLiveMsg.Create( LMREQ_SCRIPT_DIR, '' )
               );
      if Assigned( Reply ) then try
         if ( Reply.MsgType = LMREP_OK ) then begin
            SL.Text := Reply.MsgData;
            for i := 0 to SL.Count - 1 do begin
               s  := SL[i]; // <time> TAB <size> TAB <path/name>
               s1 := UnDQuoteStr( NextSepPart( s, TAB ) );
               s1 := UnDQuoteStr( NextSepPart( s, TAB ) );
               s1 := UnDQuoteStr( NextSepPart( s, TAB ) );
               if (Pos('.hsc',s1) > 0) and (Pos('\',s1) = 0) then begin
                  lbTasks.Items.Add( USERTASKCMD_SCRIPT + ' ' + '"' + s1 + '"' );
                  lbTasks.Items.Add( USERTASKCMD_SCRIPT_WAIT + ' ' + '"' + s1 + '"' );
               end;
            end;
         end;
      finally FreeAndNil(Reply) end;

   finally SL.Free end;
end;

procedure TfrmUserTasks.btnEditClick(Sender: TObject);
var  EditName: String;
begin
   EditName := SelectedUserTaskName;
   if EditName = '' then exit;
   
   Editing := True;
   ReflectEditmode;
   lbUserTasks.SetFocus;
end;

procedure TfrmUserTasks.btnAddClick(Sender: TObject);
var  NewName, s: String;
     Reply: TLiveMsg;
     i: Integer;
     c: Char;
     ok: Boolean;
begin
   // get and check new name
   NewName := '';
   if not InputDlgStr( 'Add User Task', 'Name:', NewName, 0 ) then exit;
   NewName := TrimWhSpace( NewName );
   if NewName = '' then exit;

   if IndexOfUserTaskName( NewName ) > 0 then begin
      HMessageDlg( 'User Task "' + NewName + '" already exists!',
                   'Add User Task', mtWarning, [mbOK] );
      exit;
   end;

   ok := True;
   for i := 1 to length(NewName) do begin
      c := NewName[i];
      if c < #32 then ok := False;
      if c = '\' then ok := False;
      if c = ':' then ok := False;
      if c = '<' then ok := False;
      if c = '>' then ok := False;
      if c = '|' then ok := False;
      if not ok then begin
         s := 'Invalid User Task name "' + NewName + '"!' + #13#10
            + 'Character ' + c + ' is not allowed!';
         HMessageDlg( PChar(s), 'Add User Task', mtWarning, [mbOK] );
         exit;
      end;
   end;

   // create new user task
   Reply := LiveConnector.RCLiveRequest(
      TLiveMsg.Create( LMREQ_USERTASKS_SET, NewName + CRLF + '' )
   );
   if Assigned( Reply ) then try
      if Reply.MsgType <> LMREP_OK then begin
         HMessageDlg( 'User Task "' + NewName + '" could not be created!',
                      'Add User Task', mtError, [mbOK] );
         exit;
      end;
   finally Reply.Free end;

   // refresh list and select new name
   if not RefreshList( NewName ) then exit;

   // perform edit with new item
   btnEdit.Click;
end;

procedure TfrmUserTasks.btnDeleteClick(Sender: TObject);
var  DelName, s: String;
     Reply: TLiveMsg;
begin
   DelName := SelectedUserTaskName;
   if DelName = '' then exit;

   s := Format( 'Delete User Task "%s" now?', [DelName] );
   if HMessageDlg( PChar(s), mtConfirmation, [mbYes,mbNo] ) <> mrYes then exit;

   Reply := LiveConnector.RCLiveRequest(
      TLiveMsg.Create( LMREQ_USERTASKS_DEL, DelName )
   );
   if Assigned( Reply ) then try
      // if Reply.MsgType <> LMREP_OK then exit;
   finally Reply.Free end;

   RefreshList( '' );
end;

procedure TfrmUserTasks.cbUserTasksClick(Sender: TObject);
begin
   LoadAndShowUserTask( SelectedUserTaskName );
end;

procedure TfrmUserTasks.btnSaveClick(Sender: TObject);
var  EditName: String;
begin
   EditName := SelectedUserTaskName;
   if EditName = '' then exit;

   if not LiveConnector.RCLiveRequestOK(
             TLiveMsg.Create( LMREQ_USERTASKS_SET,
                              EditName + #13#10 + lbUserTasks.Items.Text )
          ) then begin
      HMessageDlg( 'User Task "' + EditName + '" could not be saved!',
                   'Save User Tasks', mtError, [mbOK] );
      exit;
   end;

   Editing := False;
   ReflectEditmode;
   cbUserTasks.SetFocus;
end;

procedure TfrmUserTasks.btnUndoClick(Sender: TObject);
begin
   LoadAndShowUserTask( SelectedUserTaskName );
   Editing := False;
   ReflectEditmode;
   cbUserTasks.SetFocus;
end;

procedure TfrmUserTasks.lbUserTasksClick(Sender: TObject);
begin
   ShowInfo( lbUserTasks );
end;

procedure TfrmUserTasks.lbTasksClick(Sender: TObject);
begin
   ShowInfo( lbTasks );
end;

procedure TfrmUserTasks.ShowInfo( ctrl: TControl );

   function GetKeyword( lb: TListBox ): String;
   var  s: String;
        i: Integer;
   begin
      Result := '';
      if lb.ItemIndex < 0 then exit;
      if lb.ItemIndex >= lb.Items.Count then exit;
      s := lb.Items[ lb.ItemIndex ];
      i := PosWhSpace( s );
      if i > 0 then SetLength( s, i - 1 );
      Result := s;
   end;
   
var  k, s: String;
begin
   k := '';
   if ctrl = nil then ctrl := ActiveControl;
   if ctrl is TListBox then k := GetKeyword( ctrl as TListBox );

   s := '';
   if k <> '' then begin

      if      k=USERTASKCMD_RAS_DIAL      then
         s:='Dial given connection; skip if already connected'

      else if k=USERTASKCMD_RAS_HANGUP    then
         s:='Hangup connection (only if task itself has dialed)'

      else if k=USERTASKCMD_RAS_REQUIRED  then
         s:='Stop here if there is no RAS connection established'

      else if k=USERTASKCMD_MAILS_SEND    then
         s:='Send mails to given server'

      else if k=USERTASKCMD_MAILS_SEND_MX then
         s:='Send mails by MX'

      else if k=USERTASKCMD_MAILS_FETCH   then
         s:='Fetch mails from given server'

      else if k=USERTASKCMD_NEWS          then
         s:='Transfer news messages from/to given newsserver'

      else if k=USERTASKCMD_WAIT          then
         s:='Wait until previous tasks are finished'

      else if k=USERTASKCMD_WAIT_ALL      then
         s:='Wait until all currently running tasks are finished'

      else if k=USERTASKCMD_SCRIPT        then
         s:='Start script and continue immediately'

      else if k=USERTASKCMD_SCRIPT_WAIT   then
         s:='Start script and wait until it''s finished'
         
      else
         s:='(unknown command)';

      s := ' ' + k + ':  ' + s;
   end;

   StatusBar1.SimpleText := s;
end;

procedure TfrmUserTasks.btnSelectClick(Sender: TObject);
var  i: Integer;
     s: String;
begin
   i := lbTasks.ItemIndex;
   if i < 0 then exit;
   if i >= lbTasks.Items.Count then exit;

   s := lbTasks.Items[i];
   i := lbUserTasks.Items.Add( s );
   lbUserTasks.ItemIndex := i;
end;

procedure TfrmUserTasks.btnUnselectClick(Sender: TObject);
var  i: Integer;
begin
   i := lbUserTasks.ItemIndex;
   if i < 0 then exit;
   if i >= lbUserTasks.Items.Count then exit;

   lbUserTasks.Items.Delete( i );
   if i >= lbUserTasks.Items.Count then dec(i);
   if i >= 0 then lbUserTasks.ItemIndex := i;
end;

procedure TfrmUserTasks.btnUpClick(Sender: TObject);
var  i, k: Integer;
begin
   i := lbUserTasks.ItemIndex;
   if i < 0 then exit;
   if i >= lbUserTasks.Items.Count then exit;

   k := i - 1;
   if k < 0 then exit;

   lbUserTasks.Items.Exchange( i, k );
   lbUserTasks.ItemIndex := k;
end;

procedure TfrmUserTasks.btnDownClick(Sender: TObject);
var  i, k: Integer;
begin
   i := lbUserTasks.ItemIndex;
   if i < 0 then exit;
   if i >= lbUserTasks.Items.Count then exit;

   k := i + 1;
   if k >= lbUserTasks.Items.Count then exit;

   lbUserTasks.Items.Exchange( i, k );
   lbUserTasks.ItemIndex := k;
end;

procedure TfrmUserTasks.lbUserTasksDblClick(Sender: TObject);
begin
   if btnUnselect.Enabled then btnUnselect.Click;
end;

procedure TfrmUserTasks.lbTasksDblClick(Sender: TObject);
begin
   if btnSelect.Enabled then btnSelect.Click;
end;

end.
