unit fService;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls;

type
  TfrmService = class(TForm)
    Label1: TLabel;
    emComputer: TEdit;
    Label2: TLabel;
    Label3: TLabel;
    emService: TEdit;
    btnCheck: TButton;
    btnStart: TButton;
    btnStop: TButton;
    lblStatus: TLabel;
    procedure FormCreate(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure btnCheckClick(Sender: TObject);
    procedure btnStartClick(Sender: TObject);
    procedure btnStopClick(Sender: TObject);
  private
    { Private-Deklarationen }
    procedure ShowStatus( ok: Boolean; msg: String ); overload;
    procedure ShowStatus( msg: String ); overload;
    function AppServiceIsInstalled( const computerName, serviceName: String;
                                    out ErrMsg: String;
                                    out Running: Boolean ): Boolean;
    function AppServiceStart  ( const computerName, serviceName: String;
                                out ErrMsg: String ): Boolean;
    function AppServiceStop   ( const computerName, serviceName: String;
                                out ErrMsg: String ): Boolean;
    function AppServiceControl( const computerName, serviceName: String;
                                const fdwControl: DWORD;
                                out ErrMsg: String ): Boolean;
  public
    { Public-Deklarationen }
  end;


implementation

uses uWinSvc, uGlobal, uTools;

{$R *.DFM}

procedure TfrmService.FormCreate(Sender: TObject);
begin
   emComputer.Text  := SvcComputerName;
   emService.Text   := SvcServiceName;
   btnCheck.Enabled := IsWinSvcAvailable;
end;

procedure TfrmService.FormClose(Sender: TObject; var Action: TCloseAction);
begin
   Action := caFree;
end;

procedure TfrmService.btnCheckClick(Sender: TObject);
var  cn, sn, errMsg: String;
     running: Boolean;
begin
   cn := Trim( emComputer.Text );
   sn := Trim( emService.Text );
   if length(sn) = 0 then sn := 'HamsterService';

   btnStart.Enabled := False;
   btnStop.Enabled  := False;
   ShowStatus( 'Checking service status ...' );
   Application.ProcessMessages;

   if AppServiceIsInstalled( cn, sn, errMsg, running ) then begin
      if running then begin
         ShowStatus( True, 'Service is installed and running.' );
         btnStop.Enabled := True;
      end else begin
         ShowStatus( True, 'Service is installed but stopped.' );
         btnStart.Enabled := True;
      end;
      SaveLocalSettings;
   end else begin
      ShowStatus( False, errMsg );
   end;
end;

procedure TfrmService.btnStartClick(Sender: TObject);
var  cn, sn, errMsg: String;
begin
   cn := Trim( emComputer.Text );
   sn := Trim( emService.Text );
   if length(sn) = 0 then sn := 'HamsterService';

   ShowStatus( 'Starting service ...' );
   Application.ProcessMessages;

   if AppServiceStart( cn, sn, errMsg ) then begin
      ShowStatus( True, 'Service started' );
      btnStart.Enabled := False;
      SaveLocalSettings;
   end else begin
      ShowStatus( False, errMsg );
   end;
end;

procedure TfrmService.btnStopClick(Sender: TObject);
var  cn, sn, errMsg: String;
begin
   cn := Trim( emComputer.Text );
   sn := Trim( emService.Text );
   if length(sn) = 0 then sn := 'HamsterService';

   ShowStatus( 'Stopping service ...' );
   Application.ProcessMessages;

   if AppServiceStop( cn, sn, errMsg ) then begin
      ShowStatus( True, 'Service stopped' );
      btnStop.Enabled := False;
      SaveLocalSettings;
   end else begin
      ShowStatus( False, errMsg );
   end;
end;

function TfrmService.AppServiceIsInstalled( const computerName, serviceName: String;
                                            out ErrMsg: String;
                                            out Running: Boolean ): Boolean;
var  schSCManager, schService: SC_HANDLE;
     stat: TServiceStatus;
begin
   Result  := False;
   ErrMsg  := '';
   Running := False;
   
   if not IsWinSvcAvailable then begin
      ErrMsg := 'Services are not supported on this Windows platform!';
      exit;
   end;

   schSCManager := OpenSCManager( PChar(computerName), nil, SC_MANAGER_ALL_ACCESS );
   if schSCManager=0 then begin
      ErrMsg := 'Check.OpenSCManager failed: ' + WinErrMsg;
      exit;
   end;

   schService := OpenService( schSCManager,
                              PChar( serviceName ),
                              SERVICE_ALL_ACCESS ); // SERVICE_QUERY_CONFIG );
   if schService = 0 then begin
      ErrMsg := 'Check.OpenService failed: ' + WinErrMsg;
      CloseServiceHandle( schSCManager );
      exit;
   end;

   if not QueryServiceStatus( schService, stat ) then begin
      ErrMsg := 'Check.QueryServiceStatus failed: ' + WinErrMsg;
      CloseServiceHandle( schService );
      CloseServiceHandle( schSCManager );
      exit;
   end;

   Running := (stat.dwCurrentState = SERVICE_RUNNING);
   Result := True;

   CloseServiceHandle( schService );
   CloseServiceHandle( schSCManager );
end;

function TfrmService.AppServiceStart( const computerName, serviceName: String;
                                      out ErrMsg: String ): Boolean;
var  ssStatus: TServiceStatus;
     schSCManager, schService: SC_HANDLE;
     dwServiceArgs, dwOldCheckPoint: DWORD;
     lpServiceArgs: PChar;
begin
   Result := False;
   ErrMsg := '';

   schSCManager := OpenSCManager( PChar(computerName), nil, SC_MANAGER_ALL_ACCESS );
   if schSCManager=0 then begin
      ErrMsg := 'Start.OpenSCManager failed: ' + WinErrMsg;
      exit;
   end;

   schService := OpenService( schSCManager,
                              PChar( serviceName ),
                              SERVICE_ALL_ACCESS );
   if schService = 0 then begin
      ErrMsg := 'Start.OpenService failed: ' + WinErrMsg;
      CloseServiceHandle( schSCManager );
      exit;
   end;

   dwServiceArgs := 0;
   lpServiceArgs := nil;

   if StartService( schService, dwServiceArgs, lpServiceArgs ) then begin

      if QueryServiceStatus( schService, ssStatus) then begin

         while ssStatus.dwCurrentState = SERVICE_START_PENDING do begin
            dwOldCheckPoint := ssStatus.dwCheckPoint;
            Sleep( ssStatus.dwWaitHint );
            if not QueryServiceStatus( schService, ssStatus ) then break;
            if dwOldCheckPoint >= ssStatus.dwCheckPoint then break;
         end;

         if ssStatus.dwCurrentState = SERVICE_RUNNING then begin
            Result := True;
         end else begin
            ErrMsg := 'Start.WaitTilRunning failed: '
            + '  State='   + inttostr ( ssStatus.dwCurrentState )
            + '  SvcCode=' + inttostr ( ssStatus.dwServiceSpecificExitCode )
            + '  WinCode=' + WinErrMsg( ssStatus.dwWin32ExitCode );
         end;

      end else begin
         ErrMsg := 'Start.QueryServiceStatus failed ' + WinErrMsg;
      end;

   end else begin
      ErrMsg := 'Start.StartService failed: ' + WinErrMsg;
   end;

   CloseServiceHandle( schService );
   CloseServiceHandle( schSCManager );
end;

function TfrmService.AppServiceControl( const computerName, serviceName: String;
                                        const fdwControl: DWORD;
                                        out   ErrMsg: String ): Boolean;
var  ssStatus: TServiceStatus;
     fdwAccess: DWORD;
     schSCManager, schService: SC_HANDLE;
begin
   Result := False;
   ErrMsg := '';
   
   if not IsWinSvcAvailable then begin
      ErrMsg := 'Services are not supported on this platform!';
      exit;
   end;

   schSCManager := OpenSCManager( PChar(computerName), nil, SC_MANAGER_ALL_ACCESS );
   if schSCManager=0 then begin
      ErrMsg := 'Control.OpenSCManager failed: ' + WinErrMsg;
      exit;
   end;

   case fdwControl of
      SERVICE_CONTROL_STOP:        fdwAccess := SERVICE_STOP;
      SERVICE_CONTROL_PAUSE:       fdwAccess := SERVICE_PAUSE_CONTINUE;
      SERVICE_CONTROL_CONTINUE:    fdwAccess := SERVICE_PAUSE_CONTINUE;
      SERVICE_CONTROL_INTERROGATE: fdwAccess := SERVICE_INTERROGATE;
      else                         fdwAccess := SERVICE_INTERROGATE;
   end;

   schService := OpenService( schSCManager,
                              PChar( serviceName ),
                              fdwAccess );
   if schService = 0 then begin

      ErrMsg := 'Control.OpenService failed: ' + WinErrMsg;

   end else begin

      if not ControlService( schService, fdwControl, ssStatus ) then begin
         ErrMsg := 'Control.ControlService failed: ' + WinErrMsg;
      end else begin
         Result := True;
      end;

      CloseServiceHandle( schService );
      
   end;

   CloseServiceHandle( schSCManager );
end;

function TfrmService.AppServiceStop( const computerName, serviceName: String;
                                     out ErrMsg: String ): Boolean;
begin
   Result := AppServiceControl( computerName, serviceName,
                                SERVICE_CONTROL_STOP, ErrMsg );
end;

procedure TfrmService.ShowStatus( ok: Boolean; msg: String );
begin
   lblStatus.Caption := msg;
   if ok then begin
      lblStatus.Color := clLime;
   end else begin
      lblStatus.Color := clRed;
   end;
end;

procedure TfrmService.ShowStatus( msg: String );
begin
   lblStatus.Caption := msg;
   lblStatus.Color := clSilver;
end;

end.
