unit fScheduler;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, uConst, uType, cSchedulerCommon, ExtCtrls;

type
   TEditSchedulerEntry = class( TGlobalSchedulerEntry )
      public
         property ActionKey: String    read FActionKey write FActionKey;
         property FromTime : TDateTime read FFromTime  write FFromTime;
         property TilTime  : TDateTime read FTilTime   write FTilTime;
         property Weekdays : String    read FWeekdays  write FWeekdays;
         property Minutes  : Integer   read FMinutes   write FMinutes;
   end;

   TEditScheduler = class( TGlobalSchedulerCustom )
      public
         property List: TThreadList read FList;
   end;

type
  TfrmScheduler = class(TForm)
    grpDetail: TGroupBox;
    ckMon: TCheckBox;
    ckTue: TCheckBox;
    ckWed: TCheckBox;
    ckThu: TCheckBox;
    ckFri: TCheckBox;
    ckSat: TCheckBox;
    ckSun: TCheckBox;
    Label2: TLabel;
    Label3: TLabel;
    Label4: TLabel;
    lblMinutes: TLabel;
    Label6: TLabel;
    cbAction: TComboBox;
    Label7: TLabel;
    cbMinutes: TComboBox;
    cbFrom: TComboBox;
    cbTil: TComboBox;
    grpMaster: TGroupBox;
    lbEntries: TListBox;
    Label1: TLabel;
    btnEdit: TButton;
    btnAdd: TButton;
    btnDelete: TButton;
    btnSave: TButton;
    btnUndo: TButton;
    Timer1: TTimer;
    procedure FormCreate(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure btnEditClick(Sender: TObject);
    procedure btnAddClick(Sender: TObject);
    procedure btnDeleteClick(Sender: TObject);
    procedure btnSaveClick(Sender: TObject);
    procedure btnUndoClick(Sender: TObject);
    procedure lbEntriesClick(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure lbEntriesDrawItem(Control: TWinControl; Index: Integer;
      Rect: TRect; State: TOwnerDrawState);
    procedure Timer1Timer(Sender: TObject);
    procedure cbMinutesChange(Sender: TObject);
    procedure FormResize(Sender: TObject);
  private
    { Private-Deklarationen }
    Editing: Boolean;
    Entries: TEditScheduler;
    ColWidth: array[0..10] of Integer;
    procedure RefreshSchedulerEntries;
    procedure ShowSelected;
    procedure ReflectEditmode;
  public
    { Public-Deklarationen }
    class function ShowDialog( AOwner: TComponent ): Integer;
  end;

var
  frmScheduler: TfrmScheduler;

implementation

{$R *.DFM}

uses uTools, uGlobal, cLiveMsg, dInput, dDialogs, cUserTasks, uDateTime;


class function TfrmScheduler.ShowDialog( AOwner: TComponent ): Integer;
begin
   with TfrmScheduler.Create( AOwner ) do try
      Result := ShowModal;
   finally Free end;
end;


procedure TfrmScheduler.FormCreate(Sender: TObject);
var  Reply: TLiveMsg;
     SL: TStringList;
     i: Integer;
     s, s1: String;
begin
   GrowDpiForm( self );

   // Fill time combos
   for i := 0 to 23 do begin
      s := Format( '%.2d:', [ i ] );
      cbFrom.Items.Add( s + '00' );
      cbFrom.Items.Add( s + '15' );
      cbFrom.Items.Add( s + '30' );
      cbFrom.Items.Add( s + '45' );
   end;
   cbFrom.Items.Add( '23:59' );
   cbTil.Items.Text := cbFrom.Items.Text;

   // Fill minutes combos
   cbMinutes.Items.Add( '1' );
   cbMinutes.Items.Add( '2' );
   cbMinutes.Items.Add( '5' );
   cbMinutes.Items.Add( '10' );
   cbMinutes.Items.Add( '15' );
   cbMinutes.Items.Add( '20' );
   cbMinutes.Items.Add( '30' );
   cbMinutes.Items.Add( '45' );
   cbMinutes.Items.Add( '60' );
   cbMinutes.Items.Add( '90' );
   cbMinutes.Items.Add( '120' );
   cbMinutes.Items.Add( '180' );
   cbMinutes.Items.Add( '1440' );

   // Fill combo with possible actions
   SL := TStringList.Create;
   try

      // No Action
      cbAction.Items.AddObject( '', Pointer(0) );

      // User Tasks
      Reply := LiveConnector.RCLiveRequest(
         TLiveMsg.Create( LMREQ_USERTASKS_LIST, '' )
      );
      if Assigned( Reply ) then try
         if Reply.MsgType = LMREP_OK then begin
            SL.Text := Reply.MsgData;
            for i := 0 to SL.Count - 1 do begin
               cbAction.Items.AddObject( SL[i] + USERTASK_EXTENSION, Pointer(1) );
            end;
         end;
      finally FreeAndNil(Reply) end;

      // Scripts
      Reply := LiveConnector.RCLiveRequest(
                  TLiveMsg.Create( LMREQ_SCRIPT_DIR, '' )
               );
      if Assigned( Reply ) then try
         if ( Reply.MsgType = LMREP_OK ) then begin
            SL.Text := Reply.MsgData;
            for i := 0 to SL.Count - 1 do begin
               s  := SL[i]; // <time> TAB <size> TAB <path/name>
               s1 := UnDQuoteStr( NextSepPart( s, TAB ) );
               s1 := UnDQuoteStr( NextSepPart( s, TAB ) );
               s1 := UnDQuoteStr( NextSepPart( s, TAB ) );
               if (Pos('.hsc',s1) > 0) and (Pos('\',s1) = 0) then begin
                  cbAction.Items.AddObject( s1, Pointer(2) );
               end;
            end;
         end;
      finally FreeAndNil(Reply) end;

   finally
      SL.Free;
   end;

   // Load and show scheduler entries
   Editing := False;
   ReflectEditmode;
   Entries := TEditScheduler.Create;
   for i := 0 to 10 do ColWidth[i] := i;
   RefreshSchedulerEntries;

   lbEntries.Canvas.Font.Assign( lbEntries.Font );
   i := lbEntries.Canvas.TextHeight('Gg');
   lbEntries.ItemHeight := i;

   if lbEntries.Items.Count > 0 then lbEntries.ItemIndex := 0;
   ShowSelected;
   Timer1.Interval := 50;
   Timer1.Enabled  := True;
end;

procedure TfrmScheduler.FormResize(Sender: TObject);
begin
   grpMaster.Width  := ClientWidth - 2 * grpMaster.Left;
   grpMaster.Height := ClientHeight - 3 * grpMaster.Top - grpDetail.Height;
   grpDetail.Width  := ClientWidth - 2 * grpDetail.Left;
   grpDetail.Top    := ClientHeight - grpDetail.Height - grpMaster.Top;
end;

procedure TfrmScheduler.FormCloseQuery( Sender: TObject; var CanClose: Boolean );
var  i: Integer;
begin
   if Editing then begin
      i := HMessageDlg( 'Save changes?', mtConfirmation, [mbYes,mbNo,mbCancel] );
      if i = mrYes then btnSaveClick(nil);
      if i = mrNo  then btnUndoClick(nil);
      if (i = mrCancel) or Editing then CanClose := False;
   end;
end;

procedure TfrmScheduler.FormClose( Sender: TObject; var Action: TCloseAction );
begin
   Action := caFree;
end;

procedure TfrmScheduler.FormDestroy(Sender: TObject);
begin
   if Assigned( Entries ) then Entries.Free;
end;

procedure TfrmScheduler.ReflectEditmode;
begin
   lbEntries.Enabled := not Editing;
   btnEdit.Enabled   := not Editing;
   btnAdd.Enabled    := not Editing;
   btnDelete.Enabled := not Editing;

   btnSave.Enabled   := Editing;
   btnUndo.Enabled   := Editing;
   ckMon.Enabled     := Editing;
   ckTue.Enabled     := Editing;
   ckWed.Enabled     := Editing;
   ckThu.Enabled     := Editing;
   ckFri.Enabled     := Editing;
   ckSat.Enabled     := Editing;
   ckSun.Enabled     := Editing;
   cbFrom.Enabled    := Editing;
   cbTil.Enabled     := Editing;
   cbAction.Enabled  := Editing;
   cbMinutes.Enabled := Editing;
end;

procedure TfrmScheduler.RefreshSchedulerEntries;
var  Reply: TLiveMsg;
     i: Integer;
begin
   // clear entries
   Entries.Clear;
   lbEntries.Clear;

   // load scheduler definition file
   Reply := LiveConnector.RCLiveRequest(
               TLiveMsg.Create( LMREQ_FILE_GET, CFGFILE_SCHEDULER )
            );
   if Assigned( Reply ) then try
      if ( Reply.MsgType = LMREP_OK ) then Entries.SetDefText( Reply.MsgData );
   finally FreeAndNil(Reply) end;

   // show entries
   with Entries.List.LockList do try
      for i := 0 to Count - 1 do begin
         lbEntries.Items.AddObject( '?', Items[i] );
      end;
   finally Entries.List.UnlockList end;

   lbEntries.Invalidate;
end;

procedure TfrmScheduler.ShowSelected;
var  e: TEditSchedulerEntry;
     s: String;
begin
   e := nil;
   if (lbEntries.ItemIndex >= 0) and
      (lbEntries.ItemIndex < lbEntries.Items.Count) then begin
      e := TEditSchedulerEntry( lbEntries.Items.Objects[ lbEntries.ItemIndex ] );
   end;

   if e = nil then s := '1111111' else s := e.Weekdays;
   ckMon.Checked := (s[1] <> '0');
   ckTue.Checked := (s[2] <> '0');
   ckWed.Checked := (s[3] <> '0');
   ckThu.Checked := (s[4] <> '0');
   ckFri.Checked := (s[5] <> '0');
   ckSat.Checked := (s[6] <> '0');
   ckSun.Checked := (s[7] <> '0');

   if e = nil then cbFrom.Text := ''
              else cbFrom.Text := TimeToTimeString( e.FromTime, 5 );
   if e = nil then cbTil.Text  := ''
              else cbTil.Text  := TimeToTimeString( e.TilTime,  5 );
   if e = nil then cbAction.Text  := ''
              else cbAction.Text  := e.ActionKey;
   if e = nil then cbMinutes.Text := ''
              else cbMinutes.Text := inttostr( e.Minutes );

   cbMinutesChange( nil );
end;

procedure TfrmScheduler.btnEditClick(Sender: TObject);
begin
   if lbEntries.ItemIndex < 0 then exit;
   if lbEntries.ItemIndex >= lbEntries.Items.Count then exit;

   Editing := True;
   ReflectEditmode;
   ckMon.SetFocus;
end;

procedure TfrmScheduler.btnAddClick(Sender: TObject);
var  e: TEditSchedulerEntry;
     i: Integer;
begin
   // add new item
   e := TEditSchedulerEntry.Create( '' );
   e.TilTime := uDateTime.TimeStringToTime( '23:59' );
   e.Restart;
   Entries.Add( e );

   // show and select new item
   i := lbEntries.Items.AddObject( 'n', e );
   lbEntries.ItemIndex := i;
   ShowSelected;

   // perform edit with new item
   btnEdit.Click;
end;

procedure TfrmScheduler.btnDeleteClick(Sender: TObject);
var  i: Integer;
     e: TEditSchedulerEntry;
     s: String;
begin
   i := lbEntries.ItemIndex;
   if i < 0 then exit;
   if i >= lbEntries.Items.Count then exit;

   e := TEditSchedulerEntry( lbEntries.Items.Objects[ i ] );

   s := 'Delete selected Scheduled Task?';
   if HMessageDlg( PChar(s), mtConfirmation, [mbYes,mbNo] ) <> mrYes then exit;

   lbEntries.Items.Delete( i );
   Entries.Remove( e );
   e.Free;

   // save modified scheduler definition file
   if not LiveConnector.RCLiveRequestOK(
      TLiveMsg.Create( LMREQ_FILE_SET,
                       CFGFILE_SCHEDULER + #13#10 + Entries.GetDefText )
   ) then begin
      HMessageDlg( 'Scheduler file "' + CFGFILE_SCHEDULER + '" could not be saved!',
                   'Update Scheduled Tasks', mtError, [mbOK] );
   end;

   if i >= lbEntries.Items.Count then dec( i );
   if i >= 0 then lbEntries.ItemIndex := i;
   ShowSelected;
end;

procedure TfrmScheduler.btnSaveClick(Sender: TObject);
var  i: Integer;
     e: TEditSchedulerEntry;
     s: String;
begin
   i := lbEntries.ItemIndex;
   if i < 0 then exit;
   if i >= lbEntries.Items.Count then exit;

   e := TEditSchedulerEntry( lbEntries.Items.Objects[ i ] );

   s := '0000000';
   if ckMon.Checked then s[1] := '1';
   if ckTue.Checked then s[2] := '1';
   if ckWed.Checked then s[3] := '1';
   if ckThu.Checked then s[4] := '1';
   if ckFri.Checked then s[5] := '1';
   if ckSat.Checked then s[6] := '1';
   if ckSun.Checked then s[7] := '1';
   e.Weekdays := s;

   e.FromTime := TimeStringToTime( cbFrom.Text );
   e.TilTime  := TimeStringToTime( cbTil.Text );
   e.ActionKey := cbAction.Text;
   e.Minutes   := strtointdef( cbMinutes.Text, 60 );

   // redraw modified list
   e.Restart;
   lbEntries.Invalidate;

   // save modified scheduler definition file
   if not LiveConnector.RCLiveRequestOK(
      TLiveMsg.Create( LMREQ_FILE_SET,
                       CFGFILE_SCHEDULER + #13#10 + Entries.GetDefText )
   ) then begin
      HMessageDlg( 'Scheduler file "' + CFGFILE_SCHEDULER + '" could not be saved!',
                   'Update Scheduled Tasks', mtError, [mbOK] );
   end;

   Editing := False;
   ReflectEditmode;
   lbEntries.SetFocus;
end;

procedure TfrmScheduler.btnUndoClick(Sender: TObject);
begin
   ShowSelected;

   Editing := False;
   ReflectEditmode;
   lbEntries.SetFocus;
end;

procedure TfrmScheduler.lbEntriesClick(Sender: TObject);
begin
   ShowSelected;
end;

procedure TfrmScheduler.lbEntriesDrawItem(Control: TWinControl;
  Index: Integer; Rect: TRect; State: TOwnerDrawState);

   procedure draw( col: Integer; const h: String );
   var  w: Integer;
   begin
      with (Control as TListBox).Canvas do begin
         TextOut( Rect.Left, Rect.Top, h );
         w := TextWidth( h ) + 8;
         if w > ColWidth[col] then ColWidth[col] := w;
         Rect.Left := Rect.Left + ColWidth[col];
      end;
   end;

const wd = 'MonTueWedThuFriSatSun';
var  e: TGlobalSchedulerEntry;
     s, w: String;
     i: Integer;
     d: TDateTime;
begin
   with (Control as TListBox).Canvas do FillRect( Rect );

   e := TGlobalSchedulerEntry( lbEntries.Items.Objects[ Index ] );

   w := e.Weekdays;
   if      w = '1111111' then s := 'Mon-Sun'
   else if w = '1111110' then s := 'Mon-Sat'
   else if w = '1111100' then s := 'Mon-Fri'
   else if w = '0000011' then s := 'Sat-Sun'
   else if w = '0000000' then s := '(never)'
   else begin
      s := '';
      for i := 1 to 7 do begin
         if w[i] <> '0' then begin
            if length(s) > 0 then s := s + ', ';
            s := s + copy( wd, i*3-2, 3 );
         end;
      end;
   end;
   draw( 0, s );

   s := TimeToTimeString( e.FromTime, 5 ) + ' - '
      + TimeToTimeString( e.TilTime,  5 );
   draw( 1, s );

   if e.ActionKey = ''  then s := '(do nothing)'
   else if e.IsUserTask then s := e.ActionKey + ' (user task)'
   else                      s := e.ActionKey + ' (script)';
   draw( 2, s );

   draw( 3, inttostr(e.Minutes) + ' min.' );

   s := 'next at ' + TimeToTimeString( e.NextDT, 5 ) + ' (';
   d := DateOfDateTime(e.NextDT);
   if      d = Date   then s := s + 'today'
   else if d = Date+1 then s := s + 'tomorrow'
   else                    s := s + DateToStr( e.NextDT );
   draw( 4, s + ')' );
end;

procedure TfrmScheduler.Timer1Timer(Sender: TObject);
var  i: Integer;
begin
   if not Editing then begin
      if Timer1.Interval < 10000 then Timer1.Interval := 10000;
      with Entries.List.LockList do try
         for i := 0 to Count - 1 do begin
            TGlobalSchedulerEntry( Items[i] ).Restart;
         end;
      finally Entries.List.UnlockList end;
      lbEntries.Invalidate;
   end;
end;

procedure TfrmScheduler.cbMinutesChange(Sender: TObject);
var  s: String;
     i: Integer;
begin
   i := strtointdef( cbMinutes.Text, 0 );
   s := 'minutes';
   if i >= 60 then s := s + ' (' + MinutesToDHM(i) + ')';
   lblMinutes.Caption := s;
end;

end.
