// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit fConfSrv;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  dDialogs, StdCtrls, ComCtrls, ExtCtrls, cSettings, ImgList, Menus;

type
  TfrmConfSrv = class(TForm)
    btnOK: TButton;
    btnApply: TButton;
    btnRefresh: TButton;
    btnCancel: TButton;
    lbItems: TListBox;
    Label1: TLabel;
    PageControl1: TPageControl;
    pgCommon: TTabSheet;
    pgNNTP: TTabSheet;
    pgPOP3: TTabSheet;
    pgSMTP: TTabSheet;
    grpCommon: TGroupBox;
    Label12: TLabel;
    Label15: TLabel;
    emConnectTO: TEdit;
    emCommandTO: TEdit;
    ckSocksUse: TCheckBox;
    grpNNTP: TGroupBox;
    Label5: TLabel;
    labPWBase: TLabel;
    ckNntpReadonly: TCheckBox;
    ckNntpReloadGroups: TCheckBox;
    ckNntpReloadDescs: TCheckBox;
    emNntpPullThreads: TEdit;
    ckNntpY2KFix: TCheckBox;
    cbNntpDescsMode: TComboBox;
    grpPOP3: TGroupBox;
    grpSMTP: TGroupBox;
    Label6: TLabel;
    Label45: TLabel;
    emSmtpSASL: TEdit;
    ckSmtpUseAuth: TCheckBox;
    emPop3Filter: TEdit;
    ckPop3Leave: TCheckBox;
    Label7: TLabel;
    Label8: TLabel;
    ckNntpNeverAsk: TCheckBox;
    Bevel1: TBevel;
    imglstItems: TImageList;
    popItems: TPopupMenu;
    mnuNntpAdd: TMenuItem;
    N1: TMenuItem;
    mnuSrvDel: TMenuItem;
    mnuPop3Add: TMenuItem;
    mnuSmtpAdd: TMenuItem;
    Label2: TLabel;
    lblPasswordInfo: TLabel;
    btnPWChange: TButton;
    btnPWClear: TButton;
    cbPop3Accounts: TComboBox;
    ckServerDisabled: TCheckBox;
    ckPop3UsePass: TCheckBox;
    Label3: TLabel;
    emPop3SASL: TEdit;
    ckPop3UseApop: TCheckBox;
    ckPop3UseAuth: TCheckBox;
    ckPop3AutoDistribute: TCheckBox;
    cbSslMode: TComboBox;
    Label4: TLabel;
    Label9: TLabel;
    cbSslVersion: TComboBox;
    Label10: TLabel;
    emSslCipher: TEdit;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure btnRefreshClick(Sender: TObject);
    procedure btnOKClick(Sender: TObject);
    procedure btnApplyClick(Sender: TObject);
    procedure lbItemsDrawItem(Control: TWinControl; Index: Integer;
      Rect: TRect; State: TOwnerDrawState);
    procedure lbItemsClick(Sender: TObject);
    procedure popItemsPopup(Sender: TObject);
    procedure mnuNntpAddClick(Sender: TObject);
    procedure mnuPop3AddClick(Sender: TObject);
    procedure mnuSmtpAddClick(Sender: TObject);
    procedure mnuSrvDelClick(Sender: TObject);
    procedure btnPWChangeClick(Sender: TObject);
    procedure btnPWClearClick(Sender: TObject);
    procedure FormResize(Sender: TObject);
  private
    { Private-Deklarationen }
    SelItemType: Integer;
    SelItemName: String;
    SelSettings: TSettingsPlain;
    Args       : TStringList;
    procedure ShowPasswordInfo;
    procedure ChangePasswordInfo( Identifier, User, Pass: String );
    procedure SettingsToForm;
    procedure FormToSettings;
    function  LoadItemList: Boolean;
    function  LoadSettings: Boolean;
    function  SaveSettings: Boolean;
    procedure SrvAdd( const Title, Example, DefPort: String;
                      const LQReqSrvAdd: Integer );
    procedure SrvDel( const Title, ServerName: String;
                      const ServerIndex, LQReqSrvDel: Integer );
  public
    { Public-Deklarationen }
  end;

implementation

{$R *.DFM}

uses uConst, uConstVar, uGlobal, cSettingsHdlLive, uTools, cLiveMsg, cLiveQueue,
     uDateTime, dInput;

const
   selTypeMask = $0F;
   selTypeNone = $00;
   selTypeNNTP = $01;
   selTypePOP3 = $02;
   selTypeSMTP = $03;

   selFlagMask = $F0;
   // selFlagDis  = $10;

procedure TfrmConfSrv.FormCreate(Sender: TObject);
begin
   GrowDpiForm( self );

   SelItemType := selTypeNone;
   SelItemName := '';
   SelSettings := nil;
   Args := TStringList.Create;

   PageControl1.ActivePage := pgCommon;
   LoadItemList;
   btnRefresh.Click;
end;

procedure TfrmConfSrv.FormResize(Sender: TObject);
begin
   PageControl1.Width  := ClientWidth  - 4 * lbItems.Left - lbItems.Width - Bevel1.Width;
   PageControl1.Height := ClientHeight - 3 * PageControl1.Top - btnOk.Height;
   lbItems.Height      := ClientHeight - lbItems.Top - PageControl1.Top;
   btnOk.Top      := ClientHeight - btnOk.Height - PageControl1.Top;
   btnApply.Top   := ClientHeight - btnOk.Height - PageControl1.Top;
   btnRefresh.Top := ClientHeight - btnOk.Height - PageControl1.Top;
   btnCancel.Top  := ClientHeight - btnOk.Height - PageControl1.Top;
end;

procedure TfrmConfSrv.FormDestroy(Sender: TObject);
begin
   if Assigned( SelSettings ) then SelSettings.Free;
   if Assigned( Args ) then Args.Free;
end;

procedure TfrmConfSrv.ShowPasswordInfo;
const PW_SET  = 'SET';
      PW_NONE = 'NONE SET';
var  PW_STATE: String;
     Reply   : TLiveMsg;
     Server  : String;
     i       : Integer;
begin
   PW_STATE := '(unknown)';

   if ( SelItemType <> selTypeNone ) and ( SelItemName <> '' ) then begin

      // Note: To remain compatible with previous versions, port number has
      //       to be removed from server's full name ('server/alias,port').
      Server := SelItemName;
      i := Pos( ',', Server );
      if i > 0 then SetLength( Server, i-1 );

      try
         Reply := LiveConnector.RCLiveRequest(
            TLiveMsg.Create( LMREQ_PASSWORD_GET, Server )
         );
         if Assigned( Reply ) then try
            PW_STATE := PW_NONE;
            if Reply.MsgType = LMREP_OK then begin
               ArgsWhSpaceDQuoted( Reply.MsgData, Args, 3 );
               if Args[2] = '1' then PW_STATE := PW_SET;
            end;
         finally Reply.Free end;

      except end;

   end;

   lblPasswordInfo.Caption := PW_STATE;
   btnPWChange.Enabled := ( PW_STATE = PW_SET ) or ( PW_STATE = PW_NONE );
   btnPWClear .Enabled := ( PW_STATE = PW_SET );
end;

procedure TfrmConfSrv.ChangePasswordInfo( Identifier, User, Pass: String );
var  i: Integer;
begin
   // Note: To remain compatible with previous versions, port number has
   //       to be removed from server's full name ('server/alias,port').
   i := Pos( ',', Identifier );
   if i > 0 then SetLength( Identifier, i-1 );

   try
      if not LiveConnector.RCLiveRequestOK(
         TLiveMsg.Create( LMREQ_PASSWORD_SET,
                          Identifier + CRLF + User + CRLF + Pass )
      ) then begin

         HMessageDlg( 'Changing password settings failed!', SelItemName,
                      mtWarning, [mbOK] );

      end;
   except end;
end;

procedure TfrmConfSrv.SettingsToForm;

   function DT( const s: String ): String;
   var  x: TDateTime;
   begin
      x := uDateTime.TimeStampToDateTime( s );
      if x = 0 then Result := ''
               else Result := DateTimeToStr( x );
   end;

var  s: String;
begin
   ShowPasswordInfo;

   if not Assigned( SelSettings ) then exit;

   with SelSettings do begin

      // Common
      emConnectTO.Text           := GetStr( ssTimeoutConnect );
      emCommandTO.Text           := GetStr( ssTimeoutCommand );
      ckSocksUse.Checked         := GetBoo( ssUseSocks );
      ckServerDisabled.Checked   := GetBoo( ssDisabled );
      cbSslMode.ItemIndex        := GetInt( ssSSLMode );
      cbSslVersion.ItemIndex     := GetInt( ssSSLVersion );
      emSslCipher.Text           := GetStr( ssSSLCipher );

      // NNTP
      ckNntpReadonly.Checked     := GetBoo( ssNntpReadOnly );

      ckNntpReloadGroups.Checked := GetBoo( ssNntpReloadGroups );
      ckNntpReloadGroups.Hint    := 'Last: ' + DT( GetStr( ssNntpReloadGroupsDT ) );
      ckNntpReloadDescs.Checked  := GetBoo( ssNntpReloadDescs );
      ckNntpReloadDescs.Hint     := 'Last: ' + DT( GetStr( ssNntpReloadDescsDT ) );

      ckNntpY2KFix.Checked       := GetBoo( ssNntpNewGroupsY2K );
      emNntpPullThreads.Text     := GetStr( ssNntpPullThreads );
      cbNntpDescsMode.ItemIndex  := GetInt( ssNntpLoadDescsMode );

      s := GetStr( ssNntpLastNewGroups );
      ckNntpNeverAsk.Checked := ( UpperCase(s) = 'NEVER' );
      ckNntpNeverAsk.Hint    := 'Last: ' + DT( GetStr( ssNntpLastNewGroups ) );

      // POP3
      cbPop3Accounts.Text          := GetStr( ssPop3LocalUser );
      emPop3Filter.Text            := GetStr( ssPop3FilterSection );
      ckPop3AutoDistribute.Checked := GetBoo( ssPop3AutoDistribute );
      ckPop3Leave.Checked          := GetBoo( ssPop3LeaveOnServer );
      ckPop3UsePass.Checked        := GetBoo( ssPop3UseAuthPass );
      ckPop3UseApop.Checked        := GetBoo( ssPop3UseAuthApop );
      ckPop3UseAuth.Checked        := GetBoo( ssPop3UseAuthAuth );
      emPop3SASL.Text              := GetStr( ssPop3UseSasl );

      // SMTP
      ckSmtpUseAuth.Checked := GetBoo( ssSmtpUseAuth );
      emSmtpSASL.Text       := GetStr( ssSmtpUseSasl );

   end;
end;

procedure TfrmConfSrv.FormToSettings;

   procedure TakeEM( ID: Integer; EM: TEdit ); overload;
   begin
      if not EM.Enabled then exit;
      if SelSettings.GetStr( ID ) = EM.Text then exit;
      SelSettings.SetStr( ID, EM.Text );
   end;

   procedure TakeEM( ID: Integer; EM: TComboBox ); overload;
   begin
      if not EM.Enabled then exit;
      if SelSettings.GetStr( ID ) = EM.Text then exit;
      SelSettings.SetStr( ID, EM.Text );
   end;

   procedure TakeCK( ID: Integer; CK: TCheckBox );
   begin
      if not CK.Enabled then exit;
      if SelSettings.GetBoo( ID ) = CK.Checked then exit;
      SelSettings.SetBoo( ID, CK.Checked );
   end;

   procedure TakeCB( ID: Integer; CB: TComboBox );
   begin
      if not CB.Enabled then exit;
      if SelSettings.GetInt( ID ) = CB.ItemIndex then exit;
      SelSettings.SetInt( ID, CB.ItemIndex );
   end;

var  s: String;
begin
   if SelItemType = selTypeNone   then exit;
   if SelItemName = ''            then exit;
   if not Assigned( SelSettings ) then exit;

   // Common
   TakeEM( ssTimeoutConnect, emConnectTO );
   TakeEM( ssTimeoutCommand, emCommandTO );
   TakeCK( ssUseSocks, ckSocksUse );
   TakeCK( ssDisabled, ckServerDisabled );
   TakeCB( ssSSLMode, cbSslMode );
   TakeCB( ssSSLVersion, cbSslVersion );
   TakeEM( ssSSLCipher, emSslCipher );

   // NNTP
   TakeCK( ssNntpReadOnly, ckNntpReadonly );
   TakeCK( ssNntpReloadGroups, ckNntpReloadGroups );
   TakeCK( ssNntpReloadDescs, ckNntpReloadDescs );
   TakeCK( ssNntpNewGroupsY2K, ckNntpY2KFix );
   TakeEM( ssNntpPullThreads, emNntpPullThreads );
   TakeCB( ssNntpLoadDescsMode, cbNntpDescsMode );

   s := UpperCase( SelSettings.GetStr( ssNntpLastNewGroups ) );
   if ckNntpNeverAsk.Checked then begin
      if s <> 'NEVER' then SelSettings.SetStr( ssNntpLastNewGroups, 'NEVER' );
   end else begin
      if s =  'NEVER' then SelSettings.SetStr( ssNntpLastNewGroups, '' );
   end;

   // POP3
   TakeEM( ssPop3LocalUser,      cbPop3Accounts );
   TakeEM( ssPop3FilterSection,  emPop3Filter );
   TakeCK( ssPop3AutoDistribute, ckPop3AutoDistribute );
   TakeCK( ssPop3LeaveOnServer,  ckPop3Leave );
   TakeCK( ssPop3UseAuthPass,    ckPop3UsePass );
   TakeCK( ssPop3UseAuthApop,    ckPop3UseApop );
   TakeCK( ssPop3UseAuthAuth,    ckPop3UseAuth );
   TakeEM( ssPop3UseSasl,        emPop3SASL );

   // SMTP
   TakeCK( ssSmtpUseAuth, ckSmtpUseAuth );
   TakeEM( ssSmtpUseSasl, emSmtpSASL );

end;

function TfrmConfSrv.LoadItemList: Boolean;

   procedure AddItems( LqReqId, selType: Integer );
   var  Reply: TLiveMsg;
        SL: TStringList;
        i: Integer;
   begin
      Reply := LiveConnector.RCLiveRequest(
                  TLiveMsg.Create( LqReqId, '' )
               );
      if Assigned( Reply ) then try
         if ( Reply.MsgType = LMREP_OK ) then begin
            SL := TStringList.Create;
            try
               SL.Text := Reply.MsgData;
               for i := 0 to SL.Count - 1 do begin
                  lbItems.Items.AddObject( SL[i], Pointer( selType ) );
               end;
            finally SL.Free end;
         end;
      finally Reply.Free end;
   end;

begin
   Result := False;

   try
      SelItemType := selTypeNone;
      SelItemName := '';
      SelSettings := nil;

      lbItems.Items.BeginUpdate;
      try
         lbItems.Items.Clear;
         lbItems.ItemIndex := -1;
         AddItems( LMREQ_SRVNNTP_LIST, selTypeNNTP );
         AddItems( LMREQ_SRVPOP3_LIST, selTypePOP3 );
         AddItems( LMREQ_SRVSMTP_LIST, selTypeSMTP );
      finally lbItems.Items.EndUpdate end;

      Result := True;

   except
      on E: Exception do begin
         HMessageDlg( 'Error:' + CRLF + E.Message, 'Load Item List',
                      mtError, [mbOK] );
      end;
   end;
end;

function TfrmConfSrv.LoadSettings: Boolean;

   procedure LoadAccounts;
   var  Reply: TLiveMsg;
        SL: TStringList;
        i: Integer;
   begin
      cbPop3Accounts.Clear;
      Reply := LiveConnector.RCLiveRequest(
                  TLiveMsg.Create( LMREQ_ACCOUNTS_LIST, '' )
               );
      if Assigned( Reply ) then try
         if ( Reply.MsgType = LMREP_OK ) then begin
            SL := TStringList.Create;
            try
               SL.Text := Reply.MsgData;
               for i:=0 to SL.Count-1 do begin
                  ArgsWhSpaceDQuoted( SL[i], Args, 3 );
                  cbPop3Accounts.Items.Add( Args[0] );
               end;
            finally SL.Free end;
         end;
      finally Reply.Free end;
   end;

begin
   Result := False;

   try
      if lbItems.Items.Count = 0 then LoadItemList;

      if Assigned( SelSettings ) then FreeAndNil( SelSettings );
      if SelItemName = '' then SelItemType := selTypeNone;

      case SelItemType of
         selTypeNone:
            SelSettings := TSettingsPlain.Create(
               SettingsDef_Server,
               TSettingsHandler_InMemory.Create( '' ),
               False {don't AutoFlush}
            );
         selTypeNNTP:
            SelSettings := TSettingsPlain.Create(
               SettingsDef_Server,
               TSettingsHandler_LiveRemote.Create(
                  LiveConnector, SelItemName,
                  LMREQ_SRVNNTP_SETTING_GET, LMREQ_SRVNNTP_SETTING_SET
               ),
               False {don't AutoFlush}
            );
         selTypePOP3:
            SelSettings := TSettingsPlain.Create(
               SettingsDef_Server,
               TSettingsHandler_LiveRemote.Create(
                  LiveConnector, SelItemName,
                  LMREQ_SRVPOP3_SETTING_GET, LMREQ_SRVPOP3_SETTING_SET
               ),
               False {don't AutoFlush}
            );
         selTypeSMTP:
            SelSettings := TSettingsPlain.Create(
               SettingsDef_Server,
               TSettingsHandler_LiveRemote.Create(
                  LiveConnector, SelItemName,
                  LMREQ_SRVSMTP_SETTING_GET,LMREQ_SRVSMTP_SETTING_SET
               ),
               False {don't AutoFlush}
            );
      end;

      LoadAccounts;

      SettingsToForm;

      Result := True;

   except
      on E: Exception do begin
         HMessageDlg( 'Error:' + CRLF + E.Message, 'Load Settings',
                      mtError, [mbOK] );
      end;
   end;
end;

function TfrmConfSrv.SaveSettings: Boolean;
begin
   Result := True;

   try
      if SelItemType = selTypeNone   then exit;
      if SelItemName = ''            then exit;
      if not Assigned( SelSettings ) then exit;

      FormToSettings;
      SelSettings.Flush;

   except
      on E: Exception do begin
         HMessageDlg( 'Error:' + CRLF + E.Message, 'Save Settings',
                      mtError, [mbOK] );
         Result := False;
      end;
   end;
end;

procedure TfrmConfSrv.btnRefreshClick(Sender: TObject);
begin
   btnOK.Enabled := LoadSettings;
end;

procedure TfrmConfSrv.btnOKClick(Sender: TObject);
begin
   if not SaveSettings then ModalResult := mrNone;
end;

procedure TfrmConfSrv.btnApplyClick(Sender: TObject);
begin
   SaveSettings;
end;

procedure TfrmConfSrv.lbItemsDrawItem(Control: TWinControl;
  Index: Integer; Rect: TRect; State: TOwnerDrawState);
var  SelType: Integer;
begin
   SelType := Integer( (Control as TListBox).Items.Objects[Index] ) and selTypeMask;
   // SelFlag := Integer( (Control as TListBox).Items.Objects[Index] ) and selFlagMask;

   with (Control as TListBox).Canvas do begin
      FillRect( Rect );

      imglstItems.Draw( (Control as TListBox).Canvas,
                        Rect.Left+2, Rect.Top, SelType, True );

      if odSelected in State then Font.Style := [ fsBold ]
                             else Font.Style := [ ];

      // if (SelFlag and selFlagDis) <> 0 then Font.Color := clLtGray;

      TextOut( Rect.Left+20, Rect.Top+1, (Control as TListBox).Items[Index] );
   end;
end;

procedure TfrmConfSrv.lbItemsClick(Sender: TObject);

   procedure Grp( PG: TTabSheet; GB: TGroupBox; Enabled: Boolean );
   begin
      GB.Enabled     := Enabled;
      PG.Enabled     := Enabled;
      // PG.Highlighted := Enabled;
      if Enabled then PG.Font.Color := clWindowText
                 else PG.Font.Color := clGrayText;
   end;

begin
   with lbItems do begin
      if ( ItemIndex < 0 ) or ( ItemIndex >= Items.Count ) then begin
         SelItemType := selTypeNone;
         SelItemName := '';
      end else begin
         SelItemType := Integer( Items.Objects[ ItemIndex ] ) and SelTypeMask;
         SelItemName := Items[ ItemIndex ];
      end;
   end;

   LoadSettings;

   Grp( pgCommon, grpCommon, SelItemType <> selTypeNone );
   Grp( pgNNTP,   grpNNTP,   SelItemType =  selTypeNNTP );
   Grp( pgPOP3,   grpPOP3,   SelItemType =  selTypePOP3 );
   Grp( pgSMTP,   grpSMTP,   SelItemType =  selTypeSMTP );
end;

procedure TfrmConfSrv.popItemsPopup(Sender: TObject);
begin
   with lbItems do begin
      mnuSrvDel.Enabled := ( ItemIndex >= 0 ) and ( ItemIndex < Items.Count );
   end;
end;

procedure TfrmConfSrv.SrvAdd( const Title, Example, DefPort: String;
                              const LQReqSrvAdd: Integer );
var  Nam: String;
     Idx: Integer;
begin
   Nam := Example;
   if not InputDlgStr( Title,
                       'Format: Servername[/Alias][,Port]', Nam, 0 ) then exit;

   while Pos( ' ', Nam ) > 0 do System.Delete( Nam, Pos( ' ', Nam ), 1 );
   if Nam = '' then exit;
   if Pos( ',', Nam ) = 0 then Nam := Nam + ',' + DefPort;

   Idx := lbItems.Items.IndexOf( Nam );
   if Idx >= 0 then begin
      HMessageDlg( 'Server "' + Nam + '" already exists!',
                   Title, mtWarning, [mbOK] );
      exit;
   end;

   if not LiveConnector.RCLiveRequestOK(
             TLiveMsg.Create( LQReqSrvAdd, Nam )
       ) then begin
      HMessageDlg( 'Adding server "' + Nam + '" failed!',
                   Title, mtWarning, [mbOK] );
   end;

   LoadItemList;
   btnRefresh.Click;

   Idx := lbItems.Items.IndexOf( Nam );
   if Idx >= 0 then begin lbItems.ItemIndex:=Idx; lbItemsClick(nil) end;
end;

procedure TfrmConfSrv.SrvDel( const Title, ServerName: String;
                              const ServerIndex, LQReqSrvDel: Integer );
var  Idx: Integer;
     s: String;
begin
   Idx := ServerIndex;

   s := Format( 'Delete server "%s" now?', [ServerName] );
   if HMessageDlg( PChar(s), mtConfirmation, [mbYes,mbNo] ) <> mrYes then exit;

   if not LiveConnector.RCLiveRequestOK(
      TLiveMsg.Create( LQReqSrvDel, ServerName ) ) then begin

      HMessageDlg( 'Deleting server "' + ServerName + '" failed!',
                   Title, mtWarning, [mbOK] );
   end;

   LoadItemList;
   btnRefresh.Click;   

   if Idx >= lbItems.Items.Count then dec( Idx );
   if Idx >= 0 then begin lbItems.ItemIndex:=Idx; lbItemsClick(nil) end;
end;

procedure TfrmConfSrv.mnuNntpAddClick(Sender: TObject);
begin
   SrvAdd( 'Add newsserver (NNTP)', 'news.XXX.com', '119', LMREQ_SRVNNTP_ADD );
end;

procedure TfrmConfSrv.mnuPop3AddClick(Sender: TObject);
begin
   SrvAdd( 'Add mailserver (POP3)', 'pop3.XXX.com', '110', LMREQ_SRVPOP3_ADD );
end;

procedure TfrmConfSrv.mnuSmtpAddClick(Sender: TObject);
begin
   SrvAdd( 'Add mailserver (SMTP)', 'smtp.XXX.com',  '25', LMREQ_SRVSMTP_ADD );
end;

procedure TfrmConfSrv.mnuSrvDelClick(Sender: TObject);
var  Idx, Typ: Integer;
     Nam: String;
begin
   Idx := lbItems.ItemIndex;
   if ( Idx < 0 ) or ( Idx >= lbItems.Items.Count ) then exit;

   Nam := lbItems.Items[Idx];
   Typ := Integer( lbItems.Items.Objects[ Idx ] ) and SelTypeMask;

   case Typ of
      selTypeNntp:
         SrvDel( 'Delete newsserver (NNTP)', Nam, Idx, LMREQ_SRVNNTP_DEL );
      selTypePop3:
         SrvDel( 'Delete mailserver (POP3)', Nam, Idx, LMREQ_SRVPOP3_DEL );
      selTypeSmtp:
         SrvDel( 'Delete mailserver (SMTP)', Nam, Idx, LMREQ_SRVSMTP_DEL );
   end;
end;

procedure TfrmConfSrv.btnPWChangeClick(Sender: TObject);
var  User, Pass, Pass2: String;
begin
   User  := '';
   Pass  := '';
   Pass2 := '';

   if not InputDlgStr( SelItemName, 'Username:', User, 0 ) then exit;
   if not InputDlgPwd( SelItemName, 'Password:', Pass, 0 ) then exit;
   if not InputDlgPwd( SelItemName, 'Repeat Password:', Pass2, 0 ) then exit;

   if Pass <> Pass2 then begin
      HMessageDlg( 'Given passwords were not equal!',
                   SelItemName, mtWarning, [mbOK] );
      exit;
   end;

   ChangePasswordInfo( SelItemName, User, Pass );
end;

procedure TfrmConfSrv.btnPWClearClick(Sender: TObject);
var  s: String;
begin
   s := Format( 'Clear password settings for server "%s" now?', [SelItemName] );
   if HMessageDlg( PChar(s), mtConfirmation, [mbYes,mbNo] ) <> mrYes then exit;
   ChangePasswordInfo( SelItemName, '', '' );
end;

end.
