// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit fConfMailTrap;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  dDialogs, StdCtrls, Menus, ComCtrls, ExtCtrls, cTraps, ImgList, cSettings;

type
  TfrmConfMailTrap = class(TForm)
    lbItems: TListBox;
    btnOK: TButton;
    btnApply: TButton;
    btnRefresh: TButton;
    btnCancel: TButton;
    imglstItems: TImageList;
    pnlEdit: TPanel;
    pnlFuncs: TPanel;
    btnAdd: TButton;
    btnClone: TButton;
    btnDelete: TButton;
    btnMoveUp: TButton;
    btnMoveDown: TButton;
    btnTest: TButton;
    Label1: TLabel;
    Label7: TLabel;
    Label8: TLabel;
    emScoreAccept: TEdit;
    emScoreDelete: TEdit;
    Label6: TLabel;
    rbActionAccept: TRadioButton;
    rbActionDelete: TRadioButton;
    rbActionScore: TRadioButton;
    emScoreValue: TEdit;
    UpDown1: TUpDown;
    chkDisabled: TCheckBox;
    Label5: TLabel;
    cbHeaderLine: TComboBox;
    Label3: TLabel;
    cbMatchMode: TComboBox;
    chkCaseSensitive: TCheckBox;
    Label4: TLabel;
    emPattern: TEdit;
    Label2: TLabel;
    cbComment: TComboBox;
    popupItems: TPopupMenu;
    mnuListToClipboard: TMenuItem;
    chkMatchNot: TCheckBox;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure btnRefreshClick(Sender: TObject);
    procedure btnOKClick(Sender: TObject);
    procedure btnApplyClick(Sender: TObject);
    procedure lbItemsClick(Sender: TObject);
    procedure lbItemsDrawItem(Control: TWinControl; Index: Integer;
      Rect: TRect; State: TOwnerDrawState);
    procedure cbCommentChange(Sender: TObject);
    procedure cbHeaderLineChange(Sender: TObject);
    procedure cbMatchModeChange(Sender: TObject);
    procedure chkCaseSensitiveClick(Sender: TObject);
    procedure emPatternChange(Sender: TObject);
    procedure emScoreValueChange(Sender: TObject);
    procedure rbActionAcceptClick(Sender: TObject);
    procedure rbActionDeleteClick(Sender: TObject);
    procedure rbActionScoreClick(Sender: TObject);
    procedure chkDisabledClick(Sender: TObject);
    procedure btnAddClick(Sender: TObject);
    procedure btnCloneClick(Sender: TObject);
    procedure btnMoveUpClick(Sender: TObject);
    procedure btnMoveDownClick(Sender: TObject);
    procedure btnTestClick(Sender: TObject);
    procedure btnDeleteClick(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure mnuListToClipboardClick(Sender: TObject);
    procedure chkMatchNotClick(Sender: TObject);
    procedure FormActivate(Sender: TObject);
    procedure FormShow(Sender: TObject);
  private
    { Private-Deklarationen }
    Settings: TSettingsPlain;
    Startup : Boolean;
    procedure SettingsToForm;
    function  LoadSettings: Boolean;
    function  SaveSettings: Boolean;
    procedure DoAddItem( const DefLine: String );
    function  Current: TMailTrapItem;
  public
    { Public-Deklarationen }
  end;

implementation

{$R *.DFM}

uses uConst, uConstVar, uGlobal, cSettingsHdlLive, uTools, cLiveMsg, cLiveQueue,
     uDateTime, dInput, fConfMailTrapTest, ClipBrd;

procedure TfrmConfMailTrap.FormCreate(Sender: TObject);
begin
   GrowDpiForm( self );

   Startup  := True;
   Settings := nil;

   LoadWindowState( Self, self.ClassName );
   
   cbComment.Items.Add( ' ' );

   cbHeaderLine.Items.Add( MTSpecialHeader_Contents        );
   cbHeaderLine.Items.Add( MTSpecialHeader_RawContents     );
   cbHeaderLine.Items.Add( MTSpecialHeader_AnyHeader       );
   cbHeaderLine.Items.Add( MTSpecialHeader_AnySender       );
   cbHeaderLine.Items.Add( MTSpecialHeader_AnyRecipient    );
   cbHeaderLine.Items.Add( MTSpecialHeader_MessageHeaders  );
   cbHeaderLine.Items.Add( MTSpecialHeader_MessageBody     );
   cbHeaderLine.Items.Add( MTSpecialHeader_WholeMessage    );
   cbHeaderLine.Items.Add( MTSpecialHeader_BodySubject     );
   cbHeaderLine.Items.Add( 'Subject' );
   cbHeaderLine.Items.Add( 'From' );
   cbHeaderLine.Items.Add( 'Received' );
   cbHeaderLine.Items.Add( 'Return-Path' );

   cbMatchMode.Items.Add( MailTrapMatchModes[ mtmContainsWord ] );
   cbMatchMode.Items.Add( MailTrapMatchModes[ mtmContains     ] );
   cbMatchMode.Items.Add( MailTrapMatchModes[ mtmEquals       ] );
   cbMatchMode.Items.Add( MailTrapMatchModes[ mtmMatchesRegex ] );
end;

procedure TfrmConfMailTrap.FormActivate(Sender: TObject);
begin
   if Startup then begin
      StartUp := False;
      btnRefreshClick( self );
   end;
end;

procedure TfrmConfMailTrap.FormResize(Sender: TObject);
begin
   lbItems.Height := ClientHeight - 4 * lbItems.Top - pnlEdit.Height - btnOk.Height;
   lbItems.Width  := ClientWidth - 3 * lbItems.Left - pnlFuncs.Width;

   pnlFuncs.Height := ClientHeight - 4 * lbItems.Top - pnlEdit.Height - btnOk.Height;
   pnlFuncs.Left   := ClientWidth - lbItems.Left - pnlFuncs.Width;

   pnlEdit.Width := ClientWidth - 2 * pnlEdit.Left;
   pnlEdit.Top   := ClientHeight - pnlEdit.Height - btnOk.Height - 2 * lbItems.Top;

   btnOk.Top      := ClientHeight - btnOk.Height - lbItems.Top;
   btnApply.Top   := ClientHeight - btnOk.Height - lbItems.Top;
   btnRefresh.Top := ClientHeight - btnOk.Height - lbItems.Top;
   btnCancel.Top  := ClientHeight - btnOk.Height - lbItems.Top;

   lbItems.Invalidate;
end;

procedure TfrmConfMailTrap.FormShow(Sender: TObject);
begin
   if Startup then begin
      StartUp := False;
      btnRefreshClick( self );
   end;
end;

procedure TfrmConfMailTrap.FormDestroy(Sender: TObject);
var  i: Integer;
begin
   SaveWindowState( Self, self.ClassName );

   for i := 0 to lbItems.Items.Count - 1 do begin
      TMailTrapItem( lbItems.Items.Objects[i] ).Free;
   end;
   lbItems.Clear;
   
   if Assigned( Settings ) then Settings.Free;
end;

function TfrmConfMailTrap.Current: TMailTrapItem;
var  Index: Integer;
begin
   Index := lbItems.ItemIndex;
   if (Index < 0) or (Index >= lbItems.Items.Count) then begin
      Result := nil;
   end else begin
      Result := TMailTrapItem( lbItems.Items.Objects[Index] );
   end;
end;

procedure TfrmConfMailTrap.SettingsToForm;
var  t: TMailTrapItem;
begin
   if Assigned( Settings ) then begin
      emScoreAccept.Text := Settings.GetStr( hsMailTrapAcceptScore );
      emScoreDelete.Text := Settings.GetStr( hsMailTrapDeleteScore );
   end;

   t := Current;

   if t = nil then begin

      pnlEdit.Enabled := False;

      cbComment.Text    := '';
      cbHeaderLine.Text := '';
      cbMatchMode.ItemIndex := -1;
      emPattern.Text    := '';
      emScoreValue.Text := '';
      chkMatchNot.Checked := False;
      chkCaseSensitive.Checked := False;
      rbActionAccept.Checked   := True;
      chkDisabled.Checked      := False;

   end else begin

      cbComment.Text    := t.Comment;
      cbHeaderLine.Text := t.Header;

      cbMatchMode.ItemIndex := ord( t.MatchMode );

      emPattern.Text    := t.Pattern;
      emScoreValue.Text := inttostr( t.ScoreValue );
      chkMatchNot.Checked := t.MatchNot;
      chkCaseSensitive.Checked := t.CaseSensitive;
      chkDisabled.Checked      := t.Disabled;

      case t.Action of
         mtaAccept: rbActionAccept.Checked := True;
         mtaDelete: rbActionDelete.Checked := True;
         mtaScore:  rbActionScore.Checked  := True;
      end;

      pnlEdit.Enabled := True;

   end;
end;

function TfrmConfMailTrap.LoadSettings: Boolean;

   procedure LoadFile;
   var  Reply: TLiveMsg;
        SL: TStringList;
        i: Integer;
        s: String;
        t: TMailTrapItem;
   begin
      try
         Reply := LiveConnector.RCLiveRequest(
                     TLiveMsg.Create( LMREQ_FILE_GET,
                                      DQuoteStr(CFGFILE_MAILTRAP) )
                  );
         if Assigned( Reply ) then try
            if ( Reply.MsgType = LMREP_OK ) then begin
               SL := TStringList.Create;
               try
                  SL.Text := Reply.MsgData;
                  for i := 0 to SL.Count - 1 do begin
                     if Trim( SL[i] ) <> '' then begin

                        s := SL[i];
                        t := TMailTrapItem.Create( s );
                        lbItems.Items.AddObject( s, t );

                        s := t.Comment;
                        if s = '' then s := ' '; // D7-bug
                        if cbComment.Items.IndexOf( s ) < 0 then begin
                           cbComment.Items.Add( s );
                        end;

                        if cbHeaderLine.Items.IndexOf( t.Header ) < 0 then begin
                           cbHeaderLine.Items.Add( t.Header );
                        end;

                     end;
                  end;
               finally SL.Free end;
            end;
         finally Reply.Free end;

      except
         on E:Exception do begin
            HMessageDlg( 'File ' + CFGFILE_MAILTRAP + ' could not be downloaded!'
                       + #13#10#13#10 + 'Error: ' + E.Message, mtError, [mbOK] );
         end;
      end;

   end;

var  i: Integer;
begin
   Result := False;

   try
      // Hamster Settings
      if Assigned( Settings ) then FreeAndNil( Settings );

      Settings := TSettingsPlain.Create(
         SettingsDef_Hamster,
         TSettingsHandler_LiveRemote.Create(
            LiveConnector, '',
            LMREQ_HAM_SETTING_GET, LMREQ_HAM_SETTING_SET
         ),
         False {don't AutoFlush}
      );

      // MailTrap.hst
      lbItems.Items.BeginUpdate;
      try

         for i := 0 to lbItems.Items.Count - 1 do begin
            TMailTrapItem( lbItems.Items.Objects[i] ).Free;
         end;
         lbItems.Clear;

         LoadFile;

      finally
         lbItems.Items.EndUpdate;
      end;

      if lbItems.Items.Count > 0 then lbItems.ItemIndex := 0;
      SettingsToForm;

      Result := True;

   except
      on E: Exception do begin
         HMessageDlg( 'Error:' + CRLF + E.Message,
                      'Load Mail Trap', mtError, [mbOK] );
      end;
   end;
end;

function TfrmConfMailTrap.SaveSettings: Boolean;
var  SL: TStringList;
     i: Integer;
     s: String;
     t: TMailTrapItem;
begin
   Result := False;

   for i := 0 to lbItems.Items.Count - 1 do begin
      t := TMailTrapItem( lbItems.Items.Objects[i] );
      if not t.CheckAndFixTrap then begin
         lbItems.ItemIndex := i;
         SettingsToForm;
         if t.Header  = '' then cbHeaderLine.SetFocus;
         if t.Pattern = '' then emPattern.SetFocus;
         HMessageDlg( 'Can''t save, selected trap is not completed yet!',
                      'Save Mail Trap', mtError, [mbOK] );
         exit;
      end;
   end;

   try
      // MailTrap.hst
      SL := TStringList.Create;
      try
         for i := 0 to lbItems.Items.Count - 1 do begin
            SL.Add( TMailTrapItem( lbItems.Items.Objects[i] ).AsDefLine  );
         end;

         if not LiveConnector.RCLiveRequestOK(
                   TLiveMsg.Create( LMREQ_FILE_SET,
                                    DQuoteStr(CFGFILE_MAILTRAP) + CRLF + SL.Text )
                ) then begin
            HMessageDlg( 'File "' + CFGFILE_MAILTRAP + '" could not be saved!',
                         'Save Mail Trap', mtError, [mbOK] );
         end;

      finally SL.Free end;

      // Hamster Settings
      if Assigned( Settings ) then begin

         i := hsMailTrapAcceptScore;
         s := Trim( emScoreAccept.Text );
         if Settings.GetStr( i ) <> s then Settings.SetStr( i, s );

         i := hsMailTrapDeleteScore;
         s := Trim( emScoreDelete.Text );
         if Settings.GetStr( i ) <> s then Settings.SetStr( i, s );

         Settings.Flush;
         
      end;

      Result := True;

   except
      on E: Exception do HMessageDlg( 'Error:' + CRLF + E.Message,
                                      'Save Mail Trap', mtError, [mbOK] );
   end;
end;

procedure TfrmConfMailTrap.btnRefreshClick(Sender: TObject);
begin
   btnOK.Enabled := LoadSettings;
end;

procedure TfrmConfMailTrap.btnOKClick(Sender: TObject);
begin
   if not SaveSettings then ModalResult := mrNone;
end;

procedure TfrmConfMailTrap.btnApplyClick(Sender: TObject);
begin
   SaveSettings;
end;

procedure TfrmConfMailTrap.lbItemsClick(Sender: TObject);
begin
   SettingsToForm;
end;

procedure TfrmConfMailTrap.lbItemsDrawItem(Control: TWinControl; Index: Integer;
  Rect: TRect; State: TOwnerDrawState);
var  s, c: String;
     t: TMailTrapItem;
     i: Integer;
     separator: Boolean;
begin
   t := TMailTrapItem( (Control as TListBox).Items.Objects[Index] );
   s := t.AsInfo;

   separator := False;
   i := Pos( MailTrapAsInfoCommentSeparator, s );
   if i = 0 then begin
      c := '';
   end else begin
      c := ' ' + Trim( copy( s, i, MaxInt ) );
      s := Trim( copy( s, 1, i-1 ) );
      i := Pos( '_', c );
      if i > 0 then begin
         separator := True;
         c := copy( c, 1, i-1 ) + copy( c, i+1, MaxInt );
      end;
   end;

   i := ord( t.Action ) * 2;
   if (t.Action = mtaScore) and (t.ScoreValue < 0 ) then inc( i, 2 );
   if t.Disabled then inc( i );

   with (Control as TListBox).Canvas do begin
   
      FillRect( Rect );
      imglstItems.Draw( (Control as TListBox).Canvas,
                        Rect.Left+2, Rect.Top, i, True );

      // if odSelected in State then Font.Style := [ fsBold ]
      //                        else Font.Style := [ ];
      TextOut( Rect.Left+20, Rect.Top+1, s );

      if length( c ) > 0 then begin
         Font.Color := clGray;
         TextOut( Rect.Right - TextWidth(c) - 2, Rect.Top+1, c );
         if separator then begin
            Pen.Color := clBlack;
            MoveTo( Rect.Left,  Rect.Top-1 );
            LineTo( Rect.Right, Rect.Top-1 );
            MoveTo( Rect.Left,  Rect.Top   );
            LineTo( Rect.Right, Rect.Top   );
         end;
      end;
   end;
end;

procedure TfrmConfMailTrap.cbCommentChange(Sender: TObject);
begin
   if Current <> nil then begin
      Current.Comment := cbComment.Text;
      lbItems.Invalidate;
   end;
end;

procedure TfrmConfMailTrap.cbHeaderLineChange(Sender: TObject);
begin
   if Current <> nil then begin
      Current.Header := cbHeaderLine.Text;
      lbItems.Invalidate;
   end;
end;

procedure TfrmConfMailTrap.cbMatchModeChange(Sender: TObject);
begin
   if Current <> nil then begin
      Current.MatchMode := TMailTrapMatchMode( cbMatchMode.ItemIndex );
      lbItems.Invalidate;
   end;
end;

procedure TfrmConfMailTrap.chkMatchNotClick(Sender: TObject);
begin
   if Current <> nil then begin
      Current.MatchNot := chkMatchNot.Checked;
      lbItems.Invalidate;
   end;
end;

procedure TfrmConfMailTrap.chkCaseSensitiveClick(Sender: TObject);
begin
   if Current <> nil then begin
      Current.CaseSensitive := chkCaseSensitive.Checked;
      lbItems.Invalidate;
   end;
end;

procedure TfrmConfMailTrap.emPatternChange(Sender: TObject);
begin
   if Current <> nil then begin
      Current.Pattern := emPattern.Text;
      lbItems.Invalidate;
   end;
end;

procedure TfrmConfMailTrap.emScoreValueChange(Sender: TObject);
begin
   if Current <> nil then begin
      Current.ScoreValue := strtointdef( Trim(emScoreValue.Text), 0 );
      lbItems.Invalidate;
   end;
end;

procedure TfrmConfMailTrap.rbActionAcceptClick(Sender: TObject);
begin
   if Current <> nil then begin
      Current.Action := mtaAccept;
      lbItems.Invalidate;
   end;
end;

procedure TfrmConfMailTrap.rbActionDeleteClick(Sender: TObject);
begin
   if Current <> nil then begin
      Current.Action := mtaDelete;
      lbItems.Invalidate;
   end;
end;

procedure TfrmConfMailTrap.rbActionScoreClick(Sender: TObject);
begin
   if Current <> nil then begin
      Current.Action := mtaScore;
      lbItems.Invalidate;
   end;
end;

procedure TfrmConfMailTrap.chkDisabledClick(Sender: TObject);
begin
   if Current <> nil then begin
      Current.Disabled := chkDisabled.Checked;
      lbItems.Invalidate;
   end;
end;

procedure TfrmConfMailTrap.DoAddItem( const DefLine: String );
var  t: TMailTrapItem;
     i: Integer;
begin
   t := TMailTrapItem.Create( DefLine );

   i := lbItems.ItemIndex + 1;
   if i < 0 then i := 0;
   lbItems.Items.InsertObject( i, '', t );

   lbItems.ItemIndex := i;
   SettingsToForm;
end;

procedure TfrmConfMailTrap.btnAddClick(Sender: TObject);
begin
   DoAddItem( '' );
end;

procedure TfrmConfMailTrap.btnCloneClick(Sender: TObject);
var  t: TMailTrapItem;
begin
   t := Current;
   if t <> nil then DoAddItem( t.AsDefLine );
end;

procedure TfrmConfMailTrap.btnMoveUpClick(Sender: TObject);
var  i: Integer;
begin
   if Current = nil then exit;
   i := lbItems.ItemIndex;
   if i - 1 < 0 then exit;
   lbItems.Items.Exchange( i, i - 1 );
   lbItems.ItemIndex := i - 1;
   SettingsToForm;
end;

procedure TfrmConfMailTrap.btnMoveDownClick(Sender: TObject);
var  i: Integer;
begin
   if Current = nil then exit;
   i := lbItems.ItemIndex;
   if i + 1 >= lbItems.Items.Count then exit;
   lbItems.Items.Exchange( i, i + 1 );
   lbItems.ItemIndex := i + 1;
   SettingsToForm;
end;

procedure TfrmConfMailTrap.btnTestClick(Sender: TObject);
var  ScoreAccept, ScoreDelete: Integer;
begin
   with TfrmConfMailTrapTest.Create( self ) do try
      ScoreAccept := strtointdef( Trim( emScoreAccept.Text ), +100 );
      ScoreDelete := strtointdef( Trim( emScoreDelete.Text ), -100 );
      LinkItems( ScoreAccept, ScoreDelete, self.lbItems );
      ShowModal;
   finally Free end;
end;

procedure TfrmConfMailTrap.btnDeleteClick(Sender: TObject);
var  t: TMailTrapItem;
     i: Integer;
     s: String;
begin
   t := Current;
   if t = nil then exit;

   s := 'Delete selected line?' + #13#10#13#10 + t.AsInfo;
   if HMessageDlg( PChar(s), mtConfirmation, [mbYes,mbNo] ) <> mrYes then exit;

   i := lbItems.ItemIndex;
   lbItems.Items.Delete( i );
   t.Free;

   if i > lbItems.Items.Count then dec( i );
   if i >= 0 then lbItems.ItemIndex := i;

   SettingsToForm;
end;

procedure TfrmConfMailTrap.mnuListToClipboardClick(Sender: TObject);
var  s: String;
     i: Integer;
     t: TMailTrapItem;
begin
   for i := 0 to lbItems.Items.Count - 1 do begin
      t := tMailTrapItem( lbItems.Items.Objects[ i ] );
      s := s + t.AsInfo + #13#10;
   end;
   Clipboard.AsText := s;
end;

end.
