// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit fConfML;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  dDialogs, StdCtrls, Menus, ComCtrls, ExtCtrls;

type
  TfrmConfML = class(TForm)
    Label1: TLabel;
    Bevel1: TBevel;
    lbItems: TListBox;
    btnOK: TButton;
    btnApply: TButton;
    btnRefresh: TButton;
    btnCancel: TButton;
    PageControl1: TPageControl;
    pgProps: TTabSheet;
    grpProps: TGroupBox;
    Label4: TLabel;
    Label6: TLabel;
    emListName: TEdit;
    emDescription: TEdit;
    pgMembers: TTabSheet;
    popItems: TPopupMenu;
    mnuItemAdd: TMenuItem;
    mnuItemAddClone: TMenuItem;
    N1: TMenuItem;
    mnuItemDel: TMenuItem;
    Label2: TLabel;
    lbMembers: TListBox;
    cbAccounts: TComboBox;
    popMembers: TPopupMenu;
    mnuMemberAddAccount: TMenuItem;
    mnuMemberAddAddress: TMenuItem;
    N2: TMenuItem;
    mnuMemberDelete: TMenuItem;
    Label3: TLabel;
    cbRestrictions: TComboBox;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure btnRefreshClick(Sender: TObject);
    procedure btnOKClick(Sender: TObject);
    procedure btnApplyClick(Sender: TObject);
    procedure lbItemsClick(Sender: TObject);
    procedure popItemsPopup(Sender: TObject);
    procedure mnuItemAddClick(Sender: TObject);
    procedure mnuItemAddCloneClick(Sender: TObject);
    procedure mnuItemDelClick(Sender: TObject);
    procedure lbItemsDrawItem(Control: TWinControl; Index: Integer;
      Rect: TRect; State: TOwnerDrawState);
    procedure popMembersPopup(Sender: TObject);
    procedure mnuMemberAddAccountClick(Sender: TObject);
    procedure mnuMemberAddAddressClick(Sender: TObject);
    procedure mnuMemberDeleteClick(Sender: TObject);
    procedure FormResize(Sender: TObject);
  private
    { Private-Deklarationen }
    SelItemType: Integer;
    SelItemName: String;
    SelProps   : TStringList;
    SelMembers : TStringList;
    Args       : TStringList;
    procedure SettingsToForm;
    procedure FormToSettings;
    function  LoadItemList: Boolean;
    function  LoadSettings: Boolean;
    function  SaveSettings: Boolean;
    procedure DoAddMailList( const MailListToCopy: String );
    function  IndexOfMailList( const Name: String ): Integer;
    function  MailListName( const Index: Integer ): String;
    procedure DoAddMember( const MemberName: String );
  public
    { Public-Deklarationen }
  end;

implementation

{$R *.DFM}

uses uConst, uConstVar, uGlobal, cSettingsHdlLive, uTools, cLiveMsg, cLiveQueue,
     uDateTime, dInput;

const
   selNone = 0;
   selList = 1;


procedure TfrmConfML.FormCreate(Sender: TObject);
begin
   GrowDpiForm( self );

   SelItemType := selNone;
   SelItemName := '';
   SelProps    := TStringList.Create;
   SelMembers  := TStringList.Create;
   Args := TStringList.Create;

   with cbRestrictions do begin
      Clear;
      Items.Add( '0 - any sender' );
      Items.Add( '1 - any "From:" address of member account' );
      Items.Add( '2 - authenticated member accounts only' );
      Items.Add( '3 - authenticated owner account only' );
   end;

   PageControl1.ActivePageIndex := 0;
   LoadItemList;
   btnRefresh.Click;
end;

procedure TfrmConfML.FormResize(Sender: TObject);
begin
   PageControl1.Width  := ClientWidth  - 4 * lbItems.Left - lbItems.Width - Bevel1.Width;
   PageControl1.Height := ClientHeight - 3 * PageControl1.Top - btnOk.Height;
   lbItems.Height      := ClientHeight - lbItems.Top - PageControl1.Top;
   btnOk.Top      := ClientHeight - btnOk.Height - PageControl1.Top;
   btnApply.Top   := ClientHeight - btnOk.Height - PageControl1.Top;
   btnRefresh.Top := ClientHeight - btnOk.Height - PageControl1.Top;
   btnCancel.Top  := ClientHeight - btnOk.Height - PageControl1.Top;
end;

procedure TfrmConfML.FormDestroy(Sender: TObject);
begin
   if Assigned( SelMembers ) then SelMembers.Free;
   if Assigned( SelProps ) then SelProps.Free;
   if Assigned( Args ) then Args.Free;
end;

procedure TfrmConfML.SettingsToForm;
var  s: String;
     i: Integer;
begin
   emListName.Text := SelItemName;

   s := SelProps.Values[ mlpDescription ]; 
   if s = '' then s := 'List ' + SelItemName;
   emDescription.Text := s;

   s := SelProps.Values[ mlpOwnerAccount ];
   if s = '' then s := 'admin';
   cbAccounts.ItemIndex := -1;
   for i := 0 to cbAccounts.Items.Count - 1 do begin
      if cbAccounts.Items[i] = s then cbAccounts.ItemIndex := i;
   end;

   i := strtointdef( SelProps.Values[ mlpRestrictFrom ], mlprfNoRestrict );
   if (i<0) or (i>=cbRestrictions.Items.Count) then i := mlprfNoRestrict;
   cbRestrictions.ItemIndex := i;

   lbMembers.Items.Text := SelMembers.Text;
end;

procedure TfrmConfML.FormToSettings;
var  i: Integer;
begin
   if SelItemType = selNone then exit;
   if SelItemName = ''      then exit;

   SelProps.Values[ mlpDescription ] := emDescription.Text; 

   if cbAccounts.ItemIndex < 0 then begin
      SelProps.Values[ mlpOwnerAccount ] := '';
   end else begin
      SelProps.Values[ mlpOwnerAccount ] := cbAccounts.Items[ cbAccounts.ItemIndex ]; 
   end;

   i := cbRestrictions.ItemIndex;
   if (i<0) or (i>=cbRestrictions.Items.Count) then i := mlprfNoRestrict;
   SelProps.Values[ mlpRestrictFrom ] := inttostr(i);

   SelMembers.Text := lbMembers.Items.Text;
end;

function TfrmConfML.LoadItemList: Boolean;              

   procedure AddItems( LqReqId, setId: Integer );
   var  Reply: TLiveMsg;
        SL: TStringList;
        i: Integer;
   begin
      Reply := LiveConnector.RCLiveRequest(
                  TLiveMsg.Create( LqReqId, '' )
               );
      if Assigned( Reply ) then try
         if ( Reply.MsgType = LMREP_OK ) then begin
            SL := TStringList.Create;
            try
               SL.Text := Reply.MsgData;
               for i:=0 to SL.Count-1 do begin
                  lbItems.Items.AddObject( SL[i], Pointer( setId ) );
               end;
            finally SL.Free end;
         end;
      finally Reply.Free end;
   end;

begin
   Result := False;

   try
      SelItemType := selNone;
      SelItemName := '';
      SelProps.Clear;
      SelMembers.Clear;

      lbItems.Items.BeginUpdate;
      try
         lbItems.Items.Clear;
         lbItems.ItemIndex := -1;
         AddItems( LMREQ_MAILLISTS_LIST, selList );
      finally lbItems.Items.EndUpdate end;

      Result := True;

   except
      on E: Exception do begin
         HMessageDlg( 'Error:' + CRLF + E.Message,
                      'Load Item List', mtError, [mbOK] );
      end;
   end;
end;

function TfrmConfML.LoadSettings: Boolean;

   procedure LoadAccounts;
   var  Reply: TLiveMsg;
        SL: TStringList;
        i: Integer;
   begin
      cbAccounts.Clear;
      Reply := LiveConnector.RCLiveRequest(
                  TLiveMsg.Create( LMREQ_ACCOUNTS_LIST, '' )
               );
      if Assigned( Reply ) then try
         if ( Reply.MsgType = LMREP_OK ) then begin
            SL := TStringList.Create;
            try
               SL.Text := Reply.MsgData;
               for i:=0 to SL.Count-1 do begin
                  ArgsWhSpaceDQuoted( SL[i], Args, 3 );
                  cbAccounts.Items.Add( Args[0] );
               end;
            finally SL.Free end;
         end;
      finally Reply.Free end;
   end;

   procedure LoadLists;
   var  Reply: TLiveMsg;
        SL: TStringList;
        s: String;
        i: Integer;
   begin
      Reply := LiveConnector.RCLiveRequest(
                  TLiveMsg.Create( LMREQ_MAILLISTS_GET, SelItemName )
               );
      if Assigned( Reply ) then try
         if ( Reply.MsgType = LMREP_OK ) then begin
            SL := TStringList.Create;
            try
               SL.Text := Reply.MsgData;
               for i:=0 to SL.Count-1 do begin
                  s := TrimWhSpace( SL[i] );
                  if copy( s, 1, 1 ) = '!' then begin
                     SelProps.Add( s );
                  end else begin
                     if s <> '' then SelMembers.Add( s );
                  end;
               end;
            finally SL.Free end;
         end;
      finally Reply.Free end;
   end;

begin
   Result := False;

   try
      if lbItems.Items.Count = 0 then LoadItemList;

      SelProps.Clear;
      SelMembers.Clear;
      if SelItemName = '' then SelItemType := selNone;

      case SelItemType of
         selNone: ;
         selList: LoadLists;
      end;

      LoadAccounts;

      SettingsToForm;

      Result := True;

   except
      on E: Exception do begin
         HMessageDlg( 'Error:' + CRLF + E.Message,
                      'Load Settings', mtError, [mbOK] );
      end;
   end;
end;

function TfrmConfML.SaveSettings: Boolean;
var  SL: TStringList;
     i: Integer;
begin
   Result := True;

   try
      if SelItemType = selNone       then exit;
      if SelItemName = ''            then exit;

      FormToSettings;

      SL := TStringList.Create;
      try
         for i := 0 to SelProps.Count   - 1 do SL.Add( SelProps[i]   );
         for i := 0 to SelMembers.Count - 1 do SL.Add( SelMembers[i] );

         if not LiveConnector.RCLiveRequestOK(
                   TLiveMsg.Create( LMREQ_MAILLISTS_SET,
                                    SelItemName + #13#10 + SL.Text )
                ) then begin
            HMessageDlg( 'Distribution list "' + SelItemName + '" could not be saved!',
                         'Save Settings', mtError, [mbOK] );
            Result := False;
         end;

      finally SL.Free end;

   except
      on E: Exception do begin
         HMessageDlg( 'Error:' + CRLF + E.Message,
                      'Save Settings', mtError, [mbOK] );
         Result := False;
      end;
   end;
end;

procedure TfrmConfML.btnRefreshClick(Sender: TObject);
begin
   btnOK.Enabled := LoadSettings;
end;

procedure TfrmConfML.btnOKClick(Sender: TObject);
begin
   if not SaveSettings then ModalResult := mrNone;
end;

procedure TfrmConfML.btnApplyClick(Sender: TObject);
begin
   SaveSettings;
end;

procedure TfrmConfML.lbItemsClick(Sender: TObject);
begin
   with lbItems do begin
      if ( ItemIndex < 0 ) or ( ItemIndex >= Items.Count ) then begin
         SelItemType := selNone;
         SelItemName := '';
      end else begin
         SelItemType := Integer( Items.Objects[ ItemIndex ] );
         SelItemName := MailListName( ItemIndex );
      end;
   end;

   LoadSettings;

   grpProps .Enabled  := ( SelItemType <> selNone );
   lbMembers.Enabled  := ( SelItemType <> selNone );
end;

procedure TfrmConfML.popItemsPopup(Sender: TObject);
begin
   with lbItems do begin
      mnuItemDel.Enabled := ( ItemIndex >= 0 ) and ( ItemIndex < Items.Count );
      mnuItemAddClone.Enabled := mnuItemDel.Enabled;
   end;
end;

procedure TfrmConfML.DoAddMailList( const MailListToCopy: String );
var  Nam, s: String;
     Idx, i: Integer;
begin
   Nam := '';
   if not InputDlgStr( 'Add Distribution List', 'Name:', Nam, 0 ) then exit;
   if Nam = '' then exit;

   if PosWhSpace( Nam ) > 0 then begin
      HMessageDlg( 'Invalid distribution list name "' + Nam + '"!',
                   'Add Distribution List', mtWarning, [mbOK] );
      exit;
   end;

   Idx := IndexOfMailList( Nam );
   if Idx >= 0 then begin
      HMessageDlg( 'Distribution list "' + Nam + '" already exists!',
                   'Add Distribution List', mtWarning, [mbOK] );
      exit;
   end;

   s := Nam + CRLF;
   if MailListToCopy = '' then begin
      s := s + mlpDescription  + '=List ' + Nam + CRLF  
             + mlpOwnerAccount + '=admin' + CRLF;
   end else begin
      for i := 0 to SelProps.Count   - 1 do s := s + SelProps[i]   + CRLF;
      for i := 0 to SelMembers.Count - 1 do s := s + SelMembers[i] + CRLF;
   end;
   if not LiveConnector.RCLiveRequestOK(
             TLiveMsg.Create( LMREQ_MAILLISTS_SET, s )
          ) then begin
      HMessageDlg( 'Distribution list "' + Nam + '" could not be created!',
                   'Add Distribution List', mtError, [mbOK] );
   end;

   LoadItemList;
   btnRefresh.Click;

   Idx := IndexOfMailList( Nam );
   if Idx >= 0 then begin lbItems.ItemIndex:=Idx; lbItemsClick(nil) end;
end;


procedure TfrmConfML.mnuItemAddClick(Sender: TObject);
begin
   DoAddMailList( '' );
end;

procedure TfrmConfML.mnuItemAddCloneClick(Sender: TObject);
var  Idx: Integer;
     Nam: String;
begin
   Idx := lbItems.ItemIndex;
   if ( Idx < 0 ) or ( Idx >= lbItems.Items.Count ) then exit;
   Nam := MailListName( Idx );
   DoAddMailList( Nam );
end;

procedure TfrmConfML.mnuItemDelClick(Sender: TObject);   
var  Idx: Integer;
     Nam, s: String;
begin
   Idx := lbItems.ItemIndex;
   if ( Idx < 0 ) or ( Idx >= lbItems.Items.Count ) then exit;
   Nam := MailListName( Idx );

   s := Format( 'Delete distribution list "%s" now?', [Nam] );
   if HMessageDlg( PChar(s), mtConfirmation, [mbYes,mbNo] ) <> mrYes then exit;

   if not LiveConnector.RCLiveRequestOK(
             TLiveMsg.Create( LMREQ_MAILLISTS_DEL, Nam )
          ) then begin
      HMessageDlg( 'Distribution list "' + Nam + '" could not be deleted!',
                   mtError, [mbAbort] );
   end;

   LoadItemList;
   btnRefresh.Click;

   if Idx >= lbItems.Items.Count then dec( Idx );
   if Idx >= 0 then begin lbItems.ItemIndex:=Idx; lbItemsClick(nil) end;
end;

procedure TfrmConfML.lbItemsDrawItem(Control: TWinControl; Index: Integer; 
  Rect: TRect; State: TOwnerDrawState);
var  s: String;
begin
   s := (Control as TListBox).Items[Index];
   ArgsWhSpaceDQuoted( s, Args, 3 );
   s := Args[0];

   with (Control as TListBox).Canvas do begin
      FillRect( Rect );
      if odSelected in State then Font.Style := [ fsBold ]
                             else Font.Style := [ ];
      TextOut( Rect.Left+1, Rect.Top+1, s );
   end;
end;

function TfrmConfML.IndexOfMailList( const Name: String): Integer;
var  Idx: Integer;
begin
   Result := -1;
   for Idx := 0 to lbItems.Items.Count - 1 do begin
      if MailListName( Idx ) = Name then begin Result:=Idx; break end;
   end;
end;

function TfrmConfML.MailListName( const Index: Integer ): String;   
begin
   ArgsWhSpaceDQuoted( lbItems.Items[Index], Args, 3 );
   Result := Args[0];
end;

procedure TfrmConfML.popMembersPopup(Sender: TObject);
begin
   with lbMembers do begin
      mnuMemberAddAccount.Enabled := lbMembers.Enabled;
      mnuMemberAddAddress.Enabled := lbMembers.Enabled;
      mnuMemberDelete.Enabled := lbMembers.Enabled and (ItemIndex>=0)
                                                   and (ItemIndex<Items.Count);
   end;
end;

procedure TfrmConfML.DoAddMember( const MemberName: String );
begin
   if lbMembers.Items.IndexOf( MemberName ) >= 0 then begin
      HMessageDlg( 'Member "' + MemberName + '" is already in list!',
                   'Add New Member', mtWarning, [mbOK] );
      exit;
   end;

   lbMembers.ItemIndex := lbMembers.Items.Add( MemberName );
end;

procedure TfrmConfML.mnuMemberAddAccountClick(Sender: TObject);
var  Nam: String;
     SL : TStringList;
     i  : Integer;
begin
   Nam := '';

   SL := TStringList.Create;
   try
      for i := 0 to cbAccounts.Items.Count - 1 do begin
         if lbMembers.Items.IndexOf( cbAccounts.Items[i] ) < 0 then begin
            SL.Add( cbAccounts.Items[i] );
         end;
      end;
      i := -1;
      if not InputDlgList(
                'Add Account', 'Select account to add:', i, SL.Text, 0
             ) then exit;
      if i >= 0 then Nam := SL[ i ];
   finally SL.Free end;

   if Nam <> '' then DoAddMember( Nam );
end;

procedure TfrmConfML.mnuMemberAddAddressClick(Sender: TObject);
var  Nam: String;
     OK: Boolean;
begin
   Nam := '';
   if not InputDlgStr( 'Add mail address', 'Mail address:', Nam, 0 ) then exit;
   if Nam = '' then exit;

   OK := True;
   if PosWhSpace( Nam ) > 0 then OK := False;
   if Pos( '@', Nam ) <= 1  then OK := False;

   if OK then DoAddMember( Nam )
         else HMessageDlg( 'Invalid mail address "' + Nam + '"!',
                           'Add Mail Address', mtError, [mbOK] );
end;

procedure TfrmConfML.mnuMemberDeleteClick(Sender: TObject);
var  Idx: Integer;
     Nam, s: String;
begin
   Idx := lbMembers.ItemIndex;
   if ( Idx < 0 ) or ( Idx >= lbMembers.Items.Count ) then exit;
   Nam := lbMembers.Items[ Idx ];

   s := Format( 'Delete member "%s"?', [Nam] );
   if HMessageDlg( PChar(s), mtConfirmation, [mbYes,mbNo] ) <> mrYes then exit;

   lbMembers.Items.Delete( Idx );

   if Idx >= lbMembers.Items.Count then dec( Idx );
   if Idx >= 0 then lbMembers.ItemIndex := Idx; 
end;

end.
