// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit fConfAcc;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  dDialogs, StdCtrls, Menus, cSettings, ComCtrls, ExtCtrls;

type
  TfrmConfAcc = class(TForm)
    Label1: TLabel;
    lbItems: TListBox;
    popItems: TPopupMenu;
    mnuItemAdd: TMenuItem;
    N1: TMenuItem;
    mnuItemDel: TMenuItem;
    btnOK: TButton;
    btnApply: TButton;
    btnRefresh: TButton;
    btnCancel: TButton;
    PageControl1: TPageControl;
    pgAccount: TTabSheet;
    pgNews: TTabSheet;
    pgMail: TTabSheet;
    grpAccount: TGroupBox;
    grpNews: TGroupBox;
    grpMail: TGroupBox;
    Label8: TLabel;
    emNewsRead: TEdit;
    Label9: TLabel;
    emNewsPost: TEdit;
    Label4: TLabel;
    emUsername: TEdit;
    Label6: TLabel;
    emFullname: TEdit;
    Label7: TLabel;
    emUniqueID: TEdit;
    Label5: TLabel;
    lblPasswordInfo: TLabel;
    btnPWChange: TButton;
    btnPWClear: TButton;
    ckNewsNewNews: TCheckBox;
    Bevel1: TBevel;
    ckNewsPeer: TCheckBox;
    Label2: TLabel;
    cbNewsAutoSub: TComboBox;
    ckMailSend: TCheckBox;
    ckMailbox: TCheckBox;
    Label3: TLabel;
    emMailAddress: TEdit;
    Label10: TLabel;
    cbRemoteControl: TComboBox;
    mnuItemAddClone: TMenuItem;
    Label11: TLabel;
    Label12: TLabel;
    cbFwdAccount: TComboBox;
    Label13: TLabel;
    emFwdAddress: TEdit;
    Label14: TLabel;
    cbFwdMailList: TComboBox;
    ckFwdKeepCopy: TCheckBox;
    Label15: TLabel;
    emIpRestriction: TEdit;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure btnRefreshClick(Sender: TObject);
    procedure btnOKClick(Sender: TObject);
    procedure btnApplyClick(Sender: TObject);
    procedure lbItemsClick(Sender: TObject);
    procedure popItemsPopup(Sender: TObject);
    procedure mnuItemAddClick(Sender: TObject);
    procedure mnuItemDelClick(Sender: TObject);
    procedure btnPWChangeClick(Sender: TObject);
    procedure btnPWClearClick(Sender: TObject);
    procedure mnuItemAddCloneClick(Sender: TObject);
    procedure lbItemsDrawItem(Control: TWinControl; Index: Integer;
      Rect: TRect; State: TOwnerDrawState);
    procedure FormResize(Sender: TObject);
  private
    { Private-Deklarationen }
    SelItemType: Integer;
    SelItemName: String;
    SelSettings: TSettingsPlain;
    Args       : TStringList;
    procedure ShowPasswordInfo;
    procedure SettingsToForm;
    procedure FormToSettings;
    function  LoadItemList: Boolean;
    function  LoadSettings( loadAll: Boolean ): Boolean;
    function  SaveSettings: Boolean;
    procedure DoAddAccount( const AccountToCopy: String );
    function  IndexOfAccount( const Name: String ): Integer;
    function  AccountName( const Index: Integer ): String;
  public
    { Public-Deklarationen }
  end;

implementation

{$R *.DFM}

uses uConst, uConstVar, uGlobal, cSettingsHdlLive, uTools, cLiveMsg, cLiveQueue,
     uDateTime, dInput;

const
   selNone = 0;
   selAcc  = 1;

procedure TfrmConfAcc.FormCreate(Sender: TObject);
begin
   GrowDpiForm( self );

   SelItemType := selNone;
   SelItemName := '';
   SelSettings := nil;
   Args := TStringList.Create;

   PageControl1.ActivePageIndex := 0;

   LoadItemList;
   btnRefresh.Click;
end;

procedure TfrmConfAcc.FormResize(Sender: TObject);
begin
   PageControl1.Width  := ClientWidth  - 4 * lbItems.Left - lbItems.Width - Bevel1.Width;
   PageControl1.Height := ClientHeight - 3 * PageControl1.Top - btnOk.Height;
   lbItems.Height      := ClientHeight - lbItems.Top - PageControl1.Top;
   btnOk.Top      := ClientHeight - btnOk.Height - PageControl1.Top;
   btnApply.Top   := ClientHeight - btnOk.Height - PageControl1.Top;
   btnRefresh.Top := ClientHeight - btnOk.Height - PageControl1.Top;
   btnCancel.Top  := ClientHeight - btnOk.Height - PageControl1.Top;
end;

procedure TfrmConfAcc.FormDestroy(Sender: TObject);
begin
   if Assigned( SelSettings ) then SelSettings.Free;
   if Assigned( Args ) then Args.Free;
end;

procedure TfrmConfAcc.ShowPasswordInfo; 
const PW_SET  = 'SET';
      PW_NONE = 'NONE SET';
var  PW_STATE: String;
     s: String;
begin
   PW_STATE := '(unknown)';

   if ( SelItemType <> selNone ) and ( SelItemName <> '' ) then begin
      s := SelSettings.GetStr( apPassword );
      PW_STATE := iif( s = '1', PW_SET, PW_NONE );
   end;

   lblPasswordInfo.Caption := PW_STATE;
   btnPWChange.Enabled := ( PW_STATE = PW_SET ) or ( PW_STATE = PW_NONE );
   btnPWClear .Enabled := ( PW_STATE = PW_SET );
end;

procedure TfrmConfAcc.SettingsToForm;
var  i: Integer;
     s: String;
begin
   ShowPasswordInfo;

   if not Assigned( SelSettings ) then exit;

   with SelSettings do begin

      emUsername.Text           := GetStr( apUsername );
      emFullname.Text           := GetStr( apFullname );
      emUniqueID.Text           := GetStr( apUniqueID );
      emIpRestriction.Text      := GetStr( apIpRestriction );
      cbRemoteControl.ItemIndex := GetInt( apRemoteControl );

      emNewsRead.Text           := GetStr( apNewsRead );
      emNewsPost.Text           := GetStr( apNewsPost );
      ckNewsNewNews.Checked     := GetBoo( apNewsNewNews );
      // ckNewsXHSearch.Checked    := GetBoo( apNewsXHSearch );
      cbNewsAutoSub.ItemIndex   := GetInt( apNewsAutoSub );
      ckNewsPeer.Checked        := GetBoo( apNewsPeer );

      emMailAddress.Text        := GetStr( apMailAddress );
      ckMailbox.Checked         := GetBoo( apMailbox );
      ckMailSend.Checked        := GetBoo( apMailSend );

      s := GetStr( apMailFwdAccount );
      cbFwdAccount.ItemIndex := -1;
      for i := 0 to cbFwdAccount.Items.Count - 1 do begin
         if Trim( cbFwdAccount.Items[i] ) = s then cbFwdAccount.ItemIndex := i;
      end;

      emFwdAddress.Text        := GetStr( apMailFwdAddress );

      s := GetStr( apMailFwdMailList );
      cbFwdMailList.ItemIndex := -1;
      for i := 0 to cbFwdMailList.Items.Count - 1 do begin
         if Trim( cbFwdMailList.Items[i] ) = s then cbFwdMailList.ItemIndex := i;
      end;

      ckFwdKeepCopy.Checked := ( GetStr( apMailFwdKeepCopy ) <> '0' );

   end;
end;

procedure TfrmConfAcc.FormToSettings;

   procedure TakeEM( ID: Integer; EM: TEdit );
   begin
      if not EM.Enabled then exit;
      if SelSettings.GetStr( ID ) = EM.Text then exit;
      SelSettings.SetStr( ID, EM.Text );
   end;

   procedure TakeCK( ID: Integer; CK: TCheckBox );
   begin
      if not CK.Enabled then exit;
      if SelSettings.GetBoo( ID ) = CK.Checked then exit;
      SelSettings.SetBoo( ID, CK.Checked );
   end;

   procedure TakeCB( ID: Integer; CB: TComboBox );
   begin
      if not CB.Enabled then exit;
      if SelSettings.GetInt( ID ) = CB.ItemIndex then exit;
      SelSettings.SetInt( ID, CB.ItemIndex );
   end;

   procedure TakeCBStr( ID: Integer; CB: TComboBox );
   var  s: String;
   begin
      if not CB.Enabled then exit;
      if CB.ItemIndex < 0 then s := ''
                          else s := Trim( CB.Items[ CB.ItemIndex ] );
      if SelSettings.GetStr( ID ) = s then exit;
      SelSettings.SetStr( ID, s );
   end;

begin
   if SelItemType = selNone       then exit;
   if SelItemName = ''            then exit;
   if not Assigned( SelSettings ) then exit;

   TakeEM( apFullname, emFullname );
   TakeEM( apIpRestriction, emIpRestriction );
   TakeCB( apRemoteControl, cbRemoteControl );

   TakeEM( apNewsRead, emNewsRead );
   TakeEM( apNewsPost, emNewsPost );
   TakeCK( apNewsNewNews, ckNewsNewNews );
   // TakeCK( apNewsXHSearch, ckNewsXHSearch );
   TakeCB( apNewsAutoSub, cbNewsAutoSub );
   TakeCK( apNewsPeer, ckNewsPeer );

   TakeEM( apMailAddress, emMailAddress );
   TakeCK( apMailbox, ckMailbox );
   TakeCK( apMailSend, ckMailSend );

   TakeCBStr( apMailFwdAccount,  cbFwdAccount );
   TakeEM   ( apMailFwdAddress,  emFwdAddress );
   TakeCBStr( apMailFwdMailList, cbFwdMailList );
   TakeCK   ( apMailFwdKeepCopy, ckFwdKeepCopy );
end;

function TfrmConfAcc.LoadItemList: Boolean;

   procedure AddItems( LqReqId, setId: Integer );
   var  Reply: TLiveMsg;
        SL: TStringList;
        i: Integer;
   begin
      Reply := LiveConnector.RCLiveRequest(
                  TLiveMsg.Create( LqReqId, '' )
               );
      if Assigned( Reply ) then try
         if ( Reply.MsgType = LMREP_OK ) then begin
            SL := TStringList.Create;
            try
               SL.Text := Reply.MsgData;
               for i:=0 to SL.Count-1 do begin
                  lbItems.Items.AddObject( SL[i], Pointer( setId ) );
               end;
            finally SL.Free end;
         end;
      finally Reply.Free end;
   end;

begin
   Result := False;

   try
      SelItemType := selNone;
      SelItemName := '';
      SelSettings := nil;

      lbItems.Items.BeginUpdate;
      try
         lbItems.Items.Clear;
         lbItems.ItemIndex := -1;
         AddItems( LMREQ_ACCOUNTS_LIST, selAcc );
      finally lbItems.Items.EndUpdate end;

      Result := True;

   except
      on E: Exception do begin
         HMessageDlg( 'Error:' + CRLF + E.Message,
                      'Load Item List', mtError, [mbOK] );
      end;
   end;
end;

function TfrmConfAcc.LoadSettings( loadAll: Boolean ): Boolean;

   function LoadList( LmReqCode: Integer ): TStringList;
   var  Reply: TLiveMsg;
   begin
      Result := TStringList.Create;
      Reply := LiveConnector.RCLiveRequest(
                  TLiveMsg.Create( LmReqCode, '' )
               );
      if Assigned( Reply ) then try
         if ( Reply.MsgType = LMREP_OK ) then begin
            Result.Text := Reply.MsgData;
         end;
      finally Reply.Free end;
   end;

   procedure LoadRCProfiles;
   var  i: Integer;
   begin
      cbRemoteControl.Clear;
      with LoadList( LMREQ_RCPROFILES_LIST ) do try
         for i := 0 to Count - 1 do begin
            ArgsWhSpaceDQuoted( Strings[i], Args, 2 );
            cbRemoteControl.Items.Add( Args[0] + ' - ' + Args[1] );
         end;
      finally Free end;
   end;

   procedure LoadAccounts;
   var  i: Integer;
   begin
      cbFwdAccount.Clear;
      cbFwdAccount.Items.Add( ' ' ); // clear
      with LoadList( LMREQ_ACCOUNTS_LIST ) do try
         for i := 0 to Count - 1 do begin
            ArgsWhSpaceDQuoted( Strings[i], Args, 2 );
            cbFwdAccount.Items.Add( Args[0] );
         end;
      finally Free end;
   end;

   procedure LoadMailLists;
   var  i: Integer;
   begin
      cbFwdMailList.Clear;
      cbFwdMailList.Items.Add( ' ' ); // clear
      with LoadList( LMREQ_MAILLISTS_LIST ) do try
         for i := 0 to Count - 1 do begin
            ArgsWhSpaceDQuoted( Strings[i], Args, 2 );
            cbFwdMailList.Items.Add( Args[0] );
         end;
      finally Free end;
   end;

begin
   Result := False;

   try
      if lbItems.Items.Count = 0 then LoadItemList;

      if Assigned( SelSettings ) then FreeAndNil( SelSettings );
      if SelItemName = '' then SelItemType := selNone;

      case SelItemType of
         selNone: SelSettings := TSettingsPlain.Create(
                     SettingsDef_Accounts,
                     TSettingsHandler_InMemory.Create( '' ),
                     False {don't AutoFlush}
                  );
         selAcc : SelSettings := TSettingsPlain.Create(
                     SettingsDef_Accounts,
                     TSettingsHandler_LiveRemote.Create(
                        LiveConnector, SelItemName,
                        LMREQ_ACCOUNTS_GET, LMREQ_ACCOUNTS_SET
                     ),
                     False {don't AutoFlush}
                  );
      end;

      if loadAll then begin
         LoadRCProfiles;
         LoadAccounts;
         LoadMailLists;
      end;

      SettingsToForm;

      Result := True;

   except
      on E: Exception do begin
         HMessageDlg( 'Error:' + CRLF + E.Message,
                      'Load Settings', mtError, [mbOK] );
      end;
   end;
end;

function TfrmConfAcc.SaveSettings: Boolean;
begin
   Result := True;

   try
      if SelItemType = selNone       then exit;
      if SelItemName = ''            then exit;
      if not Assigned( SelSettings ) then exit;

      FormToSettings;
      SelSettings.Flush;

   except
      on E: Exception do begin
         HMessageDlg( 'Error:' + CRLF + E.Message,
                      'Save Settings', mtError, [mbOK] );
         Result := False;
      end;
   end;
end;

procedure TfrmConfAcc.btnRefreshClick(Sender: TObject);
begin
   btnOK.Enabled := LoadSettings( True );
end;

procedure TfrmConfAcc.btnOKClick(Sender: TObject);
begin
   if not SaveSettings then ModalResult := mrNone;
end;

procedure TfrmConfAcc.btnApplyClick(Sender: TObject);
begin
   SaveSettings;
end;

procedure TfrmConfAcc.lbItemsClick(Sender: TObject);
begin
   with lbItems do begin
      if ( ItemIndex < 0 ) or ( ItemIndex >= Items.Count ) then begin
         SelItemType := selNone;
         SelItemName := '';
      end else begin
         SelItemType := Integer( Items.Objects[ ItemIndex ] );
         SelItemName := AccountName( ItemIndex );
      end;
   end;

   LoadSettings( False );

   grpAccount.Enabled := ( SelItemType <> selNone );
   grpNews   .Enabled := ( SelItemType <> selNone );
   grpMail   .Enabled := ( SelItemType <> selNone );
end;

procedure TfrmConfAcc.popItemsPopup(Sender: TObject);
begin
   with lbItems do begin
      mnuItemDel.Enabled := ( ItemIndex >= 0 ) and ( ItemIndex < Items.Count );
   end;
end;

procedure TfrmConfAcc.DoAddAccount( const AccountToCopy: String );
var  Nam, s, invalid: String;
     Idx, i: Integer;
begin
   Nam := '';
   if not InputDlgStr( 'Add account',
                       'Username (only letters, digits, "." and "_" allowed):',
                       Nam, 0 ) then exit;
   if Nam = '' then exit;

   invalid := '';
   for i := 1 to length( Nam ) do begin
      if (Nam[i] >= 'a') and (Nam[i] <= 'z') then continue;
      if (Nam[i] >= 'A') and (Nam[i] <= 'Z') then continue;
      if (Nam[i] >= '0') and (Nam[i] <= '9') then continue;
      if (Nam[i] = '.') or (Nam[i] = '_') or (Nam[i] = '-') then continue;
      invalid := invalid + Nam[i];
   end;

   if length( invalid ) > 0 then begin
      HMessageDlg( 'The following characters in "' + Nam
                   + '" are not allowed in account names: "' + Invalid + '"!',
                   'Add account', mtWarning, [mbOK] );
      exit;
   end;

   Idx := IndexOfAccount( Nam );
   if Idx >= 0 then begin
      HMessageDlg( 'Username "' + Nam + '" already exists!',
                   'Add account', mtWarning, [mbOK] );
      exit;
   end;

   s := Nam + CRLF;
   if AccountToCopy <> '' then s := s + AccountToCopy + CRLF;
   LiveConnector.RCLiveRequestOK(
      TLiveMsg.Create( LMREQ_ACCOUNTS_ADD, s )
   );

   LoadItemList;
   btnRefresh.Click;

   Idx := IndexOfAccount( Nam );
   if Idx >= 0 then begin lbItems.ItemIndex:=Idx; lbItemsClick(nil) end;
end;

procedure TfrmConfAcc.mnuItemAddClick(Sender: TObject);
begin
   DoAddAccount( '' );
end;

procedure TfrmConfAcc.mnuItemAddCloneClick(Sender: TObject);
var  Idx: Integer;
     Nam: String;
begin
   Idx := lbItems.ItemIndex;
   if ( Idx < 0 ) or ( Idx >= lbItems.Items.Count ) then exit;
   Nam := AccountName( Idx );
   DoAddAccount( Nam );
end;

procedure TfrmConfAcc.mnuItemDelClick(Sender: TObject);
var  Idx: Integer;
     Nam, s: String;
begin
   Idx := lbItems.ItemIndex;
   if ( Idx < 0 ) or ( Idx >= lbItems.Items.Count ) then exit;
   Nam := AccountName( Idx );

   if Nam = 'admin' then begin
      HMessageDlg( 'Account "admin" can''t be deleted!',
                   mtError, [mbAbort] );
      exit;
   end;

   s := Format( 'Delete account "%s" now?', [Nam] );
   if HMessageDlg( PChar(s), mtConfirmation, [mbYes,mbNo] ) <> mrYes then exit;

   if not LiveConnector.RCLiveRequestOK(
             TLiveMsg.Create( LMREQ_ACCOUNTS_DEL, Nam )
          ) then begin
      HMessageDlg( 'Account "' + Nam + '" could not be deleted!',
                   mtError, [mbAbort] );
   end;

   LoadItemList;
   btnRefresh.Click;

   if Idx >= lbItems.Items.Count then dec( Idx );
   if Idx >= 0 then begin lbItems.ItemIndex:=Idx; lbItemsClick(nil) end;
end;

procedure TfrmConfAcc.btnPWChangeClick(Sender: TObject);
var  Pass, Pass2: String;
begin
   Pass  := '';
   Pass2 := '';

   if not InputDlgPwd( SelItemName, 'Password:', Pass, 0 ) then exit;
   if not InputDlgPwd( SelItemName, 'Repeat Password:', Pass2, 0 ) then exit;

   if Pass <> Pass2 then begin
      HMessageDlg( 'Given passwords were not equal!',
                   SelItemName, mtWarning, [mbOK] );
      exit;
   end;

   SelSettings.SetStr( apPassword, Pass );
   btnApplyClick( nil );
   btnRefreshClick( nil );
end;

procedure TfrmConfAcc.btnPWClearClick(Sender: TObject);
var  s: String;
begin
   s := Format( 'Clear password for account "%s"?', [SelItemName] );
   if HMessageDlg( PChar(s), mtConfirmation, [mbYes,mbNo] ) <> mrYes then exit;
   SelSettings.SetStr( apPassword, ACTPW_NOACCESS );
   btnApplyClick( nil );
   btnRefreshClick( nil );
end;

procedure TfrmConfAcc.lbItemsDrawItem(Control: TWinControl; Index: Integer;
  Rect: TRect; State: TOwnerDrawState);
var  s: String;
begin
   s := (Control as TListBox).Items[Index];
   ArgsWhSpaceDQuoted( s, Args, 3 );
   s := Args[0] + ' (' + Args[2] + ', ID ' + Args[1] + ')';

   with (Control as TListBox).Canvas do begin
      FillRect( Rect );
      if odSelected in State then Font.Style := [ fsBold ]
                             else Font.Style := [ ];
      TextOut( Rect.Left+1, Rect.Top+1, s );
   end;
end;

function TfrmConfAcc.IndexOfAccount( const Name: String): Integer;
var  Idx: Integer;
begin
   Result := -1;
   for Idx := 0 to lbItems.Items.Count - 1 do begin
      if AccountName( Idx ) = Name then begin Result:=Idx; break end;
   end;
end;

function TfrmConfAcc.AccountName( const Index: Integer ): String;
begin
   ArgsWhSpaceDQuoted( lbItems.Items[Index], Args, 3 );
   Result := Args[0];
end;

end.

