unit fActions;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ExtCtrls, uConst, uType, cHscAction;

type
  TfrmActions = class(TForm)
    grpMaster: TGroupBox;
    Label6: TLabel;
    btnEdit: TButton;
    lbActions: TListBox;
    grpDetail: TGroupBox;
    btnSave: TButton;
    btnUndo: TButton;
    Label7: TLabel;
    lblAction: TLabel;
    Label9: TLabel;
    cbExecute: TComboBox;
    Label10: TLabel;
    ckWait: TCheckBox;
    Label11: TLabel;
    ckLock: TCheckBox;
    Label12: TLabel;
    ckSilent: TCheckBox;
    procedure FormCreate(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormDestroy(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure btnEditClick(Sender: TObject);
    procedure btnSaveClick(Sender: TObject);
    procedure btnUndoClick(Sender: TObject);
    procedure lbActionsClick(Sender: TObject);
    procedure lbActionsDrawItem(Control: TWinControl; Index: Integer;
      Rect: TRect; State: TOwnerDrawState);
    procedure FormResize(Sender: TObject);
  private
    { Private-Deklarationen }
    Editing: Boolean;
    EditActions: array[THscActionTypes] of THscAction;
    procedure RefreshActions;
    procedure ShowSelected;
    procedure ReflectEditmode;
  public
    { Public-Deklarationen }
    class function ShowDialog( AOwner: TComponent ): Integer;
  end;

var
  frmActions: TfrmActions;

implementation

{$R *.DFM}

uses uTools, uGlobal, cLiveMsg, dInput, dDialogs, cUserTasks;

class function TfrmActions.ShowDialog( AOwner: TComponent ): Integer;
begin
   with TfrmActions.Create( AOwner ) do try
      Result := ShowModal;
   finally Free end;
end;


procedure TfrmActions.FormCreate(Sender: TObject);
var  Reply: TLiveMsg;
     SL: TStringList;
     i: Integer;
     s, s1: String;
     act: THscActionTypes;
begin
   GrowDpiForm( self );

   // Fill combo with possible actions
   SL := TStringList.Create;
   try

      // No Action
      cbExecute.Items.AddObject( '', Pointer(0) );

      // User Tasks
      Reply := LiveConnector.RCLiveRequest(
         TLiveMsg.Create( LMREQ_USERTASKS_LIST, '' )
      );
      if Assigned( Reply ) then try
         if Reply.MsgType = LMREP_OK then begin
            SL.Text := Reply.MsgData;
            for i := 0 to SL.Count - 1 do begin
               cbExecute.Items.AddObject( SL[i] + USERTASK_EXTENSION, Pointer(1) );
            end;
         end;
      finally FreeAndNil(Reply) end;

      // Scripts
      Reply := LiveConnector.RCLiveRequest(
                  TLiveMsg.Create( LMREQ_SCRIPT_DIR, '' )
               );
      if Assigned( Reply ) then try
         if ( Reply.MsgType = LMREP_OK ) then begin
            SL.Text := Reply.MsgData;
            for i := 0 to SL.Count - 1 do begin
               s  := SL[i]; // <time> TAB <size> TAB <path/name>
               s1 := UnDQuoteStr( NextSepPart( s, TAB ) );
               s1 := UnDQuoteStr( NextSepPart( s, TAB ) );
               s1 := UnDQuoteStr( NextSepPart( s, TAB ) );
               if (Pos('.hsc',s1) > 0) and (Pos('\',s1) = 0) then begin
                  cbExecute.Items.AddObject( s1, Pointer(2) );
               end;
            end;
         end;
      finally FreeAndNil(Reply) end;

   finally
      SL.Free;
   end;

   // Prepare action array and listbox
   for act := Low(THscActionTypes) to High(THscActionTypes) do begin
      EditActions[act] := nil;
      if act <> actIgnore then begin
         EditActions[act] := THscAction.Create(
            THscActionNames[act], '', True, False, False
         );
         lbActions.Items.AddObject( THscActionNames[act], EditActions[act] );
      end;
   end;

   // Load and show actions
   Editing := False;
   ReflectEditmode;
   RefreshActions;
   ShowSelected;
end;

procedure TfrmActions.FormResize(Sender: TObject);
begin
   grpMaster.Width  := ClientWidth - 2 * grpMaster.Left;
   grpMaster.Height := ClientHeight - 3 * grpMaster.Top - grpDetail.Height;
   grpDetail.Width  := ClientWidth - 2 * grpDetail.Left;
   grpDetail.Top    := ClientHeight - grpDetail.Height - grpMaster.Top;

   lbActions.Invalidate;
end;

procedure TfrmActions.FormCloseQuery( Sender: TObject; var CanClose: Boolean );
var  i: Integer;
begin
   if Editing then begin
      i := HMessageDlg( 'Save changes?', mtConfirmation, [mbYes,mbNo,mbCancel] );
      if i = mrYes then btnSaveClick(nil);
      if i = mrNo  then btnUndoClick(nil);
      if (i = mrCancel) or Editing then CanClose := False;
   end;
end;

procedure TfrmActions.FormClose(Sender: TObject; var Action: TCloseAction);
begin
   Action := caFree;
end;

procedure TfrmActions.FormDestroy(Sender: TObject);
var  act: THscActionTypes;
begin
   for act := Low(THscActionTypes) to High(THscActionTypes) do begin
      if Assigned( EditActions[act] ) then EditActions[act].Free;
   end;
end;

procedure TfrmActions.ReflectEditmode;
begin
   lbActions.Enabled := not Editing;
   btnEdit.Enabled   := not Editing;

   btnSave.Enabled   := Editing;
   btnUndo.Enabled   := Editing;
   cbExecute.Enabled := Editing;
   ckWait.Enabled    := Editing;
   ckLock.Enabled    := Editing;
   ckSilent.Enabled  := Editing;
end;

procedure TfrmActions.RefreshActions;
var  Reply: TLiveMsg;
     SL: TStringList;
     act: THscActionTypes;
     ex, k, v: String;
     cw, cl, cs, found: Boolean;
     i, j: Integer;
begin
   SL := TStringList.Create;
   try

      // load action definition file
      Reply := LiveConnector.RCLiveRequest(
                  TLiveMsg.Create( LMREQ_FILE_GET, CFGFILE_HSCACTIONS )
               );
      if Assigned( Reply ) then try
         if ( Reply.MsgType = LMREP_OK ) then SL.Text := Reply.MsgData;
      finally FreeAndNil(Reply) end;

      // show actions
      for act := Low(THscActionTypes) to High(THscActionTypes) do begin
         if EditActions[act] <> nil then begin

            ex := '';
            cw := True;
            cl := False;
            cs := False;

            found := False;
            for i := 0 to SL.Count - 1 do begin
               if copy( SL[i], 1, 1 ) = '[' then begin
                  if found then break;
                  if SL[i] = '[' + THscActionNames[act] + ']' then begin
                     found := True;
                  end;
               end else if found then begin
                  j := Pos( '=', SL[i] );
                  if j > 0 then begin
                     k := LowerCase( copy( SL[i], 1, j-1 ) );
                     v := LowerCase( copy( SL[i], j+1, MaxInt ) );
                     if k = 'script' then ex := v;
                     if k = 'wait'   then cw := (v = '1');
                     if k = 'lock'   then cl := (v = '1');
                     if k = 'silent' then cs := (v = '1');
                  end;
               end;
            end;

            EditActions[act].ActionFile := ex;
            EditActions[act].WaitForEnd := cw;
            EditActions[act].LockedExec := cl;
            EditActions[act].ExecSilent := cs;

         end;
      end;

   finally
      SL.Free;
      lbActions.Invalidate;
   end;
end;

procedure TfrmActions.ShowSelected;
var  e: THscAction;
     s: String;
     i: Integer;
begin
   e := nil;
   if (lbActions.ItemIndex >= 0) and
      (lbActions.ItemIndex < lbActions.Items.Count) then begin
      e := THscAction( lbActions.Items.Objects[ lbActions.ItemIndex ] );
   end;

   if e = nil then begin

      lblAction.Caption   := '?';
      cbExecute.ItemIndex := 0;
      ckWait.Checked      := True;
      ckLock.Checked      := True;
      ckSilent.Checked    := False;

   end else begin

      s := e.ActionFile;
      i := 0;
      if s <> '' then begin
         i := cbExecute.Items.IndexOf( s );
         if i < 0 then begin
            s := 'Unknown execute-value "' + e.ActionFile + '" at action "'
               + e.ActionName + '"! Please check before saving!';
            HMessageDlg( s, mtWarning, [mbOk] );
         end;
      end;

      lblAction.Caption   := e.ActionName;
      cbExecute.ItemIndex := i;
      ckWait.Checked      := e.WaitForEnd;
      ckLock.Checked      := e.LockedExec;
      ckSilent.Checked    := e.ExecSilent;

   end;
end;

procedure TfrmActions.btnEditClick(Sender: TObject);
begin
   if lbActions.ItemIndex < 0 then exit;
   if lbActions.ItemIndex >= lbActions.Items.Count then exit;

   Editing := True;
   ReflectEditmode;
   cbExecute.SetFocus;
end;

procedure TfrmActions.btnSaveClick(Sender: TObject);
var  i: Integer;
     e: THscAction;
     SL: TStringList;
     act: THscActionTypes;
     ex: String;
     cw, cl, cs: Boolean;
begin
   i := lbActions.ItemIndex;
   if i < 0 then exit;
   if i >= lbActions.Items.Count then exit;

   e := THscAction( lbActions.Items.Objects[ i ] );
   e.ActionFile := cbExecute.Text;
   e.WaitForEnd := ckWait.Checked;
   e.LockedExec := ckLock.Checked;
   e.ExecSilent := ckSilent.Checked;

   // redraw modified list
   lbActions.Invalidate;

   // create and save action definition file
   SL := TStringList.Create;
   try
      SL.Add( '# ' + DateTimeToStr( Now ) );
      SL.Add( '' );
      
      for act := Low(THscActionTypes) to High(THscActionTypes) do begin
         if EditActions[act] <> nil then begin

            SL.Add( '[' + THscActionNames[act] + ']' );

            ex := EditActions[act].ActionFile;
            cw := EditActions[act].WaitForEnd;
            cl := EditActions[act].LockedExec;
            cs := EditActions[act].ExecSilent;

            if ex <> '' then begin
               SL.Add( 'Script=' + ex );
               SL.Add( 'Wait='   + iif( cw, '1', '0' ) );
               SL.Add( 'Lock='   + iif( cl, '1', '0' ) );
               SL.Add( 'Silent=' + iif( cs, '1', '0' ) );
            end;

            SL.Add( '' );

         end;
      end;

      // save action definition file
      if not LiveConnector.RCLiveRequestOK(
         TLiveMsg.Create( LMREQ_FILE_SET,
                          CFGFILE_HSCACTIONS + #13#10 + SL.Text )
      ) then begin
         HMessageDlg( 'Actions file "' + CFGFILE_HSCACTIONS + '" could not be saved!',
                      'Update Actions', mtError, [mbOK] );
      end;

   finally SL.Free end;

   Editing := False;
   ReflectEditmode;
   lbActions.SetFocus;
end;

procedure TfrmActions.btnUndoClick(Sender: TObject);
begin
   ShowSelected;

   Editing := False;
   ReflectEditmode;
   lbActions.SetFocus;
end;

procedure TfrmActions.lbActionsClick(Sender: TObject);
begin
   ShowSelected;
end;

procedure TfrmActions.lbActionsDrawItem( Control: TWinControl;
  Index: Integer; Rect: TRect; State: TOwnerDrawState );
var  e: THscAction;
     w: Integer;
     s: String;
begin
   e := THscAction( lbActions.Items.Objects[ Index ] );

   with (Control as TListBox).Canvas do begin

      FillRect( Rect );

      TextOut( Rect.Left, Rect.Top, e.ActionName );
      
      w := TextWidth( e.ActionName ) + 3;
      if w < Control.ClientWidth div 3 then w := Control.ClientWidth div 3;
      Rect.Left := Rect.Left + w;
      TextOut( Rect.Left, Rect.Top, e.ActionFile );

      if e.ActionFile <> '' then begin
         w := TextWidth( e.ActionFile ) + 3;
         Rect.Left := Rect.Left + w;
         s := '';
         if not e.WaitForEnd then s := s + 'NOWAIT   ';
         if not e.LockedExec then s := s + 'NOLOCK   ';
         if e.ExecSilent     then s := s + 'SILENT   ';
         if s <> '' then TextOut( Rect.Left, Rect.Top, Trim(s) );
      end;

   end;
end;

end.
