// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit dInput; // "Input"-dialog (string/password/integer-input, list-selection)

// ----------------------------------------------------------------------------
// Contains an input-dialog which either allows input of a single value
// (string, password or integer) or the selection of a list-entry.
// ----------------------------------------------------------------------------

interface

{$INCLUDE Compiler.inc}

uses Windows, SysUtils, Classes, Graphics, Forms, Controls, StdCtrls,
  Buttons, ExtCtrls;

type
  TInputDlgType = ( idtString, idtPassword, idtInteger, idtList );

  TInputDlg = class(TForm)
    OKBtn: TButton;
    CancelBtn: TButton;
    Bevel1: TBevel;
    lblPrompt: TLabel;
    emInput: TEdit;
    lbSelect: TListBox;
    procedure FormCreate(Sender: TObject);
    procedure OKBtnClick(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
    DlgType: TInputDlgType;
    IntMin, IntMax: LongInt;
    procedure PrepareForm( ACaption, APrompt, AValue: String;
                           AType: TInputDlgType;
                           AList: String );
  end;

function InputDlgStr( const ACaption, APrompt: String;
                      var Value: String; HelpID: Integer ): Boolean;

function InputDlgPwd( const ACaption, APrompt: String;
                      var Value: String; HelpID: Integer ): Boolean;

function InputDlgInt( const ACaption, APrompt: String;
                      var Value: LongInt;
                      AIntMin, AIntMax: LongInt; HelpID: Integer ): Boolean;

function InputDlgList( const ACaption, APrompt: String;
                       var Value: LongInt;
                       const AList: String; HelpID: Integer ): Boolean;

// ----------------------------------------------------------------------------

implementation

{$R *.DFM}

function ExecuteDialog( const ACaption, APrompt: String;
                        var   AValue           : String;
                        const AType            : TInputDlgType;
                        const AIntMin, AIntMax : LongInt;
                        const AList            : String ): Boolean;
var  Dlg: TInputDlg;
     i: Integer;
begin
   Result := False;
   Application.CreateForm( TInputDlg, Dlg );

   try
      Dlg.PrepareForm( ACaption, APrompt, AValue, AType, AList );
      Dlg.IntMin := AIntMin;
      Dlg.IntMax := AIntMax;

      if Dlg.ShowModal = mrOK then begin
         Result := True;
         if AType = idtList then begin
            for i:=0 to Dlg.lbSelect.Items.Count-1 do begin
               if Dlg.lbSelect.Selected[i] then begin
                  AValue := inttostr( i );
                  break;
               end;
            end;
         end else begin
            AValue := Dlg.emInput.Text;
         end;
      end;

   finally
      Dlg.Release;
   end;
end;

// ------------------------------------------------------ TInputDlgThread -----

type
  TInputDlgThread = class( TThread )
    private
       FCaption, FPrompt, FValue, FList: String;
       FIntMin, FIntMax: LongInt;
       FType: TInputDlgType;
       FResult: Boolean;
       procedure SyncDialog;
    public
       property Result: Boolean read FResult;
       property Value : String  read FValue;
       procedure Execute; override;
       constructor Create( const ACaption, APrompt, AValue: String;
                           const AType: TInputDlgType;
                           const AIntMin, AIntMax: LongInt;
                           const AList: String );
  end;

procedure TInputDlgThread.SyncDialog;
begin
   FResult := ExecuteDialog( FCaption, FPrompt, FValue, FType,
                             FIntMin, FIntMax, FList );
end;

procedure TInputDlgThread.Execute;
begin
   try Synchronize( SyncDialog ) finally Terminate end;
end;

constructor TInputDlgThread.Create( const ACaption, APrompt, AValue: String;
                                    const AType: TInputDlgType;
                                    const AIntMin, AIntMax: LongInt;
                                    const AList: String );
begin
   inherited Create( True );
   FCaption := ACaption;
   FPrompt  := APrompt;
   FValue   := AValue;
   FType    := AType;
   FIntMin  := AIntMin;
   FIntMax  := AIntMax;
   FList    := AList;
   Resume;
end;

// ------------------------------------------------------------ functions -----

function InputDlgSyncVCL( const ACaption, APrompt: String;
                          var Value: String;
                          AType: TInputDlgType;
                          AIntMin, AIntMax: LongInt;
                          const AList: String ): Boolean;
var  ThdDlg: TInputDlgThread;
begin
   if GetCurrentThreadID = MainThreadID then begin

      // running in VCL-thread, no need to synchronize
      Result := ExecuteDialog( ACaption, APrompt, Value, AType,
                               AIntMin, AIntMax, AList );

   end else begin

      // we need to synchronize dialog with VCL-thread
      ThdDlg := TInputDlgThread.Create( ACaption, APrompt, Value, AType,
                                        AIntMin, AIntMax, AList );
      try
         ThdDlg.WaitFor;
         Result := ThdDlg.Result;
         Value  := ThdDlg.Value;
      finally
         ThdDlg.Free;
      end;
      
   end;
end;

function InputDlgStr( const ACaption, APrompt: String;
                      var Value: String; HelpID: Integer ): Boolean;
begin
   Result := InputDlgSyncVCL( ACaption, APrompt, Value, idtString, -1, -1, '' );
end;

function InputDlgPwd( const ACaption, APrompt: String;
                      var Value: String; HelpID: Integer ): Boolean;
begin
   Result := InputDlgSyncVCL( ACaption, APrompt, Value, idtPassword, -1, -1, '' );
end;

function InputDlgInt( const ACaption, APrompt: String;
                      var Value: LongInt;
                      AIntMin, AIntMax: LongInt; HelpID: Integer ): Boolean;
var  sValue: String;
begin
   sValue := inttostr( Value );
   Result := InputDlgSyncVCL( ACaption, APrompt, sValue,
                              idtInteger, AIntMin, AIntMax, '' );
   if Result then Value := StrToIntDef( sValue, Value );
end;

function InputDlgList( const ACaption, APrompt: String;
                       var Value: LongInt;
                       const AList: String; HelpID: Integer ): Boolean;
var  sValue: String;
begin
   sValue := inttostr( Value );
   Result := InputDlgSyncVCL( ACaption, APrompt, sValue,
                              idtList, -1, -1, AList );
   if Result then Value := StrToIntDef( sValue, Value );
end;

// ------------------------------------------------------------ TInputDlg -----

procedure TInputDlg.PrepareForm( ACaption, APrompt, AValue: String;
                                 AType: TInputDlgType; AList: String );
var  MX, CH, TH, TW, i, p: Integer;
     s, NPrompt: String;
     X: TWinControl;
begin
     DlgType := AType;

     MX := lblPrompt.ClientWidth;
     CH := Canvas.TextHeight( 'Gg' );
     TH := 2;
     NPrompt := '';

     while APrompt<>'' do begin
        i := Pos( #13#10, APrompt );
        if i=0 then begin
           s := APrompt;
           APrompt := '';
        end else begin
           s := copy( APrompt, 1, i-1 );
           System.Delete( APrompt, 1, i+1 );
        end;

        if Canvas.TextWidth(s)<=MX then begin
           NPrompt := NPrompt + s + #13#10;
           inc( TH, CH );
        end else begin
           repeat
              p := 0;
              for i:=1 to length(s) do begin
                 if (Pos(s[i],' -,.;:!?' )>0) or (i=length(s)) then begin
                    TW := Canvas.TextWidth( copy(s,1,i) );
                    if TW>MX then begin
                       if p=0 then p:=i-1;
                       break;
                    end;
                    if TW<=MX then p:=i;
                 end;
              end;
              if p<=0 then p:=length(s);

              NPrompt := NPrompt + copy( s, 1, p ) + #13#10;
              inc( TH, CH );
              System.Delete( s, 1, p );
              if copy(s,1,1)=' ' then System.Delete( s, 1, 1 );
           until s='';
        end;
     end;

     Caption           := ACaption;
     lblPrompt.Height  := TH;
     lblPrompt.Caption := NPrompt;

     if DlgType=idtList then begin
        X := lbSelect;
        emInput.Visible  := False;
        lbSelect.Visible := True;
        lbSelect.Items.Text := AList;
        if lbSelect.Items.Count>0 then begin
           i := 0;
           p := strtointdef( AValue, 0 );
           if (p>=0) and (p<lbSelect.Items.Count) then i:=p;
           lbSelect.ItemIndex := i;
        end;
     end else begin
        X := emInput;
        emInput.Text := AValue;
     end;

     X.Top := lblPrompt.Top + lblPrompt.Height + 2;

     Bevel1.Height := X.Top + X.Height + 4;
     OKBtn.Top     := Bevel1.Top + Bevel1.Height + 4*2;
     CancelBtn.Top := OKBtn.Top;
     ClientHeight  := OKBtn.Top + OKBtn.Height + 4*2;

     if DlgType=idtPassword then emInput.PasswordChar:='*';
end;

procedure TInputDlg.FormCreate(Sender: TObject);
begin
     DlgType := idtString;
     IntMin := LongInt(MINLONG);
     IntMax := LongInt(MAXLONG);
end;

procedure TInputDlg.OKBtnClick(Sender: TObject);
var  l: LongInt;
begin
     case DlgType of
        idtInteger:
           try
              l := SysUtils.StrToInt(emInput.Text);
              if l<IntMin then ModalResult:=mrNone;
              if l>IntMax then ModalResult:=mrNone;
           except ModalResult:=mrNone end;
     end;

     if ModalResult=mrNone then begin
        emInput.Color := clRed;
        emInput.SetFocus;
     end;
end;

// ----------------------------------------------------------------------------

end.
