// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cSettingsHdlLive; // settings handler for Live connections

interface

uses cSettingsDef, cSettings, cLiveConnector;

type
   TSettingsHandler_LiveRemote = class( TSettingsHandler_InMemory )
      // Handler for plain settings, that are transferred by LiveConnector.
      // Also supports qualified settings, but only one qualifier per instance.
      protected
         FLiveConnector: TLiveConnector;
         FLQReqGet, FLQReqSet: Integer;
         FQualifier: String;

         procedure DoSet( const Qualifier : String;
                          const SettingDef: TSettingDef;
                          const NewValue  : String ); override;

      public
         constructor Create( const ALiveConnector: TLiveConnector;
                             const AQualifier: String;
                             const ALQReqGet, ALQReqSet: Integer );
   end;

implementation

uses SysUtils, uConst, cLiveQueue, cLiveMsg;

// ------------------------------------------ TSettingsHandler_LiveRemote -----

constructor TSettingsHandler_LiveRemote.Create( const ALiveConnector: TLiveConnector;
                                                const AQualifier: String;
                                                const ALQReqGet, ALQReqSet: Integer );

   function LoadInitialList: String;
   // load all settings to be buffered in FList
   var  Reply: TLiveMsg;
   begin
      Result := '';
      Reply := FLiveConnector.RCLiveRequest(
                  TLiveMsg.Create( FLQReqGet, FQualifier )
               );
      if Assigned( Reply ) then try
         if ( Reply.MsgType = LMREP_OK ) then Result := Reply.MsgData;
      finally Reply.Free end;
   end;

begin
   FLiveConnector := ALiveConnector;
   FLQReqGet := ALQReqGet;
   FLQReqSet := ALQReqSet;
   FQualifier := AQualifier;
   inherited Create( LoadInitialList );
end;

procedure TSettingsHandler_LiveRemote.DoSet( const Qualifier: String;
                                             const SettingDef: TSettingDef;
                                             const NewValue: String );
var  ReqStr: String;
begin
   // update internal FList for DoGet()
   inherited;

   // transfer changed setting
   ReqStr := inttostr( SettingDef.ID ) + '=' + NewValue;
   if FQualifier <> '' then ReqStr := FQualifier + CRLF + ReqStr;

   FLiveConnector.RCLiveRequestOK(
      TLiveMsg.Create( FLQReqSet, ReqStr )
   );
end;

end.
