// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cLiveConnector; 

interface

uses SysUtils, Classes, uConst, cLiveMsg, cLiveQueue, cClientReCo, SyncObjs;

type
   ELiveConnectorError = class( Exception );

   TLiveInfoHandlerProc = procedure of object;

   TLiveInfoHandlerThread = class( TThread )
      // Thread, that waits for new messages in the info queue and proceeds
      // them to the given handler method. The handler is assumed to be a
      // GUI/VCL one, so it is called with .Synchronize.
      protected
         FQueueInf  : TLiveQueue;
         FHandlerInf: TLiveInfoHandlerProc;

         procedure Execute; override;

      public
         constructor Create( AQueueInf  : TLiveQueue;
                             AHandlerInf: TLiveInfoHandlerProc );
   end;

   TLiveReceiverThread = class( TThread )
      // Thread, that receives all incoming Live messages. It doesn't handle
      // them, but just places them in the given info or reply queue.
      protected
         FQueueInf, FQueueRep: TLiveQueue;
         FReCoClient: TReCoClient;

         procedure Execute; override;

      public
         procedure ConnectClient( AReCoClient: TReCoClient );
         procedure DisconnectClient;

         constructor Create( AQueueInf, AQueueRep: TLiveQueue );
   end;

   TLiveConnector = class
      protected
         FLock                 : TCriticalSection;
         FQueueInf, FQueueRep  : TLiveQueue;
         FLiveInfoHandlerThread: TLiveInfoHandlerThread;
         FLiveReceiverThread   : TLiveReceiverThread;
         FReCoClient           : TReCoClient;

      public
         property QueueInf: TLiveQueue read FQueueInf;
         property QueueRep: TLiveQueue read FQueueRep;

         function  RCLiveConnect( const RCServer, RCPort,
                                        RCUser, RCPass: String;
                                  out   ErrMsg: String ): Boolean;
         procedure RCLiveDisconnect;
         function  RCLiveIsConnected: Boolean;

         function  RCLiveRequest  ( Request: TLiveMsg ): TLiveMsg;
         function  RCLiveRequestOK( Request: TLiveMsg ): Boolean;

         constructor Create( AHandlerInf: TLiveInfoHandlerProc );
         destructor Destroy; override;
   end;


implementation

uses Windows, uGlobal, uTools, uMD5;


// ----------------------------------------------- TLiveInfoHandlerThread -----

constructor TLiveInfoHandlerThread.Create( AQueueInf  : TLiveQueue;
                                           AHandlerInf: TLiveInfoHandlerProc );
begin
   inherited Create( True {=Suspended} );
   FQueueInf   := AQueueInf;
   FHandlerInf := AHandlerInf;
end;

procedure TLiveInfoHandlerThread.Execute;
begin
   while not Terminated do begin
      if FQueueInf.WaitFor( INFINITE ) then begin
         if not Terminated then Synchronize( FHandlerInf );
      end;
   end;
end;


// -------------------------------------------------- TLiveReceiverThread -----

constructor TLiveReceiverThread.Create( AQueueInf, AQueueRep: TLiveQueue );
begin
   inherited Create( True {=Suspended} );
   FQueueInf   := AQueueInf;
   FQueueRep   := AQueueRep;
   FReCoClient := nil;
end;

procedure TLiveReceiverThread.ConnectClient( AReCoClient: TReCoClient );
begin
   FReCoClient := AReCoClient;
   if Suspended then Resume;
end;

procedure TLiveReceiverThread.DisconnectClient;
begin
   FReCoClient := nil;
end;

procedure TLiveReceiverThread.Execute;
var  TransferStr: String;
     LiveMsg    : TLiveMsg;
begin
   while not Terminated do begin

      // while <connected>
      while Assigned( FReCoClient ) and
            Assigned( FReCoClient.TCPClient ) and
            FReCoClient.TCPClient.Connected do try

         // wait for next line
         TransferStr := FReCoClient.TCPClient.ReadLn( #10,
                                                      TimeoutCommandSec*1000 );

         if Terminated then break;
         if not Assigned( FReCoClient ) then break;
         if not Assigned( FReCoClient.TCPClient ) then break;

         // still connected, so check if we received a message
         if not FReCoClient.TCPClient.ReadLnTimedOut then begin

            // place received message in appropriate queue
            LiveMsg := TLiveMsg.Create( TransferStr, RCSessionKey );

            if ( LiveMsg.MsgType and LMINF ) = LMINF then begin
               // info from server -> add to info queue
               FQueueInf.Add( LiveMsg );

            end else if ( LiveMsg.MsgType and LMREP ) = LMREP then begin
               // reply from server -> add to reply queue
               FQueueRep.Add( LiveMsg );

            end else begin
               // unknown/unsupported type -> place note in info queue
               FQueueInf.Add( TLiveMsg.Create(
                  LMXXX_HC_GUI_LOG, 'Received unknown message type: '
                  + inttohex( LiveMsg.MsgType, 4 ) + ' ' + LiveMsg.MsgData
               ) );
               LiveMsg.Free;
            end;

         end;

      except
         // on E: EIdReadTimeout          do ; // timeout
         // on E: EIdConnClosedGracefully do ; // connection closed
         // on E: Exception               do ; // unknown

         on E: Exception do if not Terminated then begin
            FQueueInf.Add( TLiveMsg.Create(
               LMXXX_HC_GUI_LOG, 'Live Receiver exception: ' + E.Message
            ) );
            Sleep( 500 );
         end;

      end; // while <connected> do try

      // connection lost
      if not Terminated then begin

         // notify GUI that connection is dead
         PostMessage( MainWindowHWnd, WM_GUI_CHECKCONNECTION, 0, 0 );

         // suspend until ConnectClient() is called again
         FQueueInf.Add( TLiveMsg.Create(
            LMXXX_HC_GUI_LOG, 'Connection lost - Live Receiver suspended.'
         ) );
         Suspend;

      end;

   end; // while not Terminated

   FReCoClient := nil;
end;


// ------------------------------------------------------- TLiveConnector -----

constructor TLiveConnector.Create( AHandlerInf: TLiveInfoHandlerProc );
begin
   inherited Create;

   FReCoClient := nil;
   FLock := TCriticalSection.Create;

   // create queues
   FQueueInf := TLiveQueue.Create( 5000 );
   FQueueRep := TLiveQueue.Create( 5000 );

   // prepare receiver thread (suspended)
   FLiveReceiverThread := TLiveReceiverThread.Create( FQueueInf, FQueueRep );

   // prepare info handler thread (suspended)
   FLiveInfoHandlerThread := TLiveInfoHandlerThread.Create( FQueueInf, AHandlerInf );
end;

destructor TLiveConnector.Destroy;
begin
   // prepare receiver termination
   FLiveReceiverThread.Terminate;
   if FLiveReceiverThread.Suspended then FLiveReceiverThread.Resume;

   // prepare info handler termination
   FLiveInfoHandlerThread.Terminate;
   if FLiveInfoHandlerThread.Suspended then FLiveInfoHandlerThread.Resume;

   // terminate RC connection
   try if Assigned( FReCoClient ) then RCLiveDisconnect except end;

   // wake up threads waiting for new queue entries
   FQueueInf.Add( nil );
   FQueueRep.Add( nil );

   // terminate receiver
   if FLiveReceiverThread.Suspended then FLiveReceiverThread.Resume;
   FLiveReceiverThread.WaitFor;
   FLiveReceiverThread.Free;

   // terminate info handler
   if FLiveInfoHandlerThread.Suspended then FLiveInfoHandlerThread.Resume;
   FLiveInfoHandlerThread.WaitFor;
   FLiveInfoHandlerThread.Free;

   // destroy queues
   FQueueInf.Free;
   FQueueRep.Free;

   FLock.Free;

   inherited Destroy;
end;

function TLiveConnector.RCLiveConnect( const RCServer, RCPort,
                                             RCUser, RCPass: String;
                                       out   ErrMsg: String ): Boolean;
var  ResLine: String;
     i: Integer;
     UKey, SKey: String;
begin
   Result := False;
   ErrMsg := 'Connect failed: ';

   if Assigned( FReCoClient ) then begin
      if FReCoClient.Connected then begin Result:=True; exit end;
      RCLiveDisconnect;
   end;

   FReCoClient := TReCoClient.Create( RCServer, RCPort, RCUser, RCPass );

   try
      // connect to HService's Remote Control server
      if FReCoClient.Connect < 400 then begin

         // authenticate
         if FReCoClient.Authenticate < 400 then begin

            // enter Live Mode
            if FReCoClient.SendCmd( 'LIVE ON ENCRYPT', ResLine ) < 400 then begin

               // get session key for en-/decryption
               RCSessionKey := '(none received)';
               i := Pos( '<', ResLine );
               if i > 0 then begin
                  SKey := copy( ResLine, i+1, 255 );
                  i := Pos( '>', SKey );
                  if i > 0 then begin
                     SetLength( SKey, i-1 );
                     UKey := FReCoClient.PassMD5;
                     HMAC_MD5( SKey, UKey, RCSessionKey );
                     Result := True;
                  end;
               end;

            end else begin
               ErrMsg := ErrMsg + 'LIVE ON -> ' + FReCoClient.LastRes;
            end;

         end else begin
            ErrMsg := ErrMsg + 'AUTH -> ' + FReCoClient.LastRes;
         end;

      end else begin
         ErrMsg := ErrMsg + 'Connect -> ' + FReCoClient.LastRes;
      end;

   except
      on E: Exception do ErrMsg := ErrMsg + E.Message;
   end;

   if Result then begin

      // Success: Start receiver and info handler threads for new connection.
      FLiveReceiverThread.ConnectClient( FReCoClient );
      if FLiveInfoHandlerThread.Suspended then FLiveInfoHandlerThread.Resume;

   end else begin

      // Failed: Disconnect if still connected.
      if Assigned( FReCoClient ) then begin
         try if FReCoClient.Connected then FReCoClient.Disconnect except end;
         try FReCoClient.Free except end;
         FReCoClient := nil;
      end;
      
   end;
end;

procedure TLiveConnector.RCLiveDisconnect;
begin
   if not Assigned( FReCoClient ) then exit;

   try
      FLiveReceiverThread.DisconnectClient;
      try if FReCoClient.Connected then FReCoClient.Disconnect except end;
      try FReCoClient.Free except end;

   finally
      FReCoClient := nil;
   end;
end;

function TLiveConnector.RCLiveIsConnected: Boolean;
begin
   try
      Result := ( Assigned(FReCoClient) and FReCoClient.Connected );
   except
      Result := False;
   end;
end;

function TLiveConnector.RCLiveRequest( Request: TLiveMsg ): TLiveMsg;
begin
   Result := nil;
   FLock.Enter;
   try
      try
         if not RCLiveIsConnected then begin
            raise ELiveConnectorError.Create( 'Not connected!' );
            exit;
         end;

         // send encrypted request
         Request.SessionKey := RCSessionKey;
         FReCoClient.TCPClient.WriteLn( Request.TransferStr );

         // wait until reply is placed in reply queue
         if QueueRep.WaitFor( TimeoutCommandSec * 1000 ) then begin

            // success, remove reply from queue and return it as result
            QueueRep.Get( Result );

         end else begin

            // failed (timeout or connection lost)
            if not RCLiveIsConnected then begin
               raise ELiveConnectorError.Create( 'Request failed, connection lost!' );
            end;

         end;

      except
         RCLiveDisconnect;
         raise;
      end;

   finally
      FLock.Leave;
      Request.Free;
   end;
end;

function TLiveConnector.RCLiveRequestOK( Request: TLiveMsg ): Boolean;
var  Reply: TLiveMsg;
begin
   Result := False;
   Reply := RCLiveRequest( Request );
   if Assigned( Reply ) then begin
      Result := ( Reply.MsgType = LMREP_OK );
      Reply.Free;
   end;
end;

// ----------------------------------------------------------------------------

end.
