// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cClientReCo; 

interface

uses SysUtils, Classes, IdTCPClient, uTools;

type
   TReCoClient = class
      private
         FServer, FPort, FUser, FPass: String;
         FConnection: TIdTCPClient;
         FLastCmd, FLastRes: String;

         function GetPassMD5: String;

      public
         property Server : String read FServer;
         property Port   : String read FPort;
         property User   : String read FUser;
         property PassMD5: String read GetPassMD5;

         property TCPClient: TIdTCPClient read FConnection;

         property LastCmd: String read FLastCmd;
         property LastRes: String read FLastRes;

         function SendCmd( const Cmd: String ): Integer; overload;
         function SendCmd( const Cmd: String;
                           var   ResLine: String ): Integer; overload;
         function SendCmdList( const Cmd: String;
                               var   List: String ): Integer; overload;
         function SendCmdList( const Cmd: String;
                               var   List, ResLine: String ): Integer; overload;

         function SendList( const List: String; var ResLine: String ): Integer;

         function  Connect: Integer;
         procedure Disconnect;
         function  Connected: Boolean;

         function  Authenticate: Integer;

         constructor Create( const AServer, APort, AUser, APass: String );
         destructor  Destroy; override;
   end;

implementation

uses uEncoding, uMD5, uSASL, uConst, uGlobal;

{ TReCoClient }

constructor TReCoClient.Create( const AServer, APort, AUser, APass: String );
begin
   inherited Create;
   FConnection := nil;
   FServer := AServer;
   FPort   := APort;
   FUser   := AUser;
   FPass   := APass;
end;

destructor TReCoClient.Destroy;
begin
   if Assigned(FConnection) then Disconnect;
   inherited Destroy;
end;

function TReCoClient.GetPassMD5: String;
begin
   Result := MD5OfStr( FPass );
end;

function TReCoClient.Connect: Integer;
var  ResLine: String;
begin
   if Assigned( FConnection ) then Disconnect;

   FConnection := TIdTCPClient.Create( nil );
   FConnection.Host := FServer;
   FConnection.Port := strtointdef( FPort, 23 );
   FConnection.MaxLineLength := 0;

   FConnection.Connect;
   ResLine := TrimCrLf( FConnection.ReadLn( #10, TimeoutConnectSec*1000 ) );

   if FConnection.ReadLnTimedout then begin
      Result   := 999;
      FLastCmd := '(connect)';
      FLastRes := '999 Timeout';
   end else begin
      Result   := strtointdef( copy(ResLine,1,3), 999 );
      FLastCmd := '(connect)';
      FLastRes := ResLine;
   end;
end;

procedure TReCoClient.Disconnect;
begin
   if not Assigned( FConnection ) then exit;

   if FConnection.Connected then begin
      try FConnection.Disconnect except end;
   end;

   FConnection.Free;
   FConnection := nil;
end;

function TReCoClient.Connected: Boolean;
begin
   try
      Result := ( Assigned(FConnection) and FConnection.Connected );
   except
      Result := False;
   end;
end;

function TReCoClient.Authenticate: Integer;
var  ResLine, Challenge, Response: String;
begin
   if (FUser='') or (FPass='') then begin

      FLastCmd := '(auth.)';
      FLastRes := '999 Username or password missing!';
      Result   := 999;

   end else begin

      // Result := SendCmd( 'AUTH ' + FUser + ' ' + FPass );
      // if Result >= 400 then exit;

      Result := SendCmd( 'AUTH -SASL CRAM-MD5', ResLine );
      if Result >= 400 then exit;
      if (Result<>380) then begin Result:=990; exit end;

      Challenge := copy( ResLine, 5, 999 );
      Challenge := DecodeB64( Challenge[1], length(Challenge) );
      if Challenge='' then begin Result:=991; exit end;

      Response := AUTH_CRAM_MD5_Encode( Challenge, FUser, FPass );
      Result   := SendCmd( Response, ResLine );
      if Result >= 400 then exit;
      if Result <> 280 then begin Result:=992; exit end;
      
   end;
end;

function TReCoClient.SendCmd( const Cmd: String ): Integer;
var  ResLine: String;
begin
   Result := SendCmd( Cmd, ResLine );
end;

function TReCoClient.SendCmd( const Cmd: String; var ResLine: String ): Integer;
begin
   try
      FConnection.WriteLn( Cmd );
      ResLine := TrimCrLf( FConnection.ReadLn( #10, TimeoutCommandSec*1000 ) );
      if FConnection.ReadLnTimedout then begin
         Disconnect;
         ResLine := '999 Timeout';
      end;
   except
      on E: Exception do ResLine := '999 ERROR on "' + Cmd + '": ' + E.Message;
   end;

   Result   := strtointdef( copy(ResLine,1,3), 999 );
   FLastCmd := Cmd;
   FLastRes := ResLine;
end;

function TReCoClient.SendCmdList( const Cmd: String; var List: String ): Integer;
var  ResLine: String;
begin
   Result := SendCmdList( Cmd, List, ResLine );
end;

function TReCoClient.SendCmdList( const Cmd: String; var List, ResLine: String ): Integer;
var  s: String;
begin
   List   := '';
   Result := SendCmd( Cmd, ResLine );
   if Result >= 400 then exit;

   repeat

      s := TrimCrLf( FConnection.ReadLn( #10, TimeoutCommandSec*1000 ) );
      if FConnection.ReadLnTimedout then begin
         Disconnect;
         ResLine := '999 Timeout';
         break;
      end;

      if s = '.' then break;
      if copy( s, 1, 2 ) = '..' then System.Delete( s, 1, 1 );
      List := List + s + CRLF;

   until False;
end;

function TReCoClient.SendList( const List: String; var ResLine: String ): Integer;
var  TS: TStringList;
     i: Integer;
     s: String;
begin
   TS := TStringList.Create;

   try
      try
         TS.Text := List;
         for i:=0 to TS.Count-1 do begin
            s := TS[i];
            if copy( s, 1, 1 ) = '.' then s := '.' + s;
            FConnection.WriteLn( s );
         end;

         FConnection.WriteLn( '.' );
         ResLine := TrimCrLf( FConnection.ReadLn( #10, TimeoutCommandSec*1000 ) );

         if FConnection.ReadLnTimedout then begin
            Disconnect;
            ResLine := '999 Timeout';
         end;

      except
         on E: Exception do ResLine := '999 ERROR on .SendList: ' + E.Message;
      end;

      Result   := strtointdef( copy(ResLine,1,3), 999 );
      FLastRes := ResLine;
      FLastCmd := FLastCmd + ' (data)';

   finally TS.Free end;
end;

end.
