// ============================================================================
// Logfile-object and functions.                             
// Copyright (c) 2001, Juergen Haible. All Rights Reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cLogFile; // Logfile-object and functions.

interface

uses SyncObjs, Classes;

const
   LOGID_ERROR       = $8000;
   LOGID_WARN        = $4000;
   LOGID_SYSTEM      = $0800;
   LOGID_INFO        = $0080;
   LOGID_DETAIL      = $0040;
   LOGID_DEBUG       = $0008;
   LOGID_STATUS      = $0004;
   LOGID_FULL        = $0001;
   LOGID_UNKNOWN     = Integer(-1);
   LOGID_UNHANDLED   = Integer(-2);

   LOGMARKER_ERROR   = 'ERR';
   LOGMARKER_WARN    = 'WAR';
   LOGMARKER_SYSTEM  = 'Sys';
   LOGMARKER_INFO    = 'I  ';
   LOGMARKER_DETAIL  = 'D  ';
   LOGMARKER_DEBUG   = 'd  ';
   LOGMARKER_STATUS  = 's  ';
   LOGMARKER_FULL    = 'f  ';
   LOGMARKER_UNKNOWN = '???';

type
   TLogFile = class
      // {E/L} = not thread safe, i.e. .Enter/.Leave required while using
      private
         FLock: TCriticalSection;
         FStrm: TFileStream;
         FViewBuffer, FErrorBuffer: TStringList;
         FLogPath: String;
         FViewMax, FFileMax: Integer;
         FViewMask, FFileMask, FTaskMask: Integer;
         FLastRotateLog: TDateTime;
         FLogMSecs: Boolean;

         procedure DoOpen;                         {E/L}
         procedure DoClose;                        {E/L}
         procedure DoAppend( const ID: Integer; const Line: String ); {E/L}
         procedure DoRotateLog;                    {E/L}

         procedure SetInt( Index: Integer; NewValue: Integer );
         procedure SetStr( Index: Integer; NewValue: String  );
         function  GetErrorBuffer: String;

      protected
         procedure DoAddLogView( const ID: Integer; const Msg: String ); virtual;

      public
         property  ViewMax : Integer index 0 read FViewMax  write SetInt;
         property  FileMax : Integer index 1 read FFileMax  write SetInt;
         property  ViewMask: Integer index 2 read FViewMask write SetInt;
         property  FileMask: Integer index 3 read FFileMask write SetInt;
         property  TaskMask: Integer index 4 read FTaskMask write SetInt;
         property  LogPath : String index 0 read FLogPath write SetStr;
         property  LogMSecs: Boolean read FLogMSecs write FLogMSecs;
         property  ErrorBuffer: String read GetErrorBuffer;

         function  ViewNext( out VLine: String; out VType: Integer ): Boolean;
         procedure ViewClear;

         procedure Enter;
         procedure Leave;

         procedure Add( const ID: Integer; const OrgMsg: String;
                        const UID: Integer = 0 ); 
         function  LastLines( const MaxKByte: Integer ): String;
         function  FindLines( const Pattern: String;
                              const IsRegex: Boolean;
                              const FromDaysAgo, TilDaysAgo: Integer;
                              const MaxLines: Integer ): String;
         procedure RotateLog( Forced: Boolean = True );

         constructor Create( const ALogPath: String;
                             const AFileMax: Integer;
                             RotateLogNow: Boolean );
         destructor Destroy; override;
   end;

function LogIdToMarker( const ID: Integer ): String;
function LogMarkerToId( const Marker: String ): Integer;


implementation

uses Windows, SysUtils, Math, cPCRE;

function LogIdToMarker( const ID: Integer ): String;
begin
   case ID of
      LOGID_ERROR : Result := LOGMARKER_ERROR;
      LOGID_WARN  : Result := LOGMARKER_WARN;
      LOGID_SYSTEM: Result := LOGMARKER_SYSTEM;
      LOGID_INFO  : Result := LOGMARKER_INFO;
      LOGID_DETAIL: Result := LOGMARKER_DETAIL;
      LOGID_DEBUG : Result := LOGMARKER_DEBUG;
      LOGID_STATUS: Result := LOGMARKER_STATUS;
      LOGID_FULL  : Result := LOGMARKER_FULL;
      else          Result := LOGMARKER_UNKNOWN;
   end;
end;

function LogMarkerToId( const Marker: String ): Integer;
begin
      if      Marker = LOGMARKER_ERROR  then Result := LOGID_ERROR
      else if Marker = LOGMARKER_WARN   then Result := LOGID_WARN
      else if Marker = LOGMARKER_SYSTEM then Result := LOGID_SYSTEM
      else if Marker = LOGMARKER_INFO   then Result := LOGID_INFO
      else if Marker = LOGMARKER_DETAIL then Result := LOGID_DETAIL
      else if Marker = LOGMARKER_DEBUG  then Result := LOGID_DEBUG
      else if Marker = LOGMARKER_STATUS then Result := LOGID_STATUS
      else if Marker = LOGMARKER_FULL   then Result := LOGID_FULL

      else if Marker = 'Inf' {Classic}  then Result := LOGID_INFO
      else if Marker = 'Det' {Classic}  then Result := LOGID_DETAIL
      else if Marker = 'deb' {Classic}  then Result := LOGID_DEBUG
      else if Marker = 'sta' {Classic}  then Result := LOGID_STATUS

      else                                   Result := LOGID_UNKNOWN;
end;


constructor TLogFile.Create( const ALogPath: String;
                             const AFileMax: Integer;
                             RotateLogNow: Boolean );
begin
   inherited Create;
   FLock := TCriticalSection.Create;

   FStrm         := nil;
   FLogPath      := ALogPath;
   FViewBuffer   := TStringList.Create;

   FErrorBuffer  := TStringList.Create;
   if FileExists( FLogPath + 'errors.log' ) then begin
      try FErrorBuffer.LoadFromFile( FLogPath + 'errors.log' ) except end;
   end;

   FViewMax  := 100;
   FFileMax  := AFileMax;
   FViewMask := LOGID_ERROR or LOGID_WARN or LOGID_SYSTEM or LOGID_INFO or LOGID_STATUS;
   FFileMask := LOGID_ERROR or LOGID_WARN or LOGID_SYSTEM or LOGID_INFO or LOGID_DETAIL;
   FTaskMask := LOGID_ERROR or LOGID_WARN or LOGID_SYSTEM or LOGID_INFO or LOGID_STATUS;
   FLogMSecs := False;

   if RotateLogNow then DoRotateLog
                   else FLastRotateLog := Now;
end;

destructor TLogFile.Destroy;
begin
   Enter;
   try
      try FErrorBuffer.SaveToFile( FLogPath + 'errors.log' ) except end;
      FErrorBuffer.Free;
      FViewBuffer.Free;
      DoClose;
      if Assigned( FStrm ) then FStrm.Free;
   except
   end;
   Leave;

   FLock.Free;
   inherited;
end;

procedure TLogFile.DoOpen;
begin
   if Assigned( FStrm ) then exit;

   if not FileExists( FLogPath + '0.log' ) then begin
      try
         FStrm := TFileStream.Create( FLogPath + '0.log',
                                      fmCreate or fmShareExclusive );
         FreeAndNil( FStrm );
      except
         FStrm := nil;
      end;
   end;

   try
      FStrm := TFileStream.Create( FLogPath + '0.log',
                                   fmOpenReadWrite or fmShareDenyNone );
   except
      FStrm := nil;
   end;
end;

procedure TLogFile.DoClose;
begin
   if Assigned( FStrm ) then FreeAndNil( FStrm );
end;

procedure TLogFile.DoAppend( const ID: Integer; const Line: String );
begin
   DoOpen;
   if not Assigned( FStrm ) then exit;

   try
      try
         FStrm.Seek( 0, soFromEnd );
         if length(Line) > 0 then FStrm.Write( Line[1], length(Line) );
         FStrm.Write( #13#10, 2 );
         FlushFileBuffers( FStrm.Handle );
      except end;
   finally DoClose end;;

   if (ID and (LOGID_ERROR or LOGID_WARN)) <> 0 then begin
      FErrorBuffer.AddObject( Line, Pointer(ID) );
      if FErrorBuffer.Count > 100 then FErrorBuffer.Delete( 0 );
      try FErrorBuffer.SaveToFile( FLogPath + 'errors.log' ) except end;
   end;
end;

function TLogFile.GetErrorBuffer: String;
begin
   Enter;
   try
      Result := FErrorBuffer.Text;
   finally Leave end;
end;

procedure TLogFile.DoRotateLog;
var  i: Integer;
begin
   try
      DoClose;
      FLastRotateLog := Now;
      DeleteFile( FLogPath + inttostr(FileMax) + '.log' );
      for i:=FileMax-1 downto 0 do begin
          RenameFile( FLogPath + inttostr(i  ) + '.log',
                      FLogPath + inttostr(i+1) + '.log' );
      end;
   except
   end;
end;

function TLogFile.LastLines(const MaxKByte: Integer): String;
var  p, l: Integer;
begin
   Result := '';

   Enter;
   try
      DoOpen;
      if not Assigned( FStrm ) then exit;

      try
         p := FStrm.Seek( 0, soFromEnd ) - ( MaxKByte shl 10 );
         if p < 0 then begin
            p := 0;
            l := FStrm.Size;
         end else begin
            l := MaxKByte shl 10;
         end;

         SetLength( Result, l );
         FStrm.Seek( p, soFromBeginning );
         if l>0 then FStrm.Read( Result[1], l );

         if p > 0 then begin
            p := Pos( #10, Result );
            if p > 0 then System.Delete( Result, 1, p );
         end;

      finally
         DoClose;
      end;

   finally
      Leave;
   end;
end;

function TLogFile.FindLines( const Pattern: String;
                             const IsRegex: Boolean;
                             const FromDaysAgo, TilDaysAgo: Integer;
                             const MaxLines: Integer ): String;

   function loadLogText( const logFile: String ): String;
   var  strm: TFileStream;
   begin
      Result := '';
      
      Enter;
      try

         if not FileExists( logFile ) then exit;

         strm := TFileStream.Create( logFile, fmOpenRead or fmShareDenyNone );
         try
            if strm.Size = 0 then exit;

            SetLength( Result, strm.Size );
            strm.Seek( 0, soFromBeginning );
            strm.Read( Result[1], length(Result) );

         finally strm.Free end;

      finally Leave end;

      Sleep( 0 );
   end;

   procedure findLogLines( const matches: TStringList;
                           const regex  : TPCRE;
                           const logText: String );
   var  lines: TStringList;
        i: Integer;
   begin
      if length( logText ) = 0 then exit;

      lines := TStringList.Create;
      try

         lines.Text := logText;

         for i := 0 to lines.Count - 1 do begin

            if IsRegex then begin
               if regex.Match( PChar(Pattern), PChar(lines[i]) ) then begin
                  matches.Add( lines[i] );
                  if matches.Count >= MaxLines then break;
               end;
            end else begin
               if Pos( LowerCase(Pattern), LowerCase(lines[i]) ) > 0 then begin
                  matches.Add( lines[i] );
                  if matches.Count >= MaxLines then break;
               end;
            end;

         end;

      finally lines.Free end;
   end;

var  logNo, logNoFrom, logNoTil: Integer;
     matches: TStringList;
     logFile: String;
     regex: TPCRE;
begin
   try
      logNoFrom := Max( FromDaysAgo, TilDaysAgo );
      logNoTil  := Min( FromDaysAgo, TilDaysAgo );

      matches := TStringList.Create;
      regex   := TPCRE.Create( True, PCRE_CASELESS );
      try

         if IsRegex then regex.Compile( PChar(Pattern) );

         for logNo := logNoFrom downto logNoTil do begin
         
            logFile := FLogPath + IntToStr(logNo) + '.log';
            findLogLines( matches, regex, loadLogText( logFile ) );
            if matches.Count >= MaxLines then break;

         end;

         Result := matches.Text;

      finally
         regex.Free;
         matches.Free;
      end;

   except
      on ex: Exception do Result := 'ERROR: ' + ex.Message;
   end;
end;

procedure TLogFile.DoAddLogView( const ID: Integer; const Msg: String );
begin
   FViewBuffer.AddObject( Msg, Pointer(ID) );
   if FViewBuffer.Count > ViewMax then begin
      while FViewBuffer.Count > ViewMax-10 do FViewBuffer.Delete(0);
   end;
end;

procedure TLogFile.Add( const ID: Integer; const OrgMsg: String; const UID: Integer = 0 );
var  ThdID, Msg1, Msg2: String;
begin
     if ( ID and (ViewMask or FileMask or TaskMask) ) = 0 then exit;

     // ThdID := lowercase( inttohex( GetCurrentThreadID, 1 ) );
     if UID = 0 then ThdID := lowercase( inttohex( GetCurrentThreadID, 1 ) )
                else ThdID := inttostr( UID );

     if FLogMSecs then begin
        Msg1  := FormatDateTime( 'yyyy/mm/dd hh:nn:ss,zzz', Now );
     end else begin
        Msg1  := FormatDateTime( 'yyyy/mm/dd hh:nn:ss', Now );
     end;
     Msg2  := ' ' + '{' + ThdID + '}' + ' ' + OrgMsg;

     if (ID and FileMask) <> 0 then begin
        Enter;
        try
           if Trunc(Now) <> Trunc(FLastRotateLog) then DoRotateLog;
           DoAppend( ID, Msg1 + ' ' + LogIdToMarker(ID)
                   + StringReplace( Msg2, #10, #10#9, [rfReplaceAll] ) );
        finally
           Leave;
        end;
     end;

     if (ID and ViewMask) <> 0 then begin
        Enter;
        try
           DoAddLogView( ID, Msg1 + Msg2 );
        finally
           Leave;
        end;
     end;
end;

procedure TLogFile.SetInt( Index, NewValue: Integer );
begin
   Enter;
   try
      case Index of
         0: FViewMax  := NewValue;
         1: FFileMax  := NewValue;
         2: FViewMask := NewValue;
         3: FFileMask := NewValue;
         4: FTaskMask := NewValue;
      end;
   finally
      Leave;
   end;
end;

procedure TLogFile.SetStr( Index: Integer; NewValue: String );
begin
   Enter;
   try
      case Index of
         0: FLogPath := NewValue;
      end;
   finally
      Leave;
   end;
end;

procedure TLogFile.Enter;
begin
   FLock.Enter;
end;

procedure TLogFile.Leave;
begin
   FLock.Leave;
end;

function TLogFile.ViewNext( out VLine: String; out VType: Integer ): Boolean;
begin
   Enter;
   try
      Result := ( FViewBuffer.Count > 0 );
      if Result then begin
         VLine := FViewBuffer[ 0 ];
         VType := Integer( FViewBuffer.Objects[ 0 ] );
         FViewBuffer.Delete( 0 );
      end;
   finally
      Leave;
   end;
end;

procedure TLogFile.ViewClear;
begin
   Enter;
   try
      FViewBuffer.Clear;
   finally
      Leave;
   end;
end;

procedure TLogFile.RotateLog( Forced: Boolean = True );
var  strm: TFileStream;
     dateNow, dateLog: String;
begin
   Enter;
   try
      // keep logfile on startup if it's still the same day
      if (not Forced) and FileExists( FLogPath + '0.log' ) then try

         dateNow := FormatDateTime( 'yyyy/mm/dd', Now );
         SetLength( dateLog, length(dateNow) );

         strm := TFileStream.Create( FLogPath + '0.log',
                                     fmOpenRead or fmShareDenyNone );
         try
            strm.Seek( 0, soFromBeginning );
            strm.Read( dateLog[1], length(dateLog) );
            if dateNow = dateLog then exit; // same day, don't rotate
         finally strm.Free end;

      except end;

      // start new logfile
      DoRotateLog;

   finally
      Leave;
   end;
end;

end.
