// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cSettings;    

interface

uses SysUtils, Classes, IniFiles, cSyncObjects, cSettingsDef;

type
   ESettingsError = class( Exception );

   TSetting = class
      protected
         FID     : Integer;
         FValue  : String;
         FChanged: Boolean;

         function  GetAsInt: Integer;
         procedure SetAsInt( const NewValue: Integer );
         function  GetAsInt64: Int64;
         procedure SetAsInt64( const NewValue: Int64 );
         function  GetAsBoo: Boolean;
         procedure SetAsBoo( const NewValue: Boolean );
         function  GetAsStr: String;
         procedure SetAsStr( const NewValue: String );
         function  GetAsHexInt: Integer;
         procedure SetAsHexInt( const NewValue: Integer );
         function  GetAsDT: TDateTime;
         procedure SetAsDT( const NewValue: TDateTime );
         function  GetAsStrPtr: PChar;

      public
         property ID: Integer read FID;

         property AsInt: Integer    read GetAsInt    write SetAsInt;
         property AsBoo: Boolean    read GetAsBoo    write SetAsBoo;
         property AsStr: String     read GetAsStr    write SetAsStr;
         property AsStrPtr: PChar   read GetAsStrPtr;
         property AsHexInt: Integer read GetAsHexInt write SetAsHexInt;
         property AsInt64: Int64    read GetAsInt64  write SetAsInt64;
         property AsDT: TDateTime   read GetAsDT     write SetAsDT;

         property Changed: Boolean read FChanged write FChanged;

         constructor Create( const AID: Integer );
   end;

   ISettingsHandler = interface
      function  DoOpen( const Qualifier: String;
                        const ReadOnly : Boolean ): Boolean;
      procedure DoClose;

      function  DoGet( const Qualifier : String;
                       const SettingDef: TSettingDef ): String;
      procedure DoSet( const Qualifier : String;
                       const SettingDef: TSettingDef;
                       const NewValue  : String );
   end;

   TSettingsOnChangedProc = procedure( const Qualifier: String;
                                       const ID: Integer );

   TSettingsCustom = class
      // NOTE: Some settings might share the same file, so do NEVER
      //       overwrite the whole file but only changed settings.
      protected
         FDefinition: TSettingsDef;
         FHandler   : ISettingsHandler;
         FLock      : TReaderWriterLock;
         FList      : TList;
         FQualifier : String;
         FAutoFlush : Boolean;
         FOnChanged : TSettingsOnChangedProc;

         function GetItem( const ID: Integer ): TSetting;
         function AnyChanged: Boolean;

         procedure DoChanged( const ID: Integer );
         procedure DoInitAll;
         procedure DoFreeAll;

      public
         property Qualifier: String read FQualifier;
         property OnChanged: TSettingsOnChangedProc read FOnChanged write FOnChanged;

         procedure Discard;
         procedure Flush;

         constructor Create( ASettingsDef: TSettingsDef;
                             AHandler    : ISettingsHandler;  
                             AAutoFlush  : Boolean );
         destructor Destroy; override;
   end;

   TSettingsPlain = class( TSettingsCustom )
      // 'Plain': Global settings, all loaded on .Create and kept in memory.
      public
         function  GetAll: String;

         function  GetStr( const ID: Integer ): String;
         function  GetInt( const ID: Integer ): Integer;
         function  GetBoo( const ID: Integer ): Boolean;
         function  GetDT ( const ID: Integer ): TDateTime;
         function  GetStrPtr( const ID: Integer ): PChar;
         function  GetHexInt( const ID: Integer ): Integer;
         function  GetInt64( const ID: Integer ): Int64;

         procedure SetStr( const ID: Integer; const NewValue: String );
         procedure SetInt( const ID: Integer; const NewValue: Integer );
         procedure SetBoo( const ID: Integer; const NewValue: Boolean );
         procedure SetDT ( const ID: Integer; const NewValue: TDateTime );
         procedure SetHexInt( const ID: Integer; const NewValue: Integer );
         procedure SetInt64( const ID: Integer; const NewValue: Int64 );

         function  GetChanged( const ID: Integer ): Boolean;
         procedure SetChanged( const ID: Integer; const NewValue: Boolean );

         constructor Create( ASettingsDef: TSettingsDef;
                             AHandler    : ISettingsHandler;
                             AAutoFlush  : Boolean );
   end;

   TSettingsQualified = class( TSettingsCustom )
      // 'Qualified': Each setting depends on a qualifier (e. g. server name).
      //              Only settings of current qualifier are loaded and kept
      //              in memory until another qualifier is used.
      protected
         procedure SwitchQualifier( const NewQualifier: String );

      public
         function  GetAll( const Qualifier: String ): String;

         function  GetStr( const Qualifier: String; const ID: Integer ): String;
         function  GetInt( const Qualifier: String; const ID: Integer ): Integer;
         function  GetBoo( const Qualifier: String; const ID: Integer ): Boolean;

         procedure SetStr( const Qualifier: String; const ID: Integer; const NewValue: String );
         procedure SetInt( const Qualifier: String; const ID: Integer; const NewValue: Integer );
         procedure SetBoo( const Qualifier: String; const ID: Integer; const NewValue: Boolean );

         constructor Create( ASettingsDef: TSettingsDef;
                             AHandler    : ISettingsHandler;
                             AAutoFlush  : Boolean );
   end;

   TSettingsHandler_InMemory = class( TInterfacedObject, ISettingsHandler )
      // Handler for settings, that are only kept in memory.
      // Note: Doesn't support qualifiers.
      protected
         FList: TStringList;

         function  DoOpen( const Qualifier: String;
                           const ReadOnly: Boolean ): Boolean; virtual;
         procedure DoClose; virtual;

         function  DoGet( const Qualifier : String;
                          const SettingDef: TSettingDef ): String; virtual;
         procedure DoSet( const Qualifier : String;
                          const SettingDef: TSettingDef;
                          const NewValue  : String ); virtual;

      public
         constructor Create( const AInitialList: String );
         destructor Destroy; override;
   end;

   TSettingsHandler_IniFile = class( TInterfacedObject, ISettingsHandler )
      // Handler for settings, that are stored in a local .ini file.
      protected
         FFilename: String;
         FIniFile : TCustomIniFile;

         function  DoOpen( const Qualifier: String;
                           const ReadOnly: Boolean ): Boolean; virtual;
         procedure DoClose; virtual;

         function  DoGet( const Qualifier : String;
                          const SettingDef: TSettingDef ): String; virtual;
         procedure DoSet( const Qualifier : String;
                          const SettingDef: TSettingDef;
                          const NewValue  : String ); virtual;

      public
         constructor Create( const AFilename: String );
         destructor Destroy; override;
   end;


implementation

uses uConst, uDateTime;

{ TSetting }

constructor TSetting.Create;
begin
   inherited Create;
   FID      := AID;
   FValue   := '';
   FChanged := False;
end;

function TSetting.GetAsBoo: Boolean;
begin
   Result := ( AsInt <> 0 );
end;

function TSetting.GetAsInt: Integer;
begin
   Result := strtointdef( FValue, 0 );
end;

function TSetting.GetAsInt64: Int64;
begin
   Result := strtoint64def( FValue, 0 );
end;

function TSetting.GetAsHexInt: Integer;
begin
   Result := strtointdef( '$' + FValue, 0 );
end;

function TSetting.GetAsStr: String;
begin
   Result := FValue;
end;

function TSetting.GetAsDT: TDateTime;
begin
   if FValue = '' then Result := 0
                  else Result := uDateTime.TimeStampToDateTime( FValue );
end;

function TSetting.GetAsStrPtr: PChar;
begin
   Result := PChar( FValue );
end;

procedure TSetting.SetAsBoo( const NewValue: Boolean );
begin
   if NewValue then FValue := '1' else FValue := '0';
   FChanged := True;
end;

procedure TSetting.SetAsInt( const NewValue: Integer );
begin
   FValue := inttostr( NewValue );
   FChanged := True;
end;

procedure TSetting.SetAsInt64( const NewValue: Int64 );
begin
   FValue := inttostr( NewValue );
   FChanged := True;
end;

procedure TSetting.SetAsHexInt( const NewValue: Integer );
begin
   FValue := inttohex( NewValue, 4 );
   FChanged := True;
end;

procedure TSetting.SetAsStr( const NewValue: String );
begin
   FValue := NewValue;
   FChanged := True;
end;

procedure TSetting.SetAsDT( const NewValue: TDateTime );
begin
   if NewValue = 0 then FValue := ''
                   else FValue := uDateTime.DateTimeToTimeStamp( NewValue );
   FChanged := True;
end;


{ TSettingsCustom }

constructor TSettingsCustom.Create( ASettingsDef: TSettingsDef;
                                    AHandler    : ISettingsHandler;
                                    AAutoFlush  : Boolean );
begin
   inherited Create;

   FDefinition := ASettingsDef;
   FHandler    := AHandler;
   FQualifier  := '';
   FAutoFlush  := AAutoFlush;
   FOnChanged  := nil;

   FLock := TReaderWriterLock.Create;
   FList := TList.Create;
end;

destructor TSettingsCustom.Destroy;
begin
   if FAutoFlush then Flush else Discard;
   DoFreeAll;

   FHandler := nil;
   FList.Free;
   FLock.Free;

   inherited Destroy;
end;

procedure TSettingsCustom.Discard;
var  ID: Integer;
begin
   if not AnyChanged then exit;

   FLock.BeginWrite;
   try
      with FDefinition do begin
         for ID := MinID to MaxID do begin
            if IDExists( ID ) then begin
               GetItem(ID).Changed := False;
            end;
         end;
      end;
   finally FLock.EndWrite end;
end;

procedure TSettingsCustom.Flush;
var  ID: Integer;
begin
   if not AnyChanged then exit;

   FLock.BeginWrite;
   try
      if FHandler.DoOpen( FQualifier, False ) then try
         with FDefinition do begin
            for ID := MinID to MaxID do begin
               if IDExists( ID ) then begin
                  if GetItem(ID).Changed then begin
                     FHandler.DoSet( FQualifier, Def[ID], GetItem(ID).AsStr );
                     GetItem(ID).Changed := False;
                  end;
               end;
            end;
         end;
      finally FHandler.DoClose end;
   finally FLock.EndWrite end;
end;

procedure TSettingsCustom.DoInitAll;

   procedure DoInit( const ID: Integer; const InitValue: String );
   begin
      while ID >= FList.Count do FList.Add( nil );

      if not Assigned( FList[ID] ) then FList[ID] := TSetting.Create( ID );

      with TSetting( FList[ID] ) do begin
         AsStr   := InitValue;
         Changed := False;
      end;
   end;

var  ID: Integer;
begin
   FLock.BeginWrite;
   try
      if FHandler.DoOpen( FQualifier, True ) then try
         with FDefinition do begin
            for ID := MinID to MaxID do begin
               if IDExists( ID ) then begin
                  DoInit( ID, FHandler.DoGet( FQualifier, Def[ID] ) );
               end;
            end;
         end;
      finally FHandler.DoClose end;
   finally FLock.EndWrite end;
end;

procedure TSettingsCustom.DoFreeAll;
var  ID: Integer;
begin
   FLock.BeginWrite;
   try
      for ID := 0 to FList.Count - 1 do begin
         if Assigned( FList[ID] ) then begin
            TSetting( FList[ID] ).Free;
            FList[ID] := nil;
         end;
      end;
      FList.Clear;
   finally FLock.EndWrite end;
end;

function TSettingsCustom.GetItem( const ID: Integer ): TSetting;
begin
   Result := FList[ ID ];
end;

function TSettingsCustom.AnyChanged: Boolean;
var  ID: Integer;
begin
   FLock.BeginRead;
   try
      Result := False;
      for ID := 0 to FList.Count - 1 do begin
         if Assigned( FList[ID] ) then begin
            if GetItem(ID).Changed then begin Result := True; break end;
         end;
      end;
   finally FLock.EndRead end;
end;

procedure TSettingsCustom.DoChanged( const ID: Integer );
begin
   if not Assigned( FOnChanged ) then exit;
   FOnChanged( FQualifier, ID );
end;

{ TSettingsPlain }

constructor TSettingsPlain.Create( ASettingsDef: TSettingsDef;
                                   AHandler    : ISettingsHandler;
                                   AAutoFlush  : Boolean );
begin
   inherited Create( ASettingsDef, AHandler, AAutoFlush );
   DoInitAll;
end;

function TSettingsPlain.GetAll: String;
var  ID: Integer;
begin
   FLock.BeginRead;
   try
      Result := '';
      for ID := 0 to FList.Count - 1 do begin
         if Assigned( FList[ID] ) then begin
            Result := Result + inttostr(ID) + '=' + GetStr(ID) + CRLF;
         end;
      end;
   finally FLock.EndRead end;
end;

function TSettingsPlain.GetStr( const ID: Integer ): String;
begin
   FLock.BeginRead;
   try
      Result := GetItem( ID ).AsStr;
   finally FLock.EndRead end;
end;

function TSettingsPlain.GetStrPtr( const ID: Integer ): PChar;
begin
   FLock.BeginRead;
   try
      Result := GetItem( ID ).AsStrPtr;
   finally FLock.EndRead end;
end;

function TSettingsPlain.GetInt( const ID: Integer ): Integer;
begin
   FLock.BeginRead;
   try
      Result := GetItem( ID ).AsInt;
   finally FLock.EndRead end;
end;

function TSettingsPlain.GetInt64( const ID: Integer ): Int64;
begin
   FLock.BeginRead;
   try
      Result := GetItem( ID ).AsInt64;
   finally FLock.EndRead end;
end;

function TSettingsPlain.GetHexInt( const ID: Integer ): Integer;
begin
   FLock.BeginRead;
   try
      Result := GetItem( ID ).AsHexInt;
   finally FLock.EndRead end;
end;

function TSettingsPlain.GetBoo( const ID: Integer ): Boolean;
begin
   FLock.BeginRead;
   try
      Result := GetItem( ID ).AsBoo;
   finally FLock.EndRead end;
end;

function TSettingsPlain.GetDT( const ID: Integer ): TDateTime;
begin
   FLock.BeginRead;
   try
      Result := GetItem( ID ).AsDT;
   finally FLock.EndRead end;
end;

function TSettingsPlain.GetChanged( const ID: Integer ): Boolean;
begin
   FLock.BeginWrite;
   try
      Result := GetItem( ID ).Changed;
   finally FLock.EndWrite end;
end;

procedure TSettingsPlain.SetStr( const ID: Integer; const NewValue: String );
begin
   FLock.BeginWrite;
   try
      GetItem( ID ).AsStr := NewValue;
      DoChanged( ID );
   finally FLock.EndWrite end;
end;

procedure TSettingsPlain.SetInt( const ID: Integer; const NewValue: Integer );
begin
   FLock.BeginWrite;
   try
      GetItem( ID ).AsInt := NewValue;
      DoChanged( ID );
   finally FLock.EndWrite end;
end;

procedure TSettingsPlain.SetInt64( const ID: Integer; const NewValue: Int64 );
begin
   FLock.BeginWrite;
   try
      GetItem( ID ).AsInt64 := NewValue;
      DoChanged( ID );
   finally FLock.EndWrite end;
end;

procedure TSettingsPlain.SetBoo( const ID: Integer; const NewValue: Boolean );
begin
   FLock.BeginWrite;
   try
      GetItem( ID ).AsBoo := NewValue;
      DoChanged( ID );
   finally FLock.EndWrite end;
end;

procedure TSettingsPlain.SetDT( const ID: Integer; const NewValue: TDateTime );
begin
   FLock.BeginWrite;
   try
      GetItem( ID ).AsDT := NewValue;
      DoChanged( ID );
   finally FLock.EndWrite end;
end;

procedure TSettingsPlain.SetHexInt( const ID: Integer; const NewValue: Integer );
begin
   FLock.BeginWrite;
   try
      GetItem( ID ).AsHexInt := NewValue;
      DoChanged( ID );
   finally FLock.EndWrite end;
end;

procedure TSettingsPlain.SetChanged( const ID: Integer; const NewValue: Boolean );
begin
   FLock.BeginWrite;
   try
      GetItem( ID ).Changed := NewValue;
      DoChanged( ID );
   finally FLock.EndWrite end;
end;


{ TSettingsQualified }

constructor TSettingsQualified.Create( ASettingsDef: TSettingsDef;
                                       AHandler    : ISettingsHandler;
                                       AAutoFlush  : Boolean );
begin
   inherited Create( ASettingsDef, AHandler, AAutoFlush );
   FQualifier := '';
end;

procedure TSettingsQualified.SwitchQualifier( const NewQualifier: String );
begin
   // Note: may only be called within write-lock

   // save and clear settings of old identifier
   if FQualifier <> '' then begin
      if FAutoFlush then Flush else Discard;
   end;
   DoFreeAll;

   // set new qualifier and load its settings
   FQualifier := NewQualifier;
   DoInitAll;
end;

function TSettingsQualified.GetAll( const Qualifier: String ): String;
var  ID: Integer;
begin
   FLock.BeginWrite;
   try
      SwitchQualifier( Qualifier ); 

      Result := '';
      for ID := 0 to FList.Count - 1 do begin
         if Assigned( FList[ID] ) then begin
            Result := Result + inttostr(ID) + '=' + GetItem(ID).AsStr + CRLF;
         end;
      end;
   finally FLock.EndWrite end;
end;

function TSettingsQualified.GetStr( const Qualifier: String;
                                    const ID: Integer ): String;
begin
   FLock.BeginRead;
   try
      if Qualifier = FQualifier then begin
         Result := GetItem( ID ).AsStr;
         exit;
      end;
   finally FLock.EndRead end;

   FLock.BeginWrite;
   try
      SwitchQualifier( Qualifier );
      Result := GetItem( ID ).AsStr;
   finally FLock.EndWrite end;
end;

function TSettingsQualified.GetInt( const Qualifier: String;
                                    const ID: Integer ): Integer;
begin
   FLock.BeginRead;
   try
      if Qualifier = FQualifier then begin
         Result := GetItem( ID ).AsInt;
         exit;
      end;
   finally FLock.EndRead end;

   FLock.BeginWrite;
   try
      SwitchQualifier( Qualifier );
      Result := GetItem( ID ).AsInt;
   finally FLock.EndWrite end;
end;

function TSettingsQualified.GetBoo( const Qualifier: String;
                                    const ID: Integer ): Boolean;
begin
   FLock.BeginRead;
   try
      if Qualifier = FQualifier then begin
         Result := GetItem( ID ).AsBoo;
         exit;
      end;
   finally FLock.EndRead end;

   FLock.BeginWrite;
   try
      SwitchQualifier( Qualifier );
      Result := GetItem( ID ).AsBoo;
   finally FLock.EndWrite end;
end;

procedure TSettingsQualified.SetStr( const Qualifier: String;
                                     const ID: Integer;
                                     const NewValue: String );
begin
   FLock.BeginWrite;
   try
      if Qualifier <> FQualifier then SwitchQualifier( Qualifier );
      GetItem( ID ).AsStr := NewValue;
      DoChanged( ID );
   finally FLock.EndWrite end;
end;

procedure TSettingsQualified.SetInt( const Qualifier: String;
                                     const ID: Integer;
                                     const NewValue: Integer );
begin
   FLock.BeginWrite;
   try
      if Qualifier <> FQualifier then SwitchQualifier( Qualifier );
      GetItem( ID ).AsInt := NewValue;
      DoChanged( ID );
   finally FLock.EndWrite end;
end;

procedure TSettingsQualified.SetBoo( const Qualifier: String;
                                     const ID: Integer;
                                     const NewValue: Boolean );
begin
   FLock.BeginWrite;
   try
      if Qualifier <> FQualifier then SwitchQualifier( Qualifier );
      GetItem( ID ).AsBoo := NewValue;
      DoChanged( ID );
   finally FLock.EndWrite end;
end;


{ TSettingsHandler_IniFile }

constructor TSettingsHandler_IniFile.Create( const AFilename: String );
begin
   inherited Create;
   FFilename := AFilename;
   FIniFile  := nil;
end;

destructor TSettingsHandler_IniFile.Destroy;
begin
   if Assigned( FIniFile ) then FreeAndNil( FIniFile );
   inherited Destroy;
end;

function TSettingsHandler_IniFile.DoOpen( const Qualifier: String;
                                          const ReadOnly: Boolean ): Boolean;
begin
   {if ReadOnly then FIniFile := TMemIniFile.Create( FFilename )
               else} FIniFile := TIniFile.Create   ( FFilename );
   Result := Assigned( FIniFile );
end;

procedure TSettingsHandler_IniFile.DoClose;
begin
   if Assigned( FIniFile ) then begin
      try
         if FIniFile is TIniFile then (FIniFile as TIniFile).UpdateFile;
      except end;
      FreeAndNil( FIniFile );
   end;
end;

function TSettingsHandler_IniFile.DoGet( const Qualifier : String;
                                         const SettingDef: TSettingDef ): String;
begin
   Result := FIniFile.ReadString(
      SettingDef.Section, Qualifier + SettingDef.Keyword, SettingDef.Default
   );
end;

procedure TSettingsHandler_IniFile.DoSet( const Qualifier : String;
                                          const SettingDef: TSettingDef;
                                          const NewValue  : String );
begin
   FIniFile.WriteString(
      SettingDef.Section, Qualifier + SettingDef.Keyword, NewValue
   );
end;

{ TSettingsHandler_InMemory }

constructor TSettingsHandler_InMemory.Create( const AInitialList: String );
begin
   inherited Create;
   FList := TStringList.Create;
   if length(AInitialList) > 0 then FList.Text := AInitialList;
end;

destructor TSettingsHandler_InMemory.Destroy;
begin
   if Assigned( FList ) then FList.Free;
   inherited Destroy;
end;

function TSettingsHandler_InMemory.DoOpen( const Qualifier: String;
                                           const ReadOnly: Boolean ): Boolean;
begin
   Result := True;
end;

procedure TSettingsHandler_InMemory.DoClose;
begin
   //
end;

function TSettingsHandler_InMemory.DoGet( const Qualifier: String;
                                          const SettingDef: TSettingDef ): String;
begin
   Result := FList.Values[ inttostr( SettingDef.ID ) ];
end;

procedure TSettingsHandler_InMemory.DoSet( const Qualifier: String;
                                           const SettingDef: TSettingDef;
                                           const NewValue: String );
begin
   FList.Values[ inttostr( SettingDef.ID ) ] := NewValue;
end;

end.
