// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit tReports;

interface

{$INCLUDE Compiler.inc}

uses Windows, SysUtils, Classes, uTools, uConst, uType, tBase;

type
   TReportCreateProc = function ( const Report: TStringList;
                                  const Month: TDateTime;
                                  const Filename: String ): Boolean of object;

   TThreadReports = class( TTaskThread )

      protected
         ReportTitles: TStringList;
         CurrentMonth, PreviousMonth: TDateTime;

         function Report_DiskSpace    ( const Report: TStringList;
                                        const Month: TDateTime;
                                        const Filename: String ): Boolean;
         function Report_Newsgroups   ( const Report: TStringList;
                                        const Month: TDateTime;
                                        const Filename: String ): Boolean;
         function Report_Mails        ( const Report: TStringList;
                                        const Month: TDateTime;
                                        const Filename: String ): Boolean;
         function Report_Mailboxes    ( const Report: TStringList;
                                        const Month: TDateTime;
                                        const Filename: String ): Boolean;
         function Report_MailFilters  ( const Report: TStringList;
                                        const Month: TDateTime;
                                        const Filename: String ): Boolean;
         function Report_MailTrapUsage( const Report: TStringList;
                                        const Month: TDateTime;
                                        const Filename: String ): Boolean;

         procedure ReportHeader( const Report: TStringList; const Title: String );
         procedure ReportFooter( const Report: TStringList );

         procedure CreateReport( const FilenameBase, TitleBase: String;
                                 const Reporter: TReportCreateProc;
                                 const Month: TDateTime );
         procedure CreateIndexReport( ReportTitles: TStringList );

         procedure Execute; override;

      public
         constructor Create( FreeType: TThreadFreeTypes );

  end;


implementation

uses uConstVar, uVar, cArtFiles, uDateTime, uCRC32, cLogFileHamster, cHamster,
     uHamTools, cTraps, cHscAction;

// ------------------------------------------------------------ (helpers) -----

type
   TMailFilterHit = class

      private
         FChecked   : Boolean;
         FStillInUse: Boolean;

         function GetStillInUse: Boolean;

      public
         Origin    : String;
         Filter    : String;
         HitCount  : Integer;
         LastHit   : TDateTime;
         Group     : Integer;

         property StillInUse: Boolean read GetStillInUse;

         constructor Create( const AOrigin, AFilter: String );

   end;

   TNewsgroupInfo = class

      public
         Groupname: String;
         ArtCount : Integer;
         LastRead : TDateTime;
         LastPull : TDateTime;
         GrpClass : TActiveClass;
         Diskspace: Int64;

         constructor Create( const AGroupname: String );

   end;

   TDirInfo = class

      public
         DirName: String;
         DirSize: Int64;
         DirFile: Integer;

         constructor Create( const ADirName: String );

   end;

   TCounterInfo = class
      public
         Counter: array[0..19] of Integer;
         procedure Clear;
         constructor Create();
   end;

constructor TCounterInfo.Create();
begin
   inherited Create;
   Clear;
end;

procedure TCounterInfo.Clear;
var  i: Integer;
begin
   for i := 0 to 19 do Counter[i] := 0;
end;


constructor TMailFilterHit.Create( const AOrigin, AFilter: String );
begin
   Origin      := AOrigin;
   Filter      := AFilter;
   HitCount    := 0;
   LastHit     := 0;
   Group       := 0;
   FChecked    := False;
   FStillInUse := False;
end;

function TMailFilterHit.GetStillInUse: Boolean;
begin
   if not FChecked then begin
      FChecked := True;
      if Origin = 'MailTrap' then begin
         FStillInUse := Hamster.MailTrap.Exists( Filter );
      end;
   end;

   Result := FStillInUse;
end;


constructor TNewsgroupInfo.Create( const AGroupname: String );
begin
   inherited Create;

   Groupname := AGroupName;
   ArtCount  := 0;
   LastRead  := 0;
   LastPull  := 0;
   GrpClass  := aclUnknown;
   Diskspace := 0;
end;

constructor TDirInfo.Create(const ADirName: String);
begin
   inherited Create;
   DirName := ADirName;
   DirSize := 0;
   DirFile := 0;
end;

function ComparyByHitCount( List: TStringList; Index1, Index2: Integer): Integer;
var  h1, h2: TMailFilterHit;
begin
   h1 := TMailFilterHit( List.Objects[Index1] );
   h2 := TMailFilterHit( List.Objects[Index2] );
   Result := h2.HitCount - h1.HitCount;
   if Result = 0 then begin
      Result := Trunc( (h2.LastHit - h1.LastHit) * 1000.0 );
      if Result = 0 then begin
         Result := AnsiCompareText( h1.Filter, h2.Filter );
      end;
   end;
end;

function EnHtml( s: string ): String;
begin
   s := StringReplace( s, '<', '&lt;',   [rfReplaceAll] );
   s := StringReplace( s, '>', '&gt;',   [rfReplaceAll] );
   s := StringReplace( s, '&', '&amp;',  [rfReplaceAll] );
   s := StringReplace( s, '"', '&quot;', [rfReplaceAll] );
   Result := s;
end;

function i2n( i: Int64 ): String;
var  d: Double;
begin
   d := i;
   Result := Format( '%.0n', [ d ] )
end;

function iPercent( a, b: Int64 ): String;
var  d: Double;
begin
   if b = 0 then d := 0.0
            else d := (100.0 * a) / (1.0 * b);
   Result := Format( '%.1n%', [ d ] )
end;

function SkipPort( const s: String ): String;
var  i: Integer;
begin
   i := Pos( ',', s );
   if i = 0 then Result := s
            else Result := copy( s, 1, i-1 );
end;

function SkipKey( const s: String ): String;
var  i: Integer;
begin
   i := Pos( '=', s );
   if i = 0 then Result := s
            else Result := copy( s, i+1, MaxInt );
end;

// ------------------------------------------------------- TThreadReports -----

function TThreadReports.Report_DiskSpace(
         const Report: TStringList; const Month: TDateTime; const Filename: String ): Boolean;
var  DirList: TStringList;

   procedure AddPath( const Path: String );
   var  SR: TSearchRec;
        DI: TDirInfo;
        i64: Int64;
   begin
      if DirList.IndexOf( Path ) >= 0 then exit;

      DI := TDirInfo.Create( Path );
      DirList.AddObject( Path, DI );

      if SysUtils.FindFirst( Path + '*.*', faAnyFile, SR ) <> 0 then exit;

      repeat
         if (SR.Attr and faDirectory) <> 0 then begin
            if (SR.Name <> '.') and (SR.Name <> '..') then begin
               AddPath( Path + SR.Name + '\' );
            end;
         end else begin
            i64 := Int64(SR.FindData.nFileSizeLow)
                 + Int64(SR.FindData.nFileSizeHigh) shl 32;
            inc( DI.DirSize, i64 );
            inc( DI.DirFile );
         end;
      until SysUtils.FindNext( SR ) <> 0;

      SysUtils.FindClose( SR );
   end;

   procedure CalcSize( const Path: String; var Count: Integer; var Size: Int64 );
   var  SR: TSearchRec;
        DI: TDirInfo;
        i: Integer;
   begin
      i := DirList.IndexOf( Path );
      if i < 0 then exit;

      DI := TDirInfo( DirList.Objects[i] );
      inc( Count, DI.DirFile );
      inc( Size,  DI.DirSize );

      if SysUtils.FindFirst( Path + '*.*', faDirectory, SR ) <> 0 then exit;

      repeat
         if (SR.Attr and faDirectory) <> 0 then begin
            if (SR.Name <> '.') and (SR.Name <> '..') then begin
               CalcSize( Path + SR.Name + '\', Count, Size );
            end;
         end;
      until SysUtils.FindNext( SR ) <> 0;

      SysUtils.FindClose( SR );
   end;

   procedure TrSumSize( const Title, Path: String );
   var  Count: Integer;
        Size : Int64;
   begin
      Count := 0;
      Size  := 0;
      CalcSize( Path, Count, Size );
      Report.Add( '<tr> '
                + '<td><b>' + Title + '</b> (' + Path + ')' + '</td>'
                + '<td align=right>' + i2n( Count ) + '</td>'
                + '<td align=right>' + i2n( Size  ) + '</td>'
                + ' </tr>' );
   end;

var  asPath, i, SumCount: Integer;
     SumSize: Int64;
     DI: TDirInfo;
begin
   DirList := TStringList.Create;
   try

      // scan all Hamster directories
      for asPath := asPathBase to asPathLast do begin
         AddPath( AppSettings.GetStr( asPath ) );
      end;
      DirList.Sort;


      Report.Add( '<h2> Hamster Directory Totals </h2>' );

      Report.Add( '<table border=1 width=100%>' );
      Report.Add( '<tr> '
                 + ' <th width=50% align=left>Directory</th>'
                 + ' <th width=25% align=right>Count</th>'
                 + ' <th width=25% align=right>Size</th>'
                 + ' </tr>' );

      TrSumSize( 'Base',     AppSettings.GetStr( asPathBase ) );
      TrSumSize( 'Logs',     AppSettings.GetStr( asPathLogs ) );
      TrSumSize( 'Server',   AppSettings.GetStr( asPathServer ) );
      TrSumSize( 'Groups',   AppSettings.GetStr( asPathGroups ) );
      TrSumSize( 'Mails',    AppSettings.GetStr( asPathMails ) );
      TrSumSize( 'News.Out', AppSettings.GetStr( asPathNewsOut ) );
      TrSumSize( 'Mail.Out', AppSettings.GetStr( asPathMailOut ) );
      TrSumSize( 'News.Err', AppSettings.GetStr( asPathNewsErr ) );
      TrSumSize( 'Scripts',  AppSettings.GetStr( asPathScripts ) );
      TrSumSize( 'Modules',  AppSettings.GetStr( asPathModules ) );
      TrSumSize( 'Reports',  AppSettings.GetStr( asPathReports ) );

      Report.Add( '</table>' );


      Report.Add( '<h2> Directory Details </h2>' );

      Report.Add( '<table border=1 width=100%>' );
      Report.Add( '<tr> '
                 + ' <th width=50% align=left>Directory</th>'
                 + ' <th width=25% align=right>Count</th>'
                 + ' <th width=25% align=right>Size</th>'
                 + ' </tr>' );

      SumCount := 0;
      SumSize  := 0;
      for i := 0 to DirList.Count - 1 do begin

         DI := TDirInfo( DirList.Objects[i] );

         inc( SumCount, DI.DirFile );
         inc( SumSize,  DI.DirSize );

         Report.Add( '<tr> '
                   + '<td>' + DI.DirName + '</td>'
                   + '<td align=right>' + i2n( DI.DirFile ) + '</td>'
                   + '<td align=right>' + i2n( DI.DirSize ) + '</td>'
                   + ' </tr>' );

      end;

      Report.Add( '<tr> '
                + '<td>Total</td>'
                + '<td align=right>' + i2n( SumCount ) + '</td>'
                + '<td align=right>' + i2n( SumSize ) + '</td>'
                + ' </tr>' );
      Report.Add( '</table>' );

      Report.Add( '<p><small>' );
      Report.Add( '<b>Directory:</b> disk directory, ' );
      Report.Add( '<b>Count:</b> number of files, ' );
      Report.Add( '<b>Size:</b> disk space used' );
      Report.Add( '</small></p>' );

      Result := True;

   finally
      for i := 0 to DirList.Count - 1 do TDirInfo( DirList.Objects[i] ).Free;
      DirList.Free
   end;
end;

function TThreadReports.Report_Newsgroups(
         const Report: TStringList; const Month: TDateTime; const Filename: String ): Boolean;
var  NgList: TStringList;
     NgInfo: TNewsgroupInfo;
     SumCount, i, GrpHdl: Integer;
     SumDisk: Int64;
     StatsNow: TDateTime;
     NgName: String;
     NgClassCount: array [TActiveClass] of Integer;
     NgClass: TActiveClass;
begin
   StatsNow := Now;
   SumCount := 0;
   SumDisk  := 0;
   NgList := TStringList.Create;

   for NgClass := Low(NgClassCount) to High(NgClassCount) do begin
      NgClassCount[ NgClass ] := 0;
   end;

   try
      // get list of newsgroup names
      NgList.Text := Hamster.Config.Newsgroups.GetList( '' );

      // attach newsgroup info
      for i := 0 to NgList.Count - 1 do begin

         NgName := NgList[i];
         NgInfo := TNewsgroupInfo.Create( NgName );
         NgList.Objects[i] := NgInfo;

         GrpHdl  := Hamster.ArticleBase.Open( NgName );
         if GrpHdl >= 0 then try

            NgInfo.LastRead := Hamster.ArticleBase.GetDT( GrpHdl, gsLastClientRead );
            NgInfo.LastPull := Hamster.ArticleBase.GetDT( GrpHdl, gsLastServerPull );

            NgInfo.ArtCount := Hamster.ArticleBase.Count[GrpHdl];
            inc( SumCount, NgInfo.ArtCount );

            NgInfo.GrpClass := Hamster.Config.Newsgroups.GroupClass( NgName );
            inc( NgClassCount[ NgInfo.GrpClass ] );

            NgInfo.DiskSpace := Hamster.ArticleBase.DiskspaceUsed( NgName );
            inc( SumDisk, NgInfo.DiskSpace );

         finally Hamster.ArticleBase.Close( GrpHdl ) end;
      end;

     
      // summary report
      Report.Add( '<h2> Summary </h2>' );
      Report.Add( '' );
      Report.Add( '<table border=1 width=100%>' );

      Report.Add( '<tr> <td> Total number of articles in groups </td>'
                + '<td>' + i2n( SumCount ) + '</td> </tr>' );
      Report.Add( '<tr> <td> ... disk space used </td>'
                + '<td>' + i2n( SumDisk ) + '</td> </tr>' );

      Report.Add( '<tr> <td> Total number of entries in history </td>'
                + '<td>' + i2n( Hamster.NewsHistory.Count ) + '</td> </tr>' );
      Report.Add( '<tr> <td> ... disk space used </td>'
                + '<td>' + i2n( Hamster.NewsHistory.DiskspaceUsed ) + '</td> </tr>' );

      Report.Add( '<tr> <td> Total number of newsgroups </td>'
                + '<td>' + i2n( NgList.Count ) + '</td> </tr>' );
      Report.Add( '<tr> <td> ... unknown type </td>'
                + '<td>' + i2n( NgClassCount[aclUnknown] ) + '</td> </tr>' );
      Report.Add( '<tr> <td> ... internal </td>'
                + '<td>' + i2n( NgClassCount[aclInternal] ) + '</td> </tr>' );
      Report.Add( '<tr> <td> ... local </td>'
                + '<td>' + i2n( NgClassCount[aclLocal] ) + '</td> </tr>' );
      Report.Add( '<tr> <td> ... currently not pulled </td>'
                + '<td>' + i2n( NgClassCount[aclWasPulled] ) + '</td> </tr>' );
      Report.Add( '<tr> <td> ... currently pulled </td>'
                + '<td>' + i2n( NgClassCount[aclIsPulled] ) + '</td> </tr>' );

      Report.Add( '</table>' );
      Report.Add( '' );


      // 'by name' report
      NgList.Sort;
      Report.Add( '<h2> Newsgroups </h2>' );
      Report.Add( '' );
      Report.Add( '<table border=1 width=100%>' );
      Report.Add( '<tr>'
                + ' <th align=left>Newsgroup</th>'
                + ' <th width=10%>Count</th>'
                + ' <th width=10%>Size</th>'
                + ' <th width=10%>Last&nbsp;Read</th> '
                + ' <th width=10%>Last&nbsp;Pull</th> '
                + ' <th width=10%>Type</th>'
                + ' </tr>' );

      for i := 0 to NgList.Count - 1 do begin
         with TNewsgroupInfo( NgList.Objects[i] ) do begin
            Report.Add( '<tr>'
                      + ' <td>' + Groupname + '</td>'
                      + ' <td align=right>' + i2n( ArtCount ) + '</td>'
                      + ' <td align=right>' + i2n( DiskSpace ) + '</td>'
                      + ' <td align=right>'
                         + iif( LastRead=0, '-', IntToStr( Trunc( StatsNow - LastRead ) ) )
                         + '</td> '
                      + ' <td align=right>'
                         + iif( LastPull=0, '-', IntToStr( Trunc( StatsNow - LastPull ) ) )
                         + '</td> '
                      + ' <td align=center>' + TActiceClassDesc[GrpClass] + '</td>'
                      + ' </tr>' );
         end;
      end;

      Report.Add( '<tr>'
                + ' <td>Total: ' + i2n( NgList.Count ) + ' newsgroups </td>'
                + ' <td align=right>' + i2n( SumCount ) + '</td>'
                + ' <td align=right>' + i2n( SumDisk ) + '</td>'
                + ' <td>&nbsp;</td> '
                + ' <td>&nbsp;</td> '
                + ' <td>&nbsp;</td>'
                + ' </tr>' );

      Report.Add( '</table>' );
      
      Report.Add( '<p><small>' );
      Report.Add( '<b>Newsgroup:</b> name of group, <b>Count:</b> number of messages stored in this group, <b>Size:</b> disk space used for this group, ' );
      Report.Add( '<b>Last Read:</b> number of days since group was read by a user/newsreader, ' );
      Report.Add( '<b>Last Pull:</b> number of days since group was pulled from a remote server, ' );
      Report.Add( '<b>Type:</b> type of the group' );
      Report.Add( '</small></p>' );

      Result := True;

   finally
      for i := 0 to NgList.Count-1 do TNewsgroupInfo( NgList.Objects[i] ).Free;
      NgList.Free
   end;
end;

function TThreadReports.Report_Mailboxes(
         const Report: TStringList; const Month: TDateTime; const Filename: String ): Boolean;
var  AccList: TStringList;
     i, UID, UCount, SumCount: Integer;
     USize, SumSize: Int64;
     UName, UPath: String;
     UFirst, ULast, dt: TDateTime;
     SR: TSearchRec;
begin
   AccList := TStringList.Create;
   try

      AccList.Sorted := True;
      Hamster.Accounts.EnumUsernames( AccList );

      Report.Add( '<table border=1 width=100%>' );
      Report.Add( '<tr> '
                 + ' <th align=left>Username</th>'
                 + ' <th width=10%>Count</th>'
                 + ' <th width=10%>Size</th>'
                 + ' <th width=10%>Oldest</th>'
                 + ' <th width=10%>Newest</th>'
                 + ' </tr>' );

      SumCount := 0;
      SumSize  := 0;
      for i := 0 to AccList.Count - 1 do begin

         UName := AccList[i];
         UID   := Integer( AccList.Objects[i] ); // Hamster.Accounts.UserIDOf( UName );
         if UID <> ACTID_INVALID then begin
            if Hamster.Accounts.HasMailbox( UID ) then begin

               UPath  := Hamster.Accounts.MailboxPath( UID );
               UCount := 0;
               USize  := 0;
               UFirst := 0;
               ULast  := 0;

               if SysUtils.FindFirst( UPath + '*.msg', faAnyFile, SR ) = 0 then begin
                  repeat
                     inc( UCount );
                     inc( USize, SR.Size );
                     dt := FileDateToDateTime( SR.Time );
                     if UFirst = 0  then UFirst := dt;
                     if UFirst > dt then UFirst := dt;
                     if ULast  = 0  then ULast  := dt;
                     if ULast  < dt then ULast  := dt;
                  until SysUtils.FindNext( SR ) <> 0;

                  SysUtils.FindClose( SR );
               end;

               inc( SumCount, UCount );
               inc( SumSize, USize );
               Report.Add( '<tr> '
                         + '<td>' + UName + '</td>'
                         + '<td align=right>' + iif( UCount=0, '-', i2n( UCount ) ) + '</td>'
                         + '<td align=right>' + iif( UCount=0, '-', i2n( USize  ) ) + '</td>'
                         + '<td align=right>' + iif( UCount=0, '-', inttostr( Trunc(Now-UFirst) ) ) + '</td>'
                         + '<td align=right>' + iif( UCount=0, '-', inttostr( Trunc(Now-ULast ) ) ) + '</td>'
                         + ' </tr>' );

            end;
         end;
         
      end;

      Report.Add( '<tr> '
                + '<td>&nbsp;</td>'
                + '<td align=right>' + i2n( SumCount ) + '</td>'
                + '<td align=right>' + i2n( SumSize ) + '</td>'
                + '<td>&nbsp;</td>'
                + '<td>&nbsp;</td>'
                + ' </tr>' );

      Report.Add( '</table>' );

      Report.Add( '<p><small>' );
      Report.Add( '<b>Username:</b> mailbox-/username, ' );
      Report.Add( '<b>Count:</b> number of waiting mails, ' );
      Report.Add( '<b>Size:</b> disk space occupied by waiting mails, ' );
      Report.Add( '<b>Oldest:</b> age of oldest waiting mail (in days), ' );
      Report.Add( '<b>Newest:</b> age of newest waiting mail (in days)' );
      Report.Add( '</small></p>' );

      Result := True;

   finally AccList.Free end;
end;

function TThreadReports.Report_MailFilters(
         const Report: TStringList; const Month: TDateTime; const Filename: String ): Boolean;
var  Fields, Hits: TStringList;
     LogFile, line, key: String;
     iHit, i: Integer;
     f: Text;
begin
   Result := False;
   
   // MailFilters.log available?
   LogFile := AppSettings.GetStr( asPathLogs )
            + LOGFILE_MAILFILTERS
            + FormatDateTime( '"-"yyyy"-"mm', Month )
            + LOGFILE_EXTENSION;
   if not FileExists( LogFile ) then exit;

   // is report already up to date?
   if FileExists( Filename ) then begin
      if FileDateToDateTime( FileAge( Filename ) ) >=
         FileDateToDateTime( FileAge( LogFile  ) ) then exit;
   end;

   Fields  := TStringList.Create;

   Hits := TStringList.Create;
   Hits.Sorted := True;

   try

      // load MailFilters.log
      HamFileEnter;
      try

         AssignFile( f, LogFile );
         Reset( f );
         try
         
            while not Eof( f ) do begin

               ReadLn( f, line );
               if Trim( line ) = '' then continue;

               ArgsSplitChar( line, Fields, #9, 10 );
               // [0]=timstamp [1]=serveralias [2]=origin [3]=marker [4]=filter

               key  := '[' + Fields[2] + '] ' + Fields[4];
               iHit := Hits.IndexOf( key );
               if iHit < 0 then begin
                  iHit := Hits.AddObject( key, TMailFilterHit.Create( Fields[2], Fields[4] ) );
               end;
               with TMailFilterHit( Hits.Objects[iHit] ) do begin
                  inc( HitCount );
                  LastHit := LogTimeToDateTime( Fields[0] );
               end;

            end;

         finally CloseFile( f ) end;

      finally HamFileLeave end;

      Hits.Sorted := False; // just used for quicker lookups

      // report filtered mails
      Report.Add( '<p> This reports shows all mail filters, that were actually involved when deleting mails. </p>' );

      Report.Add( '<table border=1 width=100%>' );
      Report.Add( '<tr> '
                + '<th width=5%>Count</th>'
                + '<th width=5%>Days</th>'
                + '<th width=5%>Origin</th>'
                + '<th align=left>Filter</th>'
                + ' </tr>' );

      Hits.CustomSort( ComparyByHitCount );
      for iHit := 0 to Hits.Count - 1 do begin
         with TMailFilterHit( Hits.Objects[iHit] ) do begin
            Report.Add( '<tr> '
                      + '<td align=center>' + inttostr(HitCount) + '</td> '
                      + '<td align=center>' + IntToStr( Trunc( Now - LastHit ) ) + '</td> '
                      + '<td align=center>' + Origin + '</td>'
                      + '<td>' + EnHtml( Filter ) + '</td>'
                      + ' </tr>' );
         end;
      end;
      Report.Add( '</table>' );

      Report.Add( '<p><small>' );
      Report.Add( '<b>Count:</b> total number of hits, ' );
      Report.Add( '<b>Days:</b> number of days since last hit, ' );
      Report.Add( '<b>Origin:</b> type of filter, ' );
      Report.Add( '<b>Filter:</b> matching filter-line' );
      Report.Add( '</small></p>' );

      Result := True;

   finally
      for i := 0 to Hits.Count - 1 do TMailFilterHit( Hits.Objects[i] ).Free;
      Hits.Free;
      Fields.Free;
   end;
end;

function TThreadReports.Report_MailTrapUsage(
         const Report: TStringList; const Month: TDateTime; const Filename: String ): Boolean;
var  Fields, Hits: TStringList;
     LogFile, line, key, s: String;
     iLine, iHit, i, n: Integer;
     f: Text;
begin
   Result := False;
   
   // MailTrapHits.log available?
   LogFile := AppSettings.GetStr( asPathLogs )
            + LOGFILE_MAILTRAPHITS
            + FormatDateTime( '"-"yyyy"-"mm', Month )
            + LOGFILE_EXTENSION;
   if not FileExists( LogFile ) then exit;

   // is report already up to date?
   if FileExists( Filename ) then begin
      if FileDateToDateTime( FileAge( Filename ) ) >=
         FileDateToDateTime( FileAge( LogFile  ) ) then exit;
   end;

   Fields  := TStringList.Create;

   Hits := TStringList.Create;
   Hits.Sorted := True;

   try

      // load MailTrapHits.log
      HamFileEnter;
      try

         AssignFile( f, LogFile );
         Reset( f );
         try

            while not Eof( f ) do begin

               ReadLn( f, line );
               if Trim( line ) = '' then continue;

               ArgsSplitChar( line, Fields, #9, 10 );
               // [0]=timstamp [1]=trap-action [2]=score [3]=filter

               key  := Fields[3];
               iHit := Hits.IndexOf( key );
               if iHit < 0 then begin
                  iHit := Hits.AddObject( key, TMailFilterHit.Create( 'MailTrap', key ) );

                  with TMailFilterHit( Hits.Objects[iHit] ) do begin
                     s := copy(Fields[1]+'?',1,1) + copy(Fields[2]+'?',1,1);
                     if (s[1] = '0') then Group := 0;
                     if (s[1] = '1') then Group := 1;
                     if (s[1] = '2') then begin
                        if s[2] = '-' then Group := 3 else Group := 2;
                     end;
                  end;

               end;
               with TMailFilterHit( Hits.Objects[iHit] ) do begin
                  inc( HitCount );
                  LastHit := LogTimeToDateTime( Fields[0] );
               end;

            end;

         finally CloseFile( f ) end;

      finally HamFileLeave end;

      // add all currently active traps without any hits so far
      (*
      Hamster.MailTrap.Lock.BeginRead;
      try
         for iLine := 0 to Hamster.MailTrap.List.Count - 1 do begin
            with TMailTrapItem( Hamster.MailTrap.List[iLine] ) do begin
               key  := AsInfo;
               iHit := Hits.IndexOf( key );
               if iHit < 0 then begin
                  iHit := Hits.AddObject( key, TMailFilterHit.Create( 'MailTrap', key ) );
                  with TMailFilterHit( Hits.Objects[iHit] ) do begin
                     if Action = mtaAccept then Group := 0;
                     if Action = mtaDelete then Group := 1;
                     if Action = mtaScore then begin
                        if ScoreValue < 0 then Group := 3 else Group := 2;
                     end;
                  end;
               end;
            end;
         end;
      finally Hamster.MailTrap.Lock.EndRead end;
      *)

      // report filtered mails
      Report.Add( '<p> This reports shows all hits for any mail traps, that were actually tested (i. e. not including those following a final decision).</p>' );

      // resort by hits+age
      Hits.Sorted := False; // just used for quicker lookups
      Hits.CustomSort( ComparyByHitCount );

      for iLine := 0 to 3 do begin // ACCEPT/DELETE/SCORE+/SCORE-

         case iLine of
            0: Report.Add( '<h2> ACCEPT traps </h2>' );
            1: Report.Add( '<h2> DELETE traps </h2>' );
            2: Report.Add( '<h2> SCORE (+) traps </h2>' );
            3: Report.Add( '<h2> SCORE (-) traps </h2>' );
         end;
         Report.Add( '' );

         Report.Add( '<table border=1 width=100%>' );
         Report.Add( '<tr> '
                   + '<th width=5%>Count</th> <th width=5%>Days</th> <th align=left>Trap</th>'
                   + ' </tr>' );

         n := 0;
         for iHit := 0 to Hits.Count - 1 do begin
            with TMailFilterHit( Hits.Objects[iHit] ) do begin
               if iLine = Group then begin
                  Report.Add( '<tr> '
                            + '<td align=center>'
                               + iif( HitCount=0, '-', inttostr(HitCount) )
                            + '</td> '
                            + '<td align=center>'
                               + iif( StillInUse, '', '(' )
                               + iif( LastHit=0, '-', IntToStr( Trunc( Now - LastHit ) ) )
                               + iif( StillInUse, '', ')' )
                            + '</td> '
                            + '<td>' + EnHtml( Filter ) + '</td>'
                            + ' </tr>' );
                  inc( n );
               end;
            end;
         end;
         if n = 0 then begin
            Report.Add( '<tr> '
                      + '<td align=center> - </td> '
                      + '<td align=center> - </td> '
                      + '<td> - </td>'
                      + ' </tr>' );
         end;
         Report.Add( '</table>' );

         Report.Add( '<p><small>' );
         Report.Add( '<b>Count:</b> total number of hits, ' );
         Report.Add( '<b>Days:</b> number of days since last hit (a value in parens marks traps, that are not in use any more), ' );
         Report.Add( '<b>Trap:</b> matching trap' );
         Report.Add( '</small></p>' );

      end;

      Result := True;

   finally
      for i := 0 to Hits.Count - 1 do TMailFilterHit( Hits.Objects[i] ).Free;
      Hits.Free;
      Fields.Free;
   end;
end;

function TThreadReports.Report_Mails(
         const Report: TStringList; const Month: TDateTime; const Filename: String ): Boolean;
var  Fields, Hits: TStringList;
     Sums: TCounterInfo;
     LogFile, line, key: String;
     iHit: Integer;
     f: Text;
begin
   Result := False;

   Fields  := TStringList.Create;
   Hits    := TStringList.Create;
   Sums    := TCounterInfo.Create;
   try

      // ---------- Smtp server usage ------------------------------
      LogFile := AppSettings.GetStr( asPathLogs )
               + LOGFILE_SMTPSERVER
               + FormatDateTime( '"-"yyyy"-"mm', Month )
               + LOGFILE_EXTENSION;
      if FileExists( LogFile ) then begin

         Report.Add( '<h2> Mails received by local SMTP server </h2>' );

         HamFileEnter;
         try

            AssignFile( f, LogFile );
            Reset( f );
            try

               Hits.Clear;
               Hits.Sorted := True;

               while not Eof( f ) do begin

                  ReadLn( f, line );
                  if Trim( line ) = '' then continue;

                  ArgsSplitChar( line, Fields, #9, 10 );
                  // [0]=timstamp [1]=user [2]=ip [3]=action [4]=result,
                  // [5]=envelope [6]=from, [7]=to [8]=subject [9]=date

                  key := SkipKey( SkipPort( Fields[1] ) )
                       + ' (' + SkipKey( SkipPort( Fields[2] ) ) + ')';
                  iHit := Hits.IndexOf( key );
                  if iHit < 0 then begin
                     Hits.AddObject( key, Pointer(1) );
                  end else begin
                     Hits.Objects[iHit] := Pointer( Integer(Hits.Objects[iHit]) + 1 );
                  end;

               end;

            finally CloseFile( f ) end;

         finally HamFileLeave end;

         Report.Add( '<table border=1 width=100%>' );
         Report.Add( '<tr> <th width=10%>Count</th> <th align=left>User (from IP)</th> </tr>' );
         for iHit := 0 to Hits.Count - 1 do begin
            Report.Add( '<tr> '
                      + '<td align=center>' + i2n(Integer(Hits.Objects[iHit])) + '</td> '
                      + '<td>' + Hits[iHit] + '</td> '
                      + ' </tr>' );
         end;
         Report.Add( '</table>' );

         Hits.Clear;

         Result := True;

      end;

      // ---------- Mails sent out ------------------------------
      LogFile := AppSettings.GetStr( asPathLogs )
               + LOGFILE_MAILOUT
               + FormatDateTime( '"-"yyyy"-"mm', Month )
               + LOGFILE_EXTENSION;
      if FileExists( LogFile ) then begin

         Report.Add( '<h2> Mails delivered to remote servers </h2>' );

         HamFileEnter;
         try

            AssignFile( f, LogFile );
            Reset( f );
            try

               Hits.Clear;
               Hits.Sorted := True;

               while not Eof( f ) do begin

                  ReadLn( f, line );
                  if Trim( line ) = '' then continue;

                  ArgsSplitChar( line, Fields, #9, 10 );
                  // [0]=timstamp [1]=file [2]=server [3]=result [4]=envelope
                  // [5]=from [6]=to, [7]=subject

                  key := SkipKey( SkipPort( Fields[2] ) )
                       + ' (' + SkipKey( SkipPort( Fields[3] ) ) + ')';
                  iHit := Hits.IndexOf( key );
                  if iHit < 0 then begin
                     Hits.AddObject( key, Pointer(1) );
                  end else begin
                     Hits.Objects[iHit] := Pointer( Integer(Hits.Objects[iHit]) + 1 );
                  end;

               end;

            finally CloseFile( f ) end;

         finally HamFileLeave end;

         Report.Add( '<table border=1 width=100%>' );
         Report.Add( '<tr> <th width=10%>Count</th> <th align=left>Remote Server (Result)</th> </tr>' );
         for iHit := 0 to Hits.Count - 1 do begin
            Report.Add( '<tr> '
                      + '<td align=center>' + i2n(Integer(Hits.Objects[iHit])) + '</td> '
                      + '<td>' + Hits[iHit] + '</td> '
                      + ' </tr>' );
         end;
         Report.Add( '</table>' );

         Hits.Clear;

         Result := True;

      end;

      // ---------- Mails received ------------------------------

      Report.Add( '<h2> Mails fetched from remote servers </h2>' );

      Hits.Clear;
      Hits.Sorted := True;

      LogFile := AppSettings.GetStr( asPathLogs )
               + LOGFILE_MAILIN
               + FormatDateTime( '"-"yyyy"-"mm', Month )
               + LOGFILE_EXTENSION;
      if FileExists( LogFile ) then begin

         HamFileEnter;
         try

            AssignFile( f, LogFile );
            Reset( f );
            try

               while not Eof( f ) do begin

                  ReadLn( f, line );
                  if Trim( line ) = '' then continue;

                  ArgsSplitChar( line, Fields, #9, 10 );
                  // [0]=timstamp [1]=server [2]=result [3]=from [4]=to
                  // [5]=subject [6]=date

                  key := SkipKey( SkipPort( Fields[1] ) );
                  iHit := Hits.IndexOf( key );
                  if iHit < 0 then begin
                     iHit := Hits.AddObject( key, TCounterInfo.Create() );
                  end;

                  inc( TCounterInfo( Hits.Objects[iHit] ).Counter[0] );
                  inc( TCounterInfo( Hits.Objects[iHit] ).Counter[1] );

               end;

            finally CloseFile( f ) end;

         finally HamFileLeave end;

      end;

      LogFile := AppSettings.GetStr( asPathLogs )
               + LOGFILE_MAILFILTERS
               + FormatDateTime( '"-"yyyy"-"mm', Month )
               + LOGFILE_EXTENSION;
      if FileExists( LogFile ) then begin
      
         HamFileEnter;
         try

            AssignFile( f, LogFile );
            Reset( f );
            try

               while not Eof( f ) do begin

                  ReadLn( f, line );
                  if Trim( line ) = '' then continue;

                  ArgsSplitChar( line, Fields, #9, 10 );
                  // [0]=timstamp [1]=serveralias [2]=origin [3]=marker [4]=filter

                  if Fields[3] = '1' then begin // '1'=first, '0'=continuation
                     key := SkipPort( Fields[1] );
                     iHit := Hits.IndexOf( key );
                     if iHit < 0 then begin
                        iHit := Hits.AddObject( key, TCounterInfo.Create() );
                     end;

                     inc( TCounterInfo( Hits.Objects[iHit] ).Counter[0] );
                     inc( TCounterInfo( Hits.Objects[iHit] ).Counter[2] );
                  end;

               end;

            finally CloseFile( f ) end;

         finally HamFileLeave end;

      end;

      Report.Add( '<table border=1 width=100%>' );
      Report.Add( '<tr> <th width=10%>Count</th> <th width=10%>Fetched</th> <th width=10%>Filtered</th> <th width=10%>Filtered</th> <th align=left>Remote Server (Result)</th> </tr>' );
      Sums.Clear;
      for iHit := 0 to Hits.Count - 1 do begin
         with TCounterInfo( Hits.Objects[iHit] ) do begin
            Report.Add( '<tr> '
                      + '<td align=center>' + i2n(Counter[0]) + '</td> '
                      + '<td align=center>' + i2n(Counter[1]) + '</td> '
                      + '<td align=center>' + i2n(Counter[2]) + '</td> '
                      + '<td align=center>' + iPercent(Counter[2],Counter[0]) + '</td> '
                      + '<td>' + Hits[iHit] + '</td> '
                      + ' </tr>' );
            inc( Sums.Counter[0], Counter[0] );
            inc( Sums.Counter[1], Counter[1] );
            inc( Sums.Counter[2], Counter[2] );
            Result := True;
         end;
      end;

      Report.Add( '<tr> '
                + '<td align=center>' + i2n(Sums.Counter[0]) + '</td> '
                + '<td align=center>' + i2n(Sums.Counter[1]) + '</td> '
                + '<td align=center>' + i2n(Sums.Counter[2]) + '</td> '
                + '<td align=center>' + iPercent(Sums.Counter[2],Sums.Counter[0]) + '</td> '
                + '<td>' + '<b>Total</b>' + '</td> '
                + ' </tr>' );

      Report.Add( '</table>' );

      for iHit := 0 to Hits.Count - 1 do TCounterInfo( Hits.Objects[iHit] ).Free;
      Hits.Clear;

   finally
      Fields.Free;
      Hits.Free;
      Sums.Free;
   end;
end;

procedure TThreadReports.ReportHeader( const Report: TStringList; const Title: String );
begin
   Report.Clear;

   Report.Add( '<html>' );
   Report.Add( '' );
   Report.Add( '<head>' );
   Report.Add( '<title> Hamster: ' + Title + ' </title>' );
   Report.Add( '<meta http-equiv="Content-Type" content="text/html; charset=iso-8859-1">' );
   Report.Add( '</head>' );
   Report.Add( '' );
   Report.Add( '<body>' );
   Report.Add( '' );
   Report.Add( '<h1> Hamster: ' + Title + ' </h1>' );
   Report.Add( '' );
end;

procedure TThreadReports.ReportFooter( const Report: TStringList );
begin
   Report.Add( '<br>' );
   Report.Add( '<p><small><i> Created ' + FormatDateTime( 'yyyy"-"mm"-"dd hh":"nn":"ss', Now )
             + ' by ' + GetMyStringFileInfo('ProductName','Hamster') + ' Vr. ' + GetExeVersion
             + '</i></small></p>' );
   Report.Add( '</body>' );
   Report.Add( '</html>' );
end;

procedure TThreadReports.CreateReport( const FilenameBase, TitleBase: String;
                                       const Reporter: TReportCreateProc;
                                       const Month: TDateTime );
var  Report: TStringList;
     Pagename, Filename, Title: String;
begin
   Report := TStringList.Create;
   try

      try
         Pagename := FilenameBase
                   + FormatDateTime( '"-"yyyy"-"mm', Month )
                   + REPFILE_EXTENSION;
         Filename := AppSettings.GetStr( asPathReports ) + Pagename;
         Title := TitleBase + FormatDateTime( ' (mm"/"yyyy)', Month );

         StateInfo := 'Create report: ' + Title;

         ReportHeader( Report, Title );
         if Reporter( Report, Month, Filename ) then begin

            TLog( LOGID_INFO, Filename + ' - ' + Title );
            ReportTitles.Values[ ExtractFilename(Filename) ] := Title;
            ReportFooter( Report );
            Report.SaveToFile( Filename );

         end;

      except
         on E:Exception do TLog( LOGID_ERROR, 'Error: ' + E.Message );
      end;

   finally Report.Free end;
end;

procedure TThreadReports.CreateIndexReport( ReportTitles: TStringList );
var  IndexReport, Pages: TStringList;
     IndexFile, IndexTitle: String;
     sr: TSearchRec;
     title, current, heading, period: String;
     i, j: Integer;
begin
   IndexReport := TStringList.Create;
   Pages := TStringList.Create;
   try

      IndexFile  := AppSettings.GetStr( asPathReports ) + REPFILE_INDEXPAGE;
      IndexTitle := 'Reports Index';
      ReportHeader( IndexReport, IndexTitle );

      if FindFirst( AppSettings.GetStr( asPathReports ) + '*' + REPFILE_EXTENSION,
                    faAnyfile, sr ) = 0 then try

         repeat
            if (sr.Attr and faDirectory) = 0 then begin
               if AnsiCompareText( sr.Name, REPFILE_INDEXPAGE ) <> 0 then begin
                  Pages.Add( sr.Name );
               end;
            end;
         until FindNext( sr ) <> 0;

      finally
         FindClose( sr );
      end;

      Pages.Sort;

      current := '';
      for i := 0 to Pages.Count - 1 do begin

         title := ReportTitles.Values[ Pages[i] ];
         if title = '' then title := Pages[i];

         j := Pos( '(', title );
         if j > 0 then begin
            heading := Trim( copy( title, 1, j-1 ) );
            period  := Trim( copy( title, j+1, MaxInt ) );
            if copy( period, length(period), 1 ) = ')' then System.Delete( period, length(period), 1 ); 
         end else begin
            heading := 'Other';
            period  := title;
         end;

         if heading <> current then begin
            current := heading;
            IndexReport.Add( '<h2>' + current + '</h2>' );
         end;

         IndexReport.Add( '<a href="' + Pages[i] + '">' + period + '</a> &nbsp; ' );

      end;

      TLog( LOGID_INFO, IndexFile + ' - ' + IndexTitle );
      ReportTitles.Values[ ExtractFilename(IndexFile) ] := IndexTitle;
      ReportFooter( IndexReport );
      IndexReport.SaveToFile( IndexFile );

   finally
      Pages.Free;
      IndexReport.Free;
   end;
end;

procedure TThreadReports.Execute;
var  RepPath: String;
     y, m, d: Word;
     created: Boolean;
begin
   TLog( LOGID_SYSTEM, 'Start' );
   StateInfo := 'Create reports';
   created := False;

   try
      try
         DecodeDate( Now, y, m, d );
         CurrentMonth  := EncodeDate( y, m, 1 );
         dec( m ); if m < 1 then begin dec(y); m := 12; end;
         PreviousMonth := EncodeDate( y, m, 1 );

         ReportTitles := TStringList.Create;
         try

            RepPath := AppSettings.GetStr( asPathReports );
            if FileExists( RepPath + REPFILE_REPORTTITLES ) then begin
               ReportTitles.LoadFromFile( RepPath + REPFILE_REPORTTITLES );
            end;


            CreateReport( 'Diskspace', 'Disk Space', Report_DiskSpace, CurrentMonth );

            CreateReport( 'Newsgroups', 'Newsgroups', Report_Newsgroups, CurrentMonth );

            CreateReport( 'Mailboxes', 'Mailboxes', Report_Mailboxes, CurrentMonth );

            CreateReport( 'MailTransfer', 'Mail Transfer', Report_Mails, PreviousMonth );
            CreateReport( 'MailTransfer', 'Mail Transfer', Report_Mails, CurrentMonth );

            CreateReport( 'MailFilters', 'Mail Filters', Report_MailFilters, PreviousMonth );
            CreateReport( 'MailFilters', 'Mail Filters', Report_MailFilters, CurrentMonth );

            CreateReport( 'MailTraps', 'Mail Traps', Report_MailTrapUsage, PreviousMonth );
            CreateReport( 'MailTraps', 'Mail Traps', Report_MailTrapUsage, CurrentMonth );

            CreateIndexReport( ReportTitles );

            ReportTitles.SaveToFile( RepPath + REPFILE_REPORTTITLES );

            created := True;

         finally
            ReportTitles.Free;
         end;
      
      except
         on E:Exception do TLog( LOGID_ERROR, 'Error: ' + E.Message );
      end;

      if created then Hamster.HscActions.Execute( actReportsCreated, '' );

   finally
      TLog( LOGID_SYSTEM, 'End' );
   end;
end;

constructor TThreadReports.Create( FreeType: TThreadFreeTypes );
begin
   inherited Create( attMaintenance, '{create reports}', FreeType );
   Priority := tpLower;
end;

end.
