// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit tMaintenance; // Threads for local maintenance.

// ----------------------------------------------------------------------------
// Contains threads for local maintenance. 
// ----------------------------------------------------------------------------

interface

{$INCLUDE Compiler.inc}

uses Windows, SysUtils, Classes, uTools, uConst, uType, tBase;

type
  TThreadPurge = class( TTaskThread )
    protected
      StartBits: Integer;
      Groupname: String;
      procedure ExecutePurge;
      procedure Execute; override;
    public
      constructor Create( AStartBits: Integer;
                          AGroupname: String;
                          FreeType: TThreadFreeTypes );
  end;

  TThreadHistoryRebuild = class( TTaskThread )
    protected
      procedure Execute; override;
    public
      constructor Create;
  end;

  TThreadRebuildGlobalLists = class( TTaskThread )
    protected
      LstGroups: TStringList_NoAnsi;
      LstDescs : TStringList;
      procedure Prepare_PullGroupsWithDescriptions;
      procedure SaveList_PullGroupsWithDescriptions;
      procedure Execute; override;
    public
      constructor Create( FreeType: TThreadFreeTypes );
  end;

  TThreadSpamReport = class( TTaskThread )
    protected
      procedure Execute; override;
    public
      constructor Create( FreeType: TThreadFreeTypes );
  end;

  TThreadStatistics = class( TTaskThread )
    protected
      procedure Execute; override;
    public
      constructor Create( FreeType: TThreadFreeTypes );
  end;

  TThreadAutoUnsubscribe = class( TTaskThread )
    protected
      procedure Execute; override;
    public
      constructor Create( FreeType: TThreadFreeTypes );
  end;

  TThreadDailyMaintenance = class( TTaskThread )
    protected
      procedure Execute; override;
    public
      constructor Create;
  end;

implementation

uses uConstVar, uVar, cArtFiles, uDateTime, uCRC32, cLogFileHamster, cHamster,
     uHamTools, cTraps, tReports, cMailDispatcher, uEncoding;

// --------------------------------------------------------- TThreadPurge -----

procedure TThreadPurge.ExecutePurge;
var  LfdGroup: Integer;
     GrpHdl  : LongInt;
     StrLst  : TStringList;
     i       : Integer;
     Parser  : TParser;
     Base    : TDateTime;
     Days    : LongInt;
     PurgeCount: Integer;
     GrpName : String;
begin
     if (StartBits and HAM_PURGEOPT_DONEWS)<>0 then begin
        StrLst := TStringList.Create;
        Hamster.Config.BeginRead;
        try
           for LfdGroup:=0 to Hamster.Config.Newsgroups.Count-1 do begin
              Sleep( 1 );
              if Terminated then break;
              GrpName := Hamster.Config.Newsgroups.Name[LfdGroup];
              If (GroupName = '') or (LowerCase(GrpName) = LowerCase(Groupname)) then begin
                 if Hamster.ArticleBase.UseCount( GrpName ) = 0 then begin
                    GrpHdl := Hamster.ArticleBase.Open( GrpName );
                    Hamster.ArticleBase.Purge( GrpHdl );
                    Hamster.ArticleBase.Close( GrpHdl );
                 end else begin
                    TLog( LOGID_INFO, Format(
                       '%s delayed: group is in use.', [GrpName] ));
                    StrLst.Add( GrpName );
                 end
              end
           end;
           for LfdGroup:=0 to StrLst.Count-1 do begin
              Sleep( 1 );
              if Terminated then break;
              GrpName := StrLst[LfdGroup];
              Sleep( 1000 );
              if Hamster.ArticleBase.UseCount( GrpName ) = 0 then begin
                 StateInfo := 'Purge group ' + GrpName;
                 GrpHdl := Hamster.ArticleBase.Open( GrpName );
                 Hamster.ArticleBase.Purge( GrpHdl );
                 Hamster.ArticleBase.Close( GrpHdl );
              end else begin
                 TLog( LOGID_INFO, Format(
                    '%s skipped: group still in use.', [GrpName] ));
              end;
           end;
        finally
           Hamster.Config.EndRead;
           StrLst.Free;
        end;
     end;

     if (StartBits and HAM_PURGEOPT_DOHISTORY)<>0 then begin
        if not Terminated then begin
           StateInfo := 'Purge news history';
           Hamster.NewsHistory.Purge;
        end;
     end;

     if (StartBits and HAM_PURGEOPT_DOMHISTORY)<>0 then begin
        if not Terminated then begin
           StateInfo := 'Purge mail history';
           Hamster.MailHistory.Purge;
        end;
     end;

     if (StartBits and HAM_PURGEOPT_DOKILLS)<>0 then begin
        if not Terminated then begin
           StateInfo := 'Purge killfile log';
           PurgeCount := 0;

           if FileExists( AppSettings.GetStr(asPathGroups) + GRPFILE_KILLSLOG ) then begin
              Parser := TParser.Create;
              StrLst := TStringList.Create;
              StrLst.LoadFromFile( AppSettings.GetStr(asPathGroups) + GRPFILE_KILLSLOG );

              for i:=StrLst.Count-1 downto 0 do begin
                 Parser.Parse( StrLst[i], #9 );
                 // 0:Server 1:Group 2:Score 3-:Overview
                 // 3:No. 4:Subject 5:From 6:Date 7:Message-ID 8:References 9:Bytes 10:Lines [11:Xref]
                 Base := RfcDateTimeToDateTimeGMT( Parser.sPart(6,'') );
                 Days := Trunc( Now - Base );
                 if ( Days > Hamster.Config.Settings.GetInt(hsPurgeKillsKeepDays) ) and
                    ( Hamster.Config.Settings.GetInt(hsPurgeKillsKeepDays) > 0 ) then begin
                    StrLst.Delete( i );
                    inc( PurgeCount );
                 end;
              end;

              if PurgeCount>0 then StrLst.SaveToFile( AppSettings.GetStr(asPathGroups) + GRPFILE_KILLSLOG );
              StrLst.Free;
              Parser.Free;
           end;

           Log( LOGID_INFO, Format(
             'Purge Kills.log (%sd): %s entries purged.',
             [inttostr(Hamster.Config.Settings.GetInt(hsPurgeKillsKeepDays)), inttostr(PurgeCount)]))
        end;
     end;
end;

procedure TThreadPurge.Execute;
begin
     TLog( LOGID_SYSTEM, 'Start' );
     ExecutePurge;
     TLog( LOGID_SYSTEM, 'End' );
end;

constructor TThreadPurge.Create( AStartBits: Integer;
                                 AGroupname: String;
                                 FreeType: TThreadFreeTypes );
begin
     if AGroupname = '' then begin
        inherited Create( attMaintenance,
                          '{purge all groups}', FreeType );
     end else begin
        inherited Create( attMaintenance,
                          Format('{purge %s}', [AGroupname] ), FreeType );
     end;
     StartBits := AStartBits;
     GroupName := AGroupName;
end;

// ------------------------------------------------ TThreadHistoryRebuild -----

procedure TThreadHistoryRebuild.Execute;
begin
     TLog( LOGID_SYSTEM, 'Start' );
     StateInfo := 'Rebuild news history';
     Hamster.NewsHistory.Rebuild;
     TLog( LOGID_SYSTEM, 'End' );
end;

constructor TThreadHistoryRebuild.Create;
begin
   inherited Create( attMaintenance, '{rebuild history}', tftFreeOnTerminate );
end;

// ----------------------------------------------- TThreadRebuildPullList -----

function GroupChunk( const Groupname: String ): Integer;
var  i: Integer;
begin
   Result := 0;
   for i := 1 to length(Groupname) do begin
      inc( Result, ord( Groupname[i] ) );
   end;
   Result := Result and 7;
end;

procedure TThreadRebuildGlobalLists.Prepare_PullGroupsWithDescriptions;
// Prepare internal list 'LstGroups' containing all unique groups from all
// pull-servers. Its .Objects[]-property points to the group's description
// (-1 if none), which is stored in list 'LstDescs'.
var  SrvCount, SrvCurr, GrpChunk, GrpCurr, Idx, j: Integer;
     SrvName, SrvPath, GrpName, GrpDesc: String;
     SrvNames, SrvPaths: array of String;
     AllGroups: array[0..7] of TStringList_NoAnsi;
     TS: TStringList;
begin
   // prepare list of remote NNTP servers
   Hamster.Config.BeginRead;
   try
      SrvCount := Hamster.Config.NntpServers.Count;
      SetLength( SrvNames, SrvCount );
      SetLength( SrvPaths, SrvCount );
      for SrvCurr := 0 to SrvCount - 1 do begin
         SrvNames[SrvCurr] := Hamster.Config.NntpServers.AliasName[ SrvCurr ];
         SrvPaths[SrvCurr] := Hamster.Config.NntpServers.Path[ SrvCurr ];
      end;
   finally Hamster.Config.EndRead end;

   // prepare helper lists
   TS := TStringList.Create;
   for Idx := 0 to 7 do begin
      AllGroups[Idx] := TStringList_NoAnsi.Create;
      AllGroups[Idx].Sorted := True;
   end;

   try
      // loop through configured pull-servers
      for SrvCurr := 0 to SrvCount - 1 do begin
         if Terminated then break;
         SrvName := SrvNames[ SrvCurr ];
         SrvPath := SrvPaths[ SrvCurr ];

         // add active groups of server
         if FileExists( SrvPath + SRVFILE_GROUPS ) then begin
            try
               TS.LoadFromFile( SrvPath + SRVFILE_GROUPS );
            except TS.Clear end;
            TLog( LOGID_INFO, 'add ' + SrvName + ' ' + SRVFILE_GROUPS
            + ' (' + inttostr(TS.Count) + ') ...' );

            for GrpCurr := 0 to TS.Count - 1 do begin
               if Terminated then break;
               GrpName := TS[ GrpCurr ];
               j := PosWhSpace( GrpName );
               if j > 1 then begin
                  GrpName := copy( GrpName, 1, j-1 );
                  GrpChunk := GroupChunk( GrpName );
                  Idx := AllGroups[GrpChunk].IndexOf( GrpName );
                  if Idx < 0 then AllGroups[GrpChunk].AddObject( GrpName, Pointer(-1) );
               end;
            end;
         end;

         // link descriptions of this server
         if FileExists( SrvPath + SRVFILE_GRPDESCS ) then begin
            try
               TS.LoadFromFile( SrvPath + SRVFILE_GRPDESCS );
            except TS.Clear end;
            if TS.Count <= 3 then begin
               if copy(TS[0],1,1)='#' then TS.Clear; // error-marker
            end;
            TLog( LOGID_INFO, 'add ' + SrvName + ' ' + SRVFILE_GRPDESCS
                            + ' (' + inttostr(TS.Count) + ') ...' );

            for GrpCurr := 0 to TS.Count - 1 do begin
               if Terminated then break;
               GrpName := TS[ GrpCurr ];
               j := PosWhSpace( GrpName );
               if j > 1 then begin
                  GrpDesc := TrimWhSpace( copy( GrpName, j+1, 255 ) );
                  GrpName := copy( GrpName, 1, j-1 );
                  if length(GrpDesc) > 1 then begin
                     GrpChunk := GroupChunk( GrpName );
                     Idx := AllGroups[GrpChunk].IndexOf( GrpName );
                     if Idx >= 0 then begin
                        if Integer(AllGroups[GrpChunk].Objects[Idx]) < 0 then begin
                           AllGroups[GrpChunk].Objects[Idx] := Pointer( LstDescs.Add(GrpDesc) );
                        end;
                     end;
                  end;
               end;
            end;
         end;

      end;

      if not Terminated then begin

         TLog( LOGID_INFO, 'build full list' );
         for GrpChunk := 0 to 7 do begin
            TLog( LOGID_DETAIL, 'add chunk ' + inttostr(GrpChunk) + ': '
                              + inttostr(AllGroups[GrpChunk].Count) );
            for GrpCurr := 0 to AllGroups[GrpChunk].Count - 1 do begin
               LstGroups.AddObject( AllGroups[GrpChunk].Strings[GrpCurr],
                                    AllGroups[GrpChunk].Objects[GrpCurr] );
            end;
            AllGroups[GrpChunk].Clear;
         end;

         TLog( LOGID_INFO, 'sort full list' );
         LstGroups.Sort;

      end;

   finally
      SrvNames := nil;
      SrvPaths := nil;
      TS.Free;
      for Idx := 0 to 7 do AllGroups[Idx].Free;
   end;
end;

procedure TThreadRebuildGlobalLists.SaveList_PullGroupsWithDescriptions;
// Create file alldescs.txt containing all available pull-groups with their
// descriptions. This is the base list used when a user selects new pulls by
// entering search-patterns for groupnames and descriptions.
var  i, j, cnt: Integer;
     T: TextFile;
begin
   cnt := 0;

   HamFileEnter;
   try
      try
         AssignFile( T, AppSettings.GetStr(asPathServer) + SRVFILE_ALLDESCS );
         Rewrite( T );
         for i:=0 to LstGroups.Count-1 do begin
            j := Integer( LstGroups.Objects[i] );
            if j < 0 then writeln( T, LstGroups[i] + #9 + '?' )
                     else writeln( T, LstGroups[i] + #9 + LstDescs[j] );
            inc( cnt );
         end;
         CloseFile( T );

      except
         on E:Exception do TLog( LOGID_ERROR, 'Error: ' + E.Message );
      end;
   finally
      HamFileLeave;
   end;

   TLog( LOGID_INFO, SRVFILE_ALLDESCS + ': ' + inttostr(cnt) + ' entries.' );
end;

procedure TThreadRebuildGlobalLists.Execute;
begin
     TLog( LOGID_SYSTEM, 'Start' );

     LstDescs  := TStringList.Create;
     LstGroups := TStringList_NoAnsi.Create;

     CS_MAINTENANCE.Enter;
     try
        if GlobalListMarker( glTEST ) then begin
           StateInfo := 'Rebuild newsgroup lists';
           if not Terminated then Prepare_PullGroupsWithDescriptions;
           if not Terminated then SaveList_PullGroupsWithDescriptions;
           if not Terminated then GlobalListMarker( glDONE );
        end;
     finally
        CS_MAINTENANCE.Leave;
        LstGroups.Free;
        LstDescs.Free;
     end;

     TLog( LOGID_SYSTEM, 'End' );
end;

constructor TThreadRebuildGlobalLists.Create( FreeType: TThreadFreeTypes );
begin
   inherited Create( attMaintenance, '{rebuild newsgroup lists}', FreeType );
   Priority := tpLower; 
end;


// ---------------------------------------------------- TThreadSpamReport -----

function CompareByScore( List: TStringList;
                         Index1, Index2: Integer ): Integer;
begin
   Result := Integer( List.Objects[Index2] )
           - Integer( List.Objects[Index1] );
end;

procedure TThreadSpamReport.Execute;
var  LogList, Fields, Entries: TStringList;
     Filename, Report, Account, Entry, s: String;
     i, k, Score: Integer;
begin
   TLog( LOGID_SYSTEM, 'Start' );
   StateInfo := 'Create daily spam report';

   Filename := AppSettings.GetStr(asPathLogs) + LOGFILE_SPAMREPORT;
   Account  := Hamster.Config.Settings.GetStr( hsMailSpamReportAccount );
   LogList  := TStringList.Create;
   Fields   := TStringList.Create;
   Entries  := TStringList.Create;
   HamFileEnter;
   
   try
      try
         // load logfile
         if FileExists(Filename) then LogList.LoadFromFile(Filename);

         // build report entries
         for i := 0 to LogList.Count - 1 do begin

            ArgsSplitChar( LogList[i], Fields, #9, 4 );
            // [0]=timstamp [1]=origin [2]=subject [3]=from

            Entry := 'Subject: ' + Fields[2] + #13#10
                   + iif( Pos( '=?', Fields[2] ) > 0,
                     '~Subject: ' + DecodeHeaderValue( Fields[2] ) + #13#10 )
                   + 'From: '    + Fields[3] + #13#10
                   + iif( Pos( '=?', Fields[3] ) > 0,
                     '~From: ' + DecodeHeaderValue( Fields[3] ) + #13#10 )
                   + 'Filter: '  + Fields[1]
                                 + ' (' + Fields[0] + ')' + #13#10
                   + #13#10;
                   
            Score := -2000000;
            k := Pos( ' SCORE ', Fields[1] );
            if k > 0 then begin
               s := Trim( copy( Fields[1], k + length(' SCORE '), MaxInt ) );
               k := Pos( ' ', s );
               if k > 0 then s := copy( s, 1, k-1 );
               Score := strtointdef( s, -1000000 );
            end;

            Entries.AddObject( Entry, Pointer(Score) );
            
         end;

         Entries.CustomSort( CompareByScore );

         // build report
         Report := 'The following mails were filtered:' + #13#10 + #13#10;
         for i := 0 to Entries.Count - 1 do begin
            Report := Report + Entries[i];
         end;

         // mail report
         if (Account <> '') and (LogList.Count > 0) then begin
            SendLocalInfoMail( Account,
                               'Spam Report '
                               + FormatDateTime( 'yyyy"-"mm"-"dd', Now ),
                               Report );
         end;

         // delete logfile
         if FileExists(Filename) then DeleteFile(Filename);

      except end;

   finally
      HamFileLeave;
      Fields.Free;
      LogList.Free;
      Entries.Free;
   end;

   TLog( LOGID_SYSTEM, 'End' );
end;

constructor TThreadSpamReport.Create( FreeType: TThreadFreeTypes );
begin
   inherited Create( attMaintenance, '{create daily spam report}', FreeType );
   Priority := tpLower;
end;

// ---------------------------------------------------- TThreadStatistics -----

procedure TThreadStatistics.Execute;
var  MakeStats: Boolean;
     lGroupsInfo, lOutDated: TStringList;
     i, GrpHdl, CntArt, Cnt, j, p, SumArt: Integer;
     StatsNow, D: TDateTime;
     GrpName, s, StatInfo: String;
begin
   // Stats already created today?
   CS_MAINTENANCE.Enter;
   try
      StatsNow := Now;
      MakeStats := ( Hamster.Config.Settings.GetStr( hsStatsLastInfoMsg )
                     <> FormatDateTime( 'dd"."mm"."yyyy', StatsNow ) );

      if MakeStats then begin
         Hamster.Config.Settings.SetStr(
            hsStatsLastInfoMsg, FormatDateTime( 'dd"."mm"."yyyy', StatsNow ) );
         Hamster.Config.Settings.Flush;
      end;
   finally CS_MAINTENANCE.Leave end;
   if not MakeStats then exit; // Nothing to do

   // Create stats
   TLog( LOGID_SYSTEM, 'Start' );
   StateInfo := 'Create statistics';

   SumArt := 0;
   lGroupsInfo := TStringlist.Create;
   lOutDated   := TStringlist.Create;
   Hamster.Config.BeginRead;

   try
      for i:=0 to Hamster.Config.Newsgroups.Count-1 do begin

         GrpName := Hamster.Config.Newsgroups.Name[i];
         GrpHdl := Hamster.ArticleBase.Open( GrpName );

         if GrpHdl >= 0 then try

            // create list of groups not read by clients within last 3 days
            D := Hamster.ArticleBase.GetDT( GrpHdl, gsLastClientRead );
            if ( StatsNow - D ) >= 3 then begin
               if D = 0 then begin
                  s := 'Never';
                  Cnt := 0;
               end else begin
                  s := FormatDateTime( ShortDateFormat, D );
                  Cnt := Trunc( StatsNow - D );
               end;
               lOutDated.AddObject (s + ': ' + GrpName, Pointer(Cnt));
            end;

            // create list of groups with pull servers
            CntArt := Hamster.ArticleBase.Count[GrpHdl];
            inc( SumArt, CntArt );
            s := GrpName;
            Cnt := 0;
            for j := 0 to Hamster.Config.NewsPulls.Count-1 do begin
               if lowercase(Hamster.Config.NewsPulls.Group[j])=lowercase(GrpName) then begin
                  Inc(Cnt);
                  if Cnt = 1 then s := s + ' (' else s := s + ', ';
                  s := s + Hamster.Config.NewsPulls.Server[j]
               end;
            end;
            If Cnt = 0 then
               s := s + ' (' + 'local Group';
            s := s + ')';
            lGroupsInfo.AddObject( Format( '%6d  %s', [ CntArt, s ] ), Pointer(CntArt) );

            // update pull priority
            if CntArt > 0 then Hamster.NewsJobs.GroupPriority[ GrpName ] := CntArt;

            // update group description (may have been loaded afterwards)
            if length( Hamster.ArticleBase.GetStr( GrpHdl, gsDescription ) ) <= 1 then begin
               s := GlobalGroupDesc( GrpName );
               if length(s) > 1 then
                  Hamster.ArticleBase.SetStr( GrpHdl, gsDescription, s );
            end;

         finally
            Hamster.ArticleBase.Close( GrpHdl );
         end;
      end;

      // group list (sorted by name)
      StatInfo := 'Groups in alphabetical order'+':'
                  +#13#10#13#10 + lGroupsInfo.Text
                  +#13#10#13#10;

      // group list (sorted by article count)
      StatInfo := StatInfo
                  +'Groups sorted by number of articles'+':'
                  +#13#10#13#10;
      with lGroupsInfo do begin
         i := 0;
         while True do begin
            p := -1;
            Cnt := -1;
            for j := 0 to Count-1 do begin
               CntArt := LongInt( Objects[j] );
               if CntArt > Cnt then begin p := j; Cnt := CntArt end;
            end;
            if p < 0 then break;

            inc( i );
            StatInfo := StatInfo + Format('%3d. %s', [i, Strings[p]])+#13#10;
            Delete( p );
         end;
      end;

      // list of groups not read for at least 3 days
      StatInfo := StatInfo + #13#10#13#10
                 + Format(
                  'Groups not pulled from any client longer than %s days', ['3'])+':'
                  +#13#10#13#10;
      with lOutDated do begin
         for i := 0 to Count-1 do begin
            p := -1;
            Cnt := 0;
            for j := 0 to Count-1 do begin
               CntArt := LongInt( Objects[j] );
               if CntArt = 0 then begin
                  p := j; break
               end else if CntArt > Cnt then begin
                  p := j; Cnt := CntArt
               end
            end;
            if p < 0 then break;

            StatInfo := StatInfo + Strings[p] + #13#10;
            Delete( p );
         end
      end;
      StatInfo := StatInfo + #13#10;

      // counters
      StatInfo := StatInfo + #13#10 + Format( '%6d  %s', [SumArt,
          'All articles'] ) + #13#10;

      If Assigned( Hamster.NewsHistory ) then begin
         StatInfo := StatInfo + #13#10 + Format( '%6d  %s', [Hamster.NewsHistory.Count,
            'Entries in History'] ) + #13#10;
      end;

      If Assigned( Hamster.MailHistory ) then begin
         StatInfo := StatInfo + #13#10 + Format( '%6d  %s', [Hamster.MailHistory.Count,
            'Entries in Mailhistory'] ) + #13#10;
      end;

      // save stats in internal group
      s := SaveInInternalGroup( hsHamGroupStatistics, '[Hamster] Info', StatInfo );
      Hamster.Config.Settings.SetStr( hsStatsLastInfoMsg, s );
      Hamster.Config.Settings.Flush;

   finally
      Hamster.Config.EndRead;
      lGroupsInfo.free;
      lOutDated.free;
   end;

   TLog( LOGID_SYSTEM, 'End' );
end;

constructor TThreadStatistics.Create( FreeType: TThreadFreeTypes );
begin
   inherited Create( attMaintenance, '{create statistics}', FreeType );
   Priority := tpLower;
end;


// ----------------------------------------------- TThreadAutoUnsubscribe -----

procedure TThreadAutoUnsubscribe.Execute;
var  GroupList: TStringList;
     GroupName, ServerName: String;
     GroupHdl, DeletePullsDays, DeleteGroupsDays, LastReadDays, i: Integer;
     RunItNow, DeletePulls, DeleteGroup: Boolean;
     DT, LastAnyReadDT, SafeLimitDT: TDateTime;
begin
   // check if auto unsubscribe is configured
   DeletePullsDays  := Hamster.Config.Settings.GetInt(hsAutoUnsubscribePulls );
   DeleteGroupsDays := Hamster.Config.Settings.GetInt(hsAutoUnsubscribeGroups);
   if (DeletePullsDays<=0) and (DeleteGroupsDays<=0) then exit; // not enabled

   // already run today?
   CS_MAINTENANCE.Enter;
   try
      TLog( LOGID_DEBUG, 'Check' );
      DT := Now;
      RunItNow := ( Hamster.Config.Settings.GetStr( hsLastAutoUnsubscribe )
                    <> FormatDateTime( 'yyyy"-"mm"-"dd', DT ) );
      if RunItNow then begin
         Hamster.Config.Settings.SetStr(
            hsLastAutoUnsubscribe, FormatDateTime( 'yyyy"-"mm"-"dd', DT ) );
         Hamster.Config.Settings.Flush;
      end;
   finally CS_MAINTENANCE.Leave end;
   if not RunItNow then exit; // Nothing to do

   // run auto unsubscribe
   TLog( LOGID_SYSTEM, 'Start' );
   StateInfo := 'Check/execute auto unsubscribe';
   GroupList := TStringList.Create;

   try
      // create list of active groups, that are/were pulled
      Hamster.Config.BeginRead;
      try
         for i := Hamster.Config.Newsgroups.Count - 1 downto 0 do begin
            GroupName := Hamster.Config.Newsgroups.Name[i];
            if Hamster.Config.Newsgroups.GroupClass( GroupName )
               in [aclWasPulled, aclIsPulled] then
                  GroupList.AddObject( GroupName, Pointer(0) );
         end;
      finally Hamster.Config.EndRead end;

      // get timepoints of last client access (any and per group)
      SafeLimitDT := EncodeDate( 2000, 1, 1 );
      LastAnyReadDT := 0;
      for i := 0 to GroupList.Count - 1 do begin
         GroupHdl  := Hamster.ArticleBase.Open( GroupList[i] );
         if GroupHdl >= 0 then try
            DT := Hamster.ArticleBase.GetDT( GroupHdl, gsLastClientRead );
            if DT < SafeLimitDT then continue; // invalid/unreadable timepoint
            if DT > LastAnyReadDT then LastAnyReadDT := DT;

            if Hamster.ArticleBase.GetBoo( GroupHdl, gsAutoUnsubscribe ) then begin
               // auto-unsubscribe is not disabled, so allow it by setting
               // its 'LastClientRead'-time.
               GroupList.Objects[i] := Pointer( DateTimeToUnixTime( DT ) );
            end;
         finally
            Hamster.ArticleBase.Close( GroupHdl );
         end;
      end;

      // to protect groups from deleting immediately after a long period of
      // non-pulling (e.g. holidays), we'll leave it intact if no groups were
      // accessed within last two days:
      if LastAnyReadDT < Now - 2 then GroupList.Clear;

      // check all groups
      for i := 0 to GroupList.Count - 1 do begin

         GroupName := GroupList[i];

         DT := UnixTimeToDateTime( Integer( GroupList.Objects[i] ) );
         if DT < SafeLimitDT then DT := Now;
         LastReadDays := Trunc( Now - DT );

         if LastReadDays > 0 then begin
            Log( LOGID_INFO, 'Note: ' + GroupName + ' was not read for '
                           + inttostr(LastReadDays) + ' days (' + DateToStr(dt) + ').' );

            DeletePulls := (DeletePullsDays >0) and (LastReadDays>=DeletePullsDays );
            DeleteGroup := (DeleteGroupsDays>0) and (LastReadDays>=DeleteGroupsDays);

            // make sure, that pulls and group are not deleted in the same run
            if Hamster.Config.NewsPulls.ExistPullServer( GroupName ) then begin
               if DeleteGroup then DeletePulls := True;
               if DeletePulls then DeleteGroup := False;
            end else begin
               DeletePulls := False;
            end;

            // delete all pulls for the group
            if DeletePulls then begin
               repeat
                  ServerName := Hamster.Config.NewsPulls.PullServerOf( GroupName );
                  if ServerName <> '' then begin
                     Log( LOGID_WARN, 'Auto unsubscribe: Deleting pull '
                                    + ServerName + '/' + GroupName );
                     if not Hamster.Config.NewsPulls.Del( ServerName, GroupName ) then break;
                  end;
               until ServerName = '';
            end;

            // delete the group
            if DeleteGroup then begin
               Log( LOGID_WARN, 'Auto unsubscribe: Deleting group '
                              + GroupName );
               Hamster.Config.Newsgroups.Del( GroupName );
            end;
         end;

      end;

   finally
      GroupList.Free;
   end;

   TLog( LOGID_SYSTEM, 'End' );
end;

constructor TThreadAutoUnsubscribe.Create( FreeType: TThreadFreeTypes );
begin
   inherited Create( attMaintenance, '{auto unsubscribe}', FreeType );
end;


// ---------------------------------------------- TThreadDailyMaintenance -----

procedure TThreadDailyMaintenance.Execute;
var  RunItNow : Boolean;
     LastDaily: TDateTime;
     ThdReports, ThdStatistics, ThdGlobalLists, ThdSpamReport: TThread;
begin
   // check configuration
   Hamster.CheckConfiguration;

   // already run today?
   CS_MAINTENANCE.Enter;
   try
      TLog( LOGID_DEBUG, 'Check' );
      LastDaily := Now;

      RunItNow := ( Hamster.Config.Settings.GetStr( hsLastDaily )
                    <> FormatDateTime( 'yyyy"-"mm"-"dd', LastDaily ) );
      if RunItNow then begin
         Hamster.Config.Settings.SetStr(
            hsLastDaily, FormatDateTime( 'yyyy"-"mm"-"dd', LastDaily ) );
         Hamster.Config.Settings.Flush;
      end;
   finally CS_MAINTENANCE.Leave end;
   if not RunItNow then exit; // nothing to do

   // run daily maintenance
   TLog( LOGID_SYSTEM, 'Start' );
   StateInfo := 'Daily maintenance';

   // auto unsubscribe
   if Hamster.Config.Settings.GetInt(hsDailyUnsubscribe) > 0 then begin
      with TThreadAutoUnsubscribe.Create( tftFreeByCode ) do try
         Resume;
         WaitFor;
      finally Free end;
   end;

   // purge
   if Hamster.Config.Settings.GetInt(hsDailyPurge) > 0 then begin
      with TThreadPurge.Create( Hamster.Config.Settings.GetInt(hsDailyPurge),
                                '', tftFreeByCode ) do try
         Resume;
         WaitFor;
      finally Free end;
   end;

   // rebuild global lists
   if Hamster.Config.Settings.GetInt(hsDailyBuildLists) = 0 then begin
      ThdGlobalLists := nil;
   end else begin
      ThdGlobalLists := TThreadRebuildGlobalLists.Create( tftFreeByCode );
      ThdGlobalLists.Resume;
   end;

   // create reports
   ThdReports := TThreadReports.Create( tftFreeByCode );
   ThdReports.Resume;
   ThdSpamReport := TThreadSpamReport.Create( tftFreeByCode );
   ThdSpamReport.Resume;

   // create statistics posting
   if Hamster.Config.Settings.GetInt(hsDailyStatistics) = 0 then begin
      ThdStatistics := nil;
   end else begin
      ThdStatistics := TThreadStatistics.Create( tftFreeByCode );
      ThdStatistics.Resume;
   end;

   // wait until all started threads have finished
   if Assigned( ThdGlobalLists ) then
      try ThdGlobalLists.WaitFor finally ThdGlobalLists.Free end;
   if Assigned( ThdSpamReport ) then
      try ThdSpamReport .WaitFor finally ThdSpamReport .Free end;
   if Assigned( ThdReports ) then
      try ThdReports    .WaitFor finally ThdReports    .Free end;
   if Assigned( ThdStatistics ) then
      try ThdStatistics .WaitFor finally ThdStatistics .Free end;

   TLog( LOGID_SYSTEM, 'End' );
end;

constructor TThreadDailyMaintenance.Create;
begin
   inherited Create( attMaintenance, '{daily maintenance}', tftFreeOnTerminate );
end;

end.
