// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit tBase; // Base-classes for Hamster-threads

// ----------------------------------------------------------------------------
// Contains the base classes for the Hamster-threads.
// ----------------------------------------------------------------------------

interface

{$INCLUDE Compiler.inc}

uses Windows, SysUtils, Classes, uType;

type
   // base-class for all threads
   TBaseThread = class( TThread )
      protected
         FUniqueID : Integer;
         FStateInfo, FSubStateInfo: String;
         ThreadType: TActiveThreadTypes;
         ThreadName: String;

         procedure SetStateInfo( NewStateInfo: String );
         procedure TLog( ID: Word; Msg: String );

         procedure ReportSubStateInfo( NewSubState: String );

      public
         property UniqueID : Integer read FUniqueID;
         property StateInfo: String  read FStateInfo write SetStateInfo;
         property SubStateInfo: String read FSubStateInfo write FSubStateInfo;

         property Terminated;
         procedure Terminate; virtual;

         constructor Create( const AThreadType: TActiveThreadTypes;
                             const AThreadName: String;
                             const FreeType: TThreadFreeTypes );
         destructor Destroy; override;
   end;

   // base-class for all threads, that count as "active tasks"
   TTaskThread = class( TBaseThread )
      private
         FIsLimitedTask: Boolean;

      protected
         procedure WaitLimitTasks;

      public
         constructor Create( const AThreadType: TActiveThreadTypes;
                             const AThreadName: String;
                             const FreeType: TThreadFreeTypes );
         destructor Destroy; override;
   end;

   // list of all active threads
   TActiveThreads = class
      private
         FList : TThreadList;
         FTasks: Integer;

      protected
         procedure Add   ( Item: TBaseThread );
         procedure Remove( Item: TBaseThread );

      public
         property  CountActiveTasks: Integer read FTasks;

         function  CountAll: Integer;
         function  CountAllByType( ThreadType: TActiveThreadTypes ): Integer;
         function  CountAllExcept( ThreadType: TActiveThreadTypes ): Integer;

         function  ListAll: String;

         procedure StopByID( UniqueID: Integer );
         procedure StopAllByType( ThreadType: TActiveThreadTypes );
         procedure StopAll;

         function IsIdle( ExceptUniqueID: Integer ): Boolean;
         function IsRunning( UniqueID: Integer ): Boolean;

         constructor Create;
         destructor Destroy; override;
   end;

implementation

uses uConst, uVar, uTools, cLogFileHamster, cHamster, uHamTools;

// ---------------------------------------------------------- TBaseThread -----

procedure TBaseThread.SetStateInfo( NewStateInfo: String );
begin
   FStateInfo := NewStateInfo;
   FSubStateInfo := '';
   TasksChange;
end;

procedure TBaseThread.ReportSubStateInfo( NewSubState: String );
begin
   FSubStateInfo := NewSubState;
   TasksChange;
end;

procedure TBaseThread.TLog( ID: Word; Msg: String );
begin
   Log( ID, ThreadName + ' ' + Msg );
end;

procedure TBaseThread.Terminate;
begin
   inherited Terminate;
end;

constructor TBaseThread.Create( const AThreadType: TActiveThreadTypes;
                                const AThreadName: String;
                                const FreeType: TThreadFreeTypes );
begin
   inherited Create( True ); // suspended

   FUniqueID  := TasksStart;
   UidMapper.Add( FUniqueID, ThreadID );

   FStateInfo := 'Create';
   FSubStateInfo := '';
   
   ThreadType := AThreadType;
   ThreadName := AThreadName;

   Hamster.ActiveThreads.Add( Self );
   FreeOnTerminate := ( FreeType = tftFreeOnTerminate );
end;

destructor TBaseThread.Destroy;
begin
   StateInfo := 'Destroy';
   Sleep( 1 );

   // remove from threads list
   Hamster.ActiveThreads.Remove( Self );
   TasksChange;
   CounterChange;
   UidMapper.RemoveUid( FUniqueID );

   inherited Destroy;
end;

// ---------------------------------------------------------- TTaskThread -----

procedure TTaskThread.WaitLimitTasks;
var  Logged: Boolean;
begin
   if FIsLimitedTask then begin
      SEM_LIMITTASKS.Release;
      FIsLimitedTask := False;
   end;

   Logged := False;
   while not Terminated do begin
      if SEM_LIMITTASKS.Acquire( 1000 ) then begin
         if Logged then begin
            StateInfo := 'Resumed (after task limit was reached)';
            TLog( LOGID_INFO, StateInfo );
         end;
         FIsLimitedTask := True;
         break;
      end else begin
         if not Logged then begin
            StateInfo := 'Suspended (task limit reached)';
            TLog( LOGID_INFO, StateInfo );
         end;
         Logged := True;
      end;
   end;
end;

constructor TTaskThread.Create( const AThreadType: TActiveThreadTypes;
                                const AThreadName: String;
                                const FreeType: TThreadFreeTypes );
begin
   inherited Create( AThreadType, AThreadName, FreeType );
   FIsLimitedTask := False;
end;

destructor TTaskThread.Destroy;
begin
   if FIsLimitedTask then SEM_LIMITTASKS.Release;
   inherited Destroy;
end;


// ------------------------------------------------------- TActiveThreads -----

constructor TActiveThreads.Create;
begin
   inherited Create;
   FList  := TThreadList.Create;
   FTasks := 0;
end;

destructor TActiveThreads.Destroy;
begin
   FList.Free;
   inherited Destroy;
end;

procedure TActiveThreads.Add( Item: TBaseThread );
begin
   with FList.LockList do try
      Add( Item );
      if Item.ThreadType in CountAsActiveTasksSet then inc( FTasks );
   finally FList.UnlockList end;
end;

procedure TActiveThreads.Remove( Item: TBaseThread );
begin
   with FList.LockList do try
      Remove( Item );
      if Item.ThreadType in CountAsActiveTasksSet then dec( FTasks );
   finally FList.UnlockList end;
end;

function TActiveThreads.CountAll: Integer;
begin
   with FList.LockList do try
      Result := Count;
   finally FList.UnlockList end;
end;

function TActiveThreads.CountAllByType( ThreadType: TActiveThreadTypes ): Integer;
var  i: Integer;
begin
   Result := 0;
   with FList.LockList do try
      for i := 0 to Count - 1 do begin
         if TBaseThread( Items[i] ).ThreadType = ThreadType then begin
            inc( Result );
         end;
      end;
   finally FList.UnlockList end;
end;

function TActiveThreads.CountAllExcept( ThreadType: TActiveThreadTypes ): Integer;
var  i: Integer;
begin
   Result := 0;
   with FList.LockList do try
      for i := 0 to Count - 1 do begin
         if TBaseThread( Items[i] ).ThreadType <> ThreadType then begin
            inc( Result );
         end;
      end;
   finally FList.UnlockList end;
end;

function TActiveThreads.ListAll: String;
var  i: Integer;
begin
   with FList.LockList do try
      Result := '';
      for i := 0 to Count - 1 do with TBaseThread( Items[i] ) do begin
         Result := Result + inttostr( UniqueID )
                 + TAB    + inttostr( ord( ThreadID   ) )
                 + TAB    + inttostr( ord( ThreadType ) )
                 + TAB    + DQuoteStr( ThreadName )
                 + TAB    + ClassName
                 + TAB    + DQuoteStr( StateInfo + ' ' + SubStateInfo )
                 + CRLF;
      end;
   finally FList.UnlockList end;
end;

procedure TActiveThreads.StopByID( UniqueID: Integer );
var  i: Integer;
begin
   with FList.LockList do try
      for i := 0 to Count - 1 do begin
         if TBaseThread( Items[i] ).UniqueID = UniqueID then begin
            try TBaseThread( Items[i] ).Terminate; except end;
            break;
         end;
      end;
   finally FList.UnlockList end;
end;

function TActiveThreads.IsIdle( ExceptUniqueID: Integer ): Boolean;
begin
   Result := False;
   with FList.LockList do try
      if Count = 0 then begin Result := True; exit; end;
      if Count = 1 then begin
         if TBaseThread( Items[0] ).UniqueID = ExceptUniqueID then begin
            Result := True;
            exit;
         end;
      end;
   finally FList.UnlockList end;
end;

function TActiveThreads.IsRunning( UniqueID: Integer ): Boolean;
var  i: Integer;
begin
   Result := False;
   with FList.LockList do try
      for i := 0 to Count - 1 do begin
         if TBaseThread( Items[i] ).UniqueID = UniqueID then begin
            Result := True;
            exit;
         end;
      end;
   finally FList.UnlockList end;
end;

procedure TActiveThreads.StopAllByType( ThreadType: TActiveThreadTypes );
var  i: Integer;
begin
   with FList.LockList do try
      for i := 0 to Count - 1 do begin
         if TBaseThread( Items[i] ).ThreadType = ThreadType then begin
            try TBaseThread( Items[i] ).Terminate; except end;
         end;
      end;
   finally FList.UnlockList end;
end;

procedure TActiveThreads.StopAll;
var  i: Integer;
begin
   with FList.LockList do try
      for i := 0 to Count - 1 do begin
         try TBaseThread( Items[i] ).Terminate; except end;
      end;
   finally FList.UnlockList end;
end;

// ----------------------------------------------------------------------------

end.
