// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cServerBase;

interface

{$INCLUDE Compiler.inc}

uses SysUtils, Classes, IdTcpServer, IdStack, uType;

type
   TServerClientBase  = class;
   TServerClientClass = class of TServerClientBase;

   TServerBaseClass = class of TServerBase;

   TServerBase = class( TIdTCPServer )
      protected
         FServerType   : TServerTypes;
         FIPAccessScope: LongInt;
         FMaxConnPerCli: Integer;

         procedure SetActive(AValue: Boolean); override;
         function  GetActiveConnections: Integer;

         procedure SrvOnException ( AThread: TIdPeerThread; AException: Exception );
         procedure SrvOnConnect   ( AThread: TIdPeerThread );
         procedure SrvOnDisconnect( AThread: TIdPeerThread );
         procedure SrvOnExecute   ( AThread: TIdPeerThread );

      public
         property  ActiveConnections   : Integer read GetActiveConnections;
         procedure ActiveConnectionList( ListTo: TStrings );

         constructor Create( AServerType: TServerTypes;
                             AServerBind: String;
                             AServerPort: Integer;
                             AMaxClients: Integer;
                             AMaxSameCli: Integer;
                             AIPAccessScope: LongInt;
                             AServerClientClass: TServerClientClass ); reintroduce;
         destructor Destroy; override;
   end;

   TClientErrorType = ( cetRefusedDontRetry, cetRefusedTemporary,
                        cetLineTooLong, cetTextTooLarge );

   TServerClientBase = class( TIdPeerThread )
      protected
         UniqueID       : Integer;
         CurrentUserID  : Integer;
         CurrentUserName: String;
         FIPAccessScope : LongInt;
         FClientID      : String;
         FClientIP      : String;
         FClientAD      : TIdInAddr;
         FIPAccess      : LongInt;
         FIPAccount     : String;
         FLimitLineLen  : Integer;
         FLimitTextSize : Integer;
         FCountTextSize : Integer;
         FInactivitySecs: Integer;

         procedure CheckIPAccess;
         function  GetClientIPn: LongInt;
         procedure SendGreeting( var KeepConnection: Boolean;
                                 var Reason: String ); virtual;
         procedure HandleCommand( const CmdLine: String ); virtual;
         function  FormatErrorReply( const ErrType: TClientErrorType;
                                     const ErrMsg: String ): String; virtual;

      public
         property  ClientID: String  read FClientID;
         property  IPAccess: LongInt read FIPAccess;
         property  IPAccount: String read FIPAccount;
         property  ClientIPn: LongInt read GetClientIPn;

         property  LimitLineLen : Integer read FLimitLineLen  write FLimitLineLen;
         property  LimitTextSize: Integer read FLimitTextSize write FLimitTextSize;

         function  SockDesc( const EventDesc: String ): String;

         function  HConnected: Boolean;
         procedure HDisconnect;

         function  HReadLn    : String;
         function  HReadLnText: String;

         procedure HWrite   ( const Txt: String );
         procedure HWriteLn ( const Txt: String );
         procedure HWriteLnQ( const Txt: String );

         function  HRequest    ( const Txt: String ): String;
         function  HRequestText( const Txt: String ): String;

         constructor Create( ACreateSuspended: Boolean = True ); override;
         destructor Destroy; override;
   end;

function Local_SASL_Login( const ServerClient: TServerClientBase;
                           const ChallengeReplyFormat: String;
                           const AUTH_Stamp, Mechanism, Parameters: String;
                           const ClientIPn: LongInt;
                           var   CurrentUserID: Integer;
                           var   CurrentUserName: String ): Boolean;

implementation

uses uConst, uConstVar, IdGlobal, IdException, IdIOHandlerSocket, Windows,
     cLogFileHamster, cHamster, cAccounts, cIPAccess, uSASL, uTools,
     uEncoding, uMD5, uSHA1, uHamTools, uVar;

// ----------------------------------------------------------------------------

function Local_SASL_Login( const ServerClient: TServerClientBase;
                           const ChallengeReplyFormat: String;
                           const AUTH_Stamp, Mechanism, Parameters: String;
                           const ClientIPn: LongInt;
                           var   CurrentUserID: Integer;
                           var   CurrentUserName: String ): Boolean;
// SASL authentication for local SMTP-, POP3- and RECO-servers.
const SASL_CANCEL = '*';
      SASL_EMPTY  = '=';
var  s, u, p: String;
     id: Integer;
     h, nonce, cnonce, ncvalue, qop, realm, digesturi, response, a1, a2: String;
     SL: TStringList;
     ok: Boolean;
begin
   Result := False;
   CurrentUserName := '';
   CurrentUserID   := ACTID_INVALID;

   if Mechanism = 'LOGIN' then begin

      if Parameters = '' then begin
         s := 'Username:';
         s := EncodeB64( s[1], length(s) );
         s := ServerClient.HRequest( Format( ChallengeReplyFormat, [ s ] ) );
      end else begin
         s := Parameters;
      end;
      if (s='') or (s=SASL_CANCEL) then exit;
      u := DecodeB64( s[1], length(s) );

      s := 'Password:';
      s := EncodeB64( s[1], length(s) );
      s := ServerClient.HRequest( Format( ChallengeReplyFormat, [ s ] ) );
      if (s='') or (s=SASL_CANCEL) then exit;
      p := DecodeB64( s[1], length(s) );

   end else if Mechanism = 'PLAIN' then begin

      if Parameters = '' then begin
         s := ServerClient.HRequest( Format( ChallengeReplyFormat, [SASL_EMPTY] ) );
      end else begin
         s := Parameters;
      end;
      if (s='') or (s=SASL_CANCEL) then exit;
      if not AUTH_PLAIN_Decode( s, u, p ) then exit;

   end else if ( Mechanism = 'CRAM-MD5'  ) or
               ( Mechanism = 'CRAM-SHA1' ) then begin

      s := EncodeB64( AUTH_Stamp[1], length(AUTH_Stamp) );
      s := ServerClient.HRequest( Format( ChallengeReplyFormat, [ s ] ) );
      if (s='') or (s=SASL_CANCEL) then exit;

      if not AUTH_CRAM_Decode( s, u, p ) then exit;
      if (u='') or (p='') then exit;

      id := Hamster.Accounts.UserIDOf( u );
      if id = ACTID_INVALID then exit;

      s := Hamster.Accounts.Value[ id, apPassword ];
      if s = ACTPW_NOACCESS then exit;
      if s <> ACTPW_NOTNEEDED then begin
         if Mechanism = 'CRAM-MD5' then begin
            if p <> HMAC_MD5( AUTH_Stamp, s ) then exit;
         end else if Mechanism = 'CRAM-SHA1' then begin
            if p <> HMAC_SHA1( AUTH_Stamp, s ) then exit;
         end else begin
            exit;
         end;
         p := s;
      end;

   end else if Mechanism = 'DIGEST-MD5' then begin

      // build challenge
      realm := Hamster.Config.Settings.GetStr( hsFQDNforMID );
      if realm = '' then realm := 'localhost';
      s := 'realm="' + realm + '"';

      h := AUTH_Stamp + inttostr(GetTickCount) + inttostr(Random(999999) );
      h := MD5ofStr( h );
      nonce := EncodeB64( h[1], length(h) );
      s := s + ',' + 'nonce="' + nonce + '"';

      qop := 'auth';
      s := s + ',' + 'qop="' + qop + '"';

      s := s + ',' + 'algorithm=' + 'md5-sess';

      Log( LOGID_DEBUG, 'DIGEST-MD5 challenge: ' + s );
      s := EncodeB64( s[1], length(s) );

      // send challenge, get response
      s := ServerClient.HRequest( Format( ChallengeReplyFormat, [ s ] ) );
      if (s='') or (s=SASL_CANCEL) then exit;

      // check response, extract values
      ok := True;
      SL := TStringList.Create;
      try
         s := DecodeB64( s[1], length(s) );
         Log( LOGID_DEBUG, 'DIGEST-MD5 response: ' + s );
         ArgsSplitChar( s, SL, ',', True );

         u := UnDQuoteStr( SL.Values[ 'username' ] );
         if u = '' then ok := False;
         id := Hamster.Accounts.UserIDOf( u );
         if id = ACTID_INVALID then ok := False;

         h := UnDQuoteStr( SL.Values[ 'nonce' ] );
         if (h <> '') and (h <> nonce) then ok := False;

         cnonce := UnDQuoteStr( SL.Values[ 'cnonce' ] );
         if cnonce = '' then ok := False;

         ncvalue := SL.Values[ 'nc' ];
         if ncvalue <> '00000001' then ok := False;

         h := UnDQuoteStr( SL.Values[ 'qop' ] );
         if (h <> '') and (h <> 'auth') then ok := False;

         h := UnDQuoteStr( SL.Values[ 'realm' ] );
         if (h <> '') and (h <> realm) then ok := False;

         digesturi := UnDQuoteStr( SL.Values[ 'digest-uri' ] );

         response := SL.Values[ 'response' ];
         if length(response) <> 32 then ok := False;

      finally SL.Free end;
      
      if not ok then begin
         Log( LOGID_DEBUG, 'DIGEST-MD5 response: Missing/invalid values.' );
         exit;
      end;

      // build expected response and compare with received one
      h  := Hamster.Accounts.Value[ id, apPassword ];
      a1 := MD5OfStr( u + ':' + realm + ':' + h )
          + ':' + nonce + ':' + cnonce;
      a2 := 'AUTHENTICATE:' + digesturi;
      s  := MD5toHex( MD5OfStr(
               MD5toHex( MD5OfStr( A1 ) )
               + ':' + nonce + ':' + ncvalue + ':' + cnonce + ':' + qop
               + ':' + MD5toHex( MD5OfStr( A2 ) )
            ) );
      if s <> response then begin
         Log( LOGID_DEBUG, 'DIGEST-MD5 response: Invalid response value.' );
         exit; // different
      end;
      p := Hamster.Accounts.Value[ id, apPassword ]; // ok

      // build rspauth and send it
      a2 := ':' + digesturi;
      h := MD5toHex( MD5OfStr(
              MD5toHex( MD5OfStr( A1 ) )
              + ':' + nonce + ':' + ncvalue + ':' + cnonce + ':' + qop
              + ':' + MD5toHex( MD5OfStr( A2 ) )
           ) );
      s := 'rspauth=' + h;
      Log( LOGID_DEBUG, 'DIGEST-MD5 rspauth: ' + s );
      s := EncodeB64( s[1], length(s) );
      s := ServerClient.HRequest( Format( ChallengeReplyFormat, [ s ] ) );

   end;

   if (u <> '') and (p <> '') and ServerClient.HConnected then begin
      CurrentUserID := Hamster.Accounts.LoginID( u, p, ClientIPn );
      if CurrentUserID <> ACTID_INVALID then begin
         CurrentUserName := u;
         Result := True;
      end;
   end;
end;


// ---------------------------------------------------- TServerClientBase -----

function TServerClientBase.HConnected: Boolean;
begin
   Result := False;
   try
      if Assigned( Connection ) then begin
         if Connection.Connected then Result := True;
      end;
   except end;
end;

procedure TServerClientBase.HDisconnect;
begin
   if HConnected then Connection.Disconnect;
end;

function TServerClientBase.HReadLn: String;
begin
   Connection.MaxLineLength := 32 * 1024;
   Connection.MaxLineAction := maSplit;

   Connection.ReadTimeout := FInactivitySecs * 1000;

   Result := Connection.ReadLn( #10, IdTimeoutDefault );

   while Connection.ReadLnSplit do begin
      Result := Result + Connection.ReadLn( #10, IdTimeoutDefault );
      if length( Result ) > 50*1024*1024 then begin
         raise Exception.Create( 'Linelength-limit exceeds 50 MB!' );
      end;
   end;

   if Connection.ReadLnTimedOut then begin
      Log( LOGID_INFO, 'Connection timed out:' + ClientID );
      try HDisconnect except end;
      exit;
   end;

   if (LogFile.FileMask and LOGID_FULL) <> 0 then begin
      Log( LOGID_DEBUG, '>*> ' + Result );
   end;

   // check maximum line length
   if (FLimitLineLen > 0) and (length(Result) > FLimitLineLen) then begin
      HWriteLn( FormatErrorReply( cetLineTooLong, 'Linelength-limit exceeded'
         + ', length=' + IntToStr( length(Result) )
         + ', limit='  + IntToStr( FLimitLineLen ) ) );
      try HDisconnect except end;
      exit;
   end;

   // check maximum text size
   inc( FCountTextSize, length( Result ) + 2 );
   if (FLimitTextSize > 0) and (FCountTextSize > FLimitTextSize) then begin
      HWriteLn( FormatErrorReply( cetTextTooLarge, 'Textsize-limit exceeded'
         + ', length=' + IntToStr( FCountTextSize )
         + ', limit='  + IntToStr( FLimitTextSize ) ) );
      try HDisconnect except end;
      exit;
   end;
end;

function TServerClientBase.HReadLnText: String;
var  s: String;
     MS: TMemoryStream;
begin
   SetLength( Result, 0 );

   MS := TMemoryStream.Create;
   try

      while HConnected and not(Hamster.WantsToTerminate or Terminated) do begin
         s := HReadLn;
         if s = '.' then break;
         if copy( s, 1, 2 ) = '..' then System.Delete( s, 1, 1 );
         StreamWriteLn( MS, s );
      end;

      if MS.Size > 0 then StreamToString( MS, Result );

   finally MS.Free end;
end;

procedure TServerClientBase.HWrite( const Txt: String );
begin
   if (LogFile.FileMask and LOGID_FULL) <> 0 then begin
      Log( LOGID_DEBUG, '<*< ' + Txt );
   end;

   Connection.Write( Txt );
end;

procedure TServerClientBase.HWriteLn( const Txt: String );
begin
   if Txt='.' then Log( LOGID_DETAIL, '< ' + Txt )
              else Log( LOGID_INFO,   '< ' + Txt );
   HWrite( Txt + CRLF );
end;

procedure TServerClientBase.HWriteLnQ( const Txt: String );
var  Snd: String;
begin
   if not HConnected then exit;
   if copy( Txt, 1, 1 ) = '.' then Snd := '.' + Txt // quote leading dot
                              else Snd := Txt;

   HWrite( Snd + CRLF );
end;

function TServerClientBase.HRequest( const Txt: String ): String;
begin
   HWriteLn( Txt );
   Result := HReadLn;
end;

function TServerClientBase.HRequestText( const Txt: String ): String;
begin
   HWriteLn( Txt );
   Result := HReadLnText;
end;

procedure TServerClientBase.SendGreeting( var KeepConnection: Boolean;
                                          var Reason: String );
begin
   KeepConnection := False;
   Reason := 'Not implemented';
end;

procedure TServerClientBase.HandleCommand( const CmdLine: String );
begin
   // 
end;

function TServerClientBase.FormatErrorReply( const ErrType: TClientErrorType;
                                             const ErrMsg: String ): String;
begin
   Result := ErrMsg;
end;

function TServerClientBase.GetClientIPn: LongInt;
// client's IP in network byte order
begin
   Result := FClientAD.S_addr;
end;

procedure TServerClientBase.CheckIPAccess;
begin
   FIPAccess  := IPACC_ACCESS_NA;
   FIPAccount := '';

   try
      FIPAccess := Hamster.IPAccess.GetAccess( GetClientIPn,
                                               FIPAccessScope,
                                               {out} FIPAccount );

      Log( LOGID_DETAIL, SockDesc( '.CheckIPAccess' )
                       + IPAccessScopeStr( FIPAccessScope )
                       + ' -> ' + IPAccessAccessStr( IPAccess )
                       + iif( FIPAccount <> '', ' -> ' + FIPAccount )
                       );
   except
      FIPAccess := IPACC_ACCESS_NA;
      FIPAccount := '';
   end;
end;

function TServerClientBase.SockDesc( const EventDesc: String ): String;
begin
   if HConnected then
      Result := Self.Classname + EventDesc + '(' + ClientID + ') '
   else
      Result := Self.Classname + EventDesc + '(?) ';
end;

constructor TServerClientBase.Create( ACreateSuspended: Boolean );
var  i: Integer;
begin
   inherited Create( True );

   Log( LOGID_DEBUG, Self.Classname + '.Create' );

   UniqueID := ClientsStart;
   UidMapper.Add( UniqueID, ThreadID );

   CurrentUserID   := ACTID_INVALID;
   CurrentUserName := '';
   FIPAccessScope  := IPACC_SCOPE_NONE;
   FIPAccess       := IPACC_ACCESS_NA;
   FIPAccount      := '';
   FLimitLineLen   := 1000;
   FLimitTextSize  := 1000000;
   FCountTextSize  := 0;

   i := Hamster.Config.Settings.GetInt( hsLocalTimeoutInactivity ); // Min.
   if ( i < 0 ) or ( i > (MaxInt div 60000) ) then i := MaxInt div 60000;
   FInactivitySecs := i * 60;

   if not ACreateSuspended then Resume;
end;

destructor TServerClientBase.Destroy;
begin
   Log( LOGID_DEBUG, Self.Classname + '.Destroy' );

   ClientsChange;
   UidMapper.RemoveUid( UniqueID );

   inherited Destroy;
end;


// ---------------------------------------------------------- TServerBase -----

procedure TServerBase.SetActive(AValue: Boolean);
var  i, TID, UID, LPt: Integer;
     LIP: String;
begin
   inherited SetActive( AValue );
   if not AValue then exit;

   with FListenerThreads.LockList do try
      for i := 0 to Count - 1 do begin
         UID := NewUniqueId;
         TID := TIdListenerThread(Items[i]).ThreadID;
         LIP := TIdListenerThread(Items[i]).Binding.IP;
         LPt := TIdListenerThread(Items[i]).Binding.Port;
         UidMapper.Add( UID, TID );
         Log( LOGID_DETAIL, Format( '%s-Listener for Binding %s:%d',
                            [ Self.Classname, LIP, LPt ] ), UID );
      end;
   finally FListenerThreads.UnlockList end;
end;

function TServerBase.GetActiveConnections: Integer;
begin
   Result := 0;
   if not Active then exit;

   with ThreadMgr.ActiveThreads.LockList do try
      Result := Count;
   finally ThreadMgr.ActiveThreads.UnlockList end;
end;

procedure TServerBase.ActiveConnectionList( ListTo: TStrings );
var  i: Integer;
     C: TServerClientBase;
begin
   ListTo.Clear;
   if not Active then exit;

   with ThreadMgr.ActiveThreads.LockList do try
      for i := 0 to Count - 1 do try
         C := TServerClientBase( Items[i] );
         ListTo.AddObject(       inttostr( C.UniqueID )
                         + TAB + inttostr( C.ThreadID )
                         + TAB + C.ClientID
                         + TAB + C.CurrentUserName
                         , C );
      except end;
   finally ThreadMgr.ActiveThreads.UnlockList end;
end;

procedure TServerBase.SrvOnConnect( AThread: TIdPeerThread );
var  SC: TServerClientBase;
     FinalKeepConnection, KeepConnection: Boolean;
     Cnt, i: Integer;
     Reason: String;
begin
   SC := TServerClientBase( AThread );

   // identify client
   with TIdIOHandlerSocket( AThread.Connection.IOHandler ).Binding do begin
      SC.FClientIP := PeerIP;
      SC.FClientID := PeerIP + ':' + inttostr( {Server}Port );
      SC.FClientAD := GStack.StringToTInAddr( SC.FClientIP );
   end;
   Log( LOGID_INFO, 'Client ' + SC.ClientID + ' connected' );

   // check for valid settings
   // if SC.FLimitLineLen  = 0 then SC.FLimitLineLen  := 4096;
   // if SC.FLimitTextSize = 0 then SC.FLimitTextSize := MaxInt;

   // check, if connection is allowed
   FinalKeepConnection := False;
   Reason := '';

   try
      try
         KeepConnection := True;

         if ActiveConnections > MaxConnections then begin
            // server's connection limit is reached -> refuse
            KeepConnection := False;
            Reason := 'Connection limit reached - try again later.';
            SC.HWriteLn( SC.FormatErrorReply( cetRefusedTemporary, Reason ) );
         end;

         // check number of active connections from same client
         if KeepConnection then begin
            Cnt := 0;
            try
               with ThreadMgr.ActiveThreads.LockList do try
                  for i := 0 to Count - 1 do try
                     with TIdPeerThread( Items[i] ).Connection do
                        if TIdIOHandlerSocket(IOHandler).Binding.PeerIP
                           = SC.FClientIP then inc( Cnt );
                  except end;
               finally ThreadMgr.ActiveThreads.UnlockList end;
            except end;
            if Cnt > FMaxConnPerCli then begin
               // user's connection limit is reached -> refuse
               KeepConnection := False;
               Reason := 'Connection refused: Too many open connections from you!';
               SC.HWriteLn( SC.FormatErrorReply( cetRefusedTemporary, Reason ) );
            end;
         end;

         // check IPAccess
         if KeepConnection then begin
            KeepConnection := False;
            SC.FIPAccessScope := FIPAccessScope;
            SC.CheckIPAccess;
            if SC.IPAccess = IPACC_ACCESS_NA then begin
               KeepConnection := False;
               Reason := 'Permission denied - do not try again.';
               SC.HWriteLn( SC.FormatErrorReply( cetRefusedDontRetry, Reason ) );
            end else begin
               // client decides (based on .IPAccess) if connection is ok
               SC.SendGreeting( KeepConnection, Reason );
            end;
         end;

         FinalKeepConnection := KeepConnection;

      except
         on E: Exception do begin
            if length(Reason) > 0 then Reason := Reason + CRLF + TAB;
            Reason := Reason + 'Error: ' + E.Message;
         end;
      end;

   finally
      if not FinalKeepConnection then begin
         // if any check failed, disconnect client
         if length(Reason) = 0 then Reason := 'reason unknown';
         Log( LOGID_WARN, 'Connection refused: ' + SC.ClientID
                        + ' (' + Reason + ')' );
         try SC.HDisconnect except end;
      end;
   end;
end;

procedure TServerBase.SrvOnDisconnect( AThread: TIdPeerThread );
var  SC: TServerClientBase;
begin
   SC := TServerClientBase( AThread );
   Log( LOGID_INFO, 'Client ' + SC.ClientID + ' disconnected' );
end;

procedure TServerBase.SrvOnException( AThread: TIdPeerThread;
                                      AException: Exception );
var  SC: TServerClientBase;
     i : Integer;
     s : String;
begin
   SC := TServerClientBase( AThread );
   try s := SC.ClientID except s := '?' end;
   
   i := LOGID_WARN;
   if ( AException is EIdConnClosedGracefully ) and
      ( FServerType in [ stNNTP, stRECO ] ) then i := LOGID_DEBUG;
   Log( i, Self.Classname + '.SrvOnException(' + s + '): '
                          + AException.Message );

   try SC.HDisconnect except end;
end;

procedure TServerBase.SrvOnExecute( AThread: TIdPeerThread );
var  SC: TServerClientBase;
     CmdLine: String;
begin
   SC := TServerClientBase( AThread );
   SC.FCountTextSize := 0;
   CmdLine := SC.HReadLn;
   if SC.HConnected then SC.HandleCommand( CmdLine );
end;

constructor TServerBase.Create( AServerType: TServerTypes;
                                AServerBind: String;
                                AServerPort: Integer;
                                AMaxClients: Integer;
                                AMaxSameCli: Integer;
                                AIPAccessScope: LongInt;
                                AServerClientClass: TServerClientClass );
var  BindStr: String;
     BindLst: TStringList;
     BindPort, i, j: Integer;
     BindIP, s: String;
begin
   BindStr := TrimWhSpace( AServerBind );
   if BindStr = '' then BindStr := '0.0.0.0';
   Log( LOGID_DEBUG, Self.Classname + '.Create(' + BindStr + ', '
                                    + inttostr(AServerPort)+')' );
   inherited Create( nil );

   FServerType := AServerType;
   
   Bindings.Clear;
   BindLst := TStringList.Create;
   try
      ArgsWhSpace( BindStr, BindLst );

      for i:=0 to BindLst.Count-1 do begin
         s := BindLst[i]; // IP[:Port]
         j := Pos( ':', s );

         if j = 0 then begin
            BindIP   := s;
            BindPort := AServerPort;
         end else begin
            BindIP   := copy( s, 1, j-1 );
            BindPort := strtointdef( copy( s, j+1, MaxInt ), AServerPort );
         end;

         with Bindings.Add do begin
            if BindIP = '0.0.0.0' then IP := '' else IP := BindIP;
            Port := BindPort;
         end;
      end;

   finally BindLst.Free end;

   FIPAccessScope := AIPAccessScope;
   ThreadClass    := AServerClientClass;

   MaxConnections := AMaxClients;
   FMaxConnPerCli := AMaxSameCli;

   OnException  := SrvOnException;
   OnConnect    := SrvOnConnect;
   OnDisconnect := SrvOnDisconnect;
   OnExecute    := SrvOnExecute;
end;

destructor TServerBase.Destroy;
begin
   Log( LOGID_DEBUG, Self.Classname + '.Destroy' );
   inherited Destroy;
end;

// ----------------------------------------------------------------------------

initialization
   Randomize;
   
end.
