// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cHscHamster; // Hamster-specific script-engine

interface

{$INCLUDE Compiler.inc}

uses SysUtils, Classes, SyncObjs, cSyncObjects, cHscHelpers, cHscEngine,
     cArticle;

type
   THscEngineHamster = class( THscEngine )
      private
         FOutputBuffer     : TThreadStringList;
         FSearchResults    : TStringList;
         FRasOurConnection : Boolean;

         FWebProxyServer, FWebProxyUser, FWebProxyPass: String;
         FWebProxyPort: Integer;

         procedure BufferOutput  ( const s: String );

         function Report_SearchResults( const Groupname: String;
                                        const ArticleNo: LongWord;
                                        const Article  : TMess ): Boolean;

      protected
         procedure Engine_Error( const Engine: THscEngine;
                                 const Sender: TObject;
                                 const ErrNum: Integer;
                                 const ErrMsg: String;
                                 const CurrMod: String;
                                 const CurrPos: Integer;
                                 const CurrLine: String ); override;
         procedure Engine_Trace( const Engine: THscEngine;
                                 const CurrMod: String;
                                 const CurrPos: Integer;
                                 const CurrLine: String ); override;
         procedure Engine_Warning( const Engine: THscEngine;
                                   const PrintStr: String ); override;
         procedure Engine_Print( const Engine: THscEngine;
                                 const PrintStr: String ); override;
         procedure Engine_Func( const ParsedFunc: THscParsedFunc;
                                const Result: THscVariant ); override;
         function  Engine_StartScript( const ScriptFile: String;
                                       const Params    : String;
                                       const WaitForEnd: Boolean ): Integer; override;

      public
         property RasOurConnection: Boolean read FRasOurConnection;
         property OutputBuffer: TThreadStringList read FOutputBuffer;

         constructor Create( const AStopEvent: TEvent;
                             const APathHSM  : String;
                             const AOutputBuffer: TThreadStringList );
   end;


implementation

uses Windows, uTools, cLogFileHamster, cFiltersNews, tMaintenance,
     cPasswords, tTransfer, cArtFiles, cResControl, tScript, cNewsSearcher,
     uType, uConst, uConstVar, uVar, cHamster, cLiveQueue, uHamTools,
     cMailDispatcher, cLiveMsg, uWinsock, cWebChannel
     {$IFDEF H_NEED_VARIANTS} , Variants {$ENDIF} ;

// ---------------------------------------------------- THscEngineHamster -----

procedure THscEngineHamster.BufferOutput( const s: String );
begin
   if not Assigned( FOutputBuffer ) then exit;

   with FOutputBuffer.LockList do try
      while Count > 10000 do Delete( 0 );
      Add( s );
   finally
      FOutputBuffer.UnlockList;
   end;
end;

procedure THscEngineHamster.Engine_Error( const Engine: THscEngine;
                                          const Sender: TObject;
                                          const ErrNum: Integer;
                                          const ErrMsg: String;
                                          const CurrMod: String;
                                          const CurrPos: Integer;
                                          const CurrLine: String );
var  s: String;
begin
   s := Format(
        'Error in line %s of script-file "%s"', [inttostr(CurrPos), CurrMod] );
   Log( LOGID_ERROR, s );
   BufferOutput( 'E> ' + s );
   s := Format(
        'Error %s: %s [%s]', [inttostr( ErrNum ), ErrMsg, Sender.ClassName ] );
   Log( LOGID_ERROR, s );
   BufferOutput( 'E> ' + s );
   s := Format(
        'Error-line: %s', [TrimWhSpace(CurrLine)] );
   Log( LOGID_ERROR, s );
   BufferOutput( 'E> ' + s );
end;

procedure THscEngineHamster.Engine_Trace( const Engine: THscEngine;
                                          const CurrMod: String;
                                          const CurrPos: Integer;
                                          const CurrLine: String );
begin
   Log( LOGID_INFO, '[' + CurrMod + ':' + inttostr(CurrPos) + ']: ' + CurrLine );
end;

procedure THscEngineHamster.Engine_Warning( const Engine: THscEngine;
                                            const PrintStr: String );
var  s: String;
begin
   s := '!> ' + PrintStr;
   Log( LOGID_WARN, s );
   BufferOutput( s );
end;

procedure THscEngineHamster.Engine_Print( const Engine: THscEngine;
                                          const PrintStr: String );
var  s: String;
begin
   s := '> ' + PrintStr;
   Log( LOGID_INFO, s );
   BufferOutput( s );
end;

function THscEngineHamster.Report_SearchResults( const Groupname: String;
                                                 const ArticleNo: LongWord;
                                                 const Article  : TMess ): Boolean;
begin
   try
      if ( FSearchResults.Add(
               Groupname
               + #9 + inttostr(ArticleNo)
               + #9 + Article.HeaderValueByNameSL('Message-ID')
         ) mod 100 ) = 0 then begin
         Result := ( Engine.StopEvent.WaitFor(0) <> wrSignaled );
      end else begin
         Result := True;
      end;

   except
      Result := False;
   end;
end;

procedure HamAddLog( const Info: WideString; Typ: Integer );
Var x: Word;
begin
   Case Typ of
      1: x := LOGID_DEBUG;
      2: x := LOGID_DETAIL;
      3: x := LOGID_INFO;
      4: x := LOGID_SYSTEM;
      5: x := LOGID_WARN;
      6: x := LOGID_ERROR;
      7: x := LOGID_STATUS;
      else x := LOGID_INFO
   end;
   Log( x, Info )
end;

function HamGetStatus( const Typ, Item, Scale: Integer ): Integer;
var  i64: Int64;
begin
   i64 := 0;

   case Typ of
     1: { Aktiv oder nicht aktiv? }
        Case Item of
          1: if Hamster.ServerControl( stNNTP, scIsActive ) = 1 then i64 := 1;
          2: if Hamster.ServerControl( stPOP3, scIsActive ) = 1 then i64 := 1;
          3: if Hamster.ServerControl( stSMTP, scIsActive ) = 1 then i64 := 1;
          4: if Hamster.ServerControl( stRECO, scIsActive ) = 1 then i64 := 1;
          else i64 := -1
        end;
     2: Case Item of
          1: i64 := Hamster.ServerControl( stNNTP, scClientCount );
          2: i64 := Hamster.ServerControl( stPOP3, scClientCount );
          3: i64 := Hamster.ServerControl( stSMTP, scClientCount );
          4: i64 := Hamster.ServerControl( stRECO, scClientCount );
          else i64 := -1;
        end;
     3: Case Item of
          1: i64 := CounterArtNew;
          2: i64 := CounterArtLoad;
          3: i64 := CounterArtHist;
          4: i64 := CounterArtKill;
          5: i64 := CounterOutboxN;
          6: i64 := CounterMailNew;
          7: i64 := CounterOutboxM;
          else i64 := -1;
        end;
     else i64 := -1;
   end;

   if Scale > 1 then i64 := i64 div Scale;
   if i64 < 0 then begin Result := -1; exit; end;
   if i64 > $7ffffff then begin Result := -2; exit; end;
   Result := i64;
end;

function HamMessage( const WParam, LParam: Integer ): Integer;
const
   // HAM_MSG_HAMSTER_EXIT    = 1; // -
   HAM_MSG_RESET_COUNTERS  = 2; // -
   HAM_MSG_LOCALNNTP_ONOFF = 3; // 0/1
   HAM_MSG_LOCALPOP3_ONOFF = 4; // 0/1
   HAM_MSG_LOCALSMTP_ONOFF = 5; // 0/1
   // HAM_MSG_SHOWWINDOW      = 6; // 0/1 + [2..10]
   // HAM_MSG_SHOWICON        = 7; // 0/1
   HAM_MSG_STOPALLTASKS    = 8; // -
   // HAM_MSG_CONFIG_RELOAD   = 9; // 0/1
   HAM_MSG_LOCALRECO_ONOFF = 100; // 0/1/2
begin
   Result := 1; // Default=Error (0=OK, 2=Unknown ID)

   case WParam of

      HAM_MSG_LOCALNNTP_ONOFF: begin // start/stop/restart local NNTP
         if LParam=0 then begin
            if Hamster.ServerControl( stNNTP, scSTOP    )=1 then Result := 0;
         end else if LParam=1 then begin
            if Hamster.ServerControl( stNNTP, scSTART   )=1 then Result := 0;
         end else if LParam=2 then begin
            if Hamster.ServerControl( stNNTP, scRESTART )=1 then Result := 0;
         end;
      end;

      HAM_MSG_LOCALSMTP_ONOFF: begin // start/stop/restart local SMTP
         if LParam=0 then begin
            if Hamster.ServerControl( stSMTP, scSTOP    )=1 then Result := 0;
         end else if LParam=1 then begin
            if Hamster.ServerControl( stSMTP, scSTART   )=1 then Result := 0;
         end else if LParam=2 then begin
            if Hamster.ServerControl( stSMTP, scRESTART )=1 then Result := 0;
         end;
      end;

      HAM_MSG_LOCALPOP3_ONOFF: begin // start/stop/restart local POP3
         if LParam=0 then begin
            if Hamster.ServerControl( stPOP3, scSTOP    )=1 then Result := 0;
         end else if LParam=1 then begin
            if Hamster.ServerControl( stPOP3, scSTART   )=1 then Result := 0;
         end else if LParam=2 then begin
            if Hamster.ServerControl( stPOP3, scRESTART )=1 then Result := 0;
         end;
      end;

      HAM_MSG_LOCALRECO_ONOFF: begin // start/stop/restart local ReCo
         if LParam=0 then begin
            if Hamster.ServerControl( stRECO, scSTOP    )=1 then Result := 0;
         end else if LParam=1 then begin
            if Hamster.ServerControl( stRECO, scSTART   )=1 then Result := 0;
         end else if LParam=2 then begin
            if Hamster.ServerControl( stRECO, scRESTART )=1 then Result := 0;
         end;
      end;

      HAM_MSG_RESET_COUNTERS: begin // reset stat's
         if LParam in [ 0, 1 ] then
         Hamster.LiveServer.ClientRequest( nil, LMREQ_COUNTERS_RESET, '' );
         Result := 0;
      end;

      HAM_MSG_STOPALLTASKS: begin // stop all tasks
         Hamster.LiveServer.ClientRequest( nil, LMREQ_TASKS_STOPALL, '' );
         Result := 0;
      end;

      else begin
         Result := 2;
         Log( LOGID_WARN, 'Unsupported HamMessage code: ' + inttostr(WParam) );
      end;

   end;
end;

procedure THscEngineHamster.Engine_Func( const ParsedFunc: THscParsedFunc;
                                         const Result    : THscVariant );
var  i, j, k, TimeOut: Integer;
     s, t, SrvList: String;
     Identifier, ConnectionID, Username,  Password: String;
     Server, Port, User, Pass: String;
     DestUser, FiltSect, FromSel, ToSel, LeaveOnServer: String;
     OK: Boolean;
     Art: TMess;
     ScoreFile: TFiltersNews;
     XOverRec : TXOverRec;
     NewsSearcher: TNewsSearcher;
     Reply: TLiveMsg;
begin
   with ParsedFunc do begin
      if copy(FuncName,1,3) <> 'ham' then begin Result.Unassign; exit end;

      if FuncIs( 'hamversion', 0, 1 ) then begin
         i := ParI( 0, 0 );
         case i of
            0: Result.AsStr := GetExeVersion;
            1: Result.AsStr := GetMyStringFileInfo('ProductName','Hamster');
            2: Result.AsStr := GetMyStringFileInfo('Maintainer','Unknown');
            3: Result.AsStr := GetMyStringFileInfo('Download','Unknown');
            4: Result.AsStr := GetMyStringFileInfo('Comments','Unknown');
            else Result.AsStr := '';
         end;

      end else if FuncIs( 'hampath', 0, 0 ) then begin
         Result.AsStr := AppSettings.GetStr( asPathBase );
      end else if FuncIs( 'hamhscpath', 0, 0 ) then begin
         Result.AsStr := AppSettings.GetStr( asPathScripts );
      end else if FuncIs( 'hamhsmpath', 0, 0 ) then begin
         Result.AsStr := AppSettings.GetStr( asPathModules );
      end else if FuncIs( 'hamlogspath', 0, 0 ) then begin
         Result.AsStr := AppSettings.GetStr( asPathLogs );
      end else if FuncIs( 'hamserverpath', 0, 0 ) then begin
         Result.AsStr := AppSettings.GetStr( asPathServer );
      end else if FuncIs( 'hamgroupspath', 0, 0 ) then begin
         Result.AsStr := AppSettings.GetStr( asPathGroups );
      end else if FuncIs( 'hammailpath', 0, 0 ) then begin
         Result.AsStr := AppSettings.GetStr( asPathMails );
      end else if FuncIs( 'hamnewsoutpath', 0, 0 ) then begin
         Result.AsStr := AppSettings.GetStr( asPathNewsOut );
      end else if FuncIs( 'hammailsoutpath', 0, 0 ) then begin
         Result.AsStr := AppSettings.GetStr( asPathMailOut );
      end else if FuncIs( 'hamnewserrpath', 0, 0 ) then begin
         Result.AsStr := AppSettings.GetStr( asPathNewsErr );
      end else if FuncIs( 'hamstartuppath', 0, 0 ) then begin
         Result.AsStr := AppSettings.GetStr( asPathStartup );
      end else if FuncIs( 'hamreportspath', 0, 0 ) then begin
         Result.AsStr := AppSettings.GetStr( asPathReports );
      end else if FuncIs( 'hamexepath', 0, 0 ) then begin
         Result.AsStr := ExtractFilePath( ParamStr(0) );
      end else if FuncIs( 'hamrcpath', 0, 0 ) then begin // Classic-only
         Result.AsStr := AppSettings.GetStr( asPathScripts );

      end else if FuncIs( 'hamrequest', 1, 3 ) then begin
         Result.AsInt := -1;
         i := ParI(0, 0); // request code
         j := ParI(1,-1); // request data (list)
         k := ParI(2,-1); // reply data (list)

         if Engine.Lists.ListExists(j) then s := Engine.Lists.List[j].Text
                                       else s := '';
         Reply := Hamster.LiveServer.ReplyFor( TLiveMsg.Create( i, s ) );

         if Assigned( Reply ) then try
            Result.AsInt := Reply.MsgType;
            if (Reply.MsgType = LMREP_OK) and Engine.Lists.ListExists(k) then begin
               Engine.Lists.List[k].Text := Reply.MsgData;
            end;
         finally
            Reply.Free;
         end;

      end else if FuncIs( 'hamrequeststr', 1, 3 ) then begin
         Result.AsInt := -1;
         i := ParI(0, 0); // request code
         s := ParS(1,''); // request data (string)
         t := ParX(2,''); // reply data (variable)

         Reply := Hamster.LiveServer.ReplyFor( TLiveMsg.Create( i, s ) );

         if Assigned( Reply ) then try
            Result.AsInt := Reply.MsgType;
            if Reply.MsgType = LMREP_OK then begin
               if IsHscVariable(t) then Engine.Variables.Value[t].AsStr := Reply.MsgData;
            end;
         finally
            Reply.Free;
         end;

      end else if FuncIs( 'hammessage', 1, 2 ) then begin 
         i := ParI(0,0);
         j := ParI(1,0);
         Result.AsInt := HamMessage( i, j );

      end else if FuncIs( 'hamaddlog', 2, 3 ) then begin 
         Result.AsStr := ParS(0, '');
         if ParI(2, 1)=0 then HamAddLog( Result.AsStr, ParI(1, 0) )
                         else HamAddLog( '{' + ScriptName + '} '
                                       + Result.AsStr, ParI(1, 0) );

      end else if FuncIs( 'hamflush', 0, 0 ) then begin
         Result.AsInt := 0;
         Hamster.NewsHistory.SaveToFile;

      end else if FuncIs( 'hamthreadcount', 0, 0 ) then begin
         Result.AsInt := Hamster.ActiveThreads.CountActiveTasks;
      end else if FuncIs( 'hamisidle', 0, 0 ) then begin
         if Hamster.ActiveThreads.CountActiveTasks <= 0 then Result.AsInt := 1
                                                        else Result.AsInt := 0;
      end else if FuncIs( 'hamwaitidle', 0, 1 ) then begin
         TimeOut := ParI( 0, -1 );
         Result.AsInt := 0;
         while Hamster.ActiveThreads.CountActiveTasks > 0 do begin
            i := 500;
            if Timeout>=0 then begin
               i := TimeOut;
               if i>500 then i:=500;
            end;
            if StopEvent.WaitFor( i ) = wrSignaled then begin
               Result.AsInt := 2;
               break;
            end;
            if TimeOut>=0 then begin
               TimeOut := TimeOut - i;
               if TimeOut<=0 then begin Result.AsInt:=1; break; end;
            end;
         end;

      end else if FuncIs( 'hampurge', 0, 2 ) then begin
         Result.AsInt := -1;
         if Hamster.ActiveThreads.CountActiveTasks <= 0 then begin
            i := ParI( 0, HAM_PURGEOPT_DOALL );
            s := ParS( 1, '' );
            with TThreadPurge.Create( i , s, tftFreeOnTerminate ) do Resume;
            Result.AsInt := 0;
         end;
      end else if FuncIs( 'hamrebuildgloballists', 0, 0 ) then begin
         Result.AsInt := -1;
         if Hamster.ActiveThreads.CountActiveTasks <= 0 then begin
            with TThreadRebuildGlobalLists.Create( tftFreeOnTerminate ) do Resume;
            Result.AsInt := 0;
         end;
      end else if FuncIs( 'hamrebuildhistory', 0, 0 ) then begin
         Result.AsInt := -1;
         if Hamster.ActiveThreads.CountActiveTasks <= 0 then begin
            with TThreadHistoryRebuild.Create do Resume;
            Result.AsInt := 0;
         end;

      end else if FuncIs( 'hamsetlogin', 3, 3 ) then begin
         Result.AsInt := -1;
         Identifier := ParS(0,'(noname)');
         Username   := ParS(1,'?');
         Password   := ParS(2,'?');
         OK := True;
         if (copy(Username,1,1)='$') and (Password='') then begin
            s := Username;
            if not Hamster.Passwords.UsePassword( s, Username, Password ) then begin
               Log( LOGID_WARN, Format(
                  'Missing username/password for "%s"!', [s] ) );
               OK := False;
            end;
         end;
         if OK then begin
            Result.AsInt := 0;
            Hamster.Passwords.SavePassword( False, Identifier, Username, Password );
         end;


      end else if FuncIs( 'hamwebreader', 1, 6 ) then begin
         Result.AsInt := 0;
         with TThreadWebChannelReader.Create(
                 ParS( 0, '' ),     // RSS or ATOM feed URL
                 ParS( 1, '' ),     // newsgroup
                 ParS( 2, '' ),     // subject
                 ParS( 4, '' ),     // from
                 ParI( 3,  2 ),     // mode
                 False,             // !IsFile=IsUrl
                 ParI( 5, 0 ) <> 0, // Save Xml File?
                 FWebProxyServer, FWebProxyPort, FWebProxyUser, FWebProxyPass
              ) do Resume;
      end else if FuncIs( 'hamwebuseproxy', 0, 4 ) then begin
         Result.AsInt := 0;
         FWebProxyServer := ParS( 0, '' );
         FWebProxyPort   := ParI( 1, 0 );
         FWebProxyUser   := ParS( 2, '' );
         FWebProxyPass   := ParS( 3, '' );
      end else if FuncIs( 'hamwebhtmlload', 1, 1 ) then begin
         s := ParS( 0, '' );
         Result.AsStr := LoadLinkWebText( s, FWebProxyServer, FWebProxyPort,
                                             FWebProxyUser, FWebProxyPass );
      end else if FuncIs( 'hamwebhtmltotext', 1, 1 ) then begin
         s := ParS( 0, '' );
         Result.AsStr := HtmlToText( s );

         
      end else if FuncIs( 'hamnewspull', 0, 1 ) then begin
         Result.AsInt := 0;
         SrvList := ParS(0,'');
         if SrvList<>'' then SrvList := ';' + LowerCase(SrvList) + ';';
         Hamster.NewsJobs.AddPullDef( SrvList );
         Hamster.NewsJobs.StartThreads( SrvList, nil );
      end else if FuncIs( 'hamnewspost', 0, 1 ) then begin
         Result.AsInt := 0;
         SrvList := ParS(0,'');
         if SrvList<>'' then SrvList := ';' + LowerCase(SrvList) + ';';
         if Hamster.Config.NntpServers.Count>0 then begin
            if FileExists( AppSettings.GetStr(asPathNewsOut) + '*.msg' ) then begin
               Hamster.NewsJobs.AddPostDef( SrvList );
               Hamster.NewsJobs.StartThreads( SrvList, nil, True );
            end;
         end;

      end else if FuncIs( 'hamnewsjobsclear',   0, 0 ) then begin
         Result.AsInt := Hamster.NewsJobs.Clear;
      end else if FuncIs( 'hamnewsjobspulldef', 0, 1 ) then begin
         Result.AsInt := Hamster.NewsJobs.AddPullDef( ParS( 0, '' ) );
      end else if FuncIs( 'hamnewsjobspostdef', 0, 1 ) then begin
         Result.AsInt := Hamster.NewsJobs.AddPostDef( ParS( 0, '' ) );
      end else if FuncIs( 'hamnewsjobspull',    1, 2 ) then begin
         Result.AsInt := Hamster.NewsJobs.AddPullSrv( ParS(0,''), ParS(1,'') );
      end else if FuncIs( 'hamnewsjobspost',    1, 3 ) then begin
         Result.AsInt := Hamster.NewsJobs.AddPostSrv( ParS(0,''), ParS(1,''), ParS(2,'') );
      end else if FuncIs( 'hamnewsjobsfeed',    1, 2 ) then begin
         Result.AsInt := Hamster.NewsJobs.AddFeedSrv( ParS(0,''), ParS(1,'') );
      end else if FuncIs( 'hamnewsjobsstart',   0, 1 ) then begin
         Result.AsInt := Hamster.NewsJobs.StartThreads( ParS(0,''), nil );

      end else if FuncIs( 'hamfetchmail', 1, 7 ) then begin
         Result.AsInt := 0;
         Server   := ParS(0,''); if Server='' then Result.AsInt:=-1;
         Port     := ParS(1,'');
         User     := ParS(2,'');
         Pass     := ParS(3,'');
         DestUser := ParS(4,'');
         FiltSect := ParS(5,'');
         LeaveOnServer := ParS(6,'?');
         if Result.AsInt=0 then begin
             with TThreadPop3Fetch.Create(
                     Server, Port, User, Pass,
                     DestUser, FiltSect, LeaveOnServer ) do Resume;
         end;
      end else if FuncIs( 'hamsendmail', 1, 4 ) then begin
         Result.AsInt := 0;
         Server  := ParS(0,''); if Server='' then Result.AsInt:=-1;
         Port    := ParS(1,'');
         FromSel := ParS(2,'');
         ToSel   := ParS(3,'');
         if Result.AsInt = 0 then begin
            with TThreadSmtpSend.Create( False, Server, Port, '', '', FromSel, ToSel ) do Resume;
         end;
      end else if FuncIs( 'hamsendmailauth', 1, 6 ) then begin
         Result.AsInt := 0;
         Server  := ParS(0,''); if Server='' then Result.AsInt:=-1;
         Port    := ParS(1,'');
         User    := ParS(2,'');
         Pass    := ParS(3,'');
         FromSel := ParS(4,'');
         ToSel   := ParS(5,'');
         if Result.AsInt = 0 then begin
            with TThreadSmtpSend.Create( False, Server, Port, User, Pass, FromSel, ToSel ) do Resume;
         end;
      end else if FuncIs( 'hamsendmailmx', 0, 0 ) then begin 
         Result.AsInt := 0;
         with TThreadSmtpSend.Create( True, '', '', '', '', '', '' ) do Resume;
      end else if FuncIs( 'hammailexchange', 0, 1 ) then begin
         Result.AsInt := -1;
         Log( LOGID_WARN, 'Script command HamMailExchange not supported! '
                        + 'Use HamFetchMail/HamSendMail instead!' );
      end else if FuncIs( 'hamnewmail', 3, 3 ) then begin
         Result.AsInt := iif( ScriptNewMail( ParS(0,''), ParS(1,''), ParS(2,'') ), 0, -1 );


      end else if FuncIs( 'hamrasdial', 1, 3 ) then begin
         Result.AsInt := -1;
         ConnectionID := ParS(0,'');
         Username     := ParS(1,'');
         Password     := ParS(2,'');
         OK := True;
         if (copy(Username,1,1)='$') and (Password='') then begin
            s := Username;
            if not Hamster.Passwords.UsePassword( s, Username, Password ) then begin
               Log( LOGID_WARN, Format(
                  'Missing username/password for "%s"!', [s] ) );
               OK := False;
            end;
         end;
         if OK then begin
            Result.AsInt := Hamster.RasDialer.Dial( ConnectionID,
                                                    Username, Password );
            if Result.AsInt = 0 then FRasOurConnection := True;
         end;
      end else if FuncIs( 'hamrashangup', 0, 0 ) then begin
         Result.AsInt := Hamster.RasDialer.HangUp;
         FRasOurConnection := False;

      end else if FuncIs( 'hamgroupcount', 0, 0 ) then begin
         Result.AsInt := Hamster.Config.Newsgroups.Count;
      end else if FuncIs( 'hamgroupname',  1, 1 ) then begin
         i := ParI( 0, 0 );
         Result.AsStr := Hamster.Config.Newsgroups.Name[ i ];
      end else if FuncIs( 'hamgroupindex', 1, 1 ) then begin
         s := ParS( 0, '' );
         Result.AsInt := Hamster.Config.Newsgroups.IndexOf( s );
      end else if FuncIs( 'hamgroupopen',  1, 1 ) then begin
         s := ParS( 0, '' );
         if Hamster.Config.Newsgroups.IndexOf( s ) >= 0 then begin
             i := Hamster.ArticleBase.Open( s );
             if i>=0 then Engine.ResControl.Add( TResHamsterGroup.Create(i) );
             Result.AsInt := i
         end else begin
            Result.AsInt := -1;
         end;
      end else if FuncIs( 'hamgroupclose', 1, 1 ) then begin
         Result.AsInt := 0;
         i := ParI( 0, 0 );
         if i>=0 then begin
            Engine.ResControl.Remove( RESID_HamsterGroup, i );
            Hamster.ArticleBase.Close( i )
         end
      end else if FuncIs( 'hamartcount', 1, 1 ) then begin
         i := ParI( 0, 0 );
         Result.AsInt := Hamster.ArticleBase.Count[ i ];
      end else if FuncIs( 'hamartnomax', 1, 1 ) then begin
         i := ParI( 0, 0 );
         Result.AsInt := Hamster.ArticleBase.GetInt(i,gsLocalMax);
      end else if FuncIs( 'hamartnomin', 1, 1 ) then begin
         i := ParI( 0, 0 );
         Result.AsInt := Hamster.ArticleBase.GetInt(i,gsLocalMin);
      end else if FuncIs( 'hamarttext',  2, 2 ) then begin
         i := ParI( 0, 0 );
         j := ParI( 1, 0 );
         Result.AsStr := Hamster.ArticleBase.ReadArticle( i, j );
      end else if FuncIs( 'hamarttextbymid', 1, 1 ) then begin
         Result.AsStr := '';
         s := ParS( 0, '' ); // MID
         if Hamster.NewsHistory.LocateMID( s, t, i ) then begin
            j := Hamster.ArticleBase.Open( t );
            if j >= 0 then try
               Result.AsStr := Hamster.ArticleBase.ReadArticle( j, i );
            finally
               Hamster.ArticleBase.Close( j );
            end;
         end;
      end else if FuncIs( 'hamarttextexport',  2, 2 ) then begin
         i := ParI( 0, 0 );
         j := ParI( 1, 0 );
         Result.AsStr := Hamster.ArticleBase.ReadArticle( i, j );
         if Result.AsStr<>'' then begin
            Art := TMess.Create;
            Art.FullText := Result.AsStr;
            Result.AsStr := Art.ExportText;
            Art.Free;
         end;

      end else if FuncIs( 'hamartimport',  1, 4 ) then begin
         s := ParS( 0, '' ); // article
         t := ParS( 1, '' ); // override groups
         i := ParI( 2, 0  ); // ignore history
         j := ParI( 3, 0  ); // mark no-archive
         Result.AsInt := Integer( ImportArticle( s, t, (i<>0), (j<>0) ) );


      end else if FuncIs( 'hamartimportfile',  1, 4 ) then begin
         s := ParS( 0, '' ); // filename
         t := ParS( 1, '' ); // override groups
         i := ParI( 2, 0  ); // ignore history
         j := ParI( 3, 0  ); // test-only
         Result.AsInt := ImportArticlesFromFile( s, (j<>0), t, (i<>0), False );
      end else if FuncIs( 'hamartexportfile',  2, 4 ) then begin
         s := ParS( 0, '' ); // filename
         t := ParS( 1, '' ); // groupname
         i := ParI( 2, -1 ); // artno min
         j := ParI( 3, -1 ); // artno max
         Result.AsInt := ExportArticlesToFile( s, t, i, j );

      end else if FuncIs( 'hamartdeletemid',  1, 1 ) then begin
         s := ParS( 0, '' ); // MID
         ok := Hamster.ArticleBase.DeleteArticleByMID( s );
         while Hamster.ArticleBase.DeleteArticleByMID( s ) do ;
         if ok then Result.AsInt:=0 else Result.AsInt:=-1;

      end else if FuncIs( 'hamartlocatemid',  1, 3 ) then begin
         Result.AsInt := -1;
         s := ParS( 0, '' ); // MID
         if Hamster.NewsHistory.LocateMID( s, t, i ) then begin
            s := ParX( 1, '' ); // var for groupname
            if IsHscVariable(s) then Engine.Variables.Value[s].AsStr := t;
            s := ParX( 2, '' ); // var for artno
            if IsHscVariable(s) then Engine.Variables.Value[s].AsInt := i;
            Result.AsInt := 0;
         end;

      end else if FuncIs( 'hamartsearch',  2, 2 ) then begin 
         i := ParI( 0, -1 ); // selection list
         j := ParI( 1, -1 ); // result list
         if Engine.Lists.ListExists(i) and Engine.Lists.ListExists(j) then begin
            Result.AsInt := -2;
            NewsSearcher := TNewsSearcher.Create;
            try
               FSearchResults := Engine.Lists.List[j];
               try
                  NewsSearcher.SelectParamList( Engine.Lists.List[i], nil );
                  FSearchResults.Clear;
                  NewsSearcher.Search( Report_SearchResults );
                  Result.AsInt := FSearchResults.Count;
               except
                  on E: Exception do FSearchResults.Text := E.Message;
               end;
            finally
               NewsSearcher.Free;
               FSearchResults := nil;
            end;
         end else begin
            Result.AsInt := -1;
         end;

      end else if FuncIs( 'hamscorelist',  1, 1 ) then begin
         s := ParS( 0, '' ); // grpname
         ScoreFile := TFiltersNews.Create( AppSettings.GetStr(asPathBase) + CFGFILE_SCORES );
         ScoreFile.SelectSections( s );
         Result.AsStr := ScoreFile.SelectedLines;
         ScoreFile.Free;

      end else if FuncIs( 'hamscoretest',  2, 4 ) then begin
         Result.AsInt := 0;
         s := ParS( 0, '' ); // grpname
         t := ParS( 1, '' ); // arttext
         ScoreFile := TFiltersNews.Create( AppSettings.GetStr(asPathBase) + CFGFILE_SCORES );
         Art := TMess.Create;
         try
            ScoreFile.SelectSections( s );
            Art.FullText := t;
            t := '';
            ArticleToXOverRec( Art, XOverRec );
            case ParI( 3, 0 ) of
               1:   Result.AsInt := ScoreFile.ScoreBeforeLoad( XOverRec, @t );
               2:   Result.AsInt := ScoreFile.ScoreAfterLoad( Art, @t );
               else Result.AsInt := ScoreFile.ScoreBeforeLoad( XOverRec, @t )
                                  + ScoreFile.ScoreAfterLoad( Art, @t );
            end;
            s := ParX( 2, '' ); // var for matchlog
            if IsHscVariable(s) then Engine.Variables.Value[s].AsStr := t;
         finally
            ScoreFile.Free;
            Art.Free;
         end;

      end else if FuncIs( 'hamgetstatus', 1, 3 ) then begin
         i := ParI(0,0);
         j := ParI(1,0);
         k := ParI(2,1);
         Result.AsInt := HamGetStatus( i, j, k )

      end else if FuncIs( 'hamdnscachepurge', 1, 1 ) then begin
         i := ParI(0,0); // minutes
         Result.AsInt := Hamster.ScriptDnsCache.Purge( i );
      end else if FuncIs( 'hamdnslookuphostaddr', 1, 1 ) then begin
         s := ParS( 0, 'localhost' );
         t := Hamster.ScriptDnsCache.Get( s );
         if t = '' then begin
            i := LookupHostAddr( s );
            Result.AsStr := nAddrToStr( i );
         end else begin
            Result.AsStr := t;
         end;
      end else if FuncIs( 'hamdnslookuphostaddrlist', 2, 2 ) then begin
         i := ParI( 0, -1 ); // names (in)
         k := ParI( 1, -1 ); // addrs (out)
         if Lists.ListExists( i ) and Lists.ListExists( k ) then begin
            with TThreadLookupHostAddrs.Create( Lists.List[i], Lists.List[k],
                                                Hamster.ScriptDnsCache ) do begin
               Resume;
               WaitFor;
               Free;
            end;
            Result.AsInt := 1;
         end else begin
            Result.AsInt := 0;
         end;

      end else begin
         Result.Unassign;
      end
   end
end;

function THscEngineHamster.Engine_StartScript( const ScriptFile: String;
                                               const Params: String;
                                               const WaitForEnd: Boolean ): Integer;
var  ScriptThread: TThreadExecuteScript;
begin
    Result := StartNewScript( ScriptFile, Params, WaitForEnd,
                              ScriptThread, tftFreeOnTerminate, False, False );
end;

constructor THscEngineHamster.Create( const AStopEvent: TEvent;
                                      const APathHSM  : String;
                                      const AOutputBuffer: TThreadStringList );
begin
   inherited Create( APathHSM );
   StopEvent         := AStopEvent;
   FRasOurConnection := False;
   FOutputBuffer     := AOutputBuffer;

   FWebProxyServer := '';
   FWebProxyUser   := '';
   FWebProxyPass   := '';
   FWebProxyPort   := 0;
end;

end.
