// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cHamsterCfg;

interface

uses SysUtils, Classes, cSyncObjects, uType, cNewsJobs, cArtFiles, cPasswords,
     cSettings, SyncObjs;

type
   TConfigHamster = class;

   TConfigList = class
      protected
         FConfig  : TConfigHamster;
         FList    : TStringList;
         FFilename: String;

         function GetCount: Integer;

      public
         property Count: Integer read GetCount;

         procedure LoadFromFile; virtual;
         procedure SaveToFile; virtual;

         constructor Create( AConfig: TConfigHamster;
                             const AFilename: String );
         destructor Destroy; override;
   end;

   TConfigServers = class( TConfigList )
      protected
         FPort: String;
         FServersInUse: TStringList;

         function GetPath      ( Index: Integer ): String;
         function GetAliasName ( Index: Integer ): String;
         function GetSrvName   ( Index: Integer ): String;
         function GetSrvPort   ( Index: Integer ): String;
         function GetSrvUser   ( Index: Integer ): String;
         function GetSrvPass   ( Index: Integer ): String;
         function GetIsDisabled( Index: Integer ): Boolean;

      public
         property Path      [ Index: Integer ]: String  read GetPath;
         property AliasName [ Index: Integer ]: String  read GetAliasName;
         property SrvName   [ Index: Integer ]: String  read GetSrvName;
         property SrvPort   [ Index: Integer ]: String  read GetSrvPort;
         property SrvUser   [ Index: Integer ]: String  read GetSrvUser;
         property SrvPass   [ Index: Integer ]: String  read GetSrvPass;
         property IsDisabled[ Index: Integer ]: Boolean read GetIsDisabled;

         function Settings( const Index: Integer ): TSettingsPlain; overload;
         function Settings( const Server: String ): TSettingsPlain; overload;

         function SrvAdd( const ServerName: String ): Boolean; virtual;
         function SrvDel( const ServerName: String ): Boolean; virtual;

         function  UseLimit( const Index: Integer ): Integer; virtual;
         function  UseInc( const Index: Integer ): Boolean; virtual;
         procedure UseDec( const Index: Integer ); virtual;

         function  IndexOfAlias( const ServerAlias: String;
                                 const Silent: Boolean = False ): Integer;
         procedure LoadFromFile; override;

         constructor Create( AConfig: TConfigHamster;
                             const AFilename, APort: String );
         destructor Destroy; override;
   end;

   TConfigNntpServers = class( TConfigServers )
      public
         function SrvDel( const ServerName: String ): Boolean; override;

         function  IsReadOnly( const Server: String ): Boolean;
         function  PullThreads( Index: Integer;
                                LimitByJobs: Boolean ): Integer;
         function  ServerHasGroup( const Server, GroupName: String ): Boolean;
         function  AnyServerHasGroup( const GroupName: String ): Boolean;
         function  UseLimit( const Index: Integer ): Integer; override;

         constructor Create( AConfig: TConfigHamster;
                             const AFilename, APort: String );
         destructor Destroy; override;
   end;

   TConfigPop3Servers = class( TConfigServers )
      //
   end;

   TConfigSmtpServers = class( TConfigServers )
      //
   end;

   TConfigNewsgroups = class( TConfigList )
      protected
         function GetName( Index: Integer ): String;
         function GetPath( Index: Integer ): String;

      public
         property Path[ Index: Integer ]: String read GetPath;
         property Name[ Index: Integer ]: String read GetName;

         function  GetList( const Selection: String ): String;
         function  GroupClass( const Groupname: String;
                               InternalOnly: Boolean = False ): TActiveClass;
         function  IndexOf( const FindGroup: String ): Integer;
         procedure LoadFromFile; override;
         function  RefreshPrioritiesAndCount: Integer;
         function  Add( const Groupname: String ): Boolean;
         function  Del( const Groupname: String ): Boolean;
         function  HamGroupName( const hsHamGroup: Integer ): String;
   end;

   TConfigNewsPulls = class( TConfigList )
      private
         function GetServer( Index: Integer ): String;
         function GetGroup ( Index: Integer ): String;

      public
         property Server[ Index: Integer ]: String read GetServer;
         property Group [ Index: Integer ]: String read GetGroup;

         function IndexOf( const Servername, Groupname: String ): Integer;
         function ExistPull( const Servername, Groupname: String ): Boolean;
         function PullServerOf( const Groupname: String ): String;
         function ExistPullServer(const Groupname: String): Boolean;
         function IsPostServerFor( const Servername, Groupname: String ): Boolean;
         function GetPostServer( const Groupname: String ): String;
         function Add( const Servername, Groupname: String ): Boolean;
         function Del( const Servername, Groupname: String ): Boolean;
         function DelAll( const Servername: String ): Boolean;
   end;

   TConfigHamster = class
      private
         FRWLock         : TReaderWriterLock;
         FNntpServers    : TConfigNntpServers;
         FPop3Servers    : TConfigPop3Servers;
         FSmtpServers    : TConfigSmtpServers;
         FNewsgroups     : TConfigNewsgroups;
         FNewsPulls      : TConfigNewsPulls;
         FNewsJobs       : TNewsJobs;
         FArticleBase    : TArticleBase;
         FInitialArtCount: Integer;
         FPasswords      : TPasswords;
         FSettings       : TSettingsPlain;

         procedure LoadConfiguration;

      public
         property Settings   : TSettingsPlain     read FSettings;
         property NntpServers: TConfigNntpServers read FNntpServers;
         property Pop3Servers: TConfigPop3Servers read FPop3Servers;
         property SmtpServers: TConfigSmtpServers read FSmtpServers;
         property Newsgroups : TConfigNewsgroups  read FNewsgroups;
         property NewsPulls  : TConfigNewsPulls   read FNewsPulls;

         property InitialArticleCount: Integer read FInitialArtCount;

         procedure BeginRead;
         procedure EndRead;
         procedure BeginWrite;
         procedure EndWrite;

         constructor Create( ANewsJobs   : TNewsJobs;
                             AArticleBase: TArticleBase;
                             APasswords  : TPasswords );
         destructor Destroy; override;
   end;

implementation

uses uTools, uConst, uConstVar, uVar, cLogFileHamster, uHamTools, cHamster,
     uDateTime, IniFiles;

procedure ConfigSettingChanged( const Qualifier: String;
                                const ID: Integer );
begin
   case ID of

      hsLocalNntpBind, hsLocalNntpPort, hsLocalNntpClients, hsLocalNntpSameCli,
         hsLocalNntpLimitLineLen, hsLocalNntpLimitTextSize: begin
         Log( LOGID_WARN, 'Note: Server change will not take effect until NNTP server is restarted.' );
      end;
      
      hsLocalPop3Bind, hsLocalPop3Port, hsLocalPop3Clients, hsLocalPop3SameCli,
         hsLocalPop3LimitLineLen, hsLocalPop3LimitTextSize: begin
         Log( LOGID_WARN, 'Note: Server change will not take effect until POP3 server is restarted.' );
      end;

      hsLocalSmtpBind, hsLocalSmtpPort, hsLocalSmtpClients, hsLocalSmtpSameCli,
         hsLocalSmtpLimitLineLen, hsLocalSmtpLimitTextSize: begin
         Log( LOGID_WARN, 'Note: Server change will not take effect until SMTP server is restarted.' );
      end;

      hsLocalRecoBind, hsLocalRecoPort, hsLocalRecoClients, hsLocalRecoSameCli,
         hsLocalRecoLimitLineLen, hsLocalRecoLimitTextSize: begin
         Log( LOGID_WARN, 'Note: Server change will not take effect until RC server is restarted.' );
      end;

      hsRemoteTimeoutConnect, hsRemoteTimeoutCommand: begin
         Log( LOGID_WARN, 'Note: Timeout changes will not take effect for already active tasks.' );
      end;

      hsPasswordCodeBase: begin
         Log( LOGID_WARN, 'Note: Changing password code base also requires new password settings.' );
      end;

      hsTasksLimit: begin
         Hamster.Config.BeginWrite;
         try
            SEM_LIMITTASKS.Count := Hamster.Config.Settings.GetInt(hsTasksLimit);
         finally Hamster.Config.EndWrite end;
      end;

      hsHistoryChunkBits: begin
         AppSettings.SetChanged( ID, False ); // no remote change intended
      end;

   end;
end;

procedure RemoveServerDir( const SrvDir: String );
begin
   // delete known server-files
   DeleteFile( SrvDir + SRVFILE_INI );
   DeleteFile( SrvDir + SRVFILE_HELPTEXT );
   DeleteFile( SrvDir + SRVFILE_GROUPS );
   DeleteFile( SrvDir + SRVFILE_GRPDESCS );
   DeleteFile( SrvDir + SRVFILE_OVERVIEWFMT );
   DeleteFile( SrvDir + SRVFILE_GETMIDLIST );
   DeleteFile( SrvDir + SRVFILE_GREETING );
   DeleteFile( SrvDir + 'newgrps.txt' ); // used til 1.3.19

   // remove directory
   if not RemoveDir( SrvDir ) then begin
      Log( LOGID_WARN, Format( 'Couldn''t remove directory "%s"!', [SrvDir] ) );
   end;
end;

//----------------------------------------------------------- TConfigList -----

constructor TConfigList.Create( AConfig: TConfigHamster;
                                const AFilename: String );
begin
   inherited Create;

   FConfig   := AConfig;
   FFilename := AFilename;
   
   FList := TStringList.Create;
   FList.Sorted := True;
   FList.Duplicates := dupIgnore;
end;

destructor TConfigList.Destroy;
var  i: Integer;
begin
   for i:=0 to FList.Count-1 do begin
      if Assigned( FList.Objects[i] ) then TSettingsPlain( FList.Objects[i] ).Free;
   end;
   if Assigned(FList) then FList.Free;
   inherited Destroy;
end;

function TConfigList.GetCount: Integer;
begin
   FConfig.BeginRead;
   try
      Result := FList.Count;
   finally FConfig.EndRead end;
end;

procedure TConfigList.LoadFromFile;
begin
   FConfig.BeginWrite;
   try
      FList.Clear;
      if FileExists( FFilename ) then FList.LoadFromFile( FFilename );
   finally FConfig.EndWrite end;
end;

procedure TConfigList.SaveToFile;
begin
   if FFilename = '' then exit;
   FConfig.BeginWrite;
   try
      FList.SaveToFile( FFilename );
   finally FConfig.EndWrite end;
end;

//-------------------------------------------------------- TConfigServers -----

constructor TConfigServers.Create( AConfig: TConfigHamster;
                                   const AFilename, APort: String );
begin
   inherited Create( AConfig, AFilename );
   FPort := APort;

   FServersInUse := TStringList.Create;
   FServersInUse.Sorted := True;
   FServersInUse.Duplicates := dupIgnore;
end;

destructor TConfigServers.Destroy;
begin
   if Assigned( FServersInUse ) then FServersInUse.Free;
   inherited Destroy;
end;

procedure TConfigServers.LoadFromFile;
var  i: Integer;
begin
   inherited LoadFromFile;
   for i := 0 to Count-1 do ForceDirectories( GetPath(i) );
end;

function TConfigServers.IndexOfAlias( const ServerAlias: String;
                                      const Silent: Boolean = False ): Integer;
var  i: Integer;
     Srv: String;
begin
   FConfig.BeginRead;
   try

      Result := -1;

      i := Pos( SERVER_PORT_SEP, ServerAlias );
      if i > 0 then Srv := copy( ServerAlias, 1, i-1 )
               else Srv := ServerAlias;

      if length( Srv ) > 0 then begin
         for i := 0 to GetCount - 1 do begin
            if AnsiCompareText( GetAliasName(i), Srv ) = 0 then begin
               Result := i;
               break;
            end;
         end;
      end;

      if (Result < 0) and (not Silent) then begin
         Log( LOGID_WARN, 'Tried to access unknown server: '
                        + '"' + Srv + '" (' + ClassName + ')' );
         for i := 0 to GetCount - 1 do begin
            Log( LOGID_INFO, Format( 'Known server %d of %d: "%s"',
                                     [ i+1, GetCount, GetAliasName(i) ] ) );
         end;
      end;

      finally FConfig.EndRead end;
end;

function TConfigServers.SrvAdd( const ServerName: String ): Boolean;
var  Nam: String;
     Idx: Integer;
begin
   Result := False;

   Nam := ServerName;
   while Pos(' ',Nam) > 0 do System.Delete( Nam, Pos(' ',Nam), 1 );
   if Nam = '' then exit;
   if Pos( SERVER_PORT_SEP, Nam ) = 0 then Nam := Nam + SERVER_PORT_SEP + FPort;
   if IndexOfAlias( Nam, True ) >= 0 then exit;

   FConfig.BeginWrite;
   try

      Idx := FList.Add( Nam );
      ForceDirectories( GetPath(Idx) );
      SaveToFile;

      Result := True;

   finally FConfig.EndWrite end;
end;

function TConfigServers.SrvDel( const ServerName: String ): Boolean;
var  Idx: Integer;
     Dir: String;
begin
   Result := False;

   FConfig.BeginWrite;
   try
   
      Idx := IndexOfAlias( ServerName );
      if Idx < 0 then exit;

      Dir := GetPath( Idx );
      RemoveServerDir( Dir );
      FList.Delete( Idx );
      SaveToFile;

      Result := True;

   finally FConfig.EndWrite end;
end;

function TConfigServers.GetAliasName(Index: Integer): String;
// 'name/alias,port' => 'name/alias'
begin
   FConfig.BeginRead;
   try
      if (Index < 0) or (Index >= Count) then begin
         Result := '';
      end else begin
         Result := ExtractStr( FList[Index], SERVER_PORT_SEP, 0 );
      end;
   finally FConfig.EndRead end;
end;

function TConfigServers.GetSrvName(Index: Integer): String;
// 'name/alias,port' => 'name'
var  i: Integer;
begin
   FConfig.BeginRead;
   try
      Result := GetAliasName( Index );
      i := Pos( SERVER_ALIAS_SEP, Result );
      if i > 0 then Result := TrimWhSpace( copy( Result, 1, i-1 ) );
   finally FConfig.EndRead end;
end;

function TConfigServers.GetSrvPort(Index: Integer): String;
// 'name/alias,port' => 'port'
begin
   FConfig.BeginRead;
   try
      if (Index < 0) or (Index >= Count) then begin
         Result := FPort;
      end else begin
         Result := ExtractStr( FList[Index], SERVER_PORT_SEP, 1, FPort );
      end;
   finally FConfig.EndRead end;
end;

function TConfigServers.GetPath(Index: Integer): String;
begin
   FConfig.BeginRead;
   try
      Result := AppSettings.GetStr(asPathServer)
              + StringReplace( GetAliasName(Index),
                               SERVER_ALIAS_SEP, '-', [rfReplaceAll] )
              + '\';
   finally FConfig.EndRead end;
end;

function TConfigServers.GetSrvUser(Index: Integer): String;
var  s: String;
begin
   FConfig.BeginRead;
   try
      FConfig.FPasswords.UsePassword( GetAliasName(Index), Result, s );
   finally FConfig.EndRead end;
end;

function TConfigServers.GetSrvPass(Index: Integer): String;
var  s: String;
begin
   FConfig.BeginRead;
   try
      FConfig.FPasswords.UsePassword( GetAliasName(Index), s, Result );
   finally FConfig.EndRead end;
end;

function TConfigServers.Settings( const Index: Integer ): TSettingsPlain;
begin
   FConfig.BeginRead;
   try
      if not Assigned( FList.Objects[Index] ) then begin
         FList.Objects[ Index ] := TSettingsPlain.Create(
            SettingsDef_Server,
            TSettingsHandler_IniFile.Create( GetPath(Index) + SRVFILE_INI ),
            True {AutoFlush}
         );
      end;

      Result := TSettingsPlain( FList.Objects[ Index ] );
   finally FConfig.EndRead end;
end;

function TConfigServers.Settings( const Server: String ): TSettingsPlain;
var  Index: Integer;
begin
   FConfig.BeginRead;
   try
      Result := nil;
      Index  := IndexOfAlias( Server );
      if Index >= 0 then begin
         Result := Settings( Index )
      end else begin
         Log( LOGID_ERROR, 'Accessing settings of unknown server ' + Server );
      end;
   finally FConfig.EndRead end;
end;

function TConfigServers.GetIsDisabled( Index: Integer ): Boolean;
begin
   Result := Settings( Index ).GetBoo( ssDisabled );
end;

function TConfigServers.UseLimit( const Index: Integer ): Integer;
begin
   Result := 1;
end;

function TConfigServers.UseInc( const Index: Integer ): Boolean;
var  Idx, Cnt: Integer;
begin
   FConfig.BeginRead;
   try
      Idx := FServersInUse.IndexOf( GetAliasName(Index) );
      if Idx < 0 then begin
         Idx := FServersInUse.AddObject( GetAliasName(Index), Pointer(0) );
      end;
      Cnt := LongInt( FServersInUse.Objects[Idx] );

      if Cnt < UseLimit( Index ) then begin
         Result := True;
         FServersInUse.Objects[Idx] := Pointer( Cnt + 1 );
      end else begin
         Result := False;
      end;
   finally FConfig.EndRead end;
end;

procedure TConfigServers.UseDec( const Index: Integer );
var  Idx, Cnt: Integer;
begin
   FConfig.BeginRead;
   try
     Idx := FServersInUse.IndexOf( GetAliasName(Index) );
     if Idx >= 0 then begin
        Cnt := LongInt( FServersInUse.Objects[Idx] );
        if Cnt > 0 then FServersInUse.Objects[Idx] := Pointer( Cnt - 1 );
     end;
  finally FConfig.EndRead end;
end;

//---------------------------------------------------- TConfigNntpServers -----

constructor TConfigNntpServers.Create( AConfig: TConfigHamster;
                                      const AFilename, APort: String );
begin
   inherited Create( AConfig, AFilename, APort );
end;

destructor TConfigNntpServers.Destroy;
begin
   inherited Destroy;
end;

function TConfigNntpServers.SrvDel( const ServerName: String ): Boolean;
begin
   FConfig.BeginWrite;
   try
      Result := inherited SrvDel( ServerName );
      if Result then FConfig.NewsPulls.DelAll( ServerName );
      GlobalListMarker( glOPEN );
   finally FConfig.EndWrite end;
end;

function TConfigNntpServers.IsReadOnly( const Server: String ): Boolean;
var  Index: Integer;
begin
   FConfig.BeginRead;
   try
      Index := IndexOfAlias( Server );
      if Index >= 0 then begin
         Result := Settings( Index ).GetBoo( ssNntpReadOnly );
         if ( not Result ) and ( IsDisabled[Index] ) then begin
            Result := True;
            Log( LOGID_INFO,
                 'Assuming "read-only" for disabled server: ' + Server );
         end;
      end else begin
         Result := True;
         Log( LOGID_WARN,
              'Assuming "read-only" for unknown server: ' + Server );
      end;
   finally FConfig.EndRead end;
end;

function TConfigNntpServers.PullThreads( Index: Integer;
                                         LimitByJobs: Boolean ): Integer;
var  i: Integer;
begin
   Result := Settings( Index ).GetInt( ssNntpPullThreads );
   if Result < 0 then Result := 0;
   if Result > 4 then Result := 4;

   if LimitByJobs then begin
      FConfig.BeginRead;
      try
         i := FConfig.FNewsJobs.MaxThreadsFor( GetAliasName(Index) );
         if Result > i then Result := i;
      finally FConfig.EndRead end;
   end;
end;

function TConfigNntpServers.ServerHasGroup( const Server, GroupName: String ): Boolean;
var  SrvIdx, GrpCurr, j: Integer;
     s: String;
     TS: TStringList;
begin
   Result := False;
   TS := TStringList.Create;
   FConfig.BeginRead;

   try
      SrvIdx := IndexOfAlias( Server );
      if SrvIdx >= 0 then begin

         if FileExists( GetPath(SrvIdx) + SRVFILE_GROUPS ) then begin
            TS.LoadFromFile( GetPath(SrvIdx) + SRVFILE_GROUPS );
            for GrpCurr:=0 to TS.Count-1 do begin
               s := TS[ GrpCurr ];
               j := PosWhSpace( s );
               if j > 0 then s := copy( s, 1, j-1 );
               if AnsiCompareText( s, GroupName ) = 0 then begin
                  Result := True;
                  break;
               end;
            end;
         end;

      end;

   finally
      FConfig.EndRead;
      TS.Free;
   end;
end;

function TConfigNntpServers.AnyServerHasGroup( const GroupName: String ): Boolean;
var  SrvIdx: Integer;
begin
   Result := False;
   FConfig.BeginRead;
   try
      for SrvIdx := GetCount-1 downto 0 do begin
         if ServerHasGroup( GetAliasName(SrvIdx), GroupName ) then begin
            Result := True;
            break;
         end;
      end;
   finally
      FConfig.EndRead;
   end;
end;

function TConfigNntpServers.UseLimit( const Index: Integer ): Integer;
begin
   Result := PullThreads( Index, False );
end;


//----------------------------------------------------- TConfigNewsgroups -----

function TConfigNewsgroups.GetName( Index: Integer ): String;
begin
   FConfig.BeginRead;
   try
      Result := ExtractStr( FList[Index], ',', 0 );
   finally FConfig.EndRead end;
end;

function TConfigNewsgroups.GetPath( Index: Integer ): String;
begin
   FConfig.BeginRead;
   try
      Result := AppSettings.GetStr(asPathGroups) + GetName(Index) + '\';
   finally FConfig.EndRead end;
end;

function TConfigNewsgroups.GetList( const Selection: String ): String;
var  i, k, c: Integer;
     s, n: String;
     b: Boolean;
begin
   Result := '';

   b := ( length(Selection) > 0 );

   FConfig.BeginRead;
   try
      for i:=0 to GetCount-1 do begin
         s := GetName(i);
         if b then begin
            n := '';
            c := ord( GroupClass(s) );
            for k:=1 to length(Selection) do begin
               if ord(Selection[k])-ord('0') = c then begin n:=s; break end;
            end;
         end else begin
            n := s;
         end;
         if length(n) > 0 then Result := Result + n + CRLF;
      end;
   finally FConfig.EndRead end;
end;

function TConfigNewsgroups.GroupClass( const Groupname: String;
                                       InternalOnly: Boolean = False ): TActiveClass;

   function GetGroupLastPull: String;
   var  GrpHdl: Integer;
   begin
      Result := '';
      GrpHdl := FConfig.FArticleBase.Open( GroupName );
      if GrpHdl >= 0 then try
         Result := FConfig.FArticleBase.GetStr( GrpHdl, gsLastServerPull );
      finally
         FConfig.FArticleBase.Close( GrpHdl );
      end;
   end;

var  i, hsHamGroup, Index: Integer;
     s: String;
begin
   Result := aclUnknown;

   for hsHamGroup := hsHamGroupFirst to hsHamGroupLast do begin
      if AnsiCompareText( HamGroupName(hsHamGroup), Groupname ) = 0 then begin
         Result := aclInternal;
         exit;
      end;
   end;
   if InternalOnly then exit;

   FConfig.BeginRead;
   try
      i := IndexOf( Groupname );
      if i >= 0 then begin
         Result := aclLocal;
         if FConfig.FNewsPulls.ExistPullServer( Groupname ) then begin
            Result := aclIsPulled;
         end else begin
            Index  := IndexOf( Groupname );
            if Index >= 0 then begin
               s := GetGroupLastPull;
               if s <> '' then Result := aclWasPulled;
            end;
         end;

      end;
   finally FConfig.EndRead end;
end;

function TConfigNewsgroups.IndexOf( const FindGroup: String ): Integer;
var  i: Integer;
begin
   FConfig.BeginRead;
   try
      Result := -1;
      for i:=0 to GetCount-1 do begin
         if AnsiCompareText( GetName(i), FindGroup ) = 0 then begin
            Result := i;
            break;
         end;
      end;
   finally FConfig.EndRead end;
end;

procedure TConfigNewsgroups.LoadFromFile;
var  OldCount, hsHamGroup, i, GrpHdl: Integer;
     s: String;
begin
   FConfig.BeginWrite;
   try
      inherited LoadFromFile;

      // make sure, that internal-groups are in the list
      OldCount := Count;
      for hsHamGroup := hsHamGroupFirst to hsHamGroupLast do begin
         s := HamGroupName(hsHamGroup);
         if s <> '' then FList.Add( s );
      end;
      if OldCount <> Count then begin
         // internal group was added
         SaveToFile;
         if (OldCount = 0) and (Count=1) then begin // first group, set desc.
            GrpHdl := FConfig.FArticleBase.Open( HamGroupName(hsHamGroupDefault) );
            if GrpHdl >= 0 then try
               FConfig.FArticleBase.SetStr( GrpHdl, gsDescription,
                                                    'Internal Hamster Group' );
            finally
               FConfig.FArticleBase.Close( GrpHdl );
            end;
         end;
      end;

      for i := 0 to Count-1 do ForceDirectories( GetPath(i) );

   finally FConfig.EndWrite end;
end;

function TConfigNewsgroups.Add( const Groupname: String ): Boolean;
var  GrpHdl : Integer;
     Desc   : String;
     LastMax: Integer;
     TempIni: TIniFile;
begin
   Result := False;
   FConfig.BeginWrite;
   try
      if not IsNewsgroup( Groupname ) then exit;
      if IndexOf(Groupname) >= 0 then begin Result:=True; exit end;

      FList.Add( Groupname );
      SaveToFile;

      GrpHdl := FConfig.FArticleBase.Open( GroupName ); // open also creates dirs
      if GrpHdl < 0 then exit;
      try
         Desc := GlobalGroupDesc( GroupName );
         if Desc<>'' then FConfig.FArticleBase.SetStr( GrpHdl, gsDescription, Desc );

         // creation time
         FConfig.FArticleBase.SetDT( GrpHdl, gsCreated, NowGMT );

         // init article range numbers with former ones
         LastMax := 0;
         TempIni := TIniFile.Create( AppSettings.GetStr(asPathGroups) + GRPFILE_SAVEGROUP );
         try
            LastMax := TempIni.ReadInteger( 'Local.Max', GroupName, LastMax );
         finally
            TempIni.Free;
         end;
         FConfig.FArticleBase.SetInt( GrpHdl, gsLocalMin, LastMax + 1 );
         FConfig.FArticleBase.SetInt( GrpHdl, gsLocalMax, LastMax );

      finally
         FConfig.FArticleBase.Close( GrpHdl );
      end;

      FConfig.FNewsJobs.GroupPriority[ GroupName ] := MaxInt-4; // = "new group"
      Result := True;

   finally FConfig.EndWrite end;
end;

function TConfigNewsgroups.Del( const Groupname: String ): Boolean;
var  Index, hsHamGroup: Integer;
     Server: String;
begin
   Result := False;
   FConfig.BeginWrite;
   try
      Index := IndexOf( Groupname );
      if Index < 0 then exit;

      // protect internal groups from being deleted
      for hsHamGroup := hsHamGroupFirst to hsHamGroupLast do begin
         if AnsiCompareText( Groupname, HamGroupName(hsHamGroup) ) = 0 then exit;
      end;

      // remove group files and directory
      if FConfig.FArticleBase.DeleteGroup( GroupName ) = -1 then exit; // in use

      // remove all pulls
      repeat
         Server := FConfig.NewsPulls.PullServerOf( Groupname );
         if Server <> '' then begin
            if not FConfig.NewsPulls.Del( Server, GroupName ) then break;
         end;
      until Server = '';

      // remove from active
      FList.Delete( Index );
      SaveToFile;

      Result := True;
      
   finally FConfig.EndWrite end;
end;

function TConfigNewsgroups.HamGroupName( const hsHamGroup: Integer ): String;
begin
   Result := FConfig.Settings.GetStr( hsHamGroup );
   if Result = '' then Result := FConfig.Settings.GetStr( hsHamGroupDefault );
end;

function TConfigNewsgroups.RefreshPrioritiesAndCount: Integer;
var  i, LfdArt: Integer;
     GrpNew: Boolean;
begin
   FConfig.BeginWrite;
   try
      // Initialize priorities for pulling news
      Result := 0;
      for i:=0 to GetCount-1 do begin
         FConfig.FArticleBase.EstimateValues( GetName(i), LfdArt, GrpNew );
         if GrpNew then begin
            // pull new groups with a high priority
            FConfig.FNewsJobs.GroupPriority[ GetName(i) ] := MaxInt-4;
         end else begin
            FConfig.FNewsJobs.GroupPriority[ GetName(i) ] := LfdArt;
         end;
         inc( Result, LfdArt );
      end;
   finally FConfig.EndWrite end;
end;

//------------------------------------------------------ TConfigNewsPulls -----

function TConfigNewsPulls.GetServer( Index: Integer ): String;
begin
   FConfig.BeginRead;
   try
      Result := ExtractStr( FList[Index], ',', 1 );
   finally FConfig.EndRead end;
end;

function TConfigNewsPulls.GetGroup( Index: Integer ): String;
begin
   FConfig.BeginRead;
   try
      Result := ExtractStr( FList[Index], ',', 0 );
   finally FConfig.EndRead end;
end;

function TConfigNewsPulls.IndexOf( const Servername, Groupname: String ): Integer;
var  i: Integer;
begin
   FConfig.BeginRead;
   try
      Result := -1;
      for i:=0 to GetCount-1 do begin
         if AnsiCompareText( GetGroup(i), Groupname ) = 0 then begin
            if AnsiCompareText( GetServer(i), Servername ) = 0 then begin
               Result := i;
               break;
            end;
         end;
      end;
   finally FConfig.EndRead end;
end;

function TConfigNewsPulls.ExistPull( const Servername, Groupname: String ): Boolean;
begin
   FConfig.BeginRead;
   try
      Result := ( IndexOf( Servername, Groupname ) >= 0 );
   finally FConfig.EndRead end;
end;

function TConfigNewsPulls.PullServerOf( const Groupname: String ): String;
var  i: Integer;
begin
   Result := '';
   FConfig.BeginRead;
   try
      for i:=0 to GetCount-1 do begin
         if AnsiCompareText( GetGroup(i), Groupname ) = 0 then begin
            Result := GetServer(i);
            break;
         end;
      end;
   finally FConfig.EndRead end;
end;

function TConfigNewsPulls.ExistPullServer(const Groupname: String): Boolean;
begin
   Result := ( PullServerOf( Groupname ) <> '' );
end;

function TConfigNewsPulls.Add( const Servername, Groupname: String ): Boolean;
begin
   FConfig.BeginWrite;
   try
      Result := True;
      try
         if ExistPull( Servername, Groupname ) then exit;
         if FConfig.NntpServers.IndexOfAlias( Servername ) < 0 then begin Result:=False; exit end;
         if FConfig.Newsgroups.IndexOf( Groupname ) < 0 then begin // add group
            if not FConfig.Newsgroups.Add( Groupname ) then begin Result:=False; exit end;
         end;
         FList.Add( Groupname + ',' + Servername );
         SaveToFile;

      except
         Result := False;
      end;
   finally FConfig.EndWrite end;
end;

function TConfigNewsPulls.Del( const Servername, Groupname: String ): Boolean;
var  i: Integer;
begin
   FConfig.BeginWrite;
   try
      Result := True;
      try
         i := IndexOf( Servername, Groupname );
         if i < 0 then exit;
         FList.Delete( i );
         SaveToFile;

      except
         Result := False;
      end;
   finally FConfig.EndWrite end;
end;

function TConfigNewsPulls.DelAll( const Servername: String ): Boolean;
var  srv: String;
     i: Integer;
     b: Boolean;
begin
   i := Pos( SERVER_PORT_SEP, Servername );
   if i > 0 then Srv := copy( Servername, 1, i-1 )
            else Srv := Servername;

   FConfig.BeginWrite;
   try
      b := False;
      for i := FList.Count-1 downto 0 do begin
         if AnsiCompareText( GetServer(i), srv ) = 0 then begin
            FList.Delete( i );
            b := True;
         end;
      end;
      if b then SaveToFile;
      Result := True;

   finally FConfig.EndWrite end;
end;

function TConfigNewsPulls.IsPostServerFor( const Servername, Groupname: String ): Boolean;
var  Index: Integer;
begin
   FConfig.BeginRead;
   try
      Result := False;
      Index  := IndexOf( Servername, Groupname );
      if Index >= 0 then begin
         Result := not FConfig.NntpServers.IsReadOnly( Servername );
      end;
   finally FConfig.EndRead end;
end;

function TConfigNewsPulls.GetPostServer( const Groupname: String ): String;

   function GetGroupPostServer: String;
   var  GrpHdl: Integer;
   begin
      Result := '';
      GrpHdl := FConfig.FArticleBase.Open( GroupName );
      if GrpHdl >= 0 then try
         Result := FConfig.FArticleBase.GetStr( GrpHdl, gsPostServer );
      finally
         FConfig.FArticleBase.Close( GrpHdl );
      end;
   end;

var  i, Index: Integer;
     s, srv, prefMain, prefGroup: String;
     GroupFound: Boolean;
begin
   FConfig.BeginRead;
   try
   
      // get preferred post server for all groups
      prefMain := FConfig.Settings.GetStr( hsPostServer );
      i := Pos( ',', prefMain );
      if i > 0 then prefMain := copy( prefMain, 1, i-1 );

      // get preferred post server for given group
      prefGroup := '';
      Index := FConfig.Newsgroups.IndexOf( Groupname );
      if Index >= 0 then begin
         prefGroup := GetGroupPostServer;
         i := Pos( ',', prefGroup );
         if i > 0 then prefGroup := copy( prefGroup, 1, i-1 );
      end;

      // check servers in all pulls of given group
      Result := '';
      s := '(no pullserver found; keep local)';
      GroupFound := False;

      for i := 0 to GetCount - 1 do begin
         if AnsiCompareText( GetGroup(i), Groupname ) = 0 then begin

            GroupFound := True;
            srv := GetServer(i);
            
            if not FConfig.NntpServers.IsReadOnly( srv ) then begin

               if srv = PrefGroup then begin
                  Result := srv;
                  s := ' (is preferred postserver of group)';
                  break;
               end else if srv = PrefMain then begin
                  Result := srv;
                  s := ' (is preferred postserver)';
                  if PrefGroup = '' then break;
               end else if Result = '' then begin
                  Result := srv;
                  s := ' (is pullserver for group)';
                  if PrefGroup + PrefMain = '' then break; // no preferred
               end;

            end else begin

               if Result = '' then begin
                  s := '(only "read-only" pullserver found; keep local)';
               end;

            end;
            
         end;
      end;

      if not GroupFound then s := '(no pulls for group found; keep local)';

      // log results
      i := LOGID_DETAIL;
      if Result = '' then begin
         if not( FConfig.Newsgroups.GroupClass(Groupname)
                 in [ aclInternal, aclLocal ] ) then i := LOGID_INFO;
      end;
      Log( i, 'Find POST server: ' + Groupname + ' -> ' + Result + ' ' + s );

   finally FConfig.EndRead end;
end;


//-------------------------------------------------------- TConfigHamster -----

procedure TConfigHamster.LoadConfiguration;
begin
   BeginWrite;
   try
      FNntpServers.LoadFromFile;
      FPop3Servers.LoadFromFile;
      FSmtpServers.LoadFromFile;
      FNewsgroups .LoadFromFile;
      FNewsPulls  .LoadFromFile;

      // initialize priorities for pulling news
      FInitialArtCount := Newsgroups.RefreshPrioritiesAndCount;

      ForceDirectories( AppSettings.GetStr(asPathMails) + 'admin' );

   finally EndWrite end;
end;

procedure TConfigHamster.BeginRead;
begin
   FRWLock.BeginRead;
end;

procedure TConfigHamster.EndRead;
begin
   FRWLock.EndRead;
end;

procedure TConfigHamster.BeginWrite;
begin
   FRWLock.BeginWrite;
end;

procedure TConfigHamster.EndWrite;
begin
   FRWLock.EndWrite;
end;

constructor TConfigHamster.Create( ANewsJobs   : TNewsJobs;
                                   AArticleBase: TArticleBase;
                                   APasswords  : TPasswords );
begin
     inherited Create;

     FNewsJobs    := ANewsJobs;
     FArticleBase := AArticleBase;
     FPasswords   := APasswords;

     FRWLock := TReaderWriterLock.Create;

     FSettings := TSettingsPlain.Create(
        SettingsDef_Hamster,
        TSettingsHandler_IniFile.Create( AppSettings.GetStr(asPathBase) + CFGFILE_HAMSTER ),
        True {AutoFlush}
     );

     FNntpServers := TConfigNntpServers.Create( Self, AppSettings.GetStr(asPathBase) + CFGFILE_SERVER_NNTP, '119' );
     FPop3Servers := TConfigPop3Servers.Create( Self, AppSettings.GetStr(asPathBase) + CFGFILE_SERVER_POP3, '110' );
     FSmtpServers := TConfigSmtpServers.Create( Self, AppSettings.GetStr(asPathBase) + CFGFILE_SERVER_SMTP, '25'  );

     FNewsgroups := TConfigNewsgroups.Create( Self, AppSettings.GetStr(asPathBase) + CFGFILE_NEWSGROUPS );
     FNewsPulls  := TConfigNewsPulls.Create ( Self, AppSettings.GetStr(asPathBase) + CFGFILE_NEWSPULLS  );

     LoadConfiguration;
     FSettings.OnChanged := ConfigSettingChanged;
end;

destructor TConfigHamster.Destroy;
begin
     FRWLock.BeginWrite;

     FNewsgroups.Free;
     FNewsPulls.Free;

     FNntpServers.Free;
     FPop3Servers.Free;
     FSmtpServers.Free;

     FSettings.OnChanged := nil;
     FSettings.Free;

     FRWLock.Free;

     inherited Destroy;
end;

end.

