// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cArtFiles; // Storage for articles and mails

// ----------------------------------------------------------------------------
// Contains the class and various functions to access Hamster's message-base.
// ----------------------------------------------------------------------------

interface

{$INCLUDE Compiler.inc}

uses Windows, SysUtils, Classes, uConst, uType, cArticle, cArtFile, Shellapi,
     cLogFileHamster, cSettings, cHscAction;

type
   EGroupsError = class( Exception );

   TXrefArtNo = class
      GrpNam: String;
      GrpHdl: LongInt;
      ArtNo : Integer;
   end;

   TOpenArtFile = class
      ArtFile  : TArticleFile;
      UseCount : LongInt;
      GrpName  : String;
   end;

   TArticleBase = class
      private
         OpenArtFiles : TThreadList;

         procedure EnterFile( GrpHdl: LongInt; var OAF: TOpenArtFile );
         function  GetCount( GrpHdl: LongInt ): LongInt;
         function  GetName( GrpHdl: LongInt ): String;
         function  GetServerMin( GrpHdl: LongInt; const Server: String ): Integer;
         procedure SetServerMin( GrpHdl: LongInt; const Server: String; NewArtMin: Integer );
         function  GetServerMax( GrpHdl: LongInt; const Server: String ): Integer;
         procedure SetServerMax( GrpHdl: LongInt; const Server: String; NewArtMax: Integer );
         function  GetPullLimit( GrpHdl: LongInt; const Server: String ): Integer;
         function  GetFeederLast( GrpHdl: LongInt; const Server: String ): Integer;
         procedure SetFeederLast( GrpHdl: LongInt; const Server: String; NewFeederLast: Integer );

      public
         property  Count[ GrpHdl: LongInt ]: Integer read GetCount;
         property  Name [ GrpHdl: LongInt ]: String  read GetName;
         property  ServerMin[ GrpHdl: LongInt; const Server:String ]: Integer read GetServerMin write SetServerMin;
         property  ServerMax[ GrpHdl: LongInt; const Server:String ]: Integer read GetServerMax write SetServerMax;
         property  PullLimit[ GrpHdl: LongInt; const Server:String ]: Integer read GetPullLimit;
         property  FeederLast[ GrpHdl: LongInt; const Server:String ]: Integer read GetFeederLast write SetFeederLast;
         procedure SetFirstPullDone( GrpHdl: LongInt; const Server: String );

         function Settings( GrpHdl: LongInt ): TSettingsPlain;
         function ServerSettingsAll( GrpHdl: LongInt; const Server: String ): String;
         procedure ServerSetStr( GrpHdl: LongInt; const Server: String;
                                 const ID: Integer; const NewValue: String );


         function  GetStr( const GrpHdl: LongInt;
                           const ID: Integer ): String;
         procedure SetStr( const GrpHdl: LongInt;
                           const ID: Integer;
                           const NewValue: String );
         function  GetInt( const GrpHdl: LongInt;
                           const ID: Integer ): Integer;
         procedure SetInt( const GrpHdl: LongInt;
                           const ID: Integer;
                           const NewValue: Integer );
         function  GetInt64( const GrpHdl: LongInt;
                             const ID: Integer ): Int64;
         procedure SetInt64( const GrpHdl: LongInt;
                             const ID: Integer;
                             const NewValue: Int64 );
         function  GetBoo( const GrpHdl: LongInt;
                           const ID: Integer ): Boolean;
         function  GetDT ( const GrpHdl: LongInt;
                           const ID: Integer ): TDateTime;
         procedure SetDT ( const GrpHdl: LongInt;
                           const ID: Integer;
                           const NewValue: TDateTime );

         function  DTCreated( GrpHdl: LongInt ): TDateTime;
         function  GetKeyIndex( GrpHdl: LongInt; Key: LongInt ): Integer;

         function  ReadArticle ( GrpHdl: LongInt; ArtNo: Integer ): String;
         function  ReadArticleSized( GrpHdl: LongInt; ArtNo: Integer; var MaxSize: Integer ): String;
         function  ReadOverview( GrpHdl: LongInt; ArtNo: Integer; Art: TMess = nil ): String;
         function  ReserveArtNo( GrpHdl: LongInt ): LongInt;
         function  WriteArticle( GrpHdl: LongInt; ArtNo: Integer; const ArtText: String ): LongInt;
         procedure DeleteArticle( GrpHdl: LongInt; ArtNo: Integer );

         function  ReadArticleByMID( const ReadMID: String ): String;
         function  DeleteArticleByMID( const DelMID: String ): Boolean;

         function  UseCount( const GroupName: String ): LongInt;

         function  Open( const GroupName: String ): LongInt;
         procedure Close( GrpHdl: LongInt );
         procedure Purge( GrpHdl: LongInt );
         function  DeleteGroup( const GroupName: String ): Integer;

         procedure EstimateValues( const GroupName: String;
                                   out   ArtCount : LongInt;
                                   out   NewGroup : Boolean );
         function  DiskspaceUsed( const GroupName: String ): Int64;

         constructor Create;
         destructor Destroy; override;
   end;

function SaveArticle(   const Article           : TMess;
                        const CurrentGroupName  : String;
                        const CurrentGroupHandle: LongInt;
                        const ScoreMarker       : String;
                        const IgnoreHistory     : Boolean;
                        const OverrideGroups    : String;
                        const Action            : THscActionTypes ): Boolean;

function SaveInInternalGroup( const hsHamGroup: Integer;
                              const Subject, ArtText: String ): String;

function SaveUniqueNewsMsg( const DestPath: String;
                            const MsgText: String;
                            const GenerateMid: Boolean ): Boolean;

function SaveMailToNews( const Newsgroup: String;
                         const MsgText: String ): Boolean;

function ImportArticle( const ArtText       : String;
                        const OverrideGroups: String;
                        const IgnoreHistory : Boolean;
                        const MarkNoArchive : Boolean ): Boolean;

function ImportArticlesFromFile( const FileName: String;
                                 const TestOnly: Boolean;
                                 const OverrideGroups: String;
                                 const IgnoreHistory : Boolean;
                                 const MarkNoArchive : Boolean ): Integer;

function ExportArticlesToFile( const FileName: String;
                               const GroupName: String;
                               const ArtMin, ArtMax: Integer ): Integer;

implementation

uses uConstVar, uVar, uTools, cAccounts, cPCRE, uCRC32, uDateTime,
     IniFiles, uHamTools, cHamster, cMsgFile, cFileStream64;

{---------------------------------------------------------- TArticleBase -----}

procedure TArticleBase.EnterFile( GrpHdl: LongInt; var OAF: TOpenArtFile );
begin
   OAF := nil;

   with OpenArtFiles.LockList do try
      if ( GrpHdl >= 0) and ( GrpHdl < Count ) then begin
         OAF := TOpenArtFile( Items[GrpHdl] );
         if Assigned(OAF) and Assigned(OAF.ArtFile) then begin
            OAF.ArtFile.EnterThisFile;
         end else begin
            OAF := nil;
         end;
      end;
   finally OpenArtFiles.UnlockList end;

   if not Assigned( OAF ) then begin
      raise EGroupsError.CreateFmt( 'Invalid group handle %d!', [GrpHdl] );
   end;
end;

function TArticleBase.GetCount( GrpHdl: LongInt ): LongInt;
var  OAF: TOpenArtFile;
begin
   EnterFile( GrpHdl, OAF );
   try
      Result := OAF.ArtFile.Count;
   finally OAF.ArtFile.LeaveThisFile end;
end;

procedure TArticleBase.EstimateValues( const GroupName: String;
                                       out   ArtCount: LongInt;
                                       out   NewGroup: Boolean );
var  SR: TSearchRec;
     GrpHdl: Integer;
begin
   ArtCount := 0;
   NewGroup := True;

   if FindFirst( AppSettings.GetStr(asPathGroups) + GroupName + '\' + GRPFILE_IDX,
                 faAnyFile, SR ) = 0 then begin
      ArtCount := SR.Size div sizeof(TIndexRec);
      FindClose( SR );

      if ArtCount > 1 then begin
         NewGroup := False;
      end else begin
         GrpHdl := Open( GroupName );
         if GrpHdl >= 0 then try
            if GetInt( GrpHdl, gsLocalMax ) > 1 then NewGroup := False;
         finally
            Close( GrpHdl );
         end;
      end;
   end;
end;

function TArticleBase.DiskspaceUsed( const GroupName: String ): Int64;
var  d: String;
begin
   d := AppSettings.GetStr(asPathGroups) + GroupName + '\';
   Result := TFileStream64.FileSize( d + GRPFILE_DAT )
           + TFileStream64.FileSize( d + GRPFILE_IDX )
           + TFileStream64.FileSize( d + GRPFILE_INI );
end;

function TArticleBase.GetName( GrpHdl: LongInt ): String;
begin
     with OpenArtFiles.LockList do try
        if (GrpHdl>=0) and (GrpHdl<Count) then begin
           with TOpenArtFile( Items[GrpHdl] ) do Result := GrpName;
        end else begin
           Result := '';
        end;
     finally
        OpenArtFiles.UnlockList;
     end;
end;

function TArticleBase.ReadArticle( GrpHdl: LongInt; ArtNo: Integer ): String;
var  l: LongInt;
     OAF: TOpenArtFile;
begin
   EnterFile( GrpHdl, OAF );
   try
      l := 0;
      Result := OAF.ArtFile.ReadArticle( ArtNo, l );
   finally OAF.ArtFile.LeaveThisFile end;
end;

function TArticleBase.ReadArticleSized( GrpHdl: LongInt; ArtNo: Integer; var MaxSize: Integer ): String;
var  OAF: TOpenArtFile;
begin
   EnterFile( GrpHdl, OAF );
   try
      Result := OAF.ArtFile.ReadArticle( ArtNo, MaxSize );
   finally OAF.ArtFile.LeaveThisFile end;
end;

function TArticleBase.ReadOverview( GrpHdl: LongInt; ArtNo: Integer; Art: TMess = nil ): String;

   function FormatXOver( const s: String ) : String;
   // replace (remaining) NUL, TAB, LF and CR with SP
   var  i: Integer;
   begin
      Result := s;
      for i:=1 to length(Result) do begin
         if Result[i] in [ #0, #9, #10, #13 ] then Result[i] := ' ';
      end;
   end;


var  FreeArt: Boolean;
     ArtSize: Integer;
     sLines, ArtText, Xref: String;
begin
   SetLength( Result, 0 );

   FreeArt := not Assigned( Art );
   if FreeArt then Art := TMess.Create;

   try
      ArtSize := 4096;
      ArtText := ReadArticleSized( GrpHdl, ArtNo, ArtSize );
      if TMess.HasHeaderEnd( ArtText ) then begin
         Art.FullText := ArtText;
         sLines       := Art.HeaderValueByNameSL( HDR_NAME_LINES );
      end else begin
         Art.FullText := Hamster.ArticleBase.ReadArticle( GrpHdl, ArtNo );
         sLines       := inttostr( Art.BodyLines );
      end;

      if length(Art.HeaderText) > 0 then begin
         // 0:No. 1:Subject 2:From 3:Date 4:Message-ID 5:References 6:Bytes 7:Lines [8:Xref]
         Xref := Art.HeaderValueByNameSL( HDR_NAME_XREF );
         if length(Xref) > 0 then Xref := HDR_NAME_XREF + ': ' + Xref;
         Result := inttostr(ArtNo) + #9
                 + FormatXOver( Art.HeaderValueByNameSL( HDR_NAME_SUBJECT    ) ) + #9
                 + FormatXOver( Art.HeaderValueByNameSL( HDR_NAME_FROM       ) ) + #9
                 + FormatXOver( Art.HeaderValueByNameSL( HDR_NAME_DATE       ) ) + #9
                 + FormatXOver( Art.HeaderValueByNameSL( HDR_NAME_MESSAGE_ID ) ) + #9
                 + FormatXOver( Art.HeaderValueByNameSL( HDR_NAME_REFERENCES ) ) + #9
                 + FormatXOver( inttostr( ArtSize )                            ) + #9
                 + FormatXOver( sLines                                         ) + #9
                 + FormatXOver( Xref );
      end;

   finally
      if FreeArt then Art.Free;
   end;
end;

function TArticleBase.ReserveArtNo( GrpHdl: LongInt ): LongInt;
var  OAF: TOpenArtFile;
begin
   EnterFile( GrpHdl, OAF );
   try
      Result := OAF.ArtFile.ReserveArtNo;
   finally OAF.ArtFile.LeaveThisFile end;
end;

function TArticleBase.WriteArticle( GrpHdl: LongInt; ArtNo: Integer; const ArtText: String ): LongInt;
var  OAF: TOpenArtFile;
begin
   EnterFile( GrpHdl, OAF );
   try
      Result := OAF.ArtFile.WriteArticle( ArtNo, ArtText );
   finally OAF.ArtFile.LeaveThisFile end;
end;

procedure TArticleBase.DeleteArticle( GrpHdl: LongInt; ArtNo: Integer );
var  OAF: TOpenArtFile;
begin
   EnterFile( GrpHdl, OAF );
   try
      OAF.ArtFile.DeleteArticle( ArtNo );
   finally OAF.ArtFile.LeaveThisFile end;
end;

procedure TArticleBase.SetFirstPullDone( GrpHdl: LongInt; const Server: String );
var  OAF: TOpenArtFile;
begin
   EnterFile( GrpHdl, OAF );
   try
      OAF.ArtFile.SetFirstPullDone( Server );
   finally OAF.ArtFile.LeaveThisFile end;
end;

function TArticleBase.GetKeyIndex( GrpHdl: LongInt; Key: LongInt ): Integer;
var  OAF: TOpenArtFile;
begin
   EnterFile( GrpHdl, OAF );
   try
      Result := OAF.ArtFile.IndexOfKey( Key );
   finally OAF.ArtFile.LeaveThisFile end;
end;

function TArticleBase.DTCreated( GrpHdl: LongInt ): TDateTime;
var  OAF: TOpenArtFile;
begin
   EnterFile( GrpHdl, OAF );
   try
      Result := OAF.ArtFile.DTCreated;
   finally OAF.ArtFile.LeaveThisFile end;
end;

function TArticleBase.GetServerMin( GrpHdl: LongInt; const Server: String ): Integer;
var  OAF: TOpenArtFile;
begin
   EnterFile( GrpHdl, OAF );
   try
      Result := OAF.ArtFile.ServerMin[Server];
   finally OAF.ArtFile.LeaveThisFile end;
end;

procedure TArticleBase.SetServerMin( GrpHdl: LongInt; const Server: String; NewArtMin: Integer );
var  OAF: TOpenArtFile;
begin
   EnterFile( GrpHdl, OAF );
   try
      OAF.ArtFile.ServerMin[Server] := NewArtMin;
   finally OAF.ArtFile.LeaveThisFile end;
end;

function TArticleBase.GetServerMax( GrpHdl: LongInt; const Server: String ): Integer;
var  OAF: TOpenArtFile;
begin
   EnterFile( GrpHdl, OAF );
   try
      Result := OAF.ArtFile.ServerMax[Server];
   finally OAF.ArtFile.LeaveThisFile end;
end;

procedure TArticleBase.SetServerMax( GrpHdl: LongInt; const Server: String; NewArtMax: Integer );
var  OAF: TOpenArtFile;
begin
   EnterFile( GrpHdl, OAF );
   try
      OAF.ArtFile.ServerMax[Server] := NewArtMax;
   finally OAF.ArtFile.LeaveThisFile end;
end;

function TArticleBase.GetFeederLast( GrpHdl: Integer; const Server: String): Integer;
var  OAF: TOpenArtFile;
begin
   EnterFile( GrpHdl, OAF );
   try
      Result := OAF.ArtFile.FeederLast[Server];
   finally OAF.ArtFile.LeaveThisFile end;
end;

procedure TArticleBase.SetFeederLast( GrpHdl: Integer; const Server: String; NewFeederLast: Integer);
var  OAF: TOpenArtFile;
begin
   EnterFile( GrpHdl, OAF );
   try
      OAF.ArtFile.FeederLast[Server] := NewFeederLast;
   finally OAF.ArtFile.LeaveThisFile end;
end;


function TArticleBase.Settings( GrpHdl: LongInt ): TSettingsPlain;
var  OAF: TOpenArtFile;
begin
   EnterFile( GrpHdl, OAF );
   try
      Result := OAF.ArtFile.Settings; 
   finally OAF.ArtFile.LeaveThisFile end;
end;

function TArticleBase.ServerSettingsAll( GrpHdl: LongInt; const Server: String ): String;
var  OAF: TOpenArtFile;
begin
   EnterFile( GrpHdl, OAF );
   try
      Result := OAF.ArtFile.ServerSettingsAll( Server );
   finally OAF.ArtFile.LeaveThisFile end;
end;

procedure TArticleBase.ServerSetStr( GrpHdl: LongInt; const Server: String;
                                     const ID: Integer; const NewValue: String );
var  OAF: TOpenArtFile;
begin
   EnterFile( GrpHdl, OAF );
   try
      OAF.ArtFile.ServerSetStr( Server, ID, NewValue );
   finally OAF.ArtFile.LeaveThisFile end;
end;

function TArticleBase.GetStr( const GrpHdl: LongInt;
                              const ID: Integer ): String;
var  OAF: TOpenArtFile;
begin
   EnterFile( GrpHdl, OAF );
   try
      Result := OAF.ArtFile.Settings.GetStr( ID );
   finally OAF.ArtFile.LeaveThisFile end;
end;

procedure TArticleBase.SetStr( const GrpHdl: LongInt;
                               const ID: Integer;
                               const NewValue: String );
var  OAF: TOpenArtFile;
begin
   EnterFile( GrpHdl, OAF );
   try
      OAF.ArtFile.Settings.SetStr( ID, NewValue );
   finally OAF.ArtFile.LeaveThisFile end;
end;

function TArticleBase.GetInt( const GrpHdl: LongInt;
                              const ID: Integer ): Integer;
var  OAF: TOpenArtFile;
begin
   EnterFile( GrpHdl, OAF );
   try
      Result := OAF.ArtFile.Settings.GetInt( ID );
   finally OAF.ArtFile.LeaveThisFile end;
end;

procedure TArticleBase.SetInt( const GrpHdl: LongInt;
                               const ID: Integer;
                               const NewValue: Integer );
var  OAF: TOpenArtFile;
begin
   EnterFile( GrpHdl, OAF );
   try
      OAF.ArtFile.Settings.SetInt( ID, NewValue );
   finally OAF.ArtFile.LeaveThisFile end;
end;

function TArticleBase.GetInt64( const GrpHdl: LongInt;
                                const ID: Integer ): Int64;
var  OAF: TOpenArtFile;
begin
   EnterFile( GrpHdl, OAF );
   try
      Result := OAF.ArtFile.Settings.GetInt64( ID );
   finally OAF.ArtFile.LeaveThisFile end;
end;

procedure TArticleBase.SetInt64( const GrpHdl: LongInt;
                                 const ID: Integer;
                                 const NewValue: Int64 );
var  OAF: TOpenArtFile;
begin
   EnterFile( GrpHdl, OAF );
   try
      OAF.ArtFile.Settings.SetInt64( ID, NewValue );
   finally OAF.ArtFile.LeaveThisFile end;
end;

function TArticleBase.GetBoo( const GrpHdl: LongInt;
                              const ID: Integer ): Boolean;
var  OAF: TOpenArtFile;
begin
   EnterFile( GrpHdl, OAF );
   try
      Result := OAF.ArtFile.Settings.GetBoo( ID );
   finally OAF.ArtFile.LeaveThisFile end;
end;

function TArticleBase.GetDT( const GrpHdl: LongInt;
                             const ID: Integer ): TDateTime;
var  OAF: TOpenArtFile;
begin
   EnterFile( GrpHdl, OAF );
   try
      Result := OAF.ArtFile.Settings.GetDT( ID );
   finally OAF.ArtFile.LeaveThisFile end;
end;

procedure TArticleBase.SetDT( const GrpHdl: LongInt;
                              const ID: Integer;
                              const NewValue: TDateTime );
var  OAF: TOpenArtFile;
begin
   EnterFile( GrpHdl, OAF );
   try
      OAF.ArtFile.Settings.SetDT( ID, NewValue );
   finally OAF.ArtFile.LeaveThisFile end;
end;

function TArticleBase.GetPullLimit( GrpHdl: LongInt; const Server: String ): Integer;
var  OAF: TOpenArtFile;
begin
   EnterFile( GrpHdl, OAF );
   try
      Result := OAF.ArtFile.PullLimit[Server];
   finally OAF.ArtFile.LeaveThisFile end;
end;

procedure TArticleBase.Purge( GrpHdl: LongInt );
var  OAF: TOpenArtFile;
begin
   EnterFile( GrpHdl, OAF );
   try
      OAF.ArtFile.Purge;
   finally OAF.ArtFile.LeaveThisFile end;
end;

function TArticleBase.UseCount( const GroupName: String ): LongInt;
var  i: Integer;
begin
   Result := 0;

   with OpenArtFiles.LockList do try
      // search through list of open groups
      for i := 0 to Count - 1 do begin
         if AnsiCompareText( TOpenArtFile( Items[i] ).GrpName, GroupName ) = 0 then begin
            Result := TOpenArtFile( Items[i] ).UseCount;
            break;
         end;
      end;
   finally OpenArtFiles.UnlockList end;
end;

function TArticleBase.Open( const GroupName: String ): LongInt;
var  GrpHdl  : LongInt;
     ArtOpen : TOpenArtFile;
     i       : Integer;
begin
   with OpenArtFiles.LockList do try

      GrpHdl  := -1;
      ArtOpen := nil;

      // is group already in list of open groups?
      for i := 0 to Count - 1 do begin
         if AnsiCompareText( TOpenArtFile( Items[i] ).GrpName, GroupName ) = 0 then begin
            GrpHdl  := i; // yes
            ArtOpen := Items[i];
            break;
         end;
      end;

      // add to list of open article-files
      if GrpHdl = -1 then begin
         ArtOpen := TOpenArtFile.Create;
         ArtOpen.ArtFile  := nil;
         ArtOpen.UseCount := 0;
         ArtOpen.GrpName  := GroupName;
         GrpHdl := Add( ArtOpen );
      end;

      // open group or increment usage-counter
      if ArtOpen.ArtFile = nil then begin

         ArtOpen.ArtFile := TArticleFile.Create( GroupName );
         
         if ArtOpen.ArtFile.Open( ARTFILE_DEFAULTMODE ) then begin
            inc( ArtOpen.UseCount );
            Result := GrpHdl;
         end else begin
            ArtOpen.ArtFile.Free;
            ArtOpen.ArtFile := nil;
            Result := -1;
         end;

         Log( LOGID_DEBUG, 'ArtBase.Open(' + GroupName + ') -> ' + inttostr(Result) );

      end else begin

         inc( ArtOpen.UseCount );
         Result := GrpHdl;
         Log( LOGID_DEBUG, 'ArtBase.Open(' + GroupName + ') -> ' + inttostr(Result)
                                                       + ', #' + inttostr(ArtOpen.UseCount) );

      end;

   finally OpenArtFiles.UnlockList end;
end;

procedure TArticleBase.Close( GrpHdl: LongInt );
begin
   with OpenArtFiles.LockList do try
      if (GrpHdl>=0) and (GrpHdl<Count) then begin
         with TOpenArtFile( Items[GrpHdl] ) do begin

            // decrement usage-counter
            if UseCount>0 then dec( UseCount );

            if Assigned( ArtFile ) then begin
               if UseCount=0 then begin
                  // close article file if not in use any more
                  Log( LOGID_DEBUG, 'ArtBase.Close(' + inttostr(GrpHdl) + ')' );
                  try
                     ArtFile.Free;
                  except
                     on E: Exception do
                     Log( LOGID_WARN, 'ArtBase.Close(' + inttostr(GrpHdl)
                                    + ') Error: ' + E.Message );
                  end;
                  Artfile := nil;
               end else begin
                  // just flush changes
                  Log( LOGID_DEBUG, 'ArtBase.Close(' + inttostr(GrpHdl) + ')'
                                    + ', #' + inttostr(UseCount+1) );
                  ArtFile.Flush;
               end;
            end;

         end;
      end;

   finally OpenArtFiles.UnlockList end;
end;

function TArticleBase.ReadArticleByMID( const ReadMID: String ): String;
var  GrpNam       : String;
     GrpHdl, ArtNo: Integer;
begin
     Result := '';
     if not Hamster.NewsHistory.LocateMID( ReadMID, GrpNam, ArtNo ) then exit;

     GrpHdl := Open( GrpNam );
     if GrpHdl < 0 then exit;
     Result := ReadArticle( GrpHdl, ArtNo );
     Close( GrpHdl );
end;

function TArticleBase.DeleteArticleByMID( const DelMID: String ): Boolean;
var  GrpNam, Xref, s: String;
     GrpHdl, ArtNo, j: Integer;
     Art: TMess;
begin
     Result := Hamster.NewsHistory.LocateMID( DelMID, GrpNam, ArtNo );
     if not Result then exit;

     // delete history-entry
     Hamster.NewsHistory.RemoveEntry( DelMID, StrToCRC32(GrpNam), ArtNo );

     // get Xref of article and delete "main"-instance
     GrpHdl := Open( GrpNam );
     if GrpHdl < 0 then exit;
     try
        Art := TMess.Create;
        try
           Art.FullText := ReadArticle( GrpHdl, ArtNo );
           Xref := Art.HeaderValueByNameSL( HDR_NAME_XREF );
           if length(Art.HeaderText) > 0 then DeleteArticle( GrpHdl, ArtNo );
        finally
           Art.Free;
        end;
     finally
        Close( GrpHdl );
     end;

     // delete article-instances referenced in Xref
     j := PosWhSpace( Xref ); if j=0 then exit;
     System.Delete( Xref, 1, j ); // skip hostname
     while Xref<>'' do begin
        j := PosWhSpace( Xref );
        if j=0 then begin
           s := Xref;
           Xref := '';
        end else begin
           s := copy( Xref, 1, j-1 );
           System.Delete( Xref, 1, j );
        end;

        j := Pos( ':', s );
        if j>0 then begin
           ArtNo := strtoint( copy( s, j+1, 255 ) );
           s := copy( s, 1, j-1 );
           if s<>GrpNam then begin // delete additional references
              GrpHdl := Open( s );
              if GrpHdl>=0 then begin
                 DeleteArticle( GrpHdl, ArtNo );
                 Close( GrpHdl );
              end;
           end;
        end;
     end;
end;

function TArticleBase.DeleteGroup( const GroupName: String ): Integer;
var  TempHdl: Integer;
begin
     if UseCount( GroupName ) > 0 then begin
        Result := -1;
        Log( LOGID_WARN, Format(
              'Group "%s" is in use and can''t be deleted!', [GroupName] ) );
        exit;
     end;

     // save number of last article (used as default, if group is resubscribed)
     TempHdl := Open( GroupName );
     if TempHdl >= 0 then try
        with TIniFile.Create( AppSettings.GetStr(asPathGroups) + GRPFILE_SAVEGROUP ) do try
           WriteInteger( 'Local.Max', GroupName, GetInt(TempHdl,gsLocalMax) );
        finally Free end;
     finally Close(TempHdl) end;

     try
        // delete "known" files
        DeleteFile( AppSettings.GetStr(asPathGroups) + GroupName + '\' + GRPFILE_DAT );
        DeleteFile( AppSettings.GetStr(asPathGroups) + GroupName + '\' + GRPFILE_IDX );
        DeleteFile( AppSettings.GetStr(asPathGroups) + GroupName + '\' + GRPFILE_INI );

        // remove directory
        if RemoveDir( AppSettings.GetStr(asPathGroups) + GroupName ) then begin
           Result := 0;
        end else begin
           Result := -2;
           Log( LOGID_WARN, Format(
                 'Couldn''t remove directory "%s"!', [AppSettings.GetStr(asPathGroups) + GroupName] ) );
        end;
     except
        on E: Exception do begin
           Result := -9;
           Log( LOGID_WARN, Format(
                 'Exception deleting group %s: %s', [GroupName, E.Message] ) );
        end;
     end;
end;

constructor TArticleBase.Create;
begin
     inherited Create;
     OpenArtFiles := TThreadList.Create;
end;

destructor TArticleBase.Destroy;
var  i: Integer;
begin
     // alle noch geffneten Dateien schlieen
     with OpenArtFiles.LockList do try
        for i:=Count-1 downto 0 do begin
           with TOpenArtFile( Items[i] ) do begin
              while UseCount>0 do Close(i);
              TOpenArtFile( Items[i] ).Free;
           end;
        end;
     finally OpenArtFiles.UnlockList end;
     OpenArtFiles.Free;

     inherited Destroy;
end;

{-----------------------------------------------------------------------------}

function SaveArticle( const Article           : TMess;
                      const CurrentGroupName  : String;
                      const CurrentGroupHandle: LongInt;
                      const ScoreMarker       : String;
                      const IgnoreHistory     : Boolean;
                      const OverrideGroups    : String;
                      const Action            : THscActionTypes
                    ): Boolean;
var  MessageID, Newsgroups, LfdGroup, DestXref, s, XrefHostname: String;
     GrpHdl, LfdGrp, i: Integer;
     HistoryCheck     : Boolean;
     DestGroups: TStringList;
     Xrefs     : TList;
     Xref      : TXrefArtNo;
begin
   Result := False;

   // get/create Message-ID
   MessageID := Article.HeaderValueByNameSL( HDR_NAME_MESSAGE_ID );
   if MessageID='' then begin
      MessageID := MidGenerator( Hamster.Config.Settings.GetStr(hsFQDNforMID) );
      Article.InsertHeaderSL( 0, HDR_NAME_MESSAGE_ID, MessageID );
   end;

   // path
   i := Article.IndexOfHeader( HDR_NAME_PATH );
   if i<0 then Article.InsertHeaderSL( 0, HDR_NAME_PATH, 'not-for-mail' );

   // Trigger script action ("NewsIn", "MailToNews")
   if Action <> actIgnore then begin
      if Hamster.HscActions.Execute(
            Action, CurrentGroupName + CRLF + inttostr( Integer(Article) )
         ) then begin
         if not Article.IsValid then begin 
            Log( LOGID_DETAIL, 'Note: Action invalidated message ' + MessageID );
            Result := True; // ok, this is allowed
            exit;
         end;
      end;
   end;

   // precede article with Hamster's info-header
   i := Article.IndexOfHeader( HDR_NAME_X_HAMSTER_INFO );
   if i >= 0 then Article.DeleteHeaderML( i );
   s := 'Score='    + ScoreMarker + ' '
      + 'Received=' + DateTimeToTimeStamp(Now) + ' ' // =timebase for purge
      + 'UID=' + IntToStr( GetUID(0) );
   Article.InsertHeaderSL( 0, HDR_NAME_X_HAMSTER_INFO, s );

   // remove existing Xref-header
   i := Article.IndexOfHeader( HDR_NAME_X_OLD_XREF );
   if i >= 0 then Article.DeleteHeaderML( i );
   i := Article.IndexOfHeader( HDR_NAME_XREF );
   if i >= 0 then begin
      s := Article.HeaderValueByIndexSL( i );
      Article.DeleteHeaderML( i );
      if s > '' then Article.AddHeaderSL( HDR_NAME_X_OLD_XREF, s );
   end;

   Xrefs := nil;
   DestGroups := nil;
   try
      // create Xref-header by opening groups and reserving article-numbers
      DestXref := '';
      Xrefs := TList.Create;

      if OverrideGroups='' then Newsgroups := Article.HeaderValueByNameSL( HDR_NAME_NEWSGROUPS )
                           else Newsgroups := OverrideGroups;

      // prepare list of groupnames
      DestGroups := TStringList.Create;
      DestGroups.Sorted := True;
      DestGroups.Duplicates := dupIgnore;

      if (CurrentGroupName <> '') and
         (Hamster.Config.Newsgroups.IndexOf(CurrentGroupName) >= 0) then begin
         // add current group, even if not in Newsgroups (e.g. control.*)
         DestGroups.Add( CurrentGroupName );
      end;
      repeat
         LfdGroup := TrimWhSpace( NextSepPart( Newsgroups, ',' ) );
         if length( LfdGroup ) = 0 then break;
         if Hamster.Config.Newsgroups.IndexOf(LfdGroup) >= 0 then begin
            if LfdGroup<>'' then DestGroups.Add( LfdGroup );
         end;
      until length( LfdGroup ) = 0;
      if DestGroups.Count = 0 then begin // known groups?
         DestGroups.Add( Hamster.Config.Newsgroups.HamGroupName(hsHamGroupUnknownGroup) );
         Article.SetHeaderSL( HDR_NAME_DISTRIBUTION, 'local', '' );
      end;

      for i:=0 to DestGroups.Count-1 do begin
         LfdGroup := DestGroups[i];
         if AnsiCompareText( LfdGroup, CurrentGroupName ) = 0 then begin
            GrpHdl := CurrentGroupHandle;
         end else begin
            GrpHdl := Hamster.ArticleBase.Open(LfdGroup);
         end;
         if GrpHdl>=0 then begin
            Xref := TXrefArtNo.Create;
            Xref.GrpNam := LfdGroup;
            Xref.GrpHdl := GrpHdl;
            Xref.ArtNo  := Hamster.ArticleBase.ReserveArtNo( Xref.GrpHdl );
            Xrefs.Add( Xref );
            DestXref := DestXref + ' ' + LfdGroup + ':' + inttostr( Xref.ArtNo );
         end;
      end;
      // Note: already existing Xref would have been removed above
      XrefHostname := Hamster.Config.Settings.GetStr(hsFQDNforMID);
      if XrefHostname = '' then XrefHostname := 'localhost';
      Article.InsertHeaderSL( 1, HDR_NAME_XREF, XrefHostname + DestXref );

      // save article in local groups
      HistoryCheck := not IgnoreHistory;
      for LfdGrp:=0 to Xrefs.Count-1 do begin
         Xref := TXrefArtNo( Xrefs[LfdGrp] );

         try
            if HistoryCheck then begin
               if not Hamster.NewsHistory.AddEntryFirst(
                         MessageID, StrToCRC32(Xref.GrpNam), Xref.ArtNo, 0
                      ) then begin
                  break;
               end;
            end;

            if Hamster.ArticleBase.WriteArticle( Xref.GrpHdl,
                                         Xref.ArtNo,
                                         Article.FullText )<>0 then begin
               Result := True;

               if HistoryCheck then begin
                  HistoryCheck := False;
               end else begin
                  Hamster.NewsHistory.AddEntryDupes( MessageID,
                                             StrToCRC32(Xref.GrpNam),
                                             Xref.ArtNo, 0 );
               end;
            end else begin
               break;
            end;
         except
            on E: Exception do begin
               Log( LOGID_ERROR, 'SaveArticle-Error: ' + E.Message );
               raise;
            end;
         end;
      end;

   finally
      if Assigned( DestGroups ) then DestGroups.Free;
      if Assigned( Xrefs ) then try
         // close groups and free Xref-entries and -list
         for LfdGrp := Xrefs.Count-1 downto 0 do try
            Xref := TXrefArtNo( Xrefs[LfdGrp] );
            try
               if AnsiCompareText(Xref.GrpNam,CurrentGroupName) <> 0 then begin
                  Hamster.ArticleBase.Close( Xref.GrpHdl );
               end;
               Xrefs.Delete( LfdGrp );
            finally Xref.Free end;
         except end;
      finally Xrefs.Free end;
   end;
end;

function SaveInInternalGroup( const hsHamGroup: Integer;
                              const Subject, ArtText: String ): String;
var  Header, MessageID: String;
     Art              : TMess;
     Lines            : Integer;
begin
   MessageID := MidGenerator( Hamster.Config.Settings.GetStr(hsFQDNforMID) );
   Result := MessageID;

   Art := TMess.Create;

   try
      Art.FullText := 'X-Dummy: X' + #13#10#13#10 + ArtText;
      Lines        := Art.BodyLines;

      Header := HDR_NAME_SUBJECT      + ': ' + Subject                      + #13#10
              + HDR_NAME_DATE         + ': ' + DateTimeGMTToRfcDateTime(
                                               NowGMT, NowRfcTimezone )     + #13#10
              + HDR_NAME_FROM         + ': ' + 'Hamster'                    + #13#10
              + HDR_NAME_NEWSGROUPS   + ': ' + Hamster.Config.Newsgroups.HamGroupName(hsHamGroup) + #13#10
              + HDR_NAME_MESSAGE_ID   + ': ' + MessageID                    + #13#10
              + HDR_NAME_PATH         + ': ' + 'not-for-mail'               + #13#10
              + HDR_NAME_DISTRIBUTION + ': ' + 'local'               + #13#10
              + HDR_NAME_LINES        + ': ' + inttostr(Lines)              + #13#10;

      Art.FullText := Header + #13#10 + ArtText;
      SaveArticle( Art, '', -1, '(none-local)', False, '', actIgnore );

   finally
      Art.Free;
   end;
end;

{-----------------------------------------------------------------------------}

function SaveUniqueNewsMsg( const DestPath: String;
                            const MsgText: String;
                            const GenerateMid: Boolean ): Boolean;
var  t       : TextFile;
     Msg     : TMess;
     sMID, FN: String;
     iMID    : Integer;
     ok      : Boolean;
begin
   Result := False;
   ok := False;

   Msg := TMess.Create;
   try
      Msg.FullText := MsgText;

      if GenerateMid and (Pos('.',Hamster.Config.Settings.GetStr(hsFQDNforMID))>1) then begin
         iMID := Msg.IndexOfHeader( 'Message-ID:' );
         if iMID < 0 then begin
            sMID := MidGenerator( Hamster.Config.Settings.GetStr(hsFQDNforMID) );
            Msg.InsertHeaderSL( 0, HDR_NAME_MESSAGE_ID, sMID );
         end;
      end;

      FN := GetUniqueMsgFilename( DestPath, 'news' );

      if Hamster.HscActions.Execute(
            actNewsOut, FN + CRLF + inttostr( Integer(Msg) )
         ) then begin
         if not Msg.IsValid then begin
            Log( LOGID_INFO, 'Note: Action "NewsOut" invalidated message.' );
            Result := True; // ok, this is allowed
            exit;
         end;
      end;

      CS_LOCK_NEWSOUT.Enter;
      try

         try
            AssignFile( t, FN );
            Rewrite( t );
            Write( t, Msg.FullText );
            CloseFile( t );
            Result := True;
            ok     := True;
         except
            on E: Exception do begin
               Log( LOGID_WARN, 'Error saving file ' + FN + ': ' + E.Message );
               raise;
            end;
         end;

      finally CS_LOCK_NEWSOUT.Leave end;

      if ok then begin
         Hamster.HscActions.Execute(
            actNewsWaiting, FN + CRLF + inttostr( Integer(Msg) ) );
         Hamster.HscActions.Execute(
            actMsgWaiting,  FN + CRLF + inttostr( Integer(Msg) ) );
      end;

   finally Msg.Free end;
end;

{-----------------------------------------------------------------------------}

function SaveMailToNews( const Newsgroup: String; const MsgText: String ): Boolean;
var  Art: TMess;
     s  : String;
begin
   Result := False;
   Art := TMess.Create;

   try
      Art.FullText := MsgText;

      Art.SetHeaderSL( HDR_NAME_NEWSGROUPS, Newsgroup, HDR_NAME_X_NEWSGROUPS );

      s := Art.HeaderValueByNameSL( HDR_NAME_REFERENCES );
      if s='' then begin
         s := Art.HeaderValueByNameSL( HDR_NAME_IN_REPLY_TO );
         if s<>'' then begin
            s := RE_Extract( s, '\<.+?\@.+?\>', 0 );
            if s<>'' then Art.InsertHeaderSL( 0, HDR_NAME_REFERENCES, s );
         end;
      end;

      if Hamster.Config.Settings.GetBoo(hsGateAddFUpPoster) then begin
         s := Art.HeaderValueByNameSL( HDR_NAME_FOLLOWUP_TO );
         if s='' then Art.InsertHeaderSL( 0, HDR_NAME_FOLLOWUP_TO, 'poster' );
      end;

      s := Art.HeaderValueByNameSL( HDR_NAME_PATH );
      if s='' then begin
         Art.InsertHeaderSL( 0, HDR_NAME_PATH, 'not-for-mail' );
      end;

      Art.SetHeaderSL( HDR_NAME_DISTRIBUTION, 'local', '' );

      s := Art.HeaderValueByNameSL( HDR_NAME_DATE );
      if s='' then begin
         Art.InsertHeaderSL( 0, HDR_NAME_DATE,
                             DateTimeGMTToRfcDateTime( NowGMT, NowRfcTimezone ) );
      end;

      if SaveArticle( Art, '', -1,
                      '(none-mail2news)', True, '', actMailToNews ) then Result:=True;

   finally
      Art.Free;
   end;
end;

// ----------------------------------------------------------------------------

function ImportArticle( const ArtText       : String;
                        const OverrideGroups: String;
                        const IgnoreHistory : Boolean;
                        const MarkNoArchive : Boolean ): Boolean;
var  Art: TMess;
     s  : String;
begin
   Result := False;

   if Pos( #13#10#13#10, ArtText ) < 1 then begin // not a valid article
      Log( LOGID_WARN, 'Import rejected: Missing header/body separator' );
      exit;
   end;

   if copy(ArtText,1,5)='From ' then begin // mbox-format not supported here
      Log( LOGID_WARN, 'Import rejected: No mbox-format allowed here' );
      exit;
   end;

   Art := TMess.Create;

   try
      Art.FullText := ArtText;

      Art.SetHeaderSL( HDR_NAME_DISTRIBUTION, 'local', '' );

      s := '(none-import)';
      if MarkNoArchive then s := s + ' ' + 'NoArchive=1';

      if SaveArticle(
            Art, '', -1, s, IgnoreHistory, OverrideGroups, actIgnore
         ) then Result:=True;

   finally
      Art.Free;
   end;
end;

function ImportArticlesFromFile( const FileName: String;
                                 const TestOnly: Boolean;
                                 const OverrideGroups: String;
                                 const IgnoreHistory : Boolean;
                                 const MarkNoArchive : Boolean ): Integer;
var  ArtText, ArtHdr: String;
     MsgFile: TMessagefile;
     Regex: TPCRE;

     procedure TryImport;
     var  j: Integer;
     begin
        j := Pos( #13#10#13#10, ArtText ); // header-separator
        if j = 0 then begin
           Log( LOGID_WARN, 'Import rejected: Missing header/body separator' );
           exit;
        end;

        ArtHdr := copy( ArtText, 1, j+1 );
        if not Regex.Match( '^Newsgroups:.+', PChar(ArtHdr) ) then begin
           Log( LOGID_WARN, 'Import rejected: Missing "Newsgroups:" header' );
           exit;
        end;
        if not Regex.Match( '^Subject:.+',    PChar(ArtHdr) ) then begin
           Log( LOGID_WARN, 'Import rejected: Missing "Subject:" header' );
           exit;
        end;
        if not Regex.Match( '^From:.+',       PChar(ArtHdr) ) then begin
           Log( LOGID_WARN, 'Import rejected: Missing "From:" header' );
           exit;
        end;
        if not Regex.Match( '^Date:.+',       PChar(ArtHdr) ) then begin
           Log( LOGID_WARN, 'Import rejected: Missing "Date:" header' );
           exit;
        end;

        if TestOnly then begin
           inc( Result );
        end else begin
           if ImportArticle( ArtText, OverrideGroups, IgnoreHistory,
                             MarkNoArchive ) then inc( Result );
        end;
     end;

begin
   try
      Result := 0;

      if not FileExists( Filename ) then begin
         Result := -1; // file not found
         exit;
      end;

      Regex := TPCRE.Create( False, PCRE_CASELESS or PCRE_MULTILINE );
      try

         MsgFile := TMessagefile.Create( FileName, MSGFILE_READ );
         try

            if MsgFile.MessageFormat = MSGFMT_UNKNOWN then begin

               Result := -2; // unknown file format

            end else begin

               while not MsgFile.EndOfFile do begin
                  ArtText := MsgFile.ReadMessage;
                  if length( ArtText ) > 0 then TryImport;
               end;

            end;

         finally MsgFile.Free end;

      finally Regex.Free end;

   except
      on E: Exception do begin
         Result := -9; // error
         Log( LOGID_WARN, 'ImportArticlesFromFile-Error: ' + E.Message );
      end;
   end;
end;

function ExportArticlesToFile( const FileName: String;
                               const GroupName: String;
                               const ArtMin, ArtMax: Integer ): Integer;
var  GrpHdl, ANo, AMin, AMax: Integer;
     ATxt: String;
     MsgFile: TMessagefile;
begin
   try
      Result := 0;

      if Hamster.Config.Newsgroups.IndexOf( GroupName ) < 0 then begin
         Result := -1; // unknown group
         exit;
      end;

      GrpHdl := Hamster.ArticleBase.Open( GroupName );
      if GrpHdl >= 0 then try

         MsgFile := TMessagefile.Create( Filename, MSGFILE_APPEND );
         try

            MsgFile.MessageFormat := MSGFMT_MBOX;
            MsgFile.UseUnixEOL    := False;

            AMin := Hamster.ArticleBase.GetInt( GrpHdl, gsLocalMin );
            AMax := Hamster.ArticleBase.GetInt( GrpHdl, gsLocalMax );

            if (ArtMin > AMin) then AMin := ArtMin;
            if (ArtMax < AMax) and (ArtMax >= 0) then AMax := ArtMax;

            for ANo := AMin to AMax do begin
               ATxt := Hamster.ArticleBase.ReadArticle( GrpHdl, ANo );
               if length( ATxt ) > 0 then begin
                  inc( Result );
                  MsgFile.WriteMessage( ATxt );
               end;
            end;

         finally MsgFile.Free end;

      finally Hamster.ArticleBase.Close( GrpHdl ) end;

   except
      on E: Exception do begin
         Result := -9; // error
         Log( LOGID_WARN, 'ExportArticlesToFile-Error: ' + E.Message );
      end;
   end;
end;

// ----------------------------------------------------------------------------

end.
