// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit fConfFile;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  dDialogs, StdCtrls, ComCtrls, Menus, Clipbrd, ActnList, ImgList, ToolWin;

type
  TEditFileTypes = ( eftLocalText,
                     eftScores, eftMailFilt, eftIPAccess, eftKillsLog,
                     eftRCProfiles, eftHscActions,
                     eftScript, eftModule );

  TContinuePreference = ( cpPreferUpload, cpPreferIgnore );

type
  TfrmConfFile = class(TForm)
    MainMenu1: TMainMenu;
    mnuFile: TMenuItem;
    mnuEdit: TMenuItem;
    rtfEdit: TRichEdit;
    OpenDialog1: TOpenDialog;
    SaveDialog1: TSaveDialog;
    mnuFileOpen: TMenuItem;
    mnuFileSave: TMenuItem;
    mnuFileExit: TMenuItem;
    StatusBar1: TStatusBar;
    mnuEditUndo: TMenuItem;
    N2: TMenuItem;
    mnuEditCut: TMenuItem;
    mnuEditCopy: TMenuItem;
    mnuEditInsert: TMenuItem;
    mnuEditDelete: TMenuItem;
    N3: TMenuItem;
    mnuEditSelectAll: TMenuItem;
    mnuFindStart: TMenuItem;
    mnuFindNext: TMenuItem;
    N5: TMenuItem;
    N6: TMenuItem;
    mnuFilePrint: TMenuItem;
    popupEdit: TPopupMenu;
    ActionList1: TActionList;
    acFileOpen: TAction;
    acFileSave: TAction;
    acFilePrint: TAction;
    acFileExit: TAction;
    acEditUndo: TAction;
    acEditCut: TAction;
    acEditCopy: TAction;
    acEditPaste: TAction;
    acEditDelete: TAction;
    acEditFind: TAction;
    acEditFindNext: TAction;
    acEditSelectAll: TAction;
    Rckgngig1: TMenuItem;
    N8: TMenuItem;
    Ausschneiden1: TMenuItem;
    Kopieren1: TMenuItem;
    Einfgen1: TMenuItem;
    Lschen1: TMenuItem;
    N9: TMenuItem;
    Allesmarkieren1: TMenuItem;
    N10: TMenuItem;
    Suchen1: TMenuItem;
    Weitersuchen1: TMenuItem;
    ImageList1: TImageList;
    ToolBar1: TToolBar;
    ToolButton1: TToolButton;
    ToolButton2: TToolButton;
    ToolButton7: TToolButton;
    ToolButton8: TToolButton;
    ToolButton9: TToolButton;
    ToolButton10: TToolButton;
    ToolButton11: TToolButton;
    ToolButton12: TToolButton;
    ToolButton13: TToolButton;
    ToolButton14: TToolButton;
    ToolButton15: TToolButton;
    ToolButton16: TToolButton;
    acRefresh: TAction;
    acUpload: TAction;
    ToolButton17: TToolButton;
    ToolButton18: TToolButton;
    ToolButton19: TToolButton;
    N11: TMenuItem;
    RefreshDownload1: TMenuItem;
    SaveUpload1: TMenuItem;
    acDownload: TAction;
    acRunScript: TAction;
    ToolBtnRunScript: TToolButton;
    SaveandRunScript1: TMenuItem;
    mnuView: TMenuItem;
    acFont: TAction;
    FontDialog1: TFontDialog;
    Font1: TMenuItem;
    procedure rtfEditKeyUp(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure FormCreate(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure mnuEditClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure rtfEditMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure rtfEditKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure popupEditPopup(Sender: TObject);
    procedure acFileOpenExecute(Sender: TObject);
    procedure acFileSaveExecute(Sender: TObject);
    procedure acFilePrintExecute(Sender: TObject);
    procedure acFileExitExecute(Sender: TObject);
    procedure acEditUndoExecute(Sender: TObject);
    procedure acEditCutExecute(Sender: TObject);
    procedure acEditCopyExecute(Sender: TObject);
    procedure acEditPasteExecute(Sender: TObject);
    procedure acEditDeleteExecute(Sender: TObject);
    procedure acEditFindExecute(Sender: TObject);
    procedure acEditFindNextExecute(Sender: TObject);
    procedure acEditSelectAllExecute(Sender: TObject);
    procedure acRefreshExecute(Sender: TObject);
    procedure acUploadExecute(Sender: TObject);
    procedure acDownloadExecute(Sender: TObject);
    procedure acRunScriptExecute(Sender: TObject);
    procedure acFontExecute(Sender: TObject);
  private
    { Private-Deklarationen }
    EditFileType : TEditFileTypes;
    EditFileName : String;
    EditFileTitle: String;
    EditFileText : String;
    CurrFilename : String;
    CurrFindText : String;
    procedure ShowFilename;
    procedure ShowStatus;
    function  CanContinue( Prefer: TContinuePreference ): Boolean;
    procedure FindNextPosition( IsFindNew: Boolean );
    procedure EnableActions;
    function LMPrepare( out GetId: Integer; out GetStr: String;
                        out SetId: Integer; out SetStr: String ): Boolean;
    procedure PrepareFileDialog( Dialog: TOpenDialog );
  public
    { Public-Deklarationen }
    procedure StartEdit( const AEditFileType : TEditFileTypes;
                         const AEditFileName : String;
                         const AEditFileTitle: String;
                         const AEditFileText : String );
  end;

procedure EditRemoteFile( const EditFileType : TEditFileTypes;
                          const EditFileName : String;
                          const EditFileTitle: String );

procedure EditLocalText( const EditFileTitle: String;
                         const EditFileText : String );

implementation

{$R *.DFM}

uses uGlobal, uConst, uConstVar, uTools, cLiveMsg, cLiveQueue,
     uDateTime, dInput;

function rtfPosToLine( RichEdit: TRichEdit; BytePos: Integer ): Integer;
begin
     Result := SendMessage( RichEdit.Handle, EM_LINEFROMCHAR, BytePos, 0 );
end;

procedure EditRemoteFile( const EditFileType : TEditFileTypes;
                          const EditFileName : String;
                          const EditFileTitle: String );
var  frm: TfrmConfFile;
begin
   Application.CreateForm( TfrmConfFile, frm );
   frm.StartEdit( EditFileType, EditFileName, EditFileTitle, '' );
end;

procedure EditLocalText( const EditFileTitle: String;
                         const EditFileText : String );
var  frm: TfrmConfFile;
begin
   Application.CreateForm( TfrmConfFile, frm );
   frm.StartEdit( eftLocalText, '', EditFileTitle, EditFileText );
end;

procedure TfrmConfFile.StartEdit( const AEditFileType : TEditFileTypes;
                                  const AEditFileName : String;
                                  const AEditFileTitle: String;
                                  const AEditFileText : String );
begin
   EditFileType  := AEditFileType;
   EditFileName  := AEditFileName;
   EditFileTitle := AEditFileTitle;
   EditFileText  := AEditFileText;

   if EditFileType = eftLocalText then begin
      acDownload.Enabled := False;
      acUpload  .Enabled := False;
   end;

   acRunScript.Enabled := (EditFileType = eftScript);

   LoadWindowState( Self, 'Edit ' + inttostr( ord(EditFileType) ) );
   ShowFilename;
   Visible := True;

   acRefresh.Execute;
end;

procedure TfrmConfFile.FormCreate(Sender: TObject);
begin
   CurrFileName := '';
   CurrFindText := '';
   rtfEdit.Font.Assign( FontEditor );
end;

procedure TfrmConfFile.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
begin
   CanClose := CanContinue( cpPreferUpload );
end;

procedure TfrmConfFile.FormClose(Sender: TObject; var Action: TCloseAction);
begin
   SaveWindowState( Self, 'Edit ' + inttostr( ord(EditFileType) ) );
   Action := caFree;
end;

procedure TfrmConfFile.ShowFilename;
var  What: String;
begin
   if EditFileType = eftLocalText then What := '' else What := 'File: ';

   if CurrFilename = '' then begin
      Caption := What + EditFileTitle;
   end else begin
      Caption := What + EditFileTitle + '  [' + CurrFilename + ']';
   end;
end;

procedure TfrmConfFile.ShowStatus;
var  cp: TPoint;
     s : String;
begin
   cp := rtfEdit.CaretPos;
   Statusbar1.Panels[0].Text := 'L: ' + inttostr( cp.y+1 ) + ', C: ' + inttostr( cp.x+1 );

   if rtfEdit.Modified then s:='Changed' else s:='';
   Statusbar1.Panels[1].Text := s;

   Statusbar1.Panels[2].Text := ' ';
end;

function TfrmConfFile.CanContinue( Prefer: TContinuePreference ): Boolean;
var  r: Word;
begin
   Result := True;
   if not rtfEdit.Modified then exit;
   if EditFileType = eftLocalText then exit;

   Result := False;

   if Prefer = cpPreferUpload then begin
      r := HMessageDlg( 'Text has been changed!' + #13#10#13#10
                      + 'Upload changes?', mtWarning, mbYesNoCancel );
      if r=mrYes   then begin
         acUpload.Execute;
         Result := not rtfEdit.Modified;
      end;
      if r=mrNo    then Result := True;
      if r=mrAbort then exit;
   end;

   if Prefer = cpPreferIgnore then begin
      r := HMessageDlg( 'Text has been changed!' + #13#10#13#10
                      + 'Ignore changes?', mtWarning, mbYesNoCancel );
      if r=mrYes   then begin
         rtfEdit.Modified  := False;
         Result := True;
      end;
      if r=mrNo    then Result := CanContinue( cpPreferUpload );
      if r=mrAbort then exit;
   end;
end;

procedure TfrmConfFile.rtfEditKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
var  y, y1, y2, st, sl: Integer;
     s: String;
begin
   ShowStatus;

   if ( (Shift = [ssCtrl] ) and (Key = Ord('V') ) ) or
      ( (Shift = [ssShift]) and (Key = VK_INSERT) ) then begin
      Key := 0;
      if Clipboard.HasFormat(CF_TEXT) then rtfEdit.SelText := Clipboard.AsText;
   end;

   if (Shift=[ssAlt]) and (Key in [VK_LEFT,VK_RIGHT,VK_UP,VK_DOWN]) then begin
      rtfEdit.Lines.BeginUpdate;
      st := rtfEdit.SelStart;
      sl := rtfEdit.SelLength;
      y1 := rtfPosToLine( rtfEdit, st    );
      y2 := rtfPosToLine( rtfEdit, st+sl );

      try
         s := rtfEdit.Text;
         if (y2>y1) and (s[st+sl] in [#10,#13]) then dec(y2);

         if Key=VK_LEFT then begin
            for y:=y1 to y2 do begin
               s := rtfEdit.Lines[y];
               if (s<>'') and (s[1] in [#9,#32]) then begin
                  rtfEdit.Lines[y] := copy( s, 2, length(s)-1 );
                  dec( sl );
               end;
            end;
         end;
         if Key=VK_RIGHT then begin
            for y:=y1 to y2 do begin
               rtfEdit.Lines[y] := ' ' + rtfEdit.Lines[y];
               inc( sl );
            end;
         end;
         if (Key=VK_UP) and (y1>0) then begin
            s := rtfEdit.Lines[y1-1];
            for y:=y1 to y2 do begin
               rtfEdit.Lines[y-1] := rtfEdit.Lines[y];
            end;
            rtfEdit.Lines[y2] := s;
            dec( st, length(s)+2 );
         end;
         if (Key=VK_DOWN) and (y2+1<rtfEdit.Lines.Count) then begin
            s := rtfEdit.Lines[y2+1];
            for y:=y2 downto y1 do begin
               rtfEdit.Lines[y+1] := rtfEdit.Lines[y];
            end;
            rtfEdit.Lines[y1] := s;
            inc( st, length(s)+2 );
         end;
      except
      end;
      rtfEdit.SelStart  := st;
      rtfEdit.SelLength := sl;
      rtfEdit.Lines.EndUpdate;
   end;
end;

procedure TfrmConfFile.rtfEditKeyUp(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
   ShowStatus;
end;

procedure TfrmConfFile.rtfEditMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
   ShowStatus;
end;

procedure TfrmConfFile.EnableActions;
begin
   acEditUndo.Enabled     := rtfEdit.CanUndo;
   acEditCut.Enabled      := ( rtfEdit.SelLength > 0 );
   acEditCopy.Enabled     := ( rtfEdit.SelLength > 0 );
   acEditDelete.Enabled   := ( rtfEdit.SelLength > 0 );
   acEditPaste.Enabled    := Clipboard.HasFormat( CF_TEXT );
   acEditFindNext.Enabled := ( CurrFindText <> '' );
end;

procedure TfrmConfFile.FindNextPosition( IsFindNew: Boolean );
var  t, s, h: String;
     p, f   : Integer;
begin
   t := LowerCase( rtfEdit.Text );
   s := Lowercase( CurrFindText );
   p := rtfEdit.SelStart + 1;
   if not IsFindNew then inc( p );

   h := t;
   System.Delete( h, 1, p-1 );
   f := Pos( s, h );
   if f<>0 then begin
      rtfEdit.SelStart  := p + f - 2;
      rtfEdit.SelLength := Length( CurrFindText );
      exit;
   end;

   f := Pos( s, t );
   if f<>0 then begin
      rtfEdit.SelStart  := f - 1;
      rtfEdit.SelLength := Length( CurrFindText );
      exit;
   end;

   rtfEdit.SelLength := 0;
end;

procedure TfrmConfFile.mnuEditClick(Sender: TObject);
begin
   EnableActions;
end;

procedure TfrmConfFile.popupEditPopup(Sender: TObject);
begin
   EnableActions;
end;

function TfrmConfFile.LMPrepare( out GetId: Integer; out GetStr: String;
                                 out SetId: Integer; out SetStr: String ): Boolean;
begin
   Result := True;

   GetStr := DQuoteStr( EditFileName );
   SetStr := DQuoteStr( EditFileName ) + CRLF + rtfEdit.Lines.Text;

   case EditFileType of

      eftScores, eftMailFilt, eftIPAccess, eftKillsLog,
      eftRCProfiles, eftHscActions: begin
         GetId := LMREQ_FILE_GET;
         SetId := LMREQ_FILE_SET;
      end;

      eftScript, eftModule: begin
         GetId  := LMREQ_SCRIPT_GET;
         SetId  := LMREQ_SCRIPT_PUT;
      end;

      else Result := False;

   end;
end;

procedure TfrmConfFile.acRunScriptExecute(Sender: TObject);
var  scriptPars, scriptFile, paramTitle, s, metaCommand: String;
     reply: TLiveMsg;
     scriptLines, metaParts: TStringList;
     paramCount, i, k: Integer;
     paramPrompt, paramDefault: array of string;
begin
   if EditFileType <> eftScript then exit;

   if not acUpload.Execute then exit;

   scriptFile   := EditFileName;
   scriptPars   := '';
   scriptLines  := TStringList.Create;
   metaParts    := TStringList.Create;
   paramTitle   := 'Start Script';
   paramCount   := 0;
   paramPrompt  := nil;
   paramDefault := nil;

   try

      // preload script text
      try
         reply := LiveConnector.RCLiveRequest(
            TLiveMsg.Create( LMREQ_SCRIPT_GET, DQuoteStr( scriptFile ) )
         );
         if Assigned( reply ) then try
            if ( reply.MsgType = LMREP_OK ) then begin
               scriptLines.Text := reply.MsgData;
            end;
         finally reply.Free end;
      except
         on E:Exception do begin
            HMessageDlg( 'Script ' + scriptFile + ' could not be downloaded!'
               + #13#10#13#10 + 'Error: ' + E.Message, mtError, [mbOK] );
         end;
      end;

      // look for any '#!meta' lines
      for i := 0 to scriptLines.Count - 1 do begin
         if Pos( '#!meta', LowerCase( scriptLines[i] ) ) = 1  then begin

            s := TrimWhSpace( copy( scriptLines[i], 7, MaxInt ) );
            ArgsWhSpaceDQuoted( s, metaParts, 9 );
            metaCommand := LowerCase( metaParts[0] );

            if metaCommand = 'paramcount' then begin
               // #!meta paramcount <count>
               paramCount := strtointdef( metaParts[1], 0 );
               SetLength( ParamPrompt,  paramCount+1 );
               SetLength( ParamDefault, paramCount+1 );

            end else if metaCommand = 'paramtitle' then begin
               // #!meta paramtitle <text>
               paramTitle := metaParts[1];

            end else if metaCommand = 'paramdefault' then begin
               // paramdefault <number> <prompt>
               k := strtointdef( metaParts[1], 0 );
               if (k >= 1) and (k <= paramCount) then begin
                  paramDefault[k] := metaParts[2];
               end else begin
                  // GuiLog( '[META] Index out of range: ' + scriptLines[i] );
                  paramCount := 0;
                  break;
               end;

            end else if metaCommand = 'paramprompt' then begin
               // paramprompt <number> <prompt>
               k := strtointdef( metaParts[1], 0 );
               if (k >= 1) and (k <= paramCount) then begin
                  paramPrompt[k] := metaParts[2];
               end else begin
                  // GuiLog( '[META] Index out of range: ' + scriptLines[i] );
                  paramCount := 0;
                  break;
               end;

            end;

         end;
      end;

      // get and prepare script parameters
      if paramCount > 0 then begin

         // get meta parameters
         for i := 1 to paramCount do begin
            s := paramDefault[i];
            if paramPrompt[i] = '' then paramPrompt[i] := 'Parameter:';
            if not InputDlgStr( paramTitle, paramPrompt[i], s, 0 ) then exit;
            if i > 1 then scriptPars := scriptPars + #9;
            scriptPars := scriptPars + s;
         end;

      (*
      end else if (Sender = acScriptStartPar) and (paramCount = 0) then begin

         // get space separated parameters
         if not InputDlgStr( 'Start Script', 'Parameters:', scriptPars, 0 ) then exit;
         SL := TStringList.Create;
         try
            ArgsWhSpaceDQuoted( scriptPars, SL );
            scriptPars := StringReplace( SL.Text, #13#10, #9, [rfReplaceAll] );
            if (length(scriptPars) > 0) and (scriptPars[length(scriptPars)] = #9) then
               System.Delete( scriptPars, length(scriptPars), 1 );
         finally SL.Free end;
      *)

      end;

   finally
      paramPrompt  := nil;
      paramDefault := nil;
      metaParts.Free;
      scriptLines.Free;
   end;


   try
      // start script
      LiveConnector.RCLiveRequestOK(
         TLiveMsg.Create(
            LMREQ_SCRIPT_START, DQuoteStr( scriptFile ) + CRLF + scriptPars
         )
      );

   except
      on E:Exception do begin
         HMessageDlg( 'Script ' + EditFileName + ' could not be started!'
                    + #13#10#13#10 + 'Error: ' + E.Message, mtError, [mbOK] );
      end;
   end;

   ShowStatus;
end;

procedure TfrmConfFile.acRefreshExecute(Sender: TObject);
begin
   if EditFileType = eftLocalText then begin

      rtfEdit.Lines.Text := EditFileText;
      rtfEdit.ClearUndo;
      rtfEdit.Modified  := False;
      rtfEdit.SelStart  := 0;
      rtfEdit.SelLength := 0;
      SendMessage( rtfEdit.Handle, EM_SCROLLCARET, 0, 0 );
      ShowStatus;

   end else begin

      acDownloadExecute(Sender);

   end;
end;

procedure TfrmConfFile.acDownloadExecute(Sender: TObject);
var  GetId,  SetId : Integer;
     GetStr, SetStr: String;
     Reply: TLiveMsg;
begin
   if EditFileType = eftLocalText then exit;

   if not CanContinue( cpPreferIgnore ) then exit;

   if not LMPrepare( GetId, GetStr, SetId, SetStr ) then exit;

   try
      Reply := LiveConnector.RCLiveRequest(
                  TLiveMsg.Create( GetId, GetStr )
               );
      if Assigned( Reply ) then try
         if ( Reply.MsgType = LMREP_OK ) then begin
            rtfEdit.Lines.Text := Reply.MsgData;
            rtfEdit.ClearUndo;
            rtfEdit.Modified  := False;
            rtfEdit.SelStart  := 0;
            rtfEdit.SelLength := 0;
            SendMessage( rtfEdit.Handle, EM_SCROLLCARET, 0, 0 );
         end;
      finally Reply.Free end;

   except
      on E:Exception do begin
         HMessageDlg( 'File ' + EditFileName + ' could not be downloaded!'
                    + #13#10#13#10 + 'Error: ' + E.Message, mtError, [mbOK] );
      end;
   end;

   ShowStatus;
end;

procedure TfrmConfFile.acUploadExecute(Sender: TObject);
var  GetId,  SetId : Integer;
     GetStr, SetStr: String;
     Reply: TLiveMsg;
begin
   if EditFileType = eftLocalText then exit;

   if not LMPrepare( GetId, GetStr, SetId, SetStr ) then exit;

   try
      Reply := LiveConnector.RCLiveRequest(
                  TLiveMsg.Create( SetId, SetStr )
               );
      if Assigned( Reply ) then try
         if ( Reply.MsgType = LMREP_OK ) then begin
            rtfEdit.ClearUndo;
            rtfEdit.Modified := False;
         end;
      finally Reply.Free end;

   except
      on E:Exception do begin
         HMessageDlg( 'File ' + EditFileName + ' could not be uploaded!'
                    + #13#10#13#10 + 'Error: ' + E.Message, mtError, [mbOK] );
      end;
   end;

   ShowStatus;
end;

procedure TfrmConfFile.PrepareFileDialog( Dialog: TOpenDialog );
begin
   with Dialog do begin

      Filter := 'Hamster Setup Files (*.hst)|*.hst'   + '|'  {1}
              + 'Hamster Log Files (*.log)|*.log'     + '|'  {2}
              + 'Hamster Script Files (*.hsc)|*.hsc'  + '|'  {3}
              + 'Hamster Modules Files (*.hsm)|*.hsm' + '|'  {4}
              + 'Text Files (*.txt)|*.txt'            + '|'  {5}
              + 'All Files (*.*)|*.*';                       {6}

      case EditFileType of
         eftScores, eftMailFilt, eftIPAccess, eftRCProfiles, eftHscActions:
            begin FilterIndex := 1; DefaultExt := 'hst' end;
         eftKillsLog:
            begin FilterIndex := 2; DefaultExt := 'log' end;
         eftScript:
            begin FilterIndex := 3; DefaultExt := 'hsc' end;
         eftModule:
            begin FilterIndex := 4; DefaultExt := 'hsm' end;
         else
            begin FilterIndex := 6; DefaultExt := ''    end;
      end;

   end;
end;

procedure TfrmConfFile.acFileOpenExecute(Sender: TObject);
begin
   if not CanContinue( cpPreferUpload ) then exit;

   PrepareFileDialog( OpenDialog1 );
   if not OpenDialog1.Execute then exit;
   CurrFilename := OpenDialog1.FileName;

   ShowFilename;
   try
      rtfEdit.Lines.LoadFromFile( CurrFilename );
      rtfEdit.ClearUndo;
      rtfEdit.Modified  := False;
      rtfEdit.SelStart  := 0;
      rtfEdit.SelLength := 0;
      SendMessage( rtfEdit.Handle, EM_SCROLLCARET, 0, 0 );
   except
      on E:Exception do begin
         HMessageDlg( 'File ' + CurrFilename + ' could not be loaded!'
                    + #13#10#13#10 + 'Error: ' + E.Message, mtError, [mbOK] );
         CurrFilename := '';
         ShowFilename;
      end;
   end;
   ShowStatus;
end;

procedure TfrmConfFile.acFileSaveExecute(Sender: TObject);
begin
   PrepareFileDialog( SaveDialog1 );
   SaveDialog1.FileName := CurrFilename;
   if not SaveDialog1.Execute then exit; 
   CurrFilename := SaveDialog1.FileName;

   ShowFilename;
   try
      rtfEdit.Lines.SaveToFile( CurrFilename );
   except
      on E:Exception do begin
         HMessageDlg( 'File ' + CurrFilename + ' could not be saved!'
                    + #13#10#13#10 + 'Error: ' + E.Message, mtError, [mbOK] );
         CurrFilename := '';
         ShowFilename;
      end;
   end;
   ShowStatus;
end;

procedure TfrmConfFile.acFilePrintExecute(Sender: TObject);
begin
   rtfEdit.Print( EditFileTitle );
end;

procedure TfrmConfFile.acFileExitExecute(Sender: TObject);
begin
   if not CanContinue( cpPreferUpload ) then exit;
   Close;
end;

procedure TfrmConfFile.acEditUndoExecute(Sender: TObject);
begin
   rtfEdit.Undo;
end;

procedure TfrmConfFile.acEditCutExecute(Sender: TObject);
begin
   rtfEdit.CutToClipboard;
end;

procedure TfrmConfFile.acEditCopyExecute(Sender: TObject);
begin
   rtfEdit.CopyToClipboard;
end;

procedure TfrmConfFile.acEditPasteExecute(Sender: TObject);
begin
   rtfEdit.PasteFromClipboard;
end;

procedure TfrmConfFile.acEditDeleteExecute(Sender: TObject);
begin
   rtfEdit.ClearSelection;
end;

procedure TfrmConfFile.acEditSelectAllExecute(Sender: TObject);
begin
   rtfEdit.SelectAll;
end;

procedure TfrmConfFile.acEditFindExecute(Sender: TObject);
var  s, t: String;
     p, i: Integer;
begin
   // get word at cursor
   s := '';
   t := rtfEdit.Text;
   p := rtfEdit.SelStart + 1;
   if copy( t, p, 1 ) = ' ' then dec( p );
   i := p - 1;
   while (i>0) and (t[i] in ['a'..'z','A'..'Z','0'..'9','_']) do begin
      s := t[i] + s;
      dec( i );
   end;
   i := p;
   while (i<=length(t)) and (t[i] in ['a'..'z','A'..'Z','0'..'9','_']) do begin
      s := s + t[i];
      inc( i );
   end;
   CurrFindText := s;

   if not InputDlgStr( 'Find', 'Text:', CurrFindText, 0 ) then exit;
   if CurrFindText='' then exit;
   FindNextPosition( True );
end;

procedure TfrmConfFile.acEditFindNextExecute(Sender: TObject);
begin
   FindNextPosition( False );
end;

procedure TfrmConfFile.acFontExecute(Sender: TObject);
begin
   FontDialog1.Font.Assign( FontEditor );
   if not FontDialog1.Execute then exit;
   FontEditor.Assign( FontDialog1.Font );
   rtfEdit.Font.Assign( FontEditor );
   SaveLocalSettings;
end;

end.
