
#====================================================================
# NewsWatcher: Watch for own articles and followups.
#
# Usage: a) Rename script so it will not be overwritten next time
#           you update Hamster.
#        b) Check/set values in "Settings" part below.
#        c) Run script manually or with RunScript("NewsWatcher.hsc")
#           after pulling news.
#====================================================================

#!load hamrequests.hsm

#--------------------------------------------------------------------
# Settings
#--------------------------------------------------------------------

# Name of an internal newsgroup, where a copy of each matching
# article should be stored. If no name is given, matches are just
# reported as warnings.
# varset( $ResultGroup, "" ) # no copy, just report
varset( $ResultGroup, "internal.misc" )

# Regular expression for the newsgroup names to watch (".*"=all).
varset( $REGroupSelect, ".*" )

# Regular expression for the newsgroup names to ignore. It has a
# higher priority than $REGroupsSelect above.
varset( $REGroupIgnore, "^(internal|control)\." )

# FQDN (domain part of Message-IDs) to watch for in "Message-ID:"
# and "References:" headers.
varset( $FQDN, HamGetFQDN ) # get FQDN from Hamster


#====================================================================

# prepare regular expression for FQDN
Assert( len($FQDN) > 0, "No FQDN to watch for given!" )
varset( $REFQDN, "\@" + Replace( $FQDN, ".", "\.", True ) + "\>" )

# check if a valid group was given
Assert( ($ResultGroup="") || (HamGroupIndex($ResultGroup)>=0), _
        "Group " + $ResultGroup + " does not exist!" )

# prepare filename to store group markers in
varset( $IniFile, Replace( ParamStr(0), ".hsc", ".ini" ) )

# start scanning of groups
print( "NewsWatcher: Scanning groups ..." )

varset( $Count, $Found, 0 )
varset( $GrpNam, $ArtTxt, "" )
varset( $Msg, MsgAlloc )
varset( $GrpIdx, HamGroupCount - 1 )

while( $GrpIdx >= 0 )

   $GrpNam = HamGroupName( $GrpIdx )

   if( RE_Match( $GrpNam, $REGroupSelect ) )
      if( !RE_Match( $GrpNam, $REGroupIgnore ) )
         # print( "Scanning ", $GrpNam, " ..." )
         ScanGroup( $GrpNam )
      endif
   endif

   dec( $GrpIdx )

endwhile

MsgFree( $Msg )

print( "NewsWatcher: ", $Count, " tested, ",  $Found, " found." )

quit

#====================================================================

sub ScanGroup( $GrpNam )

   varset( $GrpHdl, HamGroupOpen( $GrpNam ) )
   if( $GrpHdl < 0 )
      warning( "Could not open group ", $GrpNam, "!" )
      return
   endif
   
   varset( $ArtMin,  HamArtNoMin( $GrpHdl ) )
   varset( $ArtMax,  HamArtNoMax( $GrpHdl ) )
   varset( $ArtLast, int( IniRead($IniFile,"groups",$GrpNam,"0") ) )

   if( $ArtLast = 0 )
      $ArtLast = $ArtMax - 500     # max. 500 for new groups
   else
      if( $ArtMax < $ArtLast )
         $ArtLast = $ArtMax - 500  # fix changed number range
      endif
   endif
   if( $ArtMin <= $ArtLast )
      $ArtMin = $ArtLast + 1       # skip already tested ones
   endif

   while( $ArtMin <= $ArtMax )

      $ArtTxt = HamArtText( $GrpHdl, $ArtMin )

      if( len($ArtTxt) > 0 )
         $ArtLast = $ArtMin
         MsgSetText( $Msg, $ArtTxt )
         ScanArticle
      endif

      inc( $ArtMin )

   endwhile

   HamGroupClose( $GrpHdl )
   IniWrite( $IniFile, "groups", $GrpNam, $ArtLast )

endsub

#--------------------------------------------------------------------

sub ScanArticle

   var( $s, $r, $l, $i )

   inc( $Count )
   if( $Count % 100 = 0 )
      print( $Found, " / ", $Count )
   endif

   # check Message-ID
   $s = MsgGetHeader( $Msg, "Message-ID:" )
   if( RE_Match( $s, $REFQDN ) )
      ReportMatch( "Own article in " + $GrpNam )
      return
   endif

   # check References
   $s = MsgGetHeader( $Msg, "References:" )
   if( RE_Match( $s, $REFQDN ) )
      # split References to get followup level
      $l = ListAlloc
      do
         RE_Split( $s, "\s+", $r, $s )
         break( ($r = "") && ($s = "") )
         ListInsert( $l, 0, $r )
      loop
      for( $i, 0, ListCount($l) - 1 )
         if( RE_Match( ListGet($l,$i), $REFQDN ) )
            ReportMatch( _
               "Followup" + iif( $i=0, "", " (+" + $i + ")" ) _
               + " in " + $GrpNam _
               + " by " + Decode( 2, MsgGetHeader( $Msg, "From:" ) ) )
            break
         endif
      endfor
      ListFree( $l )
      return
   endif

endsub

#--------------------------------------------------------------------

sub ReportMatch( $ReportAs )

   inc( $Found )

   Warning( $ReportAs )

   if( len($ResultGroup) > 0 )
      HamArtImport( $ArtTxt, $ResultGroup, True )
   endif

endsub

#====================================================================
